/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_figure.cpp 
 * @brief Implementation of the model element class.
 * 
 */

#ifdef _WINDOWS
  #include <windows.h>
#else
#endif // ifdef _WINDOWS
#include <gl/gl.h>

#include "tree.h" // libxml
#include "myx_gc_figure.h"
#include "myx_gc_model.h"
#include "myx_gc_layer.h"
#include "myx_gc_canvas.h"

//----------------- CFigure --------------------------------------------------------------------------------------------

CFigure::CFigure(CGCModel* Owner, GLuint TemplateList)
{
  FDestroying = false;
  FModel = Owner;
  FTemplateList = TemplateList;
  FList = -1;
  FParent = NULL;
  FType = NULL;

  // Initialize with useful values.
  FRotation[0] = 0;
  FRotation[1] = 0;
  FRotation[2] = 0;
  FRotation[3] = 1;

  FScaling[0] = 1;
  FScaling[1] = 1;
  FScaling[2] = 1;

  FTranslation[0] = 0;
  FTranslation[1] = 0;
  FTranslation[2] = 0;

  FDirty = true;
}

//----------------------------------------------------------------------------------------------------------------------

CFigure::~CFigure(void)
{
  FDestroying = true;
  for (CInstanceIterator Iterator = FNotificationList.begin(); Iterator != FNotificationList.end(); Iterator++)
    (*Iterator)->FreeNotification(this);
  if ((FParent != NULL) && !FParent->FDestroying)
    FParent->RemoveChild(this);
  Clear();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Adds a new figure instance to the notification list to tell them when this figure is about to be destroyed.
 *
 * @param Instance The instance to add.
 */
void CFigure::AddFreeNotification(CFigureInstance* Instance)
{
  FNotificationList.insert(Instance);
}

//----------------------------------------------------------------------------------------------------------------------

// Marks the display list for this figure as invalid, hence it will be recreated next time ValidateDisplayList is called.
// If a list already exists then it is freed.
// If the figure is connected to a parent figure then this parent is invalidated as well.
void CFigure::MakeDirty(void)
{
  if (!FDirty)
  {
    glDeleteLists(FList, 1);
    FList = -1;
    FDirty = true;
    if (FParent != NULL)
      FParent->MakeDirty();

    // Notfiy all instances of this figure.
    for (CInstanceIterator Iterator = FNotificationList.begin(); Iterator != FNotificationList.end(); Iterator++)
      (*Iterator)->MakeDirty();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes the given figure instance from the notification list.
 *
 * @param Instance The instance to remove.
 */
void CFigure::RemoveFreeNotification(CFigureInstance* Instance)
{
  for (CInstanceIterator Iterator = FNotificationList.begin(); Iterator != FNotificationList.end(); Iterator++)
  {
    if (*Iterator == Instance)
    {
      FNotificationList.erase(Iterator);
      break;
    };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates the display list of this figure (and all child figures) if necessary.
 */
void CFigure::ValidateDisplayList(void)
{
  if (FDirty)
  {
    FDirty = false;

    for (CFigureListIterator Iterator = FChildren.begin(); Iterator != FChildren.end(); Iterator++)
      (*Iterator)->ValidateDisplayList();

    // Create the display list. Make sure to not change the current transformation matrix.
    FList = glGenLists(1);
    glNewList(FList, GL_COMPILE);
    glPushMatrix();

    glTranslated(FTranslation[0], FTranslation[1], FTranslation[2]);
    // Index 0 contains the angle, while the other three coordinates form the axis to rotate around.
    glRotated(FRotation[0], FRotation[1], FRotation[2], FRotation[3]);
    glScaled(FScaling[0], FScaling[1], FScaling[2]);

    if (glIsList(FTemplateList))
      glCallList(FTemplateList);

    // At this point child figures have valid display lists.
    for (CFigureListIterator Iterator = FChildren.begin(); Iterator != FChildren.end(); Iterator++)
    {
      //glTranslated(0, 0, 0.01);
      (*Iterator)->Render();
    };

    glPopMatrix();
    glEndList();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Adds the given figure to the end of the child list. If Child belongs to another figure currently it is removed from 
 * the other's child list first.
 *
 * @param Child The child figure to add.
 */
void CFigure::AddChild(CFigure* Child)
{
  if (Child->FParent != NULL)
    Child->FParent->RemoveChild(Child);
  FChildren.push_back(Child);
  Child->FParent = this;
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Removes all child figures from this figure. 
 */
void CFigure::Clear(void)
{
  for (CFigureListIterator Iterator = FChildren.begin(); Iterator != FChildren.end(); Iterator++)
  {
    CFigure* Child = *Iterator;
    Child->FParent = NULL;
    delete Child;
  };
  FChildren.clear();
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Inserts a child figure at the given position in the current child list. If Child belongs to another figure currently 
 * it is removed from the other's child list first.
 *
 * @param Index The index at which to insert the new child figure.
 * @param Child The child figure to insert.
 */
void CFigure::InsertChild(int Index, CFigure* Child)
{
  if (Child->FParent != NULL)
    Child->FParent->RemoveChild(Child);

  if (Index < 0)
    Index = 0;
  if ((size_t) Index > FChildren.size() - 1)
    Index = int(FChildren.size() - 1);
  CFigureListIterator Iterator;
  for (Iterator = FChildren.begin(); Iterator != FChildren.end(), Index >= 0; Iterator++, Index--);
  FChildren.insert(Iterator, Child);
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes the given figure form the child list if it is currently there. No error is raised if the child does not belong 
 * to this figure.
 *
 * @param Child The child figure to be removed.
 */
void CFigure::RemoveChild(CFigure* Child)
{
  for (CFigureListIterator Iterator = FChildren.begin(); Iterator != FChildren.end(); Iterator++)
    if (*Iterator == Child)
    {
      FChildren.erase(Iterator);
      Child->FParent = NULL;
      MakeDirty();
      break;
    };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Checks the validity of the figure display list and executes it.
 */
void CFigure::Render(void)
{
  if (FDirty)
    ValidateDisplayList();

  glCallList(FList);
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Turns the figure around the given axis by the angle Angle (in radians). This version of Rotate uses a single float 
 * values in the parameter list.
 *
 * @param Angle The angle in radians to turn the figure.
 * @param Rx The x part of the rotation axis.
 * @param Ry The y part of the rotation axis.
 * @param Rz The z part of the rotation axis.
 * @note Currently there is no accumulative version of Rotate available (requires a quaternion lib, which we don't have yet).
 */
void CFigure::Rotate(double Angle, double Rx, double Ry, double Rz)
{
  FRotation[0] = Angle;
  FRotation[1] = Rx;
  FRotation[2] = Ry;
  FRotation[3] = Rz;
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Turns the figure around the given axis by the angle Angle (in radians). This version of Rotate uses a vector for the
 * rotation axis in the parameter list.
 *
 * @param Angle The angle in radians to turn the figure.
 * @param Axis The axis around which the figure is to be rotated.
 * @note Currently there is no accumulative version of Rotate available (requires a quaternion lib, which we don't have yet).
 */
void CFigure::RotateV(double Angle, const double Axis[3])
{
  FRotation[0] = Angle;
  FRotation[1] = Axis[0];
  FRotation[2] = Axis[1];
  FRotation[3] = Axis[2];
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Scales the figure by the amount given in Factor. If Accumulative is true then the new scale factors are multiplied
 * with the existing values. This version of Scale uses single float values as parameters.
 *
 * @param Sx The scale factor in x direction.
 * @param Sy The scale factor in y direction.
 * @param Sz The scale factor in z direction.
 * @param Accumulative If true then the given values are added to any existing values otherwiese they are used as given.
 */
void CFigure::Scale(double Sx, double Sy, double Sz, bool Accumulative)
{
  if (Accumulative)
  {
    FScaling[0] += Sx;
    FScaling[1] += Sy;
    FScaling[2] += Sz;
  }
  else
  {
    FScaling[0] = Sx;
    FScaling[1] = Sy;
    FScaling[2] = Sz;
  };
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Scales the figure by the amount given in Factor. If Accumulative is true then the new scale factors are multiplied
 * with the existing values. This version of Scale uses an array of values in the parameter list.
 *
 * @param Factor An array containing the three scale values for x, y and z.
 * @param Accumulative If true then the given values are added to any existing values otherwiese they are used as given.
 */
void CFigure::ScaleV(const double Factor[3], bool Accumulative)
{
  if (Accumulative)
  {
    FScaling[0] += Factor[0];
    FScaling[1] += Factor[1];
    FScaling[2] += Factor[2];
  }
  else
  {
    FScaling[0] = Factor[0];
    FScaling[1] = Factor[1];
    FScaling[2] = Factor[2];
  };
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Moves the figure by the amount given in Factor. If Accumulative is true then the new translation factors are 
 * multiplied with the existing values. This version of Translate uses an array for the values in the parameter list.
 *
 * @param Tx Scale factor for the x axis.
 * @param Ty Scale factor for the y axis.
 * @param Tz Scale factor for the z axis.
 * @param Accumulative If true then the given values are added to any existing values otherwiese they are used as given.
 */
void CFigure::Translate(double Tx, double Ty, double Tz, bool Accumulative)
{
  if (Accumulative)
  {
    FTranslation[0] += Tx;
    FTranslation[1] += Ty;
    FTranslation[2] += Tz;
  }
  else
  {
    FTranslation[0] = Tx;
    FTranslation[1] = Ty;
    FTranslation[2] = Tz;
  };
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Moves the figure by the amount given in Factor. If Accumulative is true then the new translation factors are 
 * multiplied with the existing values. This version of Translate uses an array for the values in the parameter list.
 *
 * @param Factor An array of translation values, for each axis one.
 * @param Accumulative If true then the given values are added to any existing values otherwiese they are used as given.
 */
void CFigure::TranslateV(const double Factor[3], bool Accumulative)
{
  if (Accumulative)
  {
    FTranslation[0] += Factor[0];
    FTranslation[1] += Factor[1];
    FTranslation[2] += Factor[2];
  }
  else
  {
    FTranslation[0] = Factor[0];
    FTranslation[1] = Factor[1];
    FTranslation[2] = Factor[2];
  };
  MakeDirty();
}

//----------------- CFigureInstance ------------------------------------------------------------------------------------

CFigureInstance::CFigureInstance(CLayer* Owner, CFigure* Figure)
{
  FLayer = NULL;
  FFigure = Figure;
  FList = -1;
  FSelected = false;
  FValidBounds = false;

  FRotation[0] = 0;
  FRotation[1] = 0;
  FRotation[2] = 0;
  FRotation[3] = 1;

  FScaling[0] = 1;
  FScaling[1] = 1;
  FScaling[2] = 1;

  FTranslation[0] = 0;
  FTranslation[1] = 0;
  FTranslation[2] = 0;

  FCurrentBounds.Left = 0;
  FCurrentBounds.Top = 0;
  FCurrentBounds.Right = 0;
  FCurrentBounds.Bottom = 0;

  FDirty = true;
  if (Owner != NULL)
    Owner->AddInstance(this);
  if (Figure != NULL)
    Figure->AddFreeNotification(this);
}

//----------------------------------------------------------------------------------------------------------------------

CFigureInstance::~CFigureInstance(void)
{
  if (FLayer != NULL)
  {
    if (FSelected) 
      FLayer->GetCanvas()->RemoveFromSelection(this);
    if (!FLayer->IsUpdating())
      FLayer->RemoveInstance(this);
  };

  if ((FFigure != NULL) && (!FFigure->FDestroying))
    FFigure->RemoveFreeNotification(this);
  MakeDirty(); // Free display list.
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Called by a class (usually CFigure) with which we registered us as notification sink and which is
 * about to be destroyed.
 */
void CFigureInstance::FreeNotification(CFigure* Figure)
{
  if (Figure == FFigure)
    delete this;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Marks the display list for this figure instance as invalid, hence it will be recreated next time ValidateDisplayList 
 * is called. If a list already exists then it is freed. If the figure is connected to a parent figure then this parent 
 * is invalidated as well.
 */
void CFigureInstance::MakeDirty(void)
{
  if (!FDirty)
  {
    glDeleteLists(FList, 1);
    FList = -1;
    FDirty = true;
    FValidBounds = false;
    if (FLayer != NULL)
    {
      if (FSelected) 
        FLayer->GetCanvas()->InvalidateSelectionBounds(this);
      if (!FLayer->IsUpdating())
        FLayer->MakeDirty();
    };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/*
 * Renders the content fo this instance without creating a display list first. This is used to provide feedback data
 * for determination of the figures bounding box measured in its layer's coordinates.
 */
void CFigureInstance::RenderFeedback(void)
{
  glTranslated(FTranslation[0], FTranslation[1], FTranslation[2]);
  // Index 0 contains the angle, while the other three coordinates form the axis to rotate around.
  glRotated(FRotation[0], FRotation[1], FRotation[2], FRotation[3]);
  glScaled(FScaling[0], FScaling[1], FScaling[2]);

  if (FFigure != NULL)
    FFigure->Render();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates the display list of this figure instance and its associated figure if necessary.
 */
void CFigureInstance::ValidateDisplayList(void)
{
  if (FDirty)
  {
    FDirty = false;

    if (FFigure != NULL)
      FFigure->ValidateDisplayList();

    // Create the display list. Make sure not to change the current transformation matrix.
    FList = glGenLists(1);
    glNewList(FList, GL_COMPILE);
    glPushMatrix();

    glTranslated(FTranslation[0], FTranslation[1], FTranslation[2]);
    // Index 0 contains the angle, while the other three coordinates form the axis to rotate around.
    glRotated(FRotation[0], FRotation[1], FRotation[2], FRotation[3]);
    glScaled(FScaling[0], FScaling[1], FScaling[2]);

    if (FFigure != NULL)
      FFigure->Render();

    glPopMatrix();
    glEndList();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the bounds of this figure instance according to the figure it represents and all transformations.
 *
 * @return The smallest rectangle enclosing the whole content of the figure instance in the coordinate system of
 *         the layer this instance belongs to.
 */
void CFigureInstance::GetBounds(TBounds* Bounds)
{
  if (!FValidBounds)
  {
    // (Re)compute the current bounding box if it is not valid currently.
    FValidBounds = true;
    GLsizei BufferSize = 1000;
    GLfloat* Buffer = NULL;
    int Hits = 0;

    GLint Viewport[4];
    glGetIntegerv(GL_VIEWPORT, Viewport);

    // Feedback might be requested while normal rendering is already in progress.
    // So make sure to restore projection and modelview matrix once we are finished.
    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
    glLoadIdentity();
    glOrtho(0, Viewport[2], 0, Viewport[3], -100, 100);                                           

    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();

    do
    {
      Buffer = (GLfloat*) realloc(Buffer, BufferSize * sizeof(GLfloat));
      glFeedbackBuffer(BufferSize, GL_2D, Buffer);
      glRenderMode(GL_FEEDBACK);

      FLayer->RenderFeedback(this);
  
      Hits = glRenderMode(GL_RENDER);
      BufferSize <<= 1;
    }
    while (Hits < 0);

    glPopMatrix();
    glMatrixMode(GL_PROJECTION);
    glPopMatrix();

    if (Hits > 0)
    {
      #define CHECK_VERTEX(X, Y) \
        if (X < FCurrentBounds.Left) \
          FCurrentBounds.Left = X; \
        if (Y > FCurrentBounds.Top) \
          FCurrentBounds.Top = Y; \
        if (X > FCurrentBounds.Right) \
          FCurrentBounds.Right = X; \
        if (Y < FCurrentBounds.Bottom) \
          FCurrentBounds.Bottom = Y;

      int Offset = 0;
      FCurrentBounds.Left = 1e10;
      FCurrentBounds.Top = -1e10;
      FCurrentBounds.Right = -1e10;
      FCurrentBounds.Bottom = 1e10;
      while (Offset < Hits)
      {
        switch ((DWORD) Buffer[Offset++])
        {
          case GL_PASS_THROUGH_TOKEN:
            {
              float Token = Buffer[Offset++]; // Just for debugging.
              break;
            };
          case GL_POINT_TOKEN:
          case GL_BITMAP_TOKEN:
          case GL_DRAW_PIXEL_TOKEN:
          case GL_COPY_PIXEL_TOKEN:
            {
              float X = Buffer[Offset++];
              float Y = Buffer[Offset++];
              CHECK_VERTEX(X, Y);
              break;
            };
          case GL_POLYGON_TOKEN:
            {
              int Count = (int) Buffer[Offset++];
              for (int I = 0; I < Count; I++)
              {
                float X = Buffer[Offset++];
                float Y = Buffer[Offset++];
                CHECK_VERTEX(X, Y);
              };
              break;
            };
          case GL_LINE_TOKEN:
          case GL_LINE_RESET_TOKEN:
            {
              float X = Buffer[Offset++];
              float Y = Buffer[Offset++];
              CHECK_VERTEX(X, Y);

              X = Buffer[Offset++];
              Y = Buffer[Offset++];
              CHECK_VERTEX(X, Y);

              break;
            };
        };
      };
      #undef CHECK_VERTEX
    };

    free(Buffer);

    FLayer->CheckError();
  };

  *Bounds = FCurrentBounds;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Destroys this object.
 */
void CFigureInstance::Release(void)
{
  delete this;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Checks the validity of the figure instance display list and executes it.
 */
void CFigureInstance::Render(void)
{
  if (FDirty)
    ValidateDisplayList();

  // Place a reference of this class onto the name stack. This is only used when picking is enabled.
  glLoadName((GLuint) this);
  glCallList(FList);
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Rotates the figure around the given axis by the angle Angle (in radians). This version of Rotate uses a vector for the
 * rotation axis in the parameter list.
 *
 * @param Angle The rotation angle in radians.
 * @param Rx The x part of the axis around which to rotate the figure instance.
 * @param Ry The y part of the axis around which to rotate the figure instance.
 * @param Rz The z part of the axis around which to rotate the figure instance.
 * @note: Currently there is no accumulative version of Rotate available (requires a quaternion lib, which we don't have yet).
 */
void CFigureInstance::Rotate(double Angle, double Rx, double Ry, double Rz)
{
  FRotation[0] = Angle;
  FRotation[1] = Rx;
  FRotation[2] = Ry;
  FRotation[3] = Rz;
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Rotates the figure around the given axis by the angle Angle (in radians). This version of Rotate uses a vector for the
 * rotation axis in the parameter list.
 *
 * @param Angle The rotation angle in radians.
 * @param Axis The axis around which to rotate the figure instance.
 * @note: Currently there is no accumulative version of Rotate available (requires a quaternion lib, which we don't have yet).
 */
void CFigureInstance::RotateV(double Angle, const double Axis[3])
{
  FRotation[0] = Angle;
  FRotation[1] = Axis[0];
  FRotation[2] = Axis[1];
  FRotation[3] = Axis[2];
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Scales the figure by the amount given in Factor. If Accumulative is true then the new scale factors are multiplied
 * with the existing values. This version of Scale uses an array of values in the parameter list.
 *
 * @param Sx The scale value for the x-axis
 * @param Sy The scale value for the y-axis
 * @param Sz The scale value for the z-axis
 * @param Accumulative If true then the new scale values are added to any previously assigned values.
 */
void CFigureInstance::Scale(double Sx, double Sy, double Sz, bool Accumulative)
{
  if (Accumulative)
  {
    FScaling[0] += Sx;
    FScaling[1] += Sy;
    FScaling[2] += Sz;
  }
  else
  {
    FScaling[0] = Sx;
    FScaling[1] = Sy;
    FScaling[2] = Sz;
  };
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Scales the figure by the amount given in Factor. If Accumulative is true then the new scale factors are multiplied
 * with the existing values. This version of Scale uses an array of values in the parameter list.
 *
 * @param Factor Contains the scaling factors for all three axes. Index 0 contains the value for the x-axis, index 1
 *               that for the y-axis and index 2 for z.
 * @param Accumulative If true then the new scale values are added to any previously assigned values.
 */
void CFigureInstance::ScaleV(const double Factor[3], bool Accumulative)
{
  if (Accumulative)
  {
    FScaling[0] += Factor[0];
    FScaling[1] += Factor[1];
    FScaling[2] += Factor[2];
  }
  else
  {
    FScaling[0] = Factor[0];
    FScaling[1] = Factor[1];
    FScaling[2] = Factor[2];
  };
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Tells the caller whether this instance is currently selected.
 * 
 * @return True if this figure instance is currently selected, otherwise false.
 */
bool CFigureInstance::Selected(void)
{
  return FSelected;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Moves the figure by the amount given in Factor. If Accumulative is true then the new translation factors are multiplied
 * with the existing values.
 *
 * @param Tx The scale factor to apply on the x-axis.
 * @param Ty The scale factor to apply on the y-axis.
 * @param Tz The scale factor to apply on the z-axis.
 * @param Accumulative If true scaling factors are added to the values already set previously.
 */
void CFigureInstance::Translate(double Tx, double Ty, double Tz, bool Accumulative)
{
  if (Accumulative)
  {
    FTranslation[0] += Tx;
    FTranslation[1] += Ty;
    FTranslation[2] += Tz;
  }
  else
  {
    FTranslation[0] = Tx;
    FTranslation[1] = Ty;
    FTranslation[2] = Tz;
  };
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Moves the figure by the amount given in Factor. If Accumulative is true then the new translation factors are multiplied
 * with the existing values.
 *
 * @param Factor The scale factor to apply. Index 0 contains the factor for the x-axis etc.
 * @param Accumulative If true scaling factors are added to the values already set previously.
 */
void CFigureInstance::TranslateV(const double Factor[3], bool Accumulative)
{
  if (Accumulative)
  {
    FTranslation[0] += Factor[0];
    FTranslation[1] += Factor[1];
    FTranslation[2] += Factor[2];
  }
  else
  {
    FTranslation[0] = Factor[0];
    FTranslation[1] = Factor[1];
    FTranslation[2] = Factor[2];
  };
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

