unit MySQLResultSet;

interface

uses
  Windows, Messages, SysUtils, Classes, Contnrs, Controls,
  TntExtCtrls, TntComCtrls, TntClasses, TntStdCtrls, SyncObjs,
  myx_public_interface, gnugettext, auxfuncs, MyxError,
  MySQLConnection, VirtualTrees, Graphics, StrUtils,
  TextSearch;

type
  TMySQLRS = class;
  TExecuteQueryThread = class;

  RSMessageType = (RSMT_ERROR, RSMT_WARNING);

  IMySQLRSControlInterface = interface(IUnknown)
    ['{B46142F1-743F-4BB2-9A00-20C7D485D7AA}']
    procedure ClearValues;

    function GetMySQLRS: TMySQLRS;
    procedure SetMySQLRS(MySQLRS: TMySQLRS);
    function GetControl: TObject;

    procedure DoCurrentRowChanged;
    procedure DoEditStateChanged;
    procedure DoStatusCaptionChanged;
    procedure DoMessagesChanged;

    property MySQLRS: TMySQLRS read GetMySQLRS write SetMySQLRS;
    property Control: TObject read GetControl;
  end;

  TConfirmEvent = function(Sender: TObject; Msg: WideString): Boolean of object;
  TResultsetEvent = function(Sender: TObject): TMySQLRS of object;

  TMySQLRS = class(TObject)
    constructor Create;
    destructor Destroy; override;

    procedure ConnectControl(Control: IMySQLRSControlInterface);
    procedure DisconnectControl(Control: IMySQLRSControlInterface);

    procedure ExecuteQuery(sql: WideString = '');

    procedure FreeResultSet;

    procedure DoCurrentRowChanged(Sender: IMySQLRSControlInterface; CurrentRow: Integer);
    procedure DoDynamicParamsChanged;
    procedure DoEditStateChanged;
    procedure DoStatusCaptionChanged;

    procedure UpdateDynamicParams;

    procedure DoDisplaySearch(ShowReplacePage: Boolean = False);
    procedure GotoFirstRow;
    procedure GotoLastRow;
    procedure DoEdit;
    procedure DoApplyChanges;
    procedure DoDiscardChanges;

    procedure DoMessagesChanged;
    procedure AddRSMessage(Msg: WideString; MsgType: RSMessageType;
      MsgNr: Integer);
    procedure ClearRSMessages;

    procedure ConnectRS(RS: TMySQLRS);
    procedure DisconnectRS(RS: TMySQLRS);
  protected
    QueryStateMultiReadSync: TMultiReadExclusiveWriteSynchronizer;

    function GetMySQLConn: TMySQLConn;
    procedure SetMySQLConn(MySQLConn: TMySQLConn);

    function GetActive: Boolean;
    procedure SetActive(Active: Boolean);

    procedure Open;
    procedure Close;

    function GetEdited: Boolean;
    procedure SetEdited(Edited: Boolean);

    function GetStatusCaption: WideString;
    procedure SetStatusCaption(StatusCaption: WideString);

    function GetParentRS: TMySQLRS;
    procedure SetParentRS(ParentRS: TMySQLRS);

    function GetRowCount: integer;

    procedure CheckAttributes;

    function GetStopQuery: Boolean;
    procedure SetStopQuery(StopQuery: Boolean);

    function GetQueryExecuting: Boolean;
    procedure SetQueryExecuting(QueryExecuting: Boolean);

    function GetConnectedDetailRSCount: Integer;
    function GetConnectedDetailRS(I: Integer): TMySQLRS;

    function GetConnectedControlsCount: Integer;
    function GetConnectedControls(I: Integer): IMySQLRSControlInterface;

    function DoSearch(Sender: TObject;
      SearchText: WideString; ReplaceText: WideString;
      SearchOptions: TTextSearchOptions): Integer;
  private
    FConnectedControls: TInterfaceList;
    ConnectedRS: TList;

    ExecuteQueryThread: TExecuteQueryThread;

    FMySQLConn: TMySQLConn;
    FSQL: Widestring;

    FStatusCaption: WideString;

    FEdited,
    FActive: Boolean;

    FParentRS: TMySQLRS;

    FCurrentRow: Integer;

    FOnParamChange: TNotifyEvent;

    FOnQueryExecute: TNotifyEvent;
    FOnQueryExecuted: TNotifyEvent;
    FOnQuerySuccess: TNotifyEvent;
    FOnQueryStopped: TNotifyEvent;
    FOnQueryError: TNotifyEvent;

    FOnConfirmDeletion: TConfirmEvent;

    FStopQuery,
    FQueryExecuting: Boolean;

    FCreateNextRSForMultibleRSQuery: TResultsetEvent;

    FRemoveRSForMultibleRSQuery,
    FShowRSForMultibleRSQuery: TNotifyEvent;

    FGlobalParams,
    FLocalParams,
    FDynamicParams: TTntStringList;
  public
    WorkWithResultset: TCriticalSection;

    ResultSet: PMYX_RESULTSET;

    ErrorMessages: TObjectList;

    Editable,
    EditingAllowed,
    AtLeastOneRecord,
    FirstRowSelected,
    LastRowSelected: Boolean;
    ErrorOccured: Boolean;
    CurrentQuery: WideString;
    QueryHistory: TTntStringList;
    CurrentHistoryEntry: integer;

    InsertedRows,
    DeletedRows: Integer;

    function AllowedToDiscard: Boolean;

    property MySQLConn: TMySQLConn read GetMySQLConn write SetMySQLConn default nil;
    property Active: Boolean read GetActive write SetActive default False;
    property SQL: Widestring read FSQL write FSQL;
    property Edited: Boolean read GetEdited write SetEdited default False;
    property StatusCaption: WideString read GetStatusCaption write SetStatusCaption;
    property ParentRS: TMySQLRS read GetParentRS write SetParentRS;
    property RowCount: Integer read GetRowCount;

    property OnParamChange: TNotifyEvent read FOnParamChange write FOnParamChange;
    property OnQueryExecute: TNotifyEvent read FOnQueryExecute write FOnQueryExecute;
    property OnQueryExecuted: TNotifyEvent read FOnQueryExecuted write FOnQueryExecuted;
    property OnQuerySuccess: TNotifyEvent read FOnQuerySuccess write FOnQuerySuccess;
    property OnQueryStopped: TNotifyEvent read FOnQueryStopped write FOnQueryStopped;
    property OnQueryError: TNotifyEvent read FOnQueryError write FOnQueryError;

    property OnConfirmDeletion: TConfirmEvent read FOnConfirmDeletion write FOnConfirmDeletion;

    property StopQuery: Boolean read GetStopQuery write SetStopQuery default False;
    property QueryExecuting: Boolean read GetQueryExecuting write SetQueryExecuting default False;

    property ConnectedDetailRSCount: integer read GetConnectedDetailRSCount;
    property ConnectedDetailRS[I: Integer]: TMySQLRS read GetConnectedDetailRS;

    property CreateNextRSForMultibleRSQuery: TResultsetEvent read FCreateNextRSForMultibleRSQuery write FCreateNextRSForMultibleRSQuery;
    property RemoveRSForMultibleRSQuery: TNotifyEvent read FRemoveRSForMultibleRSQuery write FRemoveRSForMultibleRSQuery;
    property ShowRSForMultibleRSQuery: TNotifyEvent read FShowRSForMultibleRSQuery write FShowRSForMultibleRSQuery;

    property ConnectedControlsCount: Integer read GetConnectedControlsCount;
    property ConnectedControls[I: Integer]: IMySQLRSControlInterface read GetConnectedControls;

    property GlobalParams: TTntStringList read FGlobalParams write FGlobalParams;
    property LocalParams: TTntStringList read FLocalParams write FLocalParams;
    property DynamicParams: TTntStringList read FDynamicParams write FDynamicParams;
  end;

  TExecuteQueryThread = class(TThread)
    constructor Create(MySQLRS: TMySQLRS);
    destructor Destroy; override;

    procedure CleanUp;
    procedure BuildGrid;

    procedure QueryExecuted;
    procedure QuerySuccess;
    procedure QueryStopped;
    procedure QueryError;
  protected
    procedure Execute; override;
    procedure CreateNextRSForMultibleRSQuery;
    procedure RemoveRSForMultibleRSQuery;
    procedure ShowRSForMultibleRSQuery;
  private
    ThreadPMySQL: Pointer;
    ExceptionMsg: WideString;
  public
    MySQLRS: TMySQLRS;

    Query: WideString;

    current_row_count: Longint;
    previous_row_count: Longint;
    result_set: PMYX_RESULTSET;
  end;

  TRSError = class(TObject)
    constructor Create(Msg: WideString; MsgType: RSMessageType;
      MsgNr: Integer);
  public
    Msg: WideString;
    MsgType: RSMessageType;
    MsgNr: Integer;
  end;

  TRSFieldValue = record
    Value: PAnsiChar;
    Length: Integer;
    BinaryData: Boolean;
    IsNull: Boolean;
  end;

  function progress_row_fetch(current_row_count: Longint; previous_row_count: Longint; result_set: PMYX_RESULTSET; user_data: Pointer):Integer; cdecl;
  procedure resultset_realloc_before(user_data: Pointer); cdecl;
  procedure resultset_realloc_after(user_data: Pointer); cdecl;

implementation

uses MySQLResultSetControls, ApplicationDataModule;
                                                   
constructor TMySQLRS.Create;
begin
  inherited;

  ResultSet:=nil;
  FMySQLConn:=nil;

  FParentRS:=nil;

  FConnectedControls := TInterfaceList.Create;
  ConnectedRS := TList.Create;

  QueryHistory:=TTntStringList.Create;
  CurrentHistoryEntry:=0;
  CurrentQuery:='';

  FCurrentRow:=-1;

  InsertedRows:=0;
  DeletedRows:=0;

  FGlobalParams:=nil;
  FLocalParams:=TTntStringList.Create;
  FDynamicParams:=TTntStringList.Create;

  ErrorMessages:=TObjectList.Create;

  ErrorOccured:=False;

  Editable:=False;
  EditingAllowed:=False;
  FEdited:=False;

  QueryStateMultiReadSync:=TMultiReadExclusiveWriteSynchronizer.Create;
  FActive:=False;
  StopQuery:= False;

  WorkWithResultset:=TCriticalSection.Create;
end;

destructor TMySQLRS.Destroy;
var i: integer;
begin
  //Disconnect from ParentRS
  ParentRS:=nil;

  FreeResultSet;

  //Disconnect Controls
  for i:=0 to FConnectedControls.Count-1 do
    IMySQLRSControlInterface(FConnectedControls[0]).MySQLRS:=nil;

  ErrorMessages.Free;
  QueryHistory.Free;

  FConnectedControls.Free;
  ConnectedRS.Free;

  FLocalParams.Free;
  FDynamicParams.Free;

  QueryStateMultiReadSync.Free;

  WorkWithResultset.Free;

  inherited;
end;

procedure TMySQLRS.ConnectControl(Control: IMySQLRSControlInterface);
begin
  if FConnectedControls.IndexOf(Control) = -1 then
    FConnectedControls.Add(Control);
end;

procedure TMySQLRS.DisconnectControl(Control: IMySQLRSControlInterface);
begin
  FConnectedControls.Remove(Control);
end;

procedure TMySQLRS.FreeResultSet;
var i: integer;
begin
  for i:=0 to FConnectedControls.Count-1 do
    IMySQLRSControlInterface(FConnectedControls[i]).ClearValues;

  if(ResultSet<>nil)then
    myx_query_free_resultset(ResultSet);

  InsertedRows:=0;
  DeletedRows:=0;

  ResultSet:=nil;
end;

function TMySQLRS.GetMySQLConn: TMySQLConn;
begin
  Result:=FMySQLConn;
end;

procedure TMySQLRS.SetMySQLConn(MySQLConn: TMySQLConn);
begin
  FMySQLConn:=MySQLConn;
end;

function TMySQLRS.AllowedToDiscard: Boolean;
begin
  Result := Not(FEdited) or (ShowModalDialog(_('Unapplied Changes'),
    _('You have made changes to the resultset that have not been '+
      'applied yet. Do you want to discard the changes?'),
    myx_mtConfirmation, _('Discard')+#13#10+_('Abort')) = 1);
end;

procedure TMySQLRS.ExecuteQuery(sql: WideString);

begin
  FSQL := Trim(sql);
  
  if FSQL = '' then
  begin
    ShowModalDialog(_('No SQL Command'),
      _('You tried to execute an empty string. Please type an SQL command into the SQL edit field and execute again.'),
      myx_mtError, _('Ok'));
    Exit;
  end;

  if not AllowedToDiscard  then
    Exit;

  if MySQLConn.MySQL = nil then
    raise EInOutError.Create(_('MySQL connection not set.'));

  FreeResultSet;

  ClearRSMessages;

  EditingAllowed:=False;
  StopQuery:=False;
  QueryExecuting:=True;

  if(Assigned(FOnQueryExecute))then
    FOnQueryExecute(self);

  try
    ExecuteQueryThread:=TExecuteQueryThread.Create(self);
    try
      // Thread will be freed after execution.
      ExecuteQueryThread.FreeOnTerminate:=True;

      ExecuteQueryThread.Resume;
    except
      ExecuteQueryThread.Free;
    end;
  except
    on x: Exception do
    begin
      QueryExecuting:=False;

      if Assigned(FOnQueryError) then
        FOnQueryError(self);

      if(Assigned(FOnQueryExecuted))then
        FOnQueryExecuted(self);

      raise;
    end;
  end;
end;

procedure TMySQLRS.SetActive(Active: Boolean);
begin
  if(Active)then
    Open
  else
    Close;
end;

function TMySQLRS.GetActive: Boolean;
begin
  Result:=FActive;
end;

procedure TMySQLRS.Open;
begin
  FActive:=True;

  ExecuteQuery;
end;

procedure TMySQLRS.Close;
begin
  FreeResultSet;

  FActive:=False;
end;

function TMySQLRS.GetEdited: Boolean;
begin
  Result:=FEdited;
end;

procedure TMySQLRS.SetEdited(Edited: Boolean);
begin
  if(FEdited<>Edited)then
  begin
    FEdited:=Edited;

    DoEditStateChanged;
  end;
end;

procedure TMySQLRS.ConnectRS(RS: TMySQLRS);
begin
  if(ConnectedRS.IndexOf(RS)=-1)then
    ConnectedRS.Add(RS);
end;

procedure TMySQLRS.DisconnectRS(RS: TMySQLRS);
begin
  if(ConnectedRS.IndexOf(RS)<>-1)then
    ConnectedRS.Delete(ConnectedRS.IndexOf(RS));
end;

function TMySQLRS.GetParentRS: TMySQLRS;
begin
  Result:=FParentRS;
end;

procedure TMySQLRS.SetParentRS(ParentRS: TMySQLRS);
begin
  if(FParentRS<>nil)and(FParentRS<>ParentRS)then
    FParentRS.DisconnectRS(self);

  FParentRS:=ParentRS;

  if(FParentRS<>nil)then
    FParentRS.ConnectRS(self);
end;

function TMySQLRS.GetRowCount: integer;
begin
  if(ResultSet<>nil)then
    Result:=ResultSet.rows_num+InsertedRows-DeletedRows
  else
    Result:=0;
end;


procedure TMySQLRS.DoCurrentRowChanged(Sender: IMySQLRSControlInterface;
  CurrentRow: Integer);

var
  i: integer;

begin
  CheckAttributes;

  FirstRowSelected:=False;
  if(ResultSet<>nil)then
    if(CurrentRow=0)or((ResultSet.rows_num+InsertedRows=0)and(CurrentRow=1))then
      FirstRowSelected:=True;

  LastRowSelected:=False;
  if(ResultSet<>nil)then
    if((CurrentRow>=ResultSet.rows_num-1+InsertedRows)or
      (ResultSet.rows_num+InsertedRows=0))then
      LastRowSelected:=True;


  if(FCurrentRow=int(CurrentRow))then
    Exit;

  FCurrentRow:=CurrentRow;

  UpdateDynamicParams;

  for i:=0 to FConnectedControls.Count-1 do
    if(FConnectedControls[i]<>Sender)then
      IMySQLRSControlInterface(FConnectedControls[i]).DoCurrentRowChanged;

  for i:=0 to ConnectedRS.Count-1 do
    TMySQLRS(ConnectedRS[i]).DoDynamicParamsChanged;

  if(Assigned(FOnParamChange))then
    FOnParamChange(self);
end;

procedure TMySQLRS.UpdateDynamicParams;

var
  i: integer;
  row: PMYX_RS_ROW;
  col: PMYX_RS_COLUMN;
  field: MYX_RS_FIELD;

begin
  //Set Dynamic Parameters
  FDynamicParams.Clear;
  if (FCurrentRow>-1) and
     (ResultSet<>nil) and
     (FCurrentRow<Integer(ResultSet.rows_num))then
  begin
    row:=PMYX_RS_ROW(integer(ResultSet.rows)+sizeof(MYX_RS_ROW)*FCurrentRow);

    for i:=0 to ResultSet.columns_num-1 do
    begin
      col:=PMYX_RS_COLUMN(integer(ResultSet.columns)+(sizeof(MYX_RS_COLUMN)*i));
      if(row^.fields<>nil)then
      begin
        field:=(PMYX_RS_FIELD(integer(row^.fields)+(sizeof(MYX_RS_FIELD)*i)))^;
        //only take first 128 chars and replace returns
        FDynamicParams.Add(col^.name+'='+
          AnsiReplaceText(
            AnsiReplaceText(
              AnsiReplaceText(
                Copy(field.value, 1, 128),
                #13#10, ''),
              #10, ''),
            #13, ''));
      end
      else
        FDynamicParams.Add(col^.name+'=');
    end;
  end;
end;

procedure TMySQLRS.DoDynamicParamsChanged;
begin
  if(ResultSet<>nil)then
  begin
    //If dynamic params have been changed
    //reexecute query
    if(ResultSet.query.params_num>0)then
    begin
      ExecuteQuery(FSQL);

      if(Assigned(FOnParamChange))then
        FOnParamChange(self);
    end;
  end;
end;

procedure TMySQLRS.DoEditStateChanged;
var i: integer;
begin
  for i:=0 to FConnectedControls.Count-1 do
    IMySQLRSControlInterface(FConnectedControls[i]).DoEditStateChanged;
end;

procedure TMySQLRS.DoStatusCaptionChanged;
var i: integer;
begin
  for i:=0 to FConnectedControls.Count-1 do
    IMySQLRSControlInterface(FConnectedControls[i]).DoStatusCaptionChanged;
end;


function TMySQLRS.GetStatusCaption: WideString;
begin
  Result:=FStatusCaption;
end;

procedure TMySQLRS.SetStatusCaption(StatusCaption: WideString);
begin
  FStatusCaption:=StatusCaption;

  DoStatusCaptionChanged;
end;


procedure TMySQLRS.CheckAttributes;
begin
  if(ResultSet<>nil)then
    Editable:=(ResultSet.editable=1);

  AtLeastOneRecord:=False;
  if(ResultSet<>nil)then
    if(ResultSet.rows_num+InsertedRows>0)then
      AtLeastOneRecord:=True;
end;


procedure TMySQLRS.DoDisplaySearch(ShowReplacePage: Boolean);
begin
  if TextSearchForm=nil then
    TextSearchForm := TTextSearchForm.Create(nil,
      ApplicationDM.Options);

  if ShowReplacePage then
    TextSearchForm.PageControl.ActivePage:=
      TextSearchForm.ReplaceTabSheet;

  TextSearchForm.Show;
  TextSearchForm.OnSearch := DoSearch;
end;

function TMySQLRS.DoSearch(Sender: TObject;
  SearchText: WideString; ReplaceText: WideString;
  SearchOptions: TTextSearchOptions): Integer;

var
  i: Integer;

begin
  Result := 0;

  for i:=0 to FConnectedControls.Count-1 do
    if IMySQLRSControlInterface(FConnectedControls[i]).Control is TMySQLRSGrid then
    begin
      Result := TMySQLRSGrid(IMySQLRSControlInterface(
        FConnectedControls[i]).Control).DoSearch(Sender,
          SearchText, ReplaceText, SearchOptions);
    end;
end;

procedure TMySQLRS.GotoFirstRow;
var i: integer;
  RSGrid: TMySQLRSGrid;
begin
  for i:=0 to FConnectedControls.Count-1 do
  begin
    if((TObject(IMySQLRSControlInterface(FConnectedControls[i]).GetControl) is TMySQLRSGrid))then
    begin
      RSGrid:=TMySQLRSGrid(IMySQLRSControlInterface(FConnectedControls[i]).GetControl);

      if(RSGrid.CanFocus)then
        RSGrid.SetFocus;

      RSGrid.FocusedNode:=RSGrid.GetFirstNoInit;
      RSGrid.ClearSelection;
      RSGrid.Selected[RSGrid.FocusedNode]:=True;
    end;
  end;
end;

procedure TMySQLRS.GotoLastRow;
var i: integer;
  RSGrid: TMySQLRSGrid;
  LastNode: PVirtualNode;
begin
  for i:=0 to FConnectedControls.Count-1 do
  begin
    if((TObject(IMySQLRSControlInterface(FConnectedControls[i]).GetControl) is TMySQLRSGrid))then
    begin
      RSGrid:=TMySQLRSGrid(IMySQLRSControlInterface(FConnectedControls[i]).GetControl);

      if(RSGrid.CanFocus)then
        RSGrid.SetFocus;

      LastNode:=RSGrid.GetLastNoInit;
      if(LastNode<>nil)then
        if(LastNode.PrevSibling<>nil)then
          RSGrid.FocusedNode:=LastNode.PrevSibling
        else
          RSGrid.FocusedNode:=LastNode;
          
      RSGrid.ClearSelection;
      RSGrid.Selected[RSGrid.FocusedNode]:=True;
    end;
  end;
end;

procedure TMySQLRS.DoEdit;
begin
  if Editable then
  begin
    DoEditStateChanged;
    if not EditingAllowed then
    begin
      EditingAllowed := True;
    end
    else
    begin
      DoDiscardChanges;
    end;
  end;
end;

procedure TMySQLRS.DoApplyChanges;
var errors: PMYX_RS_ACTION_ERRORS;
  error: PMYX_RS_ACTION_ERROR;
  i: integer;
  RSGrid: TMySQLRSGrid;
  RSAction: PMYX_RS_ACTION;

begin
  ClearRSMessages;
  SetStatusCaption('');

  WorkWithResultset.Acquire;
  try
    //convert all failed actions to normal actions
    for i:=0 to ResultSet.actions.actions_num-1 do
    begin
      RSAction := PMYX_RS_ACTION(Integer(ResultSet.actions.actions) +
        sizeof(MYX_RS_ACTION) * i);

      if (RSAction.status<>MYX_RSAS_NEW) then
        RSAction.status:=MYX_RSAS_NEW;
    end;

    //Apply actions
    errors:=myx_query_apply_actions(ResultSet);
    if(errors<>nil)then
    begin
      SetStatusCaption(_('Error while applying actions.'));

      for i:=0 to errors.errors_num-1 do
      begin
        error:=PMYX_RS_ACTION_ERROR(Integer(errors.errors)+sizeof(MYX_RS_ACTION_ERROR)*i);

        AddRSMessage(error.error_text, RSMT_ERROR, error.error);
      end;
    end;

    //Delete Rows from Grids
    for i:=0 to FConnectedControls.Count-1 do
      if((TObject(IMySQLRSControlInterface(FConnectedControls[i]).GetControl) is TMySQLRSGrid))then
      begin
        RSGrid:=TMySQLRSGrid(IMySQLRSControlInterface(FConnectedControls[i]).GetControl);

        RSGrid.ApplyDeleteActionsToGrid;
      end;

    //Update resultset which will clear or re-generate a action list
    if(myx_query_update_resultset(ResultSet)<>0)then
      SetStatusCaption(_('Error while applying changes to the result set.'));

    //Update rows that had actions
    for i:=0 to FConnectedControls.Count-1 do
      if((TObject(IMySQLRSControlInterface(FConnectedControls[i]).GetControl) is TMySQLRSGrid))then
      begin
        RSGrid:=TMySQLRSGrid(IMySQLRSControlInterface(FConnectedControls[i]).GetControl);

        RSGrid.SynchronizeActions;
        //RSGrid.ApplyDeleteActions;
      end;

  finally
    WorkWithResultset.Release;
  end;


  Edited:=False;
  //EditingAllowed:=False;


  DoEditStateChanged;
end;

procedure TMySQLRS.DoDiscardChanges;
var DoIt: Boolean;
begin
  DoIt:=True;

  if(Assigned(FOnConfirmDeletion))and(Edited)then
    DoIt := FOnConfirmDeletion(Self, _('Do you really want to discard all changes?'));

  if DoIt then
  begin
    myx_query_discard_actions(ResultSet);

    Edited:=False;
    EditingAllowed:=False;

    DoEditStateChanged;
  end;
end;

procedure TMySQLRS.DoMessagesChanged;
var i: integer;
begin
  for i:=0 to FConnectedControls.Count-1 do
    IMySQLRSControlInterface(FConnectedControls[i]).DoMessagesChanged;
end;

procedure TMySQLRS.AddRSMessage(Msg: WideString; MsgType: RSMessageType;
  MsgNr: Integer);
begin
  if(MsgType=RSMT_ERROR)then
    ErrorOccured:=True;

  ErrorMessages.Add(TRSError.Create(Msg, MsgType, MsgNr));

  DoMessagesChanged;
end;

procedure TMySQLRS.ClearRSMessages;
begin
  ErrorMessages.Clear;

  DoMessagesChanged;
end;

function TMySQLRS.GetStopQuery: Boolean;
begin
  QueryStateMultiReadSync.BeginRead;
  try
    Result:=FStopQuery;
  finally
    QueryStateMultiReadSync.EndRead;
  end;
end;

procedure TMySQLRS.SetStopQuery(StopQuery: Boolean);
begin
  QueryStateMultiReadSync.BeginWrite;
  try
    FStopQuery:=StopQuery;
  finally
    QueryStateMultiReadSync.EndWrite;
  end;
end;

function TMySQLRS.GetQueryExecuting: Boolean;
begin
  QueryStateMultiReadSync.BeginRead;
  try
    Result:=FQueryExecuting;
  finally
    QueryStateMultiReadSync.EndRead;
  end;
end;

procedure TMySQLRS.SetQueryExecuting(QueryExecuting: Boolean);
begin
  QueryStateMultiReadSync.BeginWrite;
  try
    FQueryExecuting:=QueryExecuting;
  finally
    QueryStateMultiReadSync.EndWrite;
  end;
end;

function TMySQLRS.GetConnectedDetailRSCount: Integer;

begin
  Result:=ConnectedRS.Count;
end;

function TMySQLRS.GetConnectedDetailRS(I: Integer): TMySQLRS;

begin
  if (I<ConnectedRS.Count) then
    Result := ConnectedRS[I]
  else
    Result := nil;
end;

function TMySQLRS.GetConnectedControlsCount: Integer;

begin
  Result:=FConnectedControls.Count;
end;

function TMySQLRS.GetConnectedControls(I: Integer): IMySQLRSControlInterface;

begin
  if (I<FConnectedControls.Count) then
    Result := IMySQLRSControlInterface(FConnectedControls[I])
  else
    Result := nil;
end;

// -----------------------------------------------------------------------------

constructor TExecuteQueryThread.Create(MySQLRS: TMySQLRS);
begin
  inherited Create(True);

  self.MySQLRS:=MySQLRS;

  current_row_count:=0;
  previous_row_count:=0;
  result_set:=nil;
  ExceptionMsg:='';

  //When the user is in the middle of a transaction, don't create a new connection
  if(Not(MySQLRS.MySQLConn.InTransaction))then
  begin
    ThreadPMySQL:=myx_mysql_init();
    if(ThreadPMySQL=nil)then
      raise EMyxError.Create(_('Error while allocating memory for MySQL Struct in Query Execute Thread.'));


    if(myx_connect_to_instance(
      MySQLRS.MySQLConn.User_Connection.get_record_pointer,
      ThreadPMySQL)<>0)then
      raise EMyxError.Create(_('Query Execute Thread cannot connect to MySQL'));

    if(MySQLRS.MySQLConn.DefaultSchema<>'')and
      (MySQLRS.MySQLConn.DefaultSchema<>MySQLRS.MySQLConn.user_connection.schema)then
        myx_use_schema(ThreadPMySQL, MySQLRS.MySQLConn.DefaultSchema);
  end
  else
    ThreadPMySQL:=MySQLRS.MySQLConn.MySQL;
end;

destructor TExecuteQueryThread.Destroy;
begin
  CleanUp;

  inherited Destroy;
end;

procedure TExecuteQueryThread.CleanUp;
begin
  if (MySQLRS<>nil) and
    (MySQLRS.MySQLConn<>nil) and
    (Not(MySQLRS.MySQLConn.InTransaction))then
    myx_mysql_close(ThreadPMySQL);
end;

procedure TExecuteQueryThread.Execute;

var
  ErrorCode: MYX_LIB_ERROR;
  Parameters: TMYX_STRINGLIST;
  MySQLConn: TMySQLConn;
  InTransaction: Boolean;

begin
  MySQLConn := MySQLRS.MySQLConn;
  try
    try
      InTransaction := MySQLConn.InTransaction;

      //If the user is in a transaction, wait till connection is free
      if (InTransaction) then
        MySQLConn.FetchDataThreadCriticalSection.Acquire;
      try
        Query:=MySQLRS.SQL;

        Parameters:=TMYX_STRINGLIST.create;
        try
          Parameters.strings.Text := '';
          //Global Params
          if (MySQLRS.GlobalParams<>nil) then
            Parameters.strings.Text := Parameters.strings.Text+
              MySQLRS.GlobalParams.Text;
          //Local Params
          if (MySQLRS.LocalParams<>nil) then
            Parameters.strings.Text := Parameters.strings.Text+
              MySQLRS.LocalParams.Text;
          //Add dynamic parameters from parent
          if (MySQLRS.FParentRS<>nil) and
            (MySQLRS.FParentRS.DynamicParams<>nil) then
            Parameters.strings.Text:=Parameters.strings.Text+#13#10+
              MySQLRS.FParentRS.DynamicParams.Text;

          result_set := myx_query_execute(ThreadPMySQL,
            Query, Ord(ApplicationDM.Options.EnforceEditableQueries),
            Parameters.get_record_pointer,
            @ErrorCode, Addr(self), progress_row_fetch,
            resultset_realloc_before,
            resultset_realloc_after);


          //Handle multible resultsets
          while (1=1) do
          begin
            //If this was executed using a cloned MySQL Connection,
            //reassign the original one
            if (Not(MySQLRS.MySQLConn.InTransaction)) and
              (MySQLRS.ResultSet<>nil) then
              MySQLRS.ResultSet.mysql:=MySQLRS.MySQLConn.MySQL;

            //Display error/summary
            if (ErrorCode=MYX_NO_ERROR) then
              Synchronize(QuerySuccess)
            else
              if ErrorCode=MYX_STOP_EXECUTION then
                Synchronize(QueryStopped)
              else
                Synchronize(QueryError);

            //Check if there is a resultset left
            if Not( (result_set<>nil) and (result_set.has_more=1) ) then
              break;

            //Get new MySQLRS
            if Assigned(MySQLRS.CreateNextRSForMultibleRSQuery) then
            begin
              Synchronize(CreateNextRSForMultibleRSQuery);
              if MySQLRS<>nil then
              begin
                //Reset thread values
                current_row_count := 0;
                previous_row_count := 0;
                result_set:=nil;
                ExceptionMsg:='';

                //Fetch next resultset (use _ function so nil can be passed as query)
                result_set := _myx_query_execute(ThreadPMySQL,
                  nil, 0,
                  nil,
                  @ErrorCode, Addr(self), progress_row_fetch,
                  resultset_realloc_before,
                  resultset_realloc_after);

                //Handle last result in a multi resultset query
                if (ErrorCode=MYX_SQL_ERROR) and
                  (myx_mysql_errno(ThreadPMySQL)=0) then
                begin
                  Synchronize(RemoveRSForMultibleRSQuery);
                  break;
                end
                else
                  Synchronize(ShowRSForMultibleRSQuery);
              end
              else
                break;
            end
            else
              break;
          end;
        finally
          Parameters.Free;
        end;
      finally
        if(InTransaction)then
          MySQLConn.FetchDataThreadCriticalSection.Release;
      end;
    except
      Synchronize(QueryError);

      raise;
    end;
  finally
    Synchronize(QueryExecuted);
  end;
end;

procedure TExecuteQueryThread.CreateNextRSForMultibleRSQuery;
begin
  if Assigned(MySQLRS.CreateNextRSForMultibleRSQuery) then
    MySQLRS := MySQLRS.CreateNextRSForMultibleRSQuery(MySQLRS)
  else
    MySQLRS := nil;
end;

procedure TExecuteQueryThread.RemoveRSForMultibleRSQuery;
begin
  if Assigned(MySQLRS.RemoveRSForMultibleRSQuery) then
    MySQLRS.RemoveRSForMultibleRSQuery(MySQLRS);
end;

procedure TExecuteQueryThread.ShowRSForMultibleRSQuery;
begin
  if Assigned(MySQLRS.ShowRSForMultibleRSQuery) then
    MySQLRS.ShowRSForMultibleRSQuery(MySQLRS);
end;

function progress_row_fetch(current_row_count: Longint; previous_row_count: Longint; result_set: PMYX_RESULTSET; user_data: Pointer):Integer;
var PSender: ^TExecuteQueryThread;
begin
  PSender:=user_data;
  try
    PSender.current_row_count:=current_row_count;
    PSender.previous_row_count:=previous_row_count;
    PSender.result_set:=result_set;

    PSender.Synchronize(PSender.BuildGrid);

    Result:=Ord(PSender.MySQLRS.StopQuery);
  except
    on x: Exception do
    begin
      PSender.ExceptionMsg:=x.Message;
      PSender.Synchronize(PSender.QueryError);
      Result:=Ord(True);
    end;
  end;
end;

procedure resultset_realloc_before(user_data: Pointer);
var PSender: ^TExecuteQueryThread;
begin
  PSender:=user_data;
  try
    PSender.MySQLRS.WorkWithResultset.Acquire;
  except
    on x: Exception do
    begin
      PSender.MySQLRS.WorkWithResultset.Release;

      PSender.ExceptionMsg:=x.Message;
      PSender.Synchronize(PSender.QueryError);
    end;
  end;
end;

procedure resultset_realloc_after(user_data: Pointer);
var PSender: ^TExecuteQueryThread;
begin
  PSender:=user_data;
  try
    PSender.MySQLRS.WorkWithResultset.Release;
  except
    on x: Exception do
    begin
      PSender.ExceptionMsg:=x.Message;
      PSender.Synchronize(PSender.QueryError);
    end;
  end;
end;

procedure TExecuteQueryThread.BuildGrid;
var j: integer;
  RSGrid: TMySQLRSGrid;
begin
  if(MySQLRS=nil)then
    Exit;

  if(previous_row_count=0)then
    MySQLRS.ResultSet:=result_set;

  for j:=0 to MySQLRS.FConnectedControls.Count-1 do
  begin
    if(TObject(IMySQLRSControlInterface(MySQLRS.FConnectedControls[j]).GetControl).InheritsFrom(TMySQLRSGrid))then
    begin
      RSGrid:=TMySQLRSGrid(IMySQLRSControlInterface(MySQLRS.FConnectedControls[j]).GetControl);

      //If this is the first block of rows, add columns
      if(previous_row_count=0)then
      begin
        RSGrid.MySQLRS.ResultSet:=result_set;

        RSGrid.BuildColumns;
        RSGrid.FocusedColumn:=1;
      end;

      //Add rows
      RSGrid.BuildRows(previous_row_count, current_row_count-1,
        (previous_row_count=0));

      MySQLRS.CheckAttributes;

      MySQLRS.StatusCaption:=
        Format(_('%d rows fetched so far.'), [current_row_count]);
    end;
  end;
end;

// -----------------------------------------------------------------------------

procedure TExecuteQueryThread.QueryExecuted;

begin
  if(Assigned(MySQLRS.OnQueryExecuted))then
    MySQLRS.OnQueryExecuted(MySQLRS);
end;

// -----------------------------------------------------------------------------

procedure TExecuteQueryThread.QuerySuccess;

var
  AddToHistory: Boolean;
  j: integer;
  RSGrid: TMySQLRSGrid;
  S: WideString;

begin
  if(MySQLRS.ResultSet<>nil)then
    for j:=0 to MySQLRS.FConnectedControls.Count-1 do
    begin
      if(TObject(IMySQLRSControlInterface(MySQLRS.FConnectedControls[j]).GetControl).InheritsFrom(TMySQLRSGrid))then
      begin
        RSGrid:=TMySQLRSGrid(IMySQLRSControlInterface(MySQLRS.FConnectedControls[j]).GetControl);

        RSGrid.BuildRowForAppending;
      end;
    end;


  AddToHistory:=False;
  if(MySQLRS.QueryHistory.Count>0)then
  begin
    if(MySQLRS.QueryHistory[MySQLRS.QueryHistory.Count-1]<>Query)then
      AddToHistory:=True;
  end
  else
    AddToHistory:=True;

  if(AddToHistory)then
  begin
    MySQLRS.QueryHistory.Add(Query);
    MySQLRS.CurrentHistoryEntry:=0;
  end;

  MySQLRS.CheckAttributes;

  //Check if this command has commited the open Trx (if there is any)
  MySQLRS.MySQLConn.CheckConnectionStatusChange(Query);

  if current_row_count <> 1 then
    S := 's'
  else
    S := '';

  if (MySQLRS.ResultSet<>nil)then
    MySQLRS.SetStatusCaption(
      Format(_('%d row%s fetched in %.4fs (%.4fs)'),
        [current_row_count, S,
        MySQLRS.ResultSet.fetch_time,
        MySQLRS.ResultSet.query_time]))
  else
    MySQLRS.SetStatusCaption('Query returned no resultset.');

  MySQLRS.StopQuery:=False;
  MySQLRS.QueryExecuting:=False;

  if(Assigned(MySQLRS.OnQuerySuccess))then
    MySQLRS.OnQuerySuccess(MySQLRS);
end;

procedure TExecuteQueryThread.QueryStopped;
begin
  MySQLRS.CheckAttributes;

  //Check if this command has commited the open Trx (if there is any)
  MySQLRS.MySQLConn.CheckConnectionStatusChange(Query);

  if MySQLRS.ResultSet<>nil then
    MySQLRS.SetStatusCaption(
      Format(_('Query aborted. %d rows fetched so far in %.4fs (%.4fs)'),
        [current_row_count,
        MySQLRS.ResultSet.fetch_time,
        MySQLRS.ResultSet.query_time]))
  else
    MySQLRS.SetStatusCaption(
      _('Query aborted.'));

  MySQLRS.StopQuery:=False;
  MySQLRS.QueryExecuting:=False;

  if(Assigned(MySQLRS.OnQueryStopped))then
    MySQLRS.OnQueryStopped(MySQLRS);
end;

procedure TExecuteQueryThread.QueryError;
begin
  MySQLRS.FreeResultSet;

  //Check if there actually was a MySQL Error
  if myx_mysql_errno(ThreadPMySQL)>0 then
  begin
    //Check if ExceptionMsg is set
    if(ExceptionMsg<>'')then
      MySQLRS.SetStatusCaption(ExceptionMsg)
    else
      MySQLRS.SetStatusCaption(
        _('The query could not be executed.'));

    MySQLRS.AddRSMessage(myx_mysql_error(ThreadPMySQL), RSMT_ERROR,
      myx_mysql_errno(ThreadPMySQL));
  end
  else
    MySQLRS.SetStatusCaption(_('No resultset returned.'));

  MySQLRS.StopQuery:=False;
  MySQLRS.QueryExecuting:=False;

  if(Assigned(MySQLRS.OnQueryError))then
    MySQLRS.OnQueryError(MySQLRS);
end;

// ---------------------------------------------------------------

constructor TRSError.Create(Msg: WideString; MsgType: RSMessageType;
  MsgNr: Integer);
begin
  self.Msg:=Msg;
  self.MsgType:=MsgType;
  self.MsgNr:=MsgNr;
end;

end.
