/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_layer.h 
 * @brief Implementation of the GC layer class.
 * 
 */

#ifndef __GC_LAYER_H__
#define __GC_LAYER_H__

#include "myx_gc_datatypes.h"
#include "myx_gc_canvas.h"

//----------------------------------------------------------------------------------------------------------------------

class CGenericCanvas;
class CFigureInstance;
class CFigure;
class CHitResults;

class GENERIC_CANVAS_API CLayer
{
  friend class CGenericCanvas;
  friend class CFigureInstance;
  friend class CFigureInstanceEnumerator;
private:
  GLuint FList;
  wstring FName;                  // The name of the layer.
  CGenericCanvas* FCanvas;        // The canvas to which this layer belongs.
  bool FDirty;                    // True if any of the properties changed that affect the display list.
  double FScaling[3];             // The factors to scale the layer in each direction.
  double FTranslation[3];         // The factors to move the layer.
  CFigureInstanceList FInstances; // A list of figures instances.
  int FUpdateCount;               // See CGenericCanvas.
  bool FVisible;                  // true if the layer is visible, otherwise false. Default is true.
  bool FEnabled;                  // True if the layer answers to mouse and keyboard actions.
protected:
  void ApplyTransformations();
  void CheckError(void);
  void MakeDirty(void);
  void RenderFeedback(CFigureInstance* Instance);
  virtual void RenderLayerContent(void);
  void Validate(void);
  virtual void ValidateLayerContent(void);
public:
  CLayer(CGenericCanvas* Owner);
  virtual ~CLayer(void);

  virtual void __cdecl AddInstance(CFigureInstance* Instance);
  virtual void __cdecl BeginUpdate(void);
  virtual void __cdecl Clear();
  virtual CFigureInstance* __cdecl CreateInstance(CFigure* Figure);
  virtual void __cdecl EndUpdate(void);
  virtual CGenericCanvas* __cdecl GetCanvas(void);
  virtual bool __cdecl GetVisible(void);
  virtual bool __cdecl IsUpdating(void);
  virtual void __cdecl Release(void);
  virtual void __cdecl RemoveInstance(CFigureInstance* Instance);
  virtual void __cdecl Render(void);
  virtual void __cdecl Translate(double Tx, double Ty, double Tz, bool Accumulative = false);
  virtual void __cdecl TranslateV(const double Factor[3], bool Accumulative = false);
  virtual void __cdecl Scale(double Sx, double Sy, double Sz, bool Accumulative = false);
  virtual void __cdecl ScaleV(const double Factor[3], bool Accumulative = false);
  virtual void __cdecl SetEnabled(bool IsEnabled);
  virtual void __cdecl SetVisible(bool IsVisible);
};

//----------------------------------------------------------------------------------------------------------------------

/**
 * The grid layer is a special layer variant that renders itself as grid.
 */
class GENERIC_CANVAS_API CGridLayer: public CLayer
{
private:
protected:
  virtual void RenderLayerContent(void);
public:
  CGridLayer(CGenericCanvas* Owner);
};

//----------------- Selection layer and associated structures ----------------------------------------------------------

typedef struct tagSelectionEntry
{
  CFigureInstance* Instance;
  bool Dirty;
  TBounds Bounds;
} TSelectionEntry;

typedef map<CFigureInstance*, TSelectionEntry*> CSelection;
typedef map<CFigureInstance*, TSelectionEntry*>::iterator CSelectionIterator;
typedef map<CFigureInstance*, TSelectionEntry*>::reverse_iterator CSelectionIteratorReverse;

/** Interal states of the selection layer. */
#define GC_SLSTATE_RUBBERBAND 0x0001

/**
 * The selection layer is a special layer variant that renders decorations for selected figures and can be queried
 * for quick hit tests and lists of selected figures.
 */
class GENERIC_CANVAS_API CFeedbackLayer: public CLayer
{
private:
  CSelection FSelection;          // The list of currently selected figure instances.
  float FHandleSize;              // The size of the decoration handles.
  unsigned int FStates;          // A number of flags holding transient states of the layer.
  TBounds FRubberbandBounds;      // The current bounds of the rubber band (if active).
  GLuint FRubberband;             // The display list for the rubber band (if active).
  GLuint FSelectionDecoration;    // The display list for the selection decoration.
protected:
  void CreateSelectionDecoration(void);
  bool InternalAddToSelection(CFigureInstance* Instance);
  void InternalRemoveFromSelection(CFigureInstance* Instance);
  virtual void RenderLayerContent(void);
  virtual void ValidateLayerContent(void);
public:
  CFeedbackLayer(CGenericCanvas* Owner);
  virtual ~CFeedbackLayer(void);

  virtual void __cdecl AddToSelection(CFigureInstance* Instance);
  virtual void __cdecl ClearSelection(void);
  virtual bool __cdecl GetProperty(TProperty Property, double& Value);
  virtual bool __cdecl GetProperty(TProperty Property, int& Value);
  virtual TGCDirection __cdecl GetSelectionInfo(double X, double Y);
  virtual void __cdecl InvalidateBounds(CFigureInstance* Instance);
  virtual void __cdecl RemoveFromSelection(CFigureInstance* Instance);
  virtual void __cdecl ResizeFiguresStart(int X, int Y, TGCDirection Direction);
  virtual void __cdecl ResizeFiguresStop(void);
  virtual void __cdecl ResizeFiguresTo(int X, int Y);
  virtual void __cdecl RubberbandResize(int X, int Y, TRBSelectionAction Action);
  virtual void __cdecl RubberbandStart(TRubberbandStyle Style, int X, int Y, bool RemoveSelection);
  virtual void __cdecl RubberbandStop(void);
  virtual bool __cdecl SetProperty(TProperty Property, double Value);
  virtual bool __cdecl SetProperty(TProperty Property, int Value);
};

//----------------------------------------------------------------------------------------------------------------------

#endif // __GC_LAYER_H__
