//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: xml.cpp,v 1.1.1.1 2003/10/29 10:04:59 wschweer Exp $
//
//  (C) Copyright 2000 Werner Schweer (ws@seh.de)
//=========================================================

#include <stdio.h>
#include <stdarg.h>
#include <qstring.h>

#include "xml.h"

//---------------------------------------------------------
//  Note:
//    this code is a Q/D hack for reading/parsing and
//    writing XML-Files
//    - can only handle the XML subset used by MusE
//    - may not handle misformed XML (eg. when manually
//      editing MusE output)

//---------------------------------------------------------
//   Xml
//---------------------------------------------------------

Xml::Xml(FILE* _f)
      {
      f          = _f;
      _line      = 0;
      _col       = 0;
      level      = 0;
      inTag      = false;
      lbuffer[0] = 0;
      bufptr     = lbuffer;
      }

Xml::Xml(const char* buf)
      {
      f      = 0;
      _line  = 0;
      _col   = 0;
      level  = 0;
      inTag  = false;
      bufptr = buf;
      }

//---------------------------------------------------------
//   next
//---------------------------------------------------------

void Xml::next()
      {
      if (*bufptr == 0) {
            if (f == 0 || fgets(lbuffer, 512, f) == 0) {
                  c = EOF;
                  return;
                  }
            bufptr = lbuffer;
            }
      c = *bufptr++;
      if (c == '\n') {
            ++_line;
            _col = -1;
            }
      ++_col;
      }

//---------------------------------------------------------
//   nextc
//---------------------------------------------------------

void Xml::nextc()
      {
      next();
      while (c == ' ' || c == '\t' || c == '\n')
            next();
      }

//---------------------------------------------------------
//   token
//---------------------------------------------------------

void Xml::token(int cc)
      {
      char buffer[512];
      char* p = buffer;
      for (;;) {
            if (c == ' ' || c == '\t' || c == cc || c == '\n' || c == EOF)
                  break;
            *p++ = c;
            next();
            }
      *p = 0;
      _s2 = buffer;
      }

//---------------------------------------------------------
//   stoken
//---------------------------------------------------------

void Xml::stoken()
      {
      char buffer[512];
      char* p = buffer;
      *p++ = c;
      next();
      for (;;) {
            if (c == '"') {
                  *p++ = c;
                  next();
                  break;
                  }
            if (c == '\n' || c == EOF)
                  break;
            *p++ = c;
            next();
            }
      *p = 0;
      _s2 = buffer;
      }

//---------------------------------------------------------
//   strip
//---------------------------------------------------------

QString Xml::strip(const QString& s)
      {
      int l = s.length();
      if (l >= 2 && s[0]=='"')
            return s.mid(1, l-2);
      return s;
      }

//---------------------------------------------------------
//   parse
//---------------------------------------------------------

Xml::Token Xml::parse()
      {
      char buffer[512];
      char* p;
      bool endFlag = false;
      nextc();

      if (c == EOF) {
            printf("unexpected EOF reading *.med file at level %d, line %d\n",
               level, _line);
            return level == 0 ? End : Error;
            }

      _s1 = QString("");
      if (inTag) {
            //-------------------
            // parse Attributes
            //-------------------
            if (c == '/') {
                  nextc();
                  token('>');
                  if (c != '>') {
                        printf("Xml: unexpected char '%c', expected '>'\n", c);
                        goto error;
                        }
                  _s1   = _tag;
                  inTag = false;
                  --level;
                  return TagEnd;
                  }
            _s2 = QString("");
            token('=');
            _s1 = _s2;
            nextc();      // skip space
            if (c == '"')
                  stoken();
            else
                  token('>');
            if (c == '>')
                  inTag = false;
            else
                  --bufptr;
            _s2 = strip(_s2);
            return Attribut;
            }
      if (c == '<') {
            //--------------
            // parse Tag
            //--------------
            next();
            if (c == '/') {
                  endFlag = true;
                  next();
                  }
            if (c == '?') {
                  next();
                  p = buffer;
                  for (;;) {
                        if (c == '?' || c == EOF || c == '>')
                              break;
                        *p++ = c;
                        // TODO: check overflow
                        next();
                        }
                  *p = 0;
                  _s1 = QString(buffer);
                  if (c == EOF) {
                        fprintf(stderr, "XML: unexpected EOF\n");
                        goto error;
                        }
                  nextc();
                  if (c != '>') {
                        fprintf(stderr, "XML: '>' expected\n");
                        goto error;
                        }
                  next();
                  return Proc;
                  }
            p = buffer;
            for (;;) {
                  if (c == '/' || c == ' ' || c == '\t' || c == '>' || c == '\n' || c == EOF)
                        break;
                  // TODO: check overflow
                  *p++ = c;
                  next();
                  }
            *p = 0;
            _s1 = QString(buffer);
            while (c == ' ' || c == '\t' || c == '\n')
                  next();
            if (c == '/') {
                  nextc();
                  if (c == '>')
                        return Flag;
                  fprintf(stderr, "XML: '>' expected\n");
                  goto error;
                  }
            if (c == '?') {
                  nextc();
                  if (c == '>')
                        return Proc;
                  fprintf(stderr, "XML: '>' expected\n");
                  goto error;
                  }
            if (c == '>') {
                  if (endFlag) {
                        --level;
// printf("xml </%s>\n", _s1.latin1());
                        return TagEnd;
                        }
                  else {
                        ++level;
// printf("xml <%s>\n", _s1.latin1());
                        return TagStart;
                        }
                  }
            else {
                  _tag = _s1;
                  --bufptr;
                  inTag = true;
                  ++level;
                  if (!endFlag) {
// printf("xml <%s>\n", _s1.latin1());
                        return TagStart;
                        }
                  fprintf(stderr, "XML: endFlag expected\n");
                  goto error;
                  }
            }
      else {
            //--------------
            // parse Text
            //--------------
            if (level == 0) {
                  fprintf(stderr, "XML: level = 0\n");
                  goto error;
                  }
            p = buffer;
            for (;;) {
                  if (c == EOF || c == '<')
                        break;
                  if (c == '&') {
                        next();
                        if (c == '<') {         // be tolerant with old muse files
                              *p++ = '&';
                              continue;
                              }
                        char name[32];
                        char* dp = name;
                        *dp++ = c;
                        for (; dp-name < 31;) {
                              next();
                              if (c == ';')
                                    break;
                              *dp++ = c;
                              }
                        *dp = 0;
                        if (strcmp(name, "lt") == 0)
                              c = '<';
                        else if (strcmp(name, "amp") == 0)
                              c = '&';
                        else
                              c = '?';
                        }
                  *p++ = c;
                  next();
                  }
            *p = 0;
            _s1 = QString(buffer);

            if (c == '<')
                  --bufptr;
            return Text;
            }
error:
      fprintf(stderr, "XML Parse Error at line %d col %d\n", _line, _col+1);
      return Error;
      }

//---------------------------------------------------------
//   parse(QString)
//---------------------------------------------------------

QString Xml::parse(const QString& tag)
      {
      QString a;

      for (;;) {
            switch (parse()) {
                  case Error:
                  case End:
                        return a;
                  default:
                  case TagStart:
                  case Attribut:
                        break;
                  case Text:
                        a = _s1;
                        break;
                  case TagEnd:
                        if (_s1 == tag)
                              return a;
                        break;
                  }
            }
      return a;
      }

//---------------------------------------------------------
//   parse1
//---------------------------------------------------------

QString Xml::parse1()
      {
      return parse(_s1.simplifyWhiteSpace());
      }

//---------------------------------------------------------
//   parseInt
//---------------------------------------------------------

int Xml::parseInt()
      {
      return parse(_s1.simplifyWhiteSpace()).simplifyWhiteSpace().toInt();
      }

//---------------------------------------------------------
//   parseDouble
//---------------------------------------------------------

double Xml::parseDouble()
      {
      return parse(_s1.simplifyWhiteSpace()).simplifyWhiteSpace().toDouble();
      }

//---------------------------------------------------------
//   unknown
//---------------------------------------------------------

void Xml::unknown(const char* s)
      {
      printf("%s: unknown tag <%s> at line %d\n",
         s, _s1.latin1(), _line+1);
      parse1();
      }

//---------------------------------------------------------
//   header
//---------------------------------------------------------

void Xml::header()
      {
      fprintf(f, "<?xml version=\"1.0\"?>\n");
      }

//---------------------------------------------------------
//   put
//---------------------------------------------------------

void Xml::put(const char* format, ...)
      {
      va_list args;
      va_start(args, format);

      vfprintf(f, format, args);
      va_end(args);
      putc('\n', f);
      }

void Xml::put(int level, const char* format, ...)
      {
      va_list args;
      va_start(args, format);
      putLevel(level);
      vfprintf(f, format, args);
      va_end(args);
      putc('\n', f);
      }

//---------------------------------------------------------
//   nput
//---------------------------------------------------------

void Xml::nput(int level, const char* format, ...)
      {
      va_list args;
      va_start(args, format);
      putLevel(level);
      vfprintf(f, format, args);
      va_end(args);
      }

void Xml::nput(const char* format, ...)
      {
      va_list args;
      va_start(args, format);
      vfprintf(f, format, args);
      va_end(args);
      }

//---------------------------------------------------------
//   tag
//---------------------------------------------------------

void Xml::tag(int level, const char* format, ...)
      {
      va_list args;
      va_start(args, format);
      putLevel(level);
      putc('<', f);
      vfprintf(f, format, args);
      va_end(args);
      putc('>', f);
      putc('\n', f);
      }

//---------------------------------------------------------
//   etag
//---------------------------------------------------------

void Xml::etag(int level, const char* format, ...)
      {
      va_list args;
      va_start(args, format);
      putLevel(level);
      putc('<', f);
      putc('/', f);
      vfprintf(f, format, args);
      va_end(args);
      putc('>', f);
      putc('\n', f);
      }

void Xml::putLevel(int n)
      {
      for (int i = 0; i < n*2; ++i)
            putc(' ', f);
      }
void Xml::intTag(int level, const char* name, int val)
      {
      putLevel(level);
      fprintf(f, "<%s>%d</%s>\n", name, val, name);
      }
void Xml::doubleTag(int level, const char* name, double val)
      {
      putLevel(level);
      fprintf(f, "<%s>%f</%s>\n", name, val, name);
      }
void Xml::strTag(int level, const char* name, const char* val)
      {
      putLevel(level);
      fprintf(f, "<%s>", name);
      if (val) {
            while (*val) {
                  switch(*val) {
                        case '&': fprintf(f, "&amp;"); break;
                        case '<': fprintf(f, "&lt;"); break;
                        default: fputc(*val, f); break;
                        }
                  ++val;
                  }
            }
      fprintf(f, "</%s>\n", name);
      }
void Xml::strTag(int level, const char* name, const QString& val)
      {
      strTag(level, name, val.latin1());
      }

#ifdef MAIN
//---------------------------------------------------------
//   main
//---------------------------------------------------------

int main(int argc, const char* argv[])
      {
      Xml xml(stdin);
      Xml::Token token;

      for (;;) {
            token = xml.parse();
            if (token == Xml::Error || token == Xml::End)
                  break;
            switch (token) {
                  case Xml::TagStart:
                        printf("tag <%s>\n", xml.s1().latin1());
                        break;
                  case Xml::Flag:
                        printf("flag <%s>\n", xml.s1().latin1());
                        break;
                  case Xml::TagEnd:
                        printf("end tag <%s>\n", xml.s1().latin1());
                        break;
                  case Xml::Text:
                        printf("text <%s>\n", xml.s1().latin1());
                        break;
                  case Xml::Proc:
                        printf("procedure <%s>\n", xml.s1().latin1());
                        break;
                  case Xml::Attribut:
                        printf("attribut <%s>=<%s>\n",
                          xml.s1().latin1(), xml.s2().latin1());
                        break;
                  default:
                        printf("parse ???\n");
                        break;
                  }
            }
      if (token == Xml::Error) {
            printf("error line %d col %d\n", xml.line()+1, xml.col()+1);
            return -1;
            }
      return 0;
      }
#endif

