(* Copyright (C) 2002 Benjamin Monate <Benjamin.Monate@lri.fr>.  
   All rights reserved.
   
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License version 2, as published by the Free Software Foundation.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
   
   See the GNU Library General Public License version 2 for more details
   (enclosed in the file LGPL).

This product includes software developed by Christian Lindig. 
See LICENCE file for more informations.

*)

open Error			(* error handling *)
open Xml			(* XML abstract syntax *)


(*
 * [read]
 * parses an input channel [in_chan] and returns the semantic value.
 *)

let read in_chan =
    let position (line,col) = "arround line "
                            ^ (string_of_int line)
                            ^ ", column "
                            ^ (string_of_int col) in
    let lexbuf              = Lexing.from_channel in_chan in
    try
      Xmlscan.init();                           (* reset line counter *)
      Xmlparse.document Xmlscan.scan lexbuf     (* start parsing      *)
    with exn ->
        let pos = position (Xmlscan.position lexbuf) in
        begin match exn with
          | Parsing.Parse_error | Error _ -> 
	      error ("invalid glade file: not a parsable XML file.
Please report.")
          | Sys_error msg         -> error ("I/O error: " ^ msg)
          | _                     -> raise exn
        end


let usage = "mlglade filename: generates Ocaml modules for the user interface
\t\t  described in file filename with glade format."

 
let version () =
    Printf.printf 
    	"%s version %s (c) 2002 Benjamin Monate <monate@lri.fr>\n" 
	This.name 
	This.version;
  exit 0

let arg_spec =
[
  ("-v", 
  Arg.Unit version,
  ": prints the version of mlglade");
  ("-quiet", 
  Arg.Unit (function () -> quiet := true),
  ": be very quiet");
  ("-verbose", 
  Arg.Int (function i -> verbose:= i),
  ": sets the verbosity level of mlglade");
]

let filename = ref None

let get_file_name s = filename := Some s

let main () =
  Arg.parse arg_spec get_file_name usage ;
  match !filename with
    | None -> prerr_endline ("Fatal error: missing argument\n"^usage); exit 1
    | Some f ->   
	try 
	  let inc  = open_in f in
	  let doc       = read inc  in
	    Glade.process_doc f stdout doc;
	    close_in inc
	with 
	  | Error s -> prerr_endline ("Fatal error: "^s); exit 1
          | Sys_error msg         ->  prerr_endline ("Fatal I/O error: " ^ msg)
  
let _ = main ()
