/*
 * type.c
 *
 * Copyright 1999 Michael Elizabeth Chastain, <mailto:mec@shout.net>.
 * Licensed under the Gnu Public License, Version 2.
 *
 * Data types.
 */

#include <ctype.h>
#include <string.h>

#include "mconfig.h"



/*
 * Return whether a type allows a given piece or string.
 */

int 
type_allow_string(const type_type type, const char *s)
{
	piece_type      piece;

	if (s == NULL)
		error_pointer_null();

	piece.ptr = s;
	piece.len = strlen(s);

	return type_allow_piece(type, piece);
}

int 
type_allow_piece(const type_type type, const piece_type piece)
{
	int             index;

	if (piece.len > 0 && piece.ptr == NULL)
		error_pointer_null();

	switch (type) {
	case type_unset:
		error_enum_bogus();
		return 0;

	case type_bool:
		if (piece.len == 1 && piece.ptr[0] == 'n')
			return 1;
		if (piece.len == 1 && piece.ptr[0] == 'y')
			return 1;
		return 0;

	case type_hex:
		for (index = 0; index < piece.len; ++index) {
			if (!isxdigit(piece.ptr[index]))
				return 0;
		}
		return 1;

	case type_int:
		for (index = 0; index < piece.len; ++index) {
			if (index == 0) {
				if (piece.ptr[index] == '-' || piece.ptr[index] == '+') {
					if (piece.len > 1)
						continue;
					else
						return 0;
				}
			}
			if (!isdigit(piece.ptr[index]))
				return 0;
		}
		return 1;

	case type_nchoice_master:
		/* never set master directly (so far) */
		return 0;

	case type_nchoice_slave:
		if (piece.len == 1 && piece.ptr[0] == 'n')
			return 1;
		if (piece.len == 1 && piece.ptr[0] == 'y')
			return 1;
		return 0;

	case type_string:
		for (index = 0; index < piece.len; ++index) {
			if (!isprint(piece.ptr[index]))
				return 0;
			if (piece.ptr[index] == '"' || piece.ptr[index] == '\\')
				return 0;
		}

	case type_tristate:
		if (piece.len == 1 && piece.ptr[0] == 'n')
			return 1;
		if (piece.len == 1 && piece.ptr[0] == 'm')
			return 1;
		if (piece.len == 1 && piece.ptr[0] == 'y')
			return 1;
		return 0;
	}

	error_enum_bogus();
	return 0;
}



/*
 * Return the default value for a type.
 */

piece_type 
type_default_value(type_type type)
{
	switch (type) {
	case type_unset:
		error_enum_bogus();
		return piece_empty;
	case type_bool:
		return piece_n;
	case type_hex:
		return piece_0;
	case type_int:
		return piece_0;
	case type_nchoice_master:
		error_enum_bogus();
		return piece_empty;
	case type_nchoice_slave:
		return piece_n;
	case type_string:
		return piece_empty;
	case type_tristate:
		return piece_n;
	}

	error_enum_bogus();
	return piece_empty;
}
