/* Copyright (C) 1995 Bjoern Beutel. */

/* Description. =============================================================*/

/* Invoke the display process from Malaga. */

/* Includes. ================================================================*/

#define _XOPEN_SOURCE
#define _XOPEN_SOURCE_EXTENDED 1
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <setjmp.h>
#include "basic.h"
#include "files.h"
#include "input.h"
#include "commands.h"
#include "display.h"
#ifdef UNIX
#include <signal.h>
#include <unistd.h>
#include <pwd.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <sys/types.h>
#endif
#ifdef WINDOWS
#include <windows.h>
#include <fcntl.h>
#include <io.h>
#endif

/* Global variables. ========================================================*/

FILE *display_stream; /* Stream to send data to the display process. */
string_t char_set; /* Character set to use for display process. */

/* Variables. ===============================================================*/

#ifdef UNIX
static pid_t process_id; /* ID of display process. */
#endif
#ifdef WINDOWS
static HANDLE process_handle; /* Handle of display process. */
#endif
static string_t display_command_line; /* Command line for display output. */

/* Functions. ===============================================================*/

void 
stop_display_process( void )
/* Stop the Malaga display process. */
{ 
#ifdef UNIX
  close_stream( &display_stream, NULL );
  free_mem( &display_command_line );
  if (process_id != 0) 
  { 
    kill( process_id, SIGTERM );
    waitpid( process_id, NULL, 0 );
    process_id = 0;
  }
#endif

#ifdef WINDOWS
  close_stream( &display_stream, NULL );
  free_mem( &display_command_line );
  if (process_handle != NULL) 
  { 
    TerminateProcess( process_handle, 0 );
    CloseHandle( process_handle );
    process_handle = NULL;
  }
#endif
}

/*---------------------------------------------------------------------------*/

void 
start_display_process( void )
/* Start the Malaga display process by executing DISPLAY_COMMAND_LINE
 * if it is not already running. */
{ 
#ifdef UNIX
  int pipe_fd[2];
  string_t arguments, argument;
  string_t *args;
  int_t arg_count, i;

  if (process_id != 0) 
  { 
    if (waitpid( process_id, NULL, WNOHANG ) == 0) 
      return;
    process_id = 0;
    close_stream( &display_stream, NULL );
  }

  if (display_command_line == NULL) 
    display_command_line = new_string( "malshow", NULL );
  if (pipe( pipe_fd ) == -1) 
    complain( "Can't create pipe to display process: %s.", strerror( errno ) );

  signal( SIGPIPE, SIG_IGN );
  switch (process_id = fork()) 
  {
  case -1:
    complain( "Can't create display process: %s.", strerror( errno ) );
    break;
  case 0:
    dup2( pipe_fd[0], STDIN_FILENO );
    close( pipe_fd[0] );
    close( pipe_fd[1] );
    signal( SIGINT, SIG_IGN );

    /* Count arguments. */
    arg_count = 0;
    arguments = display_command_line;
    while (*arguments != EOS) 
    { 
      argument = parse_word( &arguments );
      free_mem( &argument );
      arg_count++;
    }

    /* Create argument vector. */
    args = new_vector( sizeof( string_t ), arg_count + 1 );
    arguments = display_command_line;
    for (i = 0; i < arg_count; i++) 
      args[i] = parse_word( &arguments );
    args[i] = NULL;

    execvp( args[0], args ); /* Start display program. */
    fprintf( stderr, "Can't start display process \"%s\": %s.\n", 
             args[0], strerror( errno ) );
    exit(1);
  default:
    close( pipe_fd[0] );
    display_stream = fdopen( pipe_fd[1], "w" );
    if (display_stream == NULL) 
      complain( "Can't open data stream: %s.", strerror( errno ) );
    fprintf( display_stream, "%s\n", char_set );
    break;
  }
#endif

#ifdef WINDOWS
  HANDLE child_read, parent_write, parent_write_local, old_stdin;
  SECURITY_ATTRIBUTES security;
  STARTUPINFO startup_info;
  PROCESS_INFORMATION process_info;
  DWORD status;
  int fd;

  /* Check if the process is still running. */
  if (process_handle != NULL) 
  { 
    if (GetExitCodeProcess( process_handle, &status ) 
	&& status == STILL_ACTIVE)
    {
      return;
    }
    CloseHandle( process_handle );
    process_handle = NULL;
    close_stream( &display_stream, NULL );
  }

  /* Prepare the command line. */
  if (display_command_line == NULL) 
    display_command_line = new_string( "malshow", NULL );

  /* Create a pipe with inherited handles. */
  security.nLength = sizeof( SECURITY_ATTRIBUTES );
  security.bInheritHandle = TRUE;
  security.lpSecurityDescriptor = NULL;
  if (! CreatePipe( &child_read, &parent_write, &security, 0 ) ) 
    complain( "Can't create pipe to display process." );

  /* Change STDIN for the child process. */
  old_stdin = GetStdHandle( STD_INPUT_HANDLE );
  SetStdHandle( STD_INPUT_HANDLE, child_read );

  /* Make sure the other handle is not inherited. */
  DuplicateHandle( GetCurrentProcess(), parent_write, 
		   GetCurrentProcess(), &parent_write_local,
		   0, FALSE, DUPLICATE_SAME_ACCESS );
  CloseHandle( parent_write );

  /* Start the display process as child. */
  ZeroMemory( &startup_info, sizeof( startup_info ) );
  startup_info.cb = sizeof( startup_info );
  if (! CreateProcess( NULL, display_command_line, NULL, NULL, TRUE, 
		       CREATE_NEW_PROCESS_GROUP, 
		       NULL, NULL, &startup_info, &process_info ))
  {
    complain( "Can't create the display process." );
  }
  process_handle = process_info.hProcess;
  CloseHandle( process_info.hThread );

  /* Reset STDIN. */
  SetStdHandle( STD_INPUT_HANDLE, old_stdin );
  CloseHandle( child_read );

  /* Open pipe to send. */
  fd = _open_osfhandle( (long) parent_write_local, O_TEXT );
  display_stream = _fdopen( fd, "w" );
  if (display_stream == NULL) 
    complain( "Can't open data stream: %s.", strerror( errno ) );

  fprintf( display_stream, "%s\n", char_set );
#endif
}

/*---------------------------------------------------------------------------*/

static void 
do_display_line_option( string_t arguments )
/* Set the command line to start the display process. */
{ 
  if (*arguments == EOS) 
  { 
    printf( "display-line: \"%s\"\n", 
            (display_command_line == NULL) ? "" : display_command_line );
  } 
  else 
  { 
    stop_display_process();
    display_command_line = parse_word( &arguments );
  }
  parse_end( &arguments );
}

command_t display_line_option = 
{ 
  "display-line display", do_display_line_option,
  "Usage: set display-line \"DISPLAY_COMMAND_LINE\"\n"
  "Set the command line that is used to start the display process.\n"
};

/* End of file. =============================================================*/
