/* 
**  mod_index_rss.c 
*/ 

#include "httpd.h"
#include "http_config.h"
#include "http_protocol.h"
#include "http_log.h"
#include "fnmatch.h"
#include "ap_config.h"

#define TIME_FORMAT "%a %b %d %H:%M:%S %Y"
/* This is the XML header file */
#define HEADER "<?xml version=\"1.0\" encoding=\"UTF-8\"?>

<!DOCTYPE rss PUBLIC \"-//Netscape Communications//DTD RSS 0.91//EN\"
            \"http://www.scripting.com/dtd/rss-0_91.dtd\">

<rss version=\"0.91\">

<channel>
"

#define FOOTER "</channel>
</rss>
"


#define WATCHPOINT printf("WATCHPOINT %s %d\n", __FILE__, __LINE__);

module MODULE_VAR_EXPORT index_rss_module;

typedef struct {
	int state;
	array_header *enabled;
	array_header *disabled;
} index_rss_conf;

typedef struct {
	time_t time;
	char *name;
} Container;

static void * mconfig_for_directory(pool *p, char *dir) {
	index_rss_conf *cfg = ap_pcalloc (p, sizeof (index_rss_conf));

	cfg->state = 1;
	cfg->enabled = NULL;
	cfg->disabled = NULL;

	return (void *) cfg;
}	

void print_channel(request_rec *r) {
	char *time_string = NULL;

	time_string = ap_pstrdup(r->pool, ap_ht_time(r->pool, r->finfo.st_mtime, TIME_FORMAT, 0));
	ap_rprintf(r, "<title>%s:%s</title>\n", r->hostname, r->uri);
	ap_rprintf(r, "<description>mod_index_rss channel maker</description>\n");
	ap_rprintf(r, "<pubDate>%s</pubDate>\n", time_string);
	ap_rprintf(r, "<lastBuildDate>%s</lastBuildDate>\n", time_string);
	ap_rprintf(r, "<webMaster>%s</webMaster>\n", r->server->server_admin);
	ap_rprintf(r, "<link>http://%s%s</link>\n\n", r->hostname, r->uri);
}

static int sort_container(Container *file1, Container *file2) {
	if (file1->time == file2->time)
		return 0;

	if (file1->time < file2->time)
		return -1;

	return 1;
}

static string_find(array_header *array, const char *key) {
	int x = 0;
	char **strings = NULL;

	if(!array) 
		return 0;

	if(!key) 
		return 0;

	strings = (char **) array->elts;

	for (x = 0; x < array->nelts; ++x) {
		if (!ap_fnmatch(strings[x], key, FNM_CASE_BLIND)) {
			return 1;
		}
	}

	return 0;
}

static int index_rss_fixup(request_rec *r) {
	if(!r->content_type || !r->args)
		return DECLINED;


	if(!strcmp(r->content_type, "httpd/unix-directory")) {
		if(!strcmp(r->args, "index=rss")) {
			r->handler = "index_rss";
		}
	}

	return DECLINED;
}

static int index_rss_handler(request_rec *r) {
	DIR *directory = NULL;
	struct DIR_TYPE *file = NULL;
	struct stat sbuf;
	char *filename = NULL;
	Container *container = NULL;
	Container **containers = NULL;
	array_header *containers_array = NULL;
	int x = 0;
	int state = 0;
	index_rss_conf *cfg = ap_get_module_config (r->per_dir_config, &index_rss_module);

	containers_array = ap_make_array(r->pool, 15, sizeof (Container *));

	if (!(directory = ap_popendir(r->pool, r->filename))) {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, r,
			"Can't open directory for index: %s", r->filename);
		return HTTP_FORBIDDEN;
	}

	r->content_type = "text/xml";      
	ap_send_http_header(r);

	if (r->header_only) {
		ap_pclosedir(r->pool, directory);
		return OK;
	}

	ap_rputs(HEADER, r);
	print_channel(r);
	while(file = readdir(directory)) {
		state = 0;
		if(stat(filename = ap_pstrcat(r->pool, r->filename, "/", file->d_name, NULL), &sbuf) == 0){
			if(S_ISREG(sbuf.st_mode)) {
				if(cfg->enabled || cfg->disabled) {
					/* If enabled is used, we default to trusting nothing */
					if(cfg->enabled)
						state++;

					if (string_find(cfg->enabled, file->d_name))
						state = 0;

					if (string_find(cfg->disabled, file->d_name))
						state++;
				}

				if(!state) {
					container = ap_pcalloc(r->pool, sizeof(Container));
					container->time = sbuf.st_mtime;
					container->name = ap_pstrdup (r->pool, file->d_name);
					*(Container **) ap_push_array (containers_array) = (Container *)container;
				}
			}
		}
	}
	containers = (Container **) containers_array->elts;

	qsort(containers, containers_array->nelts, sizeof(Container *),
			(int (*)(const void *, const void *)) sort_container);

	for(x = 0; x < containers_array->nelts; x++) {
		ap_rputs("<item>\n", r);
		ap_rprintf(r, "<title>%s</title>\n", ap_escape_html(r->pool, containers[x]->name));
		ap_rprintf(r, "<link>http://%s%s%s</link>\n", r->hostname, r->uri, ap_escape_html(r->pool, containers[x]->name));
		ap_rputs("</item>\n", r);
		ap_rputs("\n", r);
	}

	/* So now we have a bunch of stuff in memory, lets sort */
	ap_rputs(FOOTER, r);

	return OK;
}

static const char * add_enabled(cmd_parms * cmd, void *mconfig, char *string) {
	FILE *file_ptr;
	index_rss_conf *cfg = (index_rss_conf *) mconfig;

	if(cfg->enabled == NULL)
		cfg->enabled = ap_make_array (cmd->pool, 1, sizeof (char *));

	*(char **) ap_push_array (cfg->enabled) = ap_pstrdup (cmd->pool, string);

	return NULL;
}

static const char * add_disabled(cmd_parms * cmd, void *mconfig, char *string) {
	FILE *file_ptr;
	index_rss_conf *cfg = (index_rss_conf *) mconfig;

	if(cfg->disabled == NULL)
		cfg->disabled = ap_make_array (cmd->pool, 1, sizeof (char *));

	*(char **) ap_push_array (cfg->disabled) = ap_pstrdup (cmd->pool, string);

	return NULL;
}

/* Dispatch list of content handlers */
static const handler_rec index_rss_handlers[] = { 
    { "index_rss", index_rss_handler }, 
    { NULL, NULL }
};

static const command_rec index_rss_module_cmds[] = {
	{"IndexRSSEngine", ap_set_flag_slot, 
		(void *) XtOffsetOf(index_rss_conf, state), OR_ALL, FLAG,
		"Use this to turn on and off RSS indexes (OFF by default)."},
	{"IndexRSSEnabled", add_enabled, NULL, OR_ALL, TAKE1,
		"A regular expression to compare against files (happens first)."},
	{"IndexRSSDisabled", add_disabled, NULL, OR_ALL, TAKE1,
		"A regular expression to compare against files (happens after enabled)."},
	{NULL},
};

/* Dispatch list for API hooks */
module MODULE_VAR_EXPORT index_rss_module = {
    STANDARD_MODULE_STUFF, 
    NULL,                  /* module initializer                  */
    mconfig_for_directory, /* create per-dir    config structures */
    NULL,                  /* merge  per-dir    config structures */
    NULL,                  /* create per-server config structures */
    NULL,                  /* merge  per-server config structures */
    index_rss_module_cmds, /* table of config file commands       */
    index_rss_handlers,    /* [#8] MIME-typed-dispatched handlers */
    NULL,                  /* [#1] URI to filename translation    */
    NULL,                  /* [#4] validate user id from request  */
    NULL,                  /* [#5] check if the user is ok _here_ */
    NULL,                  /* [#3] check access by host address   */
    NULL,                  /* [#6] determine MIME type            */
    index_rss_fixup,       /* [#7] pre-run fixups                 */
    NULL,                  /* [#9] log a transaction              */
    NULL,                  /* [#2] header parser                  */
    NULL,                  /* child_init                          */
    NULL,                  /* child_exit                          */
    NULL                   /* [#0] post read-request              */
};

