/* ======================================================================
 * Copyright (c) 1998-1999 The Johns Hopkins University.
 * All rights reserved.
 * The following code was written by Theo Schlossnagle for use in the
 * Backhand project at The Center for Networking and Distributed Systems
 * at The Johns Hopkins University.
 * Please refer to the LICENSE file before using this software.
 * ======================================================================
*/

/* NOTE:: Some of this is taken from Stevens' Advanced
   Programming in the Unix Evironment */
#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/uio.h>
#include <sys/stat.h>
#include <sys/un.h>
#include <time.h>
#include <unistd.h>
#include <stddef.h>
#include <errno.h>

#include "httpd.h"
#include "http_log.h"

#define	STALE	30
#define SERVPATH "/var/tmp/bparent"
#define	CLI_PATH "/var/tmp/bchild"	/* +5 for pid = 20 chars */
#define	CLI_PERM S_IRWXU		/* rwx for user only */

#define err_sys(A)  fprintf(stderr, A)
#define err_dump(A) err_sys(A)
#define err_ret(A)  err_sys(A)
#ifndef MAXLINE
#define MAXLINE 4096
#endif

#ifdef LINUX
#define _BSDISH
#endif
#ifdef __FreeBSD__
#define _BSDISH
#endif
#ifdef __OpenBSD__
#define _BSDISH
#endif
#ifdef BSD
#define _BSDISH
#endif
#ifdef DARWIN
#define _BSDISH
#endif
#ifdef SOLARIS2
#define _SYSVISH
#endif

#ifdef _SYSVISH
#include <sys/fcntl.h>
#include <stropts.h>
int recv_fd(int servfd) {
  int newfd, nread, flag, status;
  char *ptr, buf[100];
  struct strbuf dat;
  struct strrecvfd recvfd;
  
  status = -1;
  for( ; ; ) {
    dat.buf = buf;
    dat.maxlen = MAXLINE;
    flag = 0;
    if (getmsg(servfd, NULL, &dat, &flag) < 0)
      err_sys("getmsg error");
    nread = dat.len;
    if(nread == 0) {
      err_ret("connection closed by server");
      return(-1);
    }
    for(ptr = buf; ptr < &buf[nread]; ) {
      if(*ptr++ == 0) {
	if(ptr != &buf[nread-1])
	  err_dump("message format error");
	status = *ptr & 255;
	if(status == 0) {
	  if(ioctl(servfd, I_RECVFD, &recvfd) < 0)
	    return(-1);
	  newfd = recvfd.fd;
	} else
	  newfd = -status;
	nread -= 2;
      }
    }
    if(status >= 0)
      return(newfd);
  }
}

int send_fd(int clifd, int fd) {
  char buf[2];
  buf[0] = 0;
  if(fd < 0) {
    buf[1] = -fd;
    if(buf[1] == 0)
      buf[1] = 1;
  } else {
    buf[1] = 0;
  }
  if(write(clifd, buf, 2) != 2)
    return(-1);
  if((fd >= 0) &&
     (ioctl(clifd, I_SENDFD, fd) < 0))
    return(-1);
  return(0);
}
int serv_accept(int listenfd, pid_t *pidptr) {
  struct strrecvfd recvfd;
  
  if(ioctl(listenfd, I_RECVFD, &recvfd) < 0)
    return(-1);
  /* We need th PID not the UID?  How do we get that? */
  if(pidptr != NULL)
    *pidptr = (pid_t)recvfd.uid;
  return(recvfd.fd);
}
#define FIFO_MODE  (S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP)
int serv_listen(const char *name) {
  int tempfd, fd[2], len;
  
  unlink(name);
  if((tempfd = creat(name, FIFO_MODE)) < 0)
    return(-1);
  if (close(tempfd) < 0)
    return(-2);
  
  if(pipe(fd) < 0)
    return(-3);
  if(ioctl(fd[1], I_PUSH, "connld") < 0)
    return(-4);
  if(fattach(fd[1], name) < 0)
    return(-5);

  return(fd[0]);
}
int cli_conn(const char *name) {
  int fd;

  if((fd = open(name, O_RDWR)) < 0)
    return(-1);
  if(isastream(fd) == 0)
    return(-2);
  return(fd);
}

#else
#ifdef _BSDISH
static struct cmsghdr	*cmptr = NULL;
#define	CONTROLLEN	(sizeof(struct cmsghdr) + sizeof(int))

int recv_fd(int servfd) {
  int newfd, nread, status;
  char *ptr, buf[100];
  struct iovec	iov[1];
  struct msghdr	msg;
  
  status = -1;
  for( ; ; ) {
    iov[0].iov_base = buf;
    iov[0].iov_len  = sizeof(buf);
    msg.msg_iov     = iov;
    msg.msg_iovlen  = 1;
    msg.msg_name    = NULL;
    msg.msg_namelen = 0;
    if(cmptr == NULL &&
       (cmptr = (struct cmsghdr *)malloc(CONTROLLEN)) == NULL)
      return(-1);
    msg.msg_control    = (caddr_t) cmptr;
    msg.msg_controllen = CONTROLLEN;
    if((nread = recvmsg(servfd, &msg, 0)) < 0)
      err_sys("recvmsg error");
    else if(nread == 0) {
      err_ret("connection closed by server");
      return(-1);
    }
    for(ptr = buf; ptr < &buf[nread]; ) {
      if(*ptr++ == 0) {
        if(ptr != &buf[nread-1])
          err_dump("message format error");
        status = *ptr & 255;
        if(status == 0) {
          if(msg.msg_controllen != CONTROLLEN)
            err_dump("status = 0 but no fd");
          newfd = *(int *)CMSG_DATA(cmptr);
        } else
          newfd = -status;
        nread -= 2;
      }
    }
    if(status >= 0)
      return(newfd);
  }
}

int send_fd(int clifd, int fd) {
  struct iovec iov[1];
  struct msghdr	msg;
  char buf[2];
  
  iov[0].iov_base = buf;
  iov[0].iov_len  = 2;
  msg.msg_iov     = iov;
  msg.msg_iovlen  = 1;
  msg.msg_name    = NULL;
  msg.msg_namelen = 0;
  if(fd < 0) {
    msg.msg_control    = NULL;
    msg.msg_controllen = 0;
    buf[1] = -fd;
    if(buf[1] == 0)
      buf[1] = 1;
  } else {
    if(cmptr == NULL &&
       (cmptr = (struct cmsghdr *)malloc(CONTROLLEN)) == NULL)
      return(-1);
    cmptr->cmsg_level  = SOL_SOCKET;
    cmptr->cmsg_type   = SCM_RIGHTS;
    cmptr->cmsg_len    = CONTROLLEN;
    msg.msg_control    = (caddr_t) cmptr;
    msg.msg_controllen = CONTROLLEN;
    *(int *)CMSG_DATA(cmptr) = fd;
    buf[1] = 0;
  }
  buf[0] = 0;
  if(sendmsg(clifd, &msg, 0) != 2)
    return(-1);
  return(0);
}

int serv_accept(int listenfd, pid_t *pidptr) {
  int clifd;
  unsigned int len;
  time_t staletime;
  struct sockaddr_un unix_addr;
  struct stat statbuf;
  char *cp;
  
  len = sizeof(unix_addr);
  if((clifd = accept(listenfd, (struct sockaddr *) &unix_addr, &len)) < 0)
    return(-1);
  
  /* len = SUN_LEN(&unix_addr) - sizeof(unix_addr.sun_family); */
#ifdef HAVE_SUNLEN
  len -= sizeof(unix_addr.sun_len) + sizeof(unix_addr.sun_family);
#else
  len -= sizeof(unix_addr) - sizeof(unix_addr.sun_path);
#endif
  
  unix_addr.sun_path[len] = 0;
  if(stat(unix_addr.sun_path, &statbuf) < 0) {
    close(clifd);
    return(-2);
  }
#ifdef	S_ISSOCK
  if(S_ISSOCK(statbuf.st_mode) == 0) {
    close(clifd);
    return(-3);
  }
#endif
  if((statbuf.st_mode & (S_IRWXG | S_IRWXO)) ||
     (statbuf.st_mode & S_IRWXU) != S_IRWXU) {
    close(clifd);
    return(-4);
  }
  staletime = time(NULL) - STALE;
  if(statbuf.st_atime < staletime ||
     statbuf.st_ctime < staletime ||
     statbuf.st_mtime < staletime) {
    close(clifd);
    return(-5);
  }
  
  cp = (unix_addr.sun_path + strlen(unix_addr.sun_path) -1);
  while(*cp != '-' && cp != unix_addr.sun_path)
    cp--;
  if(*cp!='-') {
    close(clifd);
    return -4;
  }
  *pidptr = atoi(cp+1);
  unlink(unix_addr.sun_path);  
  return(clifd);
}

int serv_listen(const char *name) {
  int fd, len;
  struct sockaddr_un unix_addr;
  
  /* create a Unix domain stream socket */
  if((fd = socket(AF_UNIX, SOCK_STREAM, 0)) < 0)
    return(-1);
  unlink(name);
  memset(&unix_addr, 0, sizeof(unix_addr));
  unix_addr.sun_family = AF_UNIX;
  if(name)
    strcpy(unix_addr.sun_path, name);
  else
    strcpy(unix_addr.sun_path, SERVPATH);
  len = SUN_LEN(&unix_addr);
  /* bind the name to the descriptor */
  if(bind(fd, (struct sockaddr *) &unix_addr, len) < 0)
    return(-1);
  if(listen(fd, 5) < 0)
    return(-1);
  return(fd);
}

int cli_conn(const char *name, const char *from) {
  int fd, len;
  struct sockaddr_un unix_addr;
  char oldfile[MAXPATHLEN];
  
  /* create a Unix domain stream socket */
  if((fd = socket(AF_UNIX, SOCK_STREAM, 0)) < 0)
    return(-1);
  memset(&unix_addr, 0, sizeof(unix_addr));
  unix_addr.sun_family = AF_UNIX;
  if(from)
    sprintf(unix_addr.sun_path, "%s/bchild-%05d", from, getpid());
  else
    sprintf(unix_addr.sun_path, "%s-%05d", CLI_PATH, getpid());
  len = SUN_LEN(&unix_addr);
  unlink(unix_addr.sun_path);
  strcpy(oldfile, unix_addr.sun_path); 
  if(bind(fd, (struct sockaddr *)&unix_addr, len) < 0)
    goto error;
  if(chmod(unix_addr.sun_path, CLI_PERM) < 0)
    goto error;
  memset(&unix_addr, 0, sizeof(unix_addr));
  unix_addr.sun_family = AF_UNIX;
  strcpy(unix_addr.sun_path, name);
  unix_addr.sun_path[strlen(name)]='\0';
  len = SUN_LEN(&unix_addr);
  
  if(connect(fd, (struct sockaddr *) &unix_addr, len) < 0) {
    goto error;
  }
  return(fd);
error:
  unlink(oldfile);
  close(fd);
  fd=-1;
  return(fd);
}
#else 
#error Not a supported platform... patch it and mail me.. or mail me uname -a
#endif
#endif
