#line 2 "compat.c"
/*-
 * C-SaCzech
 * Copyright (c) 1996-2002 Jaromir Dolecek <dolecek@ics.muni.cz>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Jaromir Dolecek
 *	for the CSacek project.
 * 4. The name of Jaromir Dolecek may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY JAROMIR DOLECEK ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL JAROMIR DOLECEK BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: compat.c,v 1.53 2002/02/03 11:13:41 dolecek Exp $ */

#include "csacek.h"

/* all architecture dependant bits */

/***************************************************************************/
/* implementation of functions, which can not be available on some systems */

#ifndef HAVE_STRSTR
/* 
 * returns pointer on ``search'' in ``str'' or NULL if ``str' doesn't
 * contain ``search''
 */
char *
csa_strstr(str,search)
  const char *str, *search;
{
    const char *i,*j;

    if ( str == NULL || search == NULL || *search == 0)
        return NULL;

        while(*str != '\0') {
           if ( *(str++) == *search ) {
                i = str;
                j = search+1;
                while(*j && *i && *j == *i) { j++; i++; };
                if (*j == '\0') return (char *) str-1;
           }
        }

        return NULL;
}
#endif /* ndef HAVE_STRSTR */

#ifndef HAVE_STRERROR
/*
 * some pretty old SunOS boxes don't have strerror() 
 */
char *
csa_strerror(errnum)
  int errnum;
{
   return (char *) "no error info available";
}
#endif /* ndef HAVE_STRERROR */

/***************************************************************************/
/* OS specific bits */

#ifdef __MSWIN__

#include <sys/stat.h>
#include <fcntl.h>

#ifdef CSA_MUTEXES

/* right now, we need this for WinNT with ISAPI */
csa_mutex *
csa_create_mutex(name)
  char *name;
{
	return CreateMutex(NULL, FALSE, name);;
}

void
csa_acquire_mutex(mutex_id)
  csa_mutex *mutex_id;
{
    WaitForSingleObject(mutex_id, INFINITE);
}

void
csa_release_mutex(mutex_id)
  csa_mutex *mutex_id;
{
    ReleaseMutex(mutex_id);
}
#endif /* CSA_MUTEXES */

/*
 * on the braindead MS Win systems Winsock DLL library has to be explicitly
 * loaded; stdout need to be set to binary mode too
 */
int 
csa_init_compat()
{
    int err=0;
#if !defined(CSA_MUTACE_ISAPI) && !defined(CSA_MUTACE_APACHE)
    WSADATA     Wsadata;

    /* load Winsock shared library */
    err = WSAStartup(MAKEWORD(2,0), &Wsadata);
    if (err != 0) {
	printf("Status: 500 Internal error - cannot load WINSOCK dll\n");
	printf("Content-Type: text/html\n\n");
	printf("<H1>C-SaCzech inicialization error</H1>\n");
	printf("Cannot load WINSOCK dynamic linking library.\n");
    }

    /* set mode of stdout to binary */
    if (err == 0
	&& (_setmode(fileno(stdout), _O_BINARY) < 0
		|| _setmode(fileno(stdin), _O_BINARY) < 0) )
    {
	printf("Status: 500 Internal Server error\n");
	printf("Content-Type: text/html\n\n");
	printf("<H1>C-SaCzech inicialization error</H1>\n");
	printf("Cannot set mode of stdout or stdin stream to binary.\n");
	err = -1;
    }
#endif

#ifdef CSA_DEBUG
	/* seed PRNG, csa_debug_start() uses it under Windows instead
	 * of PID, since PID tend to repeat quite often due to the
	 * way they are generated under MS Windows */
	srand((unsigned int) time(NULL));
	csa_debug_uniqid = rand();
#endif

	return err;
}


/*
 * there is a bug in MSWin's stat() - it is not able to stat() directory;
 * if the name ends with backslash
 */
int 
csa_stat(filename, pstatbuf)
  const char *filename;
  struct stat *pstatbuf;
{
    char *b, *pagefile=NULL;
    int  i;

    if (filename) /* let stat(2) choke if filename is NULL */
    {
	pagefile = (char *) alloca(strlen(filename) + 1);
	strcpy(pagefile, filename); /* safe */
	b = strchr(pagefile, 0);
	if (b!=pagefile) {
		b--;
		for(;b>pagefile;b--)
			if (*b != CSA_DIRDELIM && *b != '/') break;
		*b = '\0';
	}
    }

    i = stat(pagefile, pstatbuf);

    return i;
}

#endif /* __MSWIN__ */

#ifdef CSA_NEED_FAKE_STDIO

/* simulation of stream functions (f* functions) for braindead MSWin, which */
/* treats socket desriptor and file descriptor as two different things */
/* and native stdio f*() functions fail to work */

/* following functions do their work only when the descriptor underneth    */
/* is really socket; they call native f* otherwise */
/* there is usually no comment, what the functions do; they simply mimic   */
/* behaviour of their respective ANSI C counterparts */

int 
csa_feof(fd)
  csa_FILE_t *fd;
{
	int retval;

	if (fd->issocket) retval = (CSA_ISSET(fd->flags, CSA_IOEOF));
	else retval = feof(fd->FILE);

	return retval;
}

int 
csa_ferror(fd)
  csa_FILE_t *fd;
{
	int retval;

        if (fd->issocket) retval = (CSA_ISSET(fd->flags, CSA_IOERR));
        else retval = ferror(fd->FILE);

	return retval;
}

/*
 * MSWin fdopen() silently fails to open stream over socket descriptor
 * so this dummy mechanism was created - note csa_fdopen() has to be
 * called with totally non-standard mode "s" to open stream over
 * descriptor; it uses native fdopen() otherwise
 */
csa_FILE_t * 
csa_fdopen( p, desc, mode )
  csa_params_t *p;
  int desc;
  const char *mode;
{
	csa_FILE_t *swp;

	swp = (csa_FILE_t *) ap_palloc (p->pool_req, sizeof(csa_FILE_t));

	/* init buffer */
	if (*mode == 's') {
		swp->desc = desc;
		swp->FILE = NULL;
		swp->issocket	= 1;
		swp->flags	= 0;
	
		swp->in_buf.buf = (char *) ap_palloc(p->pool_req, CSA_BUFLEN);
		swp->out_buf.buf = (char *) ap_palloc(p->pool_req, CSA_BUFLEN);
		swp->in_buf.len = swp->out_buf.len = CSA_BUFLEN;
		swp->in_buf.index = swp->in_buf.maxindex = 0;
		swp->out_buf.index = swp->out_buf.maxindex = 0;
	}
	else {
		swp->FILE = fdopen(desc, mode);
		swp->issocket	= 0;
		swp->flags	= 0;
	}

	return swp;
}

/*
 * reads ``nmemb'' items of size ``size'' from stream ``fd'' to ``ptr''
 */
size_t 
csa_fread( where, size, nmemb, fd )
  void	*where;
  size_t size, nmemb;
  csa_FILE_t	*fd;
{
        int number=size*nmemb, num, remains, datalen;
	int *index, *maxindex;
	char *ptr = (char *)where;

	if (csa_feof(fd)||csa_ferror(fd)) return 0;

	if (!fd->issocket) return fread(ptr,size,nmemb,fd->FILE);

	index = &fd->in_buf.index;
	maxindex = &fd->in_buf.maxindex;

	remains = number;
	while(remains>0) {
		if (*index == *maxindex)
		{
			num=recv(fd->desc,fd->in_buf.buf,fd->in_buf.len, 0);
			if (num==-1) {
				CSA_SET(fd->flags, CSA_IOERR);
				break;
			}
			else if (num==0) {
				CSA_SET(fd->flags, CSA_IOEOF);
				break;
			}
			*maxindex = num;
			*index = 0;
		}
		datalen = *maxindex - *index;
		num = (remains < datalen) ? remains : datalen;
		memcpy(ptr, &fd->in_buf.buf[*index], num);
		*index += num;
		ptr += num;
		remains -= num;
	} /* while */

	return number - remains;
}

/*
 * writes ``nmemb'' items of size ``size'' from ``ptr'' to stream ``fd''
 */
size_t
csa_fwrite( src, size, nmemb, fd )
  const void *src;
  size_t size, nmemb;
  csa_FILE_t *fd;
{
        size_t number=size*nmemb, num, remains, free_space;
	int *index, *maxindex;
	const char *ptr= (char *)src;

	if (csa_feof(fd)||csa_ferror(fd)) return 0;

	if (!fd->issocket) return fwrite(ptr,size,nmemb,fd->FILE);

	index = &fd->out_buf.index;
	maxindex = &fd->out_buf.maxindex;

	remains = number;
	while(remains>0) {
		free_space = fd->out_buf.len - *index;
		num = (remains<free_space) ? remains : free_space;
		memcpy(&fd->out_buf.buf[*index], ptr, num);
		*index += num;
		ptr += num;
		remains -= num;
		if (*index == fd->out_buf.len) {
		    *index = 0;
		    do {
			/* ensure _all_ chars are sent */
			num = send(fd->desc,fd->out_buf.buf,fd->out_buf.len,0);
			if (num==-1) {
				CSA_SET(fd->flags, CSA_IOERR);
				break;
			}
			*index += num;
		    } while (*index < fd->out_buf.len);
		    *index = 0;
		    if (csa_ferror(fd)) break; /* break if error */
		}
	}

	return number - remains;
}

/*
 * note vsprintf() is used to implement this
 */
int 
csa_fprintf(csa_FILE_t *fd, char *fmt, ...)
{
	va_list arg;
	char buff[CSA_BUFLEN];
	int num;

	if (csa_feof(fd)||csa_ferror(fd)) return -1;

	va_start(arg, fmt);
#ifdef HAVE_VSNPRINTF
	num = vsnprintf(buff, CSA_BUFLEN, fmt, arg);
#else
	num = vsprintf(buff, fmt, arg);
#endif
	va_end(arg);

	if (num <= 0) return num;

	return csa_fwrite(buff, sizeof(char), strlen(buff), fd);
}

int 
csa_fclose( fd )
  csa_FILE_t *fd;
{
	if (!fd->issocket) return fclose(fd->FILE);

	csa_fflush(fd);
#ifdef CSA_UNIX_LIKE
	/* on Un*x, socket is closed by normal close(2) */
	close(fd->desc);
#else
	closesocket(fd->desc);
#endif /* CSA_UNIX_LIKE */

	return 0;
}

char * 
csa_fgets( str, size, fd )
  char *str;
  int size;
  csa_FILE_t *fd;
{
	char *targstr=str, getchar = '\0';

	if (!fd->issocket) return fgets(str, size, fd->FILE);

	/* there is some error pending or we are on the end already */
	if (csa_feof(fd)||csa_ferror(fd)) return NULL;

	size--; /* we need one more char for ending '\0' */

	while( size>0 && getchar != '\n')
	{
		if (csa_fread(&getchar, sizeof(char), 1, fd) < 1) break;
		*(targstr++) = getchar;
		size--;
	}
	*targstr = 0; /* ukonceni retezce */

	if (targstr != str || (!csa_feof(fd) && !csa_ferror(fd))) return str;
	else return NULL;
}

int
csa_fgetc( fd )
  csa_FILE_t * fd;
{
	size_t foo;
	char retchar;

	if (csa_feof(fd) || csa_ferror(fd)) return EOF;

        if (!fd->issocket) return fgetc(fd->FILE);

	foo = csa_fread(&retchar, sizeof(char), 1, fd);
	if (foo <=0) return EOF;
	else return retchar;
}

int 
csa_fflush( fd )
  csa_FILE_t *fd;
{
	int num, datalen;

	if (csa_feof(fd) || csa_ferror(fd)) return 1;

	if (!fd->issocket) return fflush(fd->FILE);

	datalen = fd->out_buf.index;
	while(datalen) {
		num = send(fd->desc, fd->out_buf.buf, datalen, 0);
		if (num == -1) {
			CSA_SET(fd->flags, CSA_IOERR);
			return -1;
		}

		fd->out_buf.index -= num;
		datalen -= num;
	}

	return 0;
}

/*
 * fills in csa_FILE_t structure to be usable as parameter to
 * csa_f* functions
 */
int
csa_set_fio(p, fd, issocket, ff, desc)
	csa_params_t *p;
	csa_FILE_t 	**fd;
	int		 issocket;
	FILE		*ff;
	int		 desc;
{
	*fd = (csa_FILE_t *) ap_pcalloc(p->pool_req, sizeof(csa_FILE_t));
	(*fd)->issocket = issocket;
	(*fd)->FILE     = ff;
	(*fd)->desc     = desc;

	return 0;
}

#endif  /* CSA_NEED_FAKE_STDIO */
