/**
 * @file libleakbug/leakbug.h Leakbug
 *
 * $Id: leakbug.h,v 1.11 2002/05/21 22:01:10 chipx86 Exp $
 *
 * @Copyright (C) 2001-2002 The GNUpdate Project. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _LEAKBUG_H_
#define _LEAKBUG_H_

/*
 * We should include these now, so that we can #undef the functions in
 * this file.
 */
#include <string.h>
#include <stdlib.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 * Allocates memory.
 *
 * This wraps around malloc().
 *
 * @param size        The size of the memory to allocate.
 * @param filename    The calling filename.
 * @param line        The calling line number.
 * @param debug_level The debug level.
 *
 * @return A pointer to the new chunk of memory.
 */
void *lbMalloc(size_t size, const char *filename, int line, int debug_level);

/**
 * Allocates and clears an array.
 *
 * This wraps around calloc().
 *
 * @param nmemb       The number of elements to allocate.
 * @param size        The size of each element.
 * @param filename    The calling filename.
 * @param line        The calling line number.
 * @param debug_level The debug level.
 *
 * @return A pointer to the first element in the new chunk of memory.
 */
void *lbCalloc(size_t nmemb, size_t size, const char *filename, int line,
               int debug_level);

/**
 * Reallocates memory.
 *
 * This wraps around realloc().
 *
 * @param data        The data to reallocate.
 * @param size        The new size of the data.
 * @param filename    The calling filename.
 * @param line        The calling line number.
 * @param debug_level The debug level.
 *
 * @return A pointer to the start of @a data.
 */
void *lbRealloc(void *data, size_t size, const char *filename, int line,
                int debug_level);

/**
 * Frees memory.
 *
 * This wraps around free().
 *
 * @param ptr         The pointer to free.
 * @param filename    The calling filename.
 * @param line        The calling line number.
 * @param debug_level The debug level.
 */
void lbFree(void *ptr, const char *filename, int line, int debug_level);

/**
 * Registers memory for leakbug to keep track of.
 *
 * This is used to register any memory not automatically handled by
 * leakbug. This is useful when leakbug dumps "Freeing unknown memory"
 * warnings in your program.
 *
 * The easiest way to use this is to define macros in your program. For
 * example:
 *
 * @code
 * #ifdef WITH_LEAKBUG
 *  #include <leakbug.h>
 *
 *  #define LB_REGISTER(ptr, dataSize) \
 *    lbRegister((ptr), (dataSize), __FILE__, __LINE__, LEAKBUG_DEBUG_LEVEL)
 * #else
 *  #define LB_REGISTER(ptr, dataSize)
 * #endif
 * @endcode
 *
 * @param ptr         The pointer to register.
 * @param dataSize    The size of the data.
 * @param filename    The calling filename.
 * @param line        The calling line number.
 * @param debug_level The debug level.
 *
 * @return The pointer passed.
 *
 * @see lbRegisterArray()
 */
void *lbRegister(void *ptr, unsigned int dataSize, const char *filename,
				 int line, int debug_level);

/**
 * Registers memory in an array for leakbug to keep track of.
 *
 * This is used to register any memory not automatically handled by
 * leakbug. This is useful when leakbug dumps "Freeing unknown memory"
 * warnings in your program.
 *
 * The easiest way to use this is to define macros in your program. For
 * example:
 *
 * @code
 * #ifdef WITH_LEAKBUG
 *  #include <leakbug.h>
 *
 *  #define LB_REGISTER_ARRAY(ptr, dataSize, numElements) \
 *    lbRegisterArray((ptr), (dataSize), (numElements) \
 *                    __FILE__, __LINE__, LEAKBUG_DEBUG_LEVEL)
 * #else
 *  #define LB_REGISTER_ARRAY(ptr, dataSize, numElements)
 * #endif
 * @endcode
 *
 * @param ptr         The array pointer to register.
 * @param dataSize    The size of the data.
 * @param numElements The number of elements in the array.
 * @param filename    The calling filename.
 * @param line        The calling line number.
 * @param debug_level The debug level.
 *
 * @return The array pointer passed.
 *
 * @see lbRegister()
 */
void **lbRegisterArray(void **ptr, unsigned int dataSize,
					   unsigned int numElements, const char *filename,
					   int line, int debug_level);

/**
 * Frees all known leaked memory.
 *
 * This is the easy way out of freeing memory, but it can be dangerous.
 * You probably shouldn't use it.
 */
void lbFreeLeaks(void);

/**
 * Displays a list of all known leaked memory.
 */
void lbDumpLeaks(void);

/**
 * @name Leakbug Debug Levels
 */
/*@{*/
#define LEAKBUG_SILENT   0  /**< Run leakbug silently.                */
#define LEAKBUG_WARNINGS 1  /**< Run leakbug with warnings outputted. */
#define LEAKBUG_ERRORS   2  /**< Run leakbug with errors outputted.   */
/*@}*/

#ifndef _DOXYGEN_
# ifndef LEAKBUG_NO_DEBUG
#  ifndef LEAKBUG_DEBUG_LEVEL
#   define LEAKBUG_DEBUG_LEVEL (LEAKBUG_WARNINGS | LEAKBUG_ERRORS)
#  endif

#  undef malloc
#  undef calloc
#  undef realloc
#  undef free
#  undef strdup

#  define malloc(x) lbMalloc((x),  __FILE__, __LINE__, LEAKBUG_DEBUG_LEVEL)
#  define calloc(n, x) \
	lbCalloc((n), (x), __FILE__, __LINE__, LEAKBUG_DEBUG_LEVEL)
#  define realloc(d, x) \
	lbRealloc((d), (x), __FILE__, __LINE__, LEAKBUG_DEBUG_LEVEL)
#  define free(x)   lbFree((x),    __FILE__, __LINE__, LEAKBUG_DEBUG_LEVEL)

#  define strdup(x) lbRegister(strdup(x), (strlen(x) + 1) * sizeof(char), \
							   __FILE__, __LINE__, LEAKBUG_DEBUG_LEVEL)
# endif /* !LEAKBUG_NO_DEBUG */
#endif

#ifdef __cplusplus
}
#endif

#endif /* _LEAKBUG_H_ */
