// =============================================================================
//
//      --- kvi_dcc_manager.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviDccManager"

#include <qfileinfo.h>

#include "kvi_app.h"
#include "kvi_console.h"
#include "kvi_dcc_chat.h"
#include "kvi_dcc_chatmessagebox.h"
#include "kvi_dcc_sendfiledialog.h"
#include "kvi_dcc_sendmessagebox.h"
#include "kvi_dcc_sendrenamemessagebox.h"
#include "kvi_dcc_sendresumemessagebox.h"
#include "kvi_dcc_manager.h"
#include "kvi_dcc_send_event.h"
#include "kvi_dcc_voice.h"
#include "kvi_fileutils.h"
#include "kvi_frame.h"
#include "kvi_irc_socket.h"
#include "kvi_irc_user.h"
#include "kvi_irc_userlist.h"
#include "kvi_locale.h"
#include "kvi_options.h"

// NOTE: uAddress in this file is in NETWORK BYTE ORDER.

// Global variable
unsigned short int g_uLastDccListenPort = 0;

KviDccManager::KviDccManager(KviFrame *frame)
	: QObject()
{
	m_pFrm = frame;
}

KviDccManager::~KviDccManager()
{
	// Nothing here
}

void KviDccManager::handleDccRequest(KviIrcUser &source, KviDccRequest *data)
{
	if( g_pOptions->m_bIgnoreDccRequests ) {
		m_pFrm->m_pConsole->output(KVI_OUT_DCCINFO,
			_i18n_("Ignoring DCC %s request from %s [%s@%s]: [%s]"),
			data->szType.ptr(), source.nick(), source.username(),
			source.host(), data->szOriginalRequest.ptr()
		);
		return;
	}

	if( g_pOptions->m_bNotifyAllDccRequestsInConsole ) {
		m_pFrm->m_pConsole->output(KVI_OUT_DCCINFO,
			_i18n_("Processing DCC %s request from %s [%s@%s]: [%s]"),
			data->szType.ptr(), source.nick(), source.username(), source.host(),
			data->szOriginalRequest.ptr()
		);
	}

	if( kvi_strEqualCI(data->szType.ptr(), "chat") ) {
		if( !kvi_strEqualCI(data->szParam.ptr(), "chat") ) {
			m_pFrm->m_pConsole->output(KVI_OUT_DCCWARNING,
				_i18n_("Invalid parameter '%s' for DCC chat request from %s [%s@%s]: should be 'chat': continuing"),
				data->szParam.ptr(), source.nick(), source.username(), source.host());
		}
		handleDccChat(source, data->uAddress, data->uPort);
	}
	else if( kvi_strEqualCI(data->szType.ptr(), "send") ) {
		bool bOk = false;
		unsigned int size = data->szLast.toUInt(&bOk);
		if( !bOk ) {
			m_pFrm->m_pConsole->output(KVI_OUT_DCCERROR,
				_i18n_("Invalid file size (%s) for DCC send request from %s [%s@%s]: ignoring"),
				data->szLast.ptr(), source.nick(), source.username(), source.host()
			);
			return;
		}
		handleDccSend(source, data->szParam, size, data->uAddress, data->uPort);
	}
	else if( kvi_strEqualCI(data->szType.ptr(), "accept") ) {
		handleDccAccept(source, data->szParam, data->szAddress.toUShort(), data->szPort.toULong());
	}
	else if( kvi_strEqualCI(data->szType.ptr(), "resume") ) {
		handleDccResume(source, data->szParam, data->szAddress.toUShort(), data->szPort.toULong());
	}
	else if( kvi_strEqualCI(data->szType.ptr(), "voice") ) {
		bool bOk = false;
		unsigned int sampleRate = data->szLast.toUInt(&bOk);
		if( !bOk ) {
			m_pFrm->m_pConsole->output(KVI_OUT_DCCWARNING,
				_i18n_("No bitrate specified for DCC voice request from %s!%s@%s: defaulting to 8 kHz"),
				source.nick(), source.username(), source.host()
			);
			sampleRate = 8000;
		}
		if( sampleRate != 8000 ) {
			m_pFrm->m_pConsole->output(KVI_OUT_DCCWARNING,
				_i18n_("Unsupported sample rate for DCC voice request from %s!%s@%s: %d Hz: defaulting to 8 kHz"),
				source.nick(), source.username(), source.host(), sampleRate
			);
			sampleRate = 8000;
		}
		if( !kvi_strEqualCI("ADPCM", data->szParam.ptr()) ) {
			m_pFrm->m_pConsole->output(KVI_OUT_DCCWARNING,
				_i18n_("Unsupported codec for DCC voice request from %s!%s@%s: %s: defaulting to ADPCM"),
				source.nick(), source.username(), source.host(), data->szParam.ptr()
			);
		}
		handleDccVoice(source, data->uAddress, data->uPort);
	}
	// TODO: More DCC Types here...
}

void KviDccManager::handleDccAccept(KviIrcUser &source, KviStr &filename, unsigned short uPort, unsigned long uResumePos)
{
	KviDccSend *send = m_pFrm->findWaitingDccSend(uPort, source.nick());
	if( !send ) {
		m_pFrm->m_pConsole->output(KVI_OUT_DCCERROR,
			_i18n_("Invalid DCC ACCEPT from %s [%s@%s]. Transfer not initiated for file %s on port %u: ignoring"),
			source.nick(), source.username(), source.host(), filename.ptr(), uPort
		);
		return;
	}
	send->output(KVI_OUT_DCCINFO, _i18n_("RESUME accepted. Transfer will initiate from position %u"), uResumePos);
	send->initiateGet();
}

void KviDccManager::handleDccChat(KviIrcUser &source, unsigned long uAddress, unsigned short uPort)
{
	if( g_pOptions->m_bAutoAcceptDccChat ) {
		m_pFrm->m_pConsole->output(KVI_OUT_DCCINFO,
			_i18n_("Accepting DCC chat request from %s [%s@%s]: [%u, %u]"),
			source.nick(), source.username(), source.host(), uAddress, uPort
		);
		acceptDccChat(source.nick(), source.username(), source.host(), uAddress, uPort);
	} else {
		KviDccChatMessageBox *box = new KviDccChatMessageBox(
			m_pFrm, source.nick(), source.username(), source.host(), uAddress, uPort, false
		);
		connect(
			box,  SIGNAL(dccRequestAccepted(const char *, const char *, const char *, unsigned long, unsigned short)),
			this, SLOT(acceptDccChat(const char *, const char *, const char *, unsigned long, unsigned short))
		);
		if( !box->isVisible() ) box->show();
	}
}

void KviDccManager::handleDccVoice(KviIrcUser &source, unsigned long uAddress, unsigned short uPort)
{
	if( g_pOptions->m_bAutoAcceptDccVoice ) {
		m_pFrm->m_pConsole->output(KVI_OUT_DCCINFO,
			_i18n_("Accepting DCC voice request from %s [%s@%s]: [%u, %u]"),
			source.nick(), source.username(), source.host(), uAddress, uPort
		);
		acceptDccVoice(source.nick(), source.username(), source.host(), uAddress, uPort);
	} else {
		KviDccChatMessageBox *box = new KviDccChatMessageBox(
			m_pFrm, source.nick(), source.username(), source.host(), uAddress, uPort, true
		);
		connect(
			box,  SIGNAL(dccRequestAccepted(const char *, const char *, const char *, unsigned long, unsigned short)),
			this, SLOT(acceptDccVoice(const char *, const char *, const char *, unsigned long, unsigned short))
		);
		if( !box->isVisible() ) box->show();
	}
}

/**
 * Slot
 */
void KviDccManager::acceptDccVoice(
	const char *nick, const char *username, const char *host, unsigned long uAddress, unsigned short uPort)
{
	KviStr tmp(KviStr::Format, "DCC-Voice-%s!%s@%s", nick, username, host);
	KviDccVoice *data = m_pFrm->createDccVoice(tmp.ptr());
	data->acceptDccRequest(nick, username, host, uAddress, uPort);
}

void KviDccManager::requestDccVoice(const char *nick)
{
	KviIrcUser *u = m_pFrm->m_pUserList->findUser(nick);
	KviStr username, host;
	if( !u ) {
		username = "*";
		host     = "*";
	} else {
		username = u->username();
		host     = u->host();
	}
	KviStr tmp(KviStr::Format, "DCC-Voice-%s!%s@%s", nick, username.ptr(), host.ptr());
	KviDccVoice *data = m_pFrm->createDccVoice(tmp.ptr());
	data->requestDcc(nick, username.ptr(), host.ptr());
}

void KviDccManager::acceptDccChat(
	const char *nick, const char *username, const char *host, unsigned long uAddress, unsigned short uPort)
{
	KviStr tmp(KviStr::Format, "DCC-Chat-%s!%s@%s", nick, username, host);
	KviDccChat *chat = m_pFrm->createDccChat(tmp.ptr());
	chat->acceptDccRequest(nick, username, host, uAddress, uPort);
}

KviDccChat *KviDccManager::requestDccChat(const char *nick, const char *userandhost)
{
	KviStr username, host;
	if( userandhost ) {
		KviStr tmp(KviStr::Format, "%s!%s", nick, userandhost);
		KviIrcUser user(tmp.ptr());
		username = user.username();
		host     = user.host();
	} else {
		KviIrcUser *u = m_pFrm->m_pUserList->findUser(nick);
		if( !u ) {
			username = "*";
			host     = "*";
		} else {
			username = u->username();
			host     = u->host();
		}
	}
	KviStr tmp(KviStr::Format, "DCC-Chat-%s!%s@%s", nick, username.ptr(), host.ptr());
	KviDccChat *chat = m_pFrm->createDccChat(tmp.ptr());
	chat->requestDcc(nick, username.ptr(), host.ptr());
	return chat;
}

void KviDccManager::handleDccSend(
	KviIrcUser &source, KviStr &filename, unsigned long fileLen, unsigned long uAddress, unsigned short uPort)
{
	KviDccSendRequestData *data = new KviDccSendRequestData();
	filename.stripWhiteSpace();
	data->nick       = source.nick();
	data->username   = source.username();
	data->host       = source.host();
	data->fileName   = filename.ptr();
	data->originalFileName = filename.ptr();
	if( filename.contains(' ') != 0 ) {
		data->originalFileName.prepend("\"");
		data->originalFileName.append("\"");
	}
	if( g_pOptions->m_bReplaceSpacesInDccSendFileNames )
		data->fileName.replaceAll(' ', "_"); // Replace all spaces with underscores
	data->fileLength = fileLen;
	data->uAddress   = uAddress;
	data->uPort      = uPort;

	if( g_pOptions->m_bAutoAcceptDccSend ) {
		m_pFrm->m_pConsole->output(KVI_OUT_DCCINFO,
			_i18n_("Accepting DCC send request from %s [%s@%s] for file %s [%u bytes]: [%u, %u]"),
			source.nick(), source.username(), source.host(), filename.ptr(), fileLen, uAddress, uPort
		);
		acceptDccSend(data);
	} else {
		data->parent      = this;
		data->requestType = KVI_DCC_SENDEVENT_ASK;
		KviDccSendMessageBox *box = new KviDccSendMessageBox(m_pFrm, data);
		if( !box->isVisible() ) box->show();
	}
}

void KviDccManager::acceptDccSend(KviDccSendRequestData *data)
{
	int idx = data->fileName.findLastIdx('/');
	if( idx != -1 )
		data->fileName.cutLeft(idx + 1); // Remove leading directories
	g_pApp->getDefaultDccSaveFilePath(data->filePath, data->fileName.ptr());

	if( g_pOptions->m_bAutoAcceptDccSend ) {
		// Auto select the file name
		data->filePath.ensureLastCharIs('/');
		data->filePath += data->fileName.ptr();
		dccAcceptFileRequest(data);
	} else {
		// Show a dir dialog starting in filePath
		data->parent      = this;
		data->requestType = KVI_DCC_SENDEVENT_RECEIVE;
		KviDccSendFileDialog *box = new KviDccSendFileDialog(m_pFrm, data, false);
		if( !box->isVisible() ) box->show();
	}
}

void KviDccManager::dccAcceptFileRequest(KviDccSendRequestData *data)
{
	// Check if the file exists on disk
	if( kvi_fileExists(data->filePath.ptr()) ) {
		// The file exists on disk;
		// check the file sizes.
		QFileInfo fi(data->filePath.ptr());

		if( (fi.size() <= data->fileLength) && (fi.size() > 0) ) {
			// The file on disk is smaller: resume, overwrite or rename?
			if( g_pOptions->m_bAutoAcceptDccSend ) {
				// No need to ask, auto accepting. Resume or rename
				if( g_pOptions->m_bEnableResumeOnAutoAccept )
					data->resumeValue = fi.size();
				else {
					// Auto-rename
					data->resumeValue = 0;
					while( kvi_fileExists(data->filePath.ptr()) )
						data->filePath.append(".rnm");
				}
			} else {
				// Not auto accepting, ask the user what to do
				data->resumeValue = fi.size();
				data->parent      = this;
				data->requestType = KVI_DCC_SENDEVENT_RESUME;
				KviDccSendResumeMessageBox *box = new KviDccSendResumeMessageBox(m_pFrm, data);
				if( !box->isVisible() ) box->show();
				return;
			}
		} else {
			// Equal file sizes or the file on disk is bigger:
			// ask whether to overwrite or rename
			data->resumeValue = 0;
			if( g_pOptions->m_bAutoAcceptDccSend ) {
				// Auto-rename
				while( kvi_fileExists(data->filePath.ptr()) )
					data->filePath.append(".rnm");
			} else {
				// Not auto accepting, ask the user what to do
				data->parent      = this;
				data->requestType = KVI_DCC_SENDEVENT_RENAME;
				KviDccSendRenameMessageBox *box = new KviDccSendRenameMessageBox(m_pFrm, data);
				if( !box->isVisible() ) box->show();
				return;
			}
		}
	} else data->resumeValue = 0;
	dccSendResumeSelectionDone(data);
}

void KviDccManager::dccSendResumeSelectionDone(KviDccSendRequestData *data)
{
	// Create the DCC now...
	KviStr tmp(KviStr::Format, "DCC-Get-%s!%s@%s", data->nick.ptr(), data->username.ptr(), data->host.ptr());
	KviDccSend *send = m_pFrm->createDccSend(tmp.ptr());
	send->acceptDccSendRequest(data);
	if( data )
		delete data;
}

void KviDccManager::requestDccSend(const char *nick, const char *filename)
{
	KviDccSendRequestData *data = new KviDccSendRequestData();
	data->nick = nick;
	KviIrcUser *u = m_pFrm->m_pUserList->findUser(nick);
	if( !u ) {
		data->username = "*";
		data->host     = "*";
	} else {
		data->username = u->username();
		data->host     = u->host();
	}

	data->filePath = filename ? filename : "";
	if( data->filePath.isEmpty() ) {
		data->parent      = this;
		data->requestType = KVI_DCC_SENDEVENT_SEND;
		KviDccSendFileDialog *box = new KviDccSendFileDialog(m_pFrm, data, true); // Existing file only
		if( !box->isVisible() ) box->show();
	} else dccSendFileRequest(data);
}

void KviDccManager::dccSendFileRequest(KviDccSendRequestData *data)
{
	KviStr tmp(KviStr::Format, "DCC-Send-%s!%s@%s", data->nick.ptr(), data->username.ptr(), data->host.ptr());
	KviDccSend *send = m_pFrm->createDccSend(tmp.ptr());
	send->requestDccSend(data);
	if( data )
		delete data;
}

void KviDccManager::handleDccResume(KviIrcUser &source, KviStr &filename, unsigned short uPort, unsigned long uResumePos)
{
	KviDccSend *send = m_pFrm->findListeningDccSend(uPort, source.nick());
	if( !send ) {
		m_pFrm->m_pConsole->output(KVI_OUT_DCCERROR,
			_i18n_("Invalid DCC RESUME from %s [%s@%s]. Transfer not initiated for file %s on port %u: ignoring"),
			source.nick(), source.username(), source.host(), filename.ptr(), uPort
		);
		return;
	}
	send->output(KVI_OUT_DCCINFO, _i18n_("Received RESUME request: position %u"), uResumePos);
	if( send->resumeForCurrentSend(uResumePos) ) {
		m_pFrm->m_pSocket->sendFmtData(
			"PRIVMSG %s :%cDCC ACCEPT %s %u %u%c",
			source.nick(), 0x01, filename.ptr(), uPort, uResumePos, 0x01
		);
	} else { // Failed to resume
		m_pFrm->m_pConsole->output(KVI_OUT_DCCERROR,
			_i18n_("Invalid DCC RESUME from %s [%s@%s]: resume position %u out of range for file %s on port %u: ignoring"),
			source.nick(), source.username(), source.host(), filename.ptr(), uResumePos, uPort
		);
	}
}

unsigned short int KviDccManager::getDccSendListenPort()
{
	if( g_pOptions->m_bDccListenOnPortsInRange ) {
		if( (g_pOptions->m_uMaxDccListenPort - g_pOptions->m_uMinDccListenPort) < 1 ) {
			g_pOptions->m_uMinDccListenPort = 1025;
			g_pOptions->m_uMaxDccListenPort = 65000;
		}

		if( g_uLastDccListenPort < g_pOptions->m_uMinDccListenPort )
			g_uLastDccListenPort = g_pOptions->m_uMinDccListenPort;
		else
			g_uLastDccListenPort++;
		if( g_uLastDccListenPort > g_pOptions->m_uMaxDccListenPort )
			g_uLastDccListenPort = g_pOptions->m_uMinDccListenPort;
		return g_uLastDccListenPort;
	} else return 0;
}

bool KviDccManager::event(QEvent *e)
{
	if( e->type() != QEvent::User )
		return QObject::event(e);

	KviDccSendEvent *ev = (KviDccSendEvent *) e;
	KviDccSendRequestData *data = ev->m_pData;
	switch( data->requestType ) {
		case KVI_DCC_SENDEVENT_RECEIVE:
			dccAcceptFileRequest(data);
			break;
		case KVI_DCC_SENDEVENT_SEND:
			dccSendFileRequest(data);
			break;
		case KVI_DCC_SENDEVENT_ASK:
			acceptDccSend(data);
			break;
		case KVI_DCC_SENDEVENT_RENAME:
			dccSendResumeSelectionDone(data);
			break;
		case KVI_DCC_SENDEVENT_RESUME:
			dccSendResumeSelectionDone(data);
			break;
	}
	return true;
}

#include "m_kvi_dcc_manager.moc"
