/***************************************************************************
                          kbearfilesyspart.cpp  -  description
                             -------------------
    begin                : Tue Feb 12 2002
    copyright            : (C) 2002 by Bjrn Sahlstrm
    email                : kbjorn@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

//////////////////////////////////////////////////////////////////////
// Qt specific include files
#include <qsplitter.h>
#include <qptrlist.h>
#include <qmap.h>
#include <qvbox.h>
#include <qpushbutton.h>
#include <qtooltip.h>
#include <qstringlist.h>
#include <qwhatsthis.h>
#include <qlayout.h>
#include <qhbox.h>
#include <qsizepolicy.h>
#include <qregexp.h>
#include <qdatetime.h>
#include <qtimer.h>
#include <qclipboard.h>
#include <qcheckbox.h>
#include <qtextcodec.h>
//////////////////////////////////////////////////////////////////////
// KDE specific include files
#include <kparts/genericfactory.h>
#include <kmessagebox.h>
#include <kpopupmenu.h>
#include <kapplication.h>
#include <kcharsets.h>
#include <kurl.h>
#include <krun.h>
#include <kurldrag.h>
#include <klocale.h>
#include <ktoolbar.h>
#include <kfiledialog.h>
#include <kdialogbase.h>
#include <kcombobox.h>
#include <kiconloader.h>
#include <kxmlguifactory.h>
#include <kio/job.h>
#include <kfile.h>
#include <kprogress.h>
#include <kanimwidget.h>
#include <kparts/partmanager.h>
#include <kparts/part.h>
#include <kio/paste.h>
#include <kuserprofile.h>
#include <kxmlguifactory.h>
#include <kpropertiesdialog.h>
#include <kparts/componentfactory.h>

#include <kdebug.h>
//////////////////////////////////////////////////////////////////////
// Application specific include files
#include "kbearfilesyspart.h"
#include "kbeardirview.h"
#include "kbeariconview.h"
#include "kbeardetailview.h"
#include "../../base/kbeardirlister.h"
#include "../../base/kbearconnectionmanager.h"
#include "kbearsearchdialog.h"
#include "kbearpropertiesdialog.h"

#include <X11/Xlib.h>

//-----------------------------------------------
typedef KParts::GenericFactory<KBearFileSysPart> KBearFileSysPartFactory;
K_EXPORT_COMPONENT_FACTORY( libkbearfilesyspart, KBearFileSysPartFactory );
//-----------------------------------------------
KBearFileSysPart::KBearFileSysPart( QWidget* parentWidget, const char* widgetName,
						QObject* parent, const char* name, const QStringList& )
	:	KBearChildViewPart( parentWidget, widgetName,parent,name),
		m_fileView( 0L ), m_leftView( 0L ), 
		m_myTime(QDateTime::currentDateTime().date(), QDateTime::currentDateTime().time()),
		m_numOfConnRetry( 0 ), m_connRetryTime( 0 ), m_lastURL( KURL() ),
		m_partViewerWidget( 0L ), m_isConnected( false ), m_isLoading( false ),
		m_guiEnabled( true )

{
	// we need an instance
	setInstance( KBearFileSysPartFactory::instance() );
	KGlobal::locale()->insertCatalogue("kbear");
	bool ok;
	m_codec = KGlobal::charsets()->codecForName( m_connection.fileSysEncoding(), ok );

	m_backStack.setAutoDelete( true );
	m_forwardStack.setAutoDelete( true );
	m_serviceActionList.setAutoDelete( true );
	setupActions();
	setup();
	setActionsEnabled( false );
	setXMLFile("kbearfilesyspart.rc");
	readProperties( kapp->config(), "View Settings"  );

	connect( connectionManager, SIGNAL( enableGUI( const QString&, bool ) ),
				this, SLOT( slotEnableGUI( const QString&, bool ) ) );
	// fill the synch list
//	QTimer::singleShot( 100, this, SLOT( updateSynchList() ) );
}
//-----------------------------------------------
KBearFileSysPart::~KBearFileSysPart(){
    QApplication::restoreOverrideCursor();

	if( m_partViewerWidget )
		delete m_partViewerWidget;
	m_partViewerWidget = 0L;
	m_partViewer = 0L;
	//showFileView();
	kdDebug()<<"KBearFileSysPart::~KBearFileSysPart()="<<this<<endl;
}
//-----------------------------------------------
KAboutData* KBearFileSysPart::createAboutData() {
	KAboutData* about = new KAboutData( "kbearfilesyspart", I18N_NOOP("KBearFileSysPart"), "1.0", 0,
							KAboutData::License_GPL, I18N_NOOP("(C) 2002, The KBear team") );
	about->addAuthor("Björn Sahlström", 0, "kbjorn@users.sourceforge.net");

	return about;
}
//-----------------------------------------------
void KBearFileSysPart::slotClipboardDataChanged() {
	KAction* act = action( KStdAction::stdName(KStdAction::Paste) );
	if( act )
		act->setEnabled( ! KIO::isClipboardEmpty() );
}
//-----------------------------------------------
void KBearFileSysPart::reconnect() {
	kdDebug()<<"KBearFileSysPart::reconnect()"<<endl;
	m_isConnected = false;
	
	m_dirLister->openConnection();
//	reload();
}
//-----------------------------------------------
bool KBearFileSysPart::openURL( const KURL& url ) {
	kdDebug()<<"KBearFileSysPart::openURL() url="<<url.prettyURL()<<endl;
		
	if( ! url.hasHost() )
		action("synch_new" )->setEnabled( false );
	// we have to stat the url since we don't know if it's a directory or a file
	m_dirLister->statURL( url );
	
	return true;
}
//-----------------------------------------------
void KBearFileSysPart::openConnection( const Connection& c ) {
	kdDebug()<<"KBearFileSysPart::openConnection() url="<<c.url().prettyURL()<<endl;
	m_connection = c;
	bool ok;
	m_codec = KGlobal::charsets()->codecForName( m_connection.fileSysEncoding(), ok );
	m_isConnected = false;
	if( m_dirLister ) {
		kdDebug()<<"KBearFileSysPart::openConnection() no dirLister, creating new"<<endl;
		delete m_dirLister;
	}
	setDirLister( new KBearDirLister( this, "KBearDirLister", c ) );
	m_leftView->setConnection( c );
	connect( m_dirLister, SIGNAL( previewPart( KParts::ReadOnlyPart*, const KURL& ) ),
						this, SLOT( slotGotPreviewPart( KParts::ReadOnlyPart*, const KURL& ) ) );
	connect( m_dirLister, SIGNAL( gotError(int) ), this, SLOT( slotConnectResult(int) ) );
	connect( m_dirLister, SIGNAL( connected() ), this, SLOT( slotConnected() ) );
	connect( m_dirLister, SIGNAL( infoMessage( const QString&) ), this, SLOT( slotInfoMessage( const QString& ) ) );
	connect( m_dirLister, SIGNAL( progress( KIO::Job*, unsigned long ) ), this, SLOT( slotProgress( KIO::Job*, unsigned long ) ) );
	m_url = m_connection.url();
	if( ! m_url.hasHost() ) {
		action("synch_new" )->setEnabled( false );
		action("clear_synch" )->setEnabled( false );
	}
	m_lastURL = m_url;
	setView( (KFile::FileView)m_currentViewKind );

	m_dirLister->openConnection();
}
//-----------------------------------------------
void KBearFileSysPart::connectDirlister() {
	m_dirLister->disconnect( this );
	m_dirLister->disconnect( m_leftView );

	connect( m_dirLister, SIGNAL( previewPart( KParts::ReadOnlyPart*, const KURL& ) ),
						this, SLOT( slotGotPreviewPart( KParts::ReadOnlyPart*, const KURL& ) ) );
	connect( m_dirLister, SIGNAL( gotError(int) ), this, SLOT( slotConnectResult(int) ) );
	connect( m_dirLister, SIGNAL( infoMessage( const QString&) ), this, SLOT( slotInfoMessage( const QString& ) ) );
	connect( m_dirLister, SIGNAL( progress( KIO::Job*, unsigned long ) ), this, SLOT( slotProgress( KIO::Job*, unsigned long ) ) );

	connect( m_dirLister, SIGNAL( progress( int ) ), this, SLOT( slotProgress( int ) ) );
	connect( m_dirLister, SIGNAL( started() ), this, SLOT( slotStartLoading() ) );
	connect( m_dirLister, SIGNAL( newItems( const KFileItemList &) ), this, SLOT( slotNewItems( const KFileItemList &) ) );
	connect( m_dirLister, SIGNAL( newItems( const KFileItemList &) ), m_leftView, SLOT( slotInsertItems( const KFileItemList &) ) );
	connect( m_dirLister, SIGNAL( finishedLoading() ), this, SLOT( slotFinishedLoading() ) );
	connect( m_dirLister, SIGNAL( deleteItem( KFileItem* ) ), this, SLOT( slotDeleteItem( KFileItem* ) ) );
	connect( m_dirLister, SIGNAL( deleteItem( KFileItem* ) ), m_leftView, SLOT( slotDeleteItem( KFileItem* )));
	connect( m_dirLister, SIGNAL( openDir( const KFileItem* ) ), this, SLOT( slotOpenDir( const KFileItem* ) ) );
	connect( m_dirLister, SIGNAL( openFile( const KFileItem* ) ), this, SLOT( slotOpenFile( const KFileItem* ) ) );
	connect( m_dirLister, SIGNAL( redirection( const KURL& ) ), this, SLOT( slotRedirected( const KURL& ) ) );
	connect( m_dirLister, SIGNAL( clear() ), this, SLOT( slotClear() ) );
	connect( m_dirLister, SIGNAL( speed(int) ), this, SLOT( slotSpeed(int) ) );
	connect( m_dirLister, SIGNAL( deleteFinished() ), this, SLOT( slotDeleteFinished() ) );
	connect( m_dirLister, SIGNAL( reload() ), this, SLOT( reload() ) );
}
//-----------------------------------------------
void KBearFileSysPart::slotConnected() {
	kdDebug()<<"KBearFileSysPart::slotConnected()"<<endl;
	QObject::disconnect( m_dirLister, SIGNAL( connected() ), this, SLOT( slotConnected() ) );

	m_isConnected = true;
	m_numOfConnRetry = 0;
	m_connRetryTime = 0;
	m_dirLister->setShowingDotFiles( showHiddenAction->isChecked() );

	connect( m_dirLister, SIGNAL( progress( int ) ), this, SLOT( slotProgress( int ) ) );
	connect( m_dirLister, SIGNAL( started() ), this, SLOT( slotStartLoading() ) );
	connect( m_dirLister, SIGNAL( newItems( const KFileItemList &) ), this, SLOT( slotNewItems( const KFileItemList &) ) );
	connect( m_dirLister, SIGNAL( newItems( const KFileItemList &) ), m_leftView, SLOT( slotInsertItems( const KFileItemList &) ) );
	connect( m_dirLister, SIGNAL( finishedLoading() ), this, SLOT( slotFinishedLoading() ) );
	connect( m_dirLister, SIGNAL( deleteItem( KFileItem* ) ), this, SLOT( slotDeleteItem( KFileItem* ) ) );
	connect( m_dirLister, SIGNAL( deleteItem( KFileItem* ) ), m_leftView, SLOT( slotDeleteItem( KFileItem* )));
	connect( m_dirLister, SIGNAL( openDir( const KFileItem* ) ), this, SLOT( slotOpenDir( const KFileItem* ) ) );
	connect( m_dirLister, SIGNAL( openFile( const KFileItem* ) ), this, SLOT( slotOpenFile( const KFileItem* ) ) );
	connect( m_dirLister, SIGNAL( redirection( const KURL& ) ), this, SLOT( slotRedirected( const KURL& ) ) );
	connect( m_dirLister, SIGNAL( clear() ), this, SLOT( slotClear() ) );
	connect( m_dirLister, SIGNAL( speed(int) ), this, SLOT( slotSpeed(int) ) );
	connect( m_dirLister, SIGNAL( deleteFinished() ), this, SLOT( slotDeleteFinished() ) );
	connect( m_dirLister, SIGNAL( reload() ), this, SLOT( reload() ) );
	// Now we're connected so we can enable everything again
	setActionsEnabled( true );

	clearHistory();
	m_pathCombo->clear();
	pathChanged();
	m_dirLister->statURL( m_url );
}
//-----------------------------------------------
bool KBearFileSysPart::disconnect() {
	kdDebug()<<"KBearFileSysPart::disconnect()"<<endl;
	if( m_guiEnabled )
		m_dirLister->stop();
	return m_guiEnabled;
}
//-----------------------------------------------
void KBearFileSysPart::clearHistory() {
	kdDebug()<<"KBearFileSysPart::clearHistory()"<<endl;
	m_backStack.clear();
	backAction->setEnabled( false );
	m_forwardStack.clear();
	forwardAction->setEnabled( false );
}
//-----------------------------------------------
void KBearFileSysPart::addToPathHistory(const QString& path ) {
	kdDebug()<<"KBearFileSysPart::addToPathHistory() path="<<path<<endl;
	QString p;
	if( m_connection.isLocal() )
		p = path;
	else
		p = m_codec->toUnicode( path );
	bool found = false;
	for( int i = 0; i < m_pathCombo->count(); i++ ) {
		if( m_pathCombo->text( i ) == p ) {
			m_pathCombo->setCurrentItem( p, true );
			found = true;
		}
		else if( m_pathCombo->text( i ) == QString::null ) {
			m_pathCombo->removeItem( i );
			--i;
		}
	}
	if( ! found ) {
		m_pathCombo->setCurrentItem( p, true );
	}
}
//-----------------------------------------------
void KBearFileSysPart::addToHistory(const KURL& url ) {
	kdDebug()<<"KBearFileSysPart::addToHistory() url="<<url.prettyURL()<<endl;
	if( ! m_backStack.isEmpty() && url == * m_backStack.top() )
		return;
	m_backStack.push(new KURL(url));
	m_forwardStack.clear();
}
//-----------------------------------------------
void KBearFileSysPart::setDir(const KURL& url ) {
	kdDebug()<<"KBearFileSysPart::setDir() url="<<url.prettyURL()<<endl;
	pathChanged();

	showFileView();

	addToPathHistory( url.path( +1 ) );
	slotSetWindowCaption( url.prettyURL() );
	m_leftView->slotSetURL( url );
	m_dirLister->openURL( url );
}
//-----------------------------------------------
void KBearFileSysPart::pathChanged() {
	kdDebug()<<"KBearFileSysPart::pathChanged()"<<endl;
	m_fileView->clear();
	m_pathCombo->completionObject()->clear();
	QApplication::restoreOverrideCursor();
	// when KIO::Job emits finished, the slot will restore the cursor
	QApplication::setOverrideCursor( waitCursor );
}
//-----------------------------------------------
void KBearFileSysPart::slotNewItems(const KFileItemList &newone) {
	kdDebug()<<"KBearFileSysPart::slotNewItems()"<<endl;
	if( newone.isEmpty() || !m_fileView )
		return;

	m_fileView->addItemList( newone );

	KFileItem* item;
	KFileItemListIterator it( newone );

	while( (item = it.current()) ) {
		QString tmp;
		if( item->isDir() ) {
			// highlight the dir we come from, if possible
			if( item->url().url(+1) == m_lastURL.url(+1) ) {
				m_fileView->setCurrentItem( item );
				m_fileView->ensureItemVisible( item );
			}
			tmp = it.current()->url().path( +1 );
		}
		else
			tmp = it.current()->url().path( -1 );
		
		if( ! m_connection.isLocal() )
			tmp = m_codec->toUnicode( tmp );
		m_pathCombo->completionObject()->addItem( tmp );
		++it;
	}
	slotUpdateDirsAndFiles( m_fileView->numDirs(), m_fileView->numFiles() );
	QTimer::singleShot(200, this, SLOT(resetCursor()));
}
//-----------------------------------------------
void KBearFileSysPart::resetCursor() {
	kdDebug()<<"KBearFileSysPart::resetCursor()"<<endl;
    QApplication::restoreOverrideCursor();
    m_progress->hide();
}
//-----------------------------------------------
void KBearFileSysPart::setupActions(){
	// First we set up the private actions
    m_privateActionCollection = new KActionCollection( 0L, this, "PrivateActionCollection" );
    actionMenu = new KActionMenu( i18n("Menu"), m_privateActionCollection, "popupMenu" );

    actionSeparator = new KActionSeparator( m_privateActionCollection, "separator" );

	new KAction( i18n("&Open"), 0, this, SLOT( slotOpen() ), m_privateActionCollection, "open" );
	new KAction( i18n("&Other..."), 0, this, SLOT( slotOpenWith() ), m_privateActionCollection, "open with" );
	openWithMenu = new KActionMenu( i18n("Open With..."), m_privateActionCollection, "openWithMenu" );


    new KAction( i18n("Properties..."), ALT+Key_Return, this,
                 SLOT(slotProperties()), m_privateActionCollection, "properties" );


    // Warning: adjust slotViewActionAdded() and slotViewActionRemoved()
    // when you add/remove actions here!

	// Now it's time for the public actions
	actionCollection()->setHighlightingEnabled( true );

	new KAction( i18n("S&ynchronize current directory"), "dirsynch_new", 0, this, SLOT( slotNewSynch() ), actionCollection(), "synch_new" );
	new KAction( i18n("&Clear List"), 0, this, SLOT( slotClearSynch() ), actionCollection(), "clear_synch" );

	mkdirAction = new KAction( i18n("New Directory..."), 0,
							this, SLOT( mkdir() ), actionCollection(), "edit_mkdir" );
	mkdirAction->setIcon( QString::fromLatin1("folder_new") );
	deleteAction = new KAction( i18n( "Delete" ), "editdelete", SHIFT+Key_Delete, this,
							SLOT( slotDelete() ), actionCollection(), "edit_delete" );
	deleteAction->setEnabled( false );
	shredAction = new KAction( i18n("&Shred"), "shredder", CTRL+SHIFT+Key_Delete, this, SLOT( slotShred() ), actionCollection(), "edit_shred" );
	shredAction->setEnabled( false );

	stopAction = new KAction( i18n("S&top"),"stop", 0, this, SLOT( stop() ), actionCollection(), "view_stop" );
	stopAction->setToolTip( i18n("Stop loading directory.") );
	stopAction->setEnabled( false );

	upAction = KStdAction::up( this, SLOT( cdUp() ), actionCollection(), KStdAction::stdName(KStdAction::Up) );
	upAction->setText( i18n("Parent Directory") );
	backAction = KStdAction::back( this, SLOT( back() ), actionCollection(), KStdAction::stdName(KStdAction::Back) );
	forwardAction = KStdAction::forward( this, SLOT(forward()), actionCollection(), KStdAction::stdName(KStdAction::Forward));
	homeAction = KStdAction::home( this, SLOT( home() ), actionCollection(), KStdAction::stdName(KStdAction::Home) );
	homeAction->setText(i18n("Home Directory"));
	reloadAction = KStdAction::redisplay( this, SLOT(reload()), actionCollection(), KStdAction::stdName(KStdAction::Redisplay) );
	reloadAction->setShortcut( Key_F5 );
	reloadAction->setText( i18n("Reload") );
	KStdAction::cut( this, SLOT( slotCut() ), actionCollection(), KStdAction::stdName(KStdAction::Cut) );
	KStdAction::copy( this, SLOT( slotCopy() ), actionCollection(), KStdAction::stdName(KStdAction::Copy));
	KStdAction::paste( this, SLOT( slotPaste() ), actionCollection(), KStdAction::stdName(KStdAction::Paste));
	KStdAction::find( this, SLOT( slotFind() ), actionCollection(), KStdAction::stdName(KStdAction::Find) );
	KStdAction::selectAll( this, SLOT( slotSelectAll() ), actionCollection(), KStdAction::stdName(KStdAction::SelectAll) );
	KStdAction::deselect( this, SLOT( slotClearSelection() ), actionCollection(), KStdAction::stdName(KStdAction::Deselect) );
	new KAction( i18n("&Invert Selection"),CTRL+Key_I, this, SLOT( slotInvertSelection() ), actionCollection(), "edit_invert_selection" );

	//Now we set up the view and sort menu
	// Note: the popupMenu is setup in runtime in slotSetupPopupMenu

	// the view menu actions
	viewActionMenu = new KActionMenu( i18n("View" ), actionCollection(), "view_view_menu" );
	shortAction = new KRadioAction( i18n("Short View"), "view_icon",
							0, actionCollection(), "short view" );
	detailedAction = new KRadioAction( i18n("Detailed View"), "view_detailed",
                                       0, actionCollection(), "detailed view" );
	treeAction = new KToggleAction( i18n("Directory &tree"), "view_tree",0,	this, 0, actionCollection(), "tree view"  );

	showHiddenAction = new KToggleAction( i18n("Show Hidden Files"), 0,
									actionCollection(), "show hidden" );
	QString viewGroup = QString::fromLatin1("view");
	shortAction->setExclusiveGroup( viewGroup );
	detailedAction->setExclusiveGroup( viewGroup );

	connect( shortAction, SIGNAL( toggled( bool ) ), SLOT( slotSimpleView() ));
	connect( detailedAction, SIGNAL( toggled( bool ) ), SLOT( slotDetailedView() ));
	connect( showHiddenAction, SIGNAL( toggled( bool ) ), SLOT( slotToggleHidden( bool ) ));
	viewActionMenu->insert( showHiddenAction );
	viewActionMenu->insert( new KActionSeparator( m_privateActionCollection, "separator1" ) );
	viewActionMenu->insert( shortAction );
	viewActionMenu->insert( detailedAction );
	viewActionMenu->insert( m_privateActionCollection->action("separator1") );
	viewActionMenu->insert( treeAction );
	connect( viewActionMenu->popupMenu(), SIGNAL( aboutToShow() ), this, SLOT( slotSetupViewMenu() ) );
	// the sort menu actions
	sortActionMenu = new KActionMenu( i18n("Sorting"), actionCollection(), "view_sorting_menu");
	byNameAction = new KRadioAction( i18n("By Name"), 0, this,
								SLOT( slotSortByName() ), actionCollection(), "by name" );
	byDateAction = new KRadioAction( i18n("By Date"), 0, this,
								SLOT( slotSortByDate() ), actionCollection(), "by date" );
	bySizeAction = new KRadioAction( i18n("By Size"), 0, this,
						SLOT( slotSortBySize() ), actionCollection(), "by size" );
	reverseAction = new KToggleAction( i18n("Reverse"), 0, this,
						SLOT( slotSortReversed() ), actionCollection(), "reversed" );

	QString sortGroup = QString::fromLatin1("sort");
	byNameAction->setExclusiveGroup( sortGroup );
	byDateAction->setExclusiveGroup( sortGroup );
	bySizeAction->setExclusiveGroup( sortGroup );

	dirsFirstAction = new KToggleAction( i18n("Directories First"), 0, actionCollection(), "dirs first");
	caseInsensitiveAction = new KToggleAction(i18n("Case Insensitive"), 0, actionCollection(), "case insensitive" );

	connect( dirsFirstAction, SIGNAL( toggled( bool ) ), SLOT( slotToggleDirsFirst() ));
	connect( caseInsensitiveAction, SIGNAL( toggled( bool ) ), SLOT( slotToggleIgnoreCase() ));

	sortActionMenu->insert( byNameAction );
	sortActionMenu->insert( byDateAction );
	sortActionMenu->insert( bySizeAction );
	sortActionMenu->insert( reverseAction );
	sortActionMenu->insert( actionSeparator );
	sortActionMenu->insert( dirsFirstAction );
	sortActionMenu->insert( caseInsensitiveAction );

	upAction->setWhatsThis( i18n("This button moves you up one step in the directory tree.") );
	backAction->setWhatsThis( i18n("This button moves you one step back in history list.") );
	forwardAction->setWhatsThis( i18n("This button moves you one step forward in history list.") );
	reloadAction->setWhatsThis( i18n("This button reloads current directory.") );
	mkdirAction->setWhatsThis( i18n("This button creates a new directory in current directory.") );
	stopAction->setWhatsThis( i18n("This button stops loading the directory.") );
	detailedAction->setWhatsThis( i18n("This button change the view to detailed list mode.") );
	shortAction->setWhatsThis( i18n("This button change the view to icon mode.") );
	homeAction->setWhatsThis( i18n("This button moves you to your home directory\n"
                                                                                    "On a remote system this will be your startup directory.") );
	treeAction->setToolTip( i18n("This button toggles the directory tree.") );
	treeAction->setWhatsThis( i18n("This button toggles the directory tree.") );
	action( KStdAction::stdName(KStdAction::Cut) )->setToolTip( i18n("Cut current selection from the active view into the clipboard.") );
	action( KStdAction::stdName(KStdAction::Copy) )->setToolTip( i18n("Copy current selection from the active view into the clipboard.") );
	action( KStdAction::stdName(KStdAction::Paste) )->setToolTip( i18n("Paste clipboard contents into the active view.") );
	action( KStdAction::stdName(KStdAction::Find) )->setToolTip( i18n("Open the Find dialog.") );
	action( KStdAction::stdName(KStdAction::SelectAll) )->setToolTip( i18n("Select all items in the active view.") );
	action( KStdAction::stdName(KStdAction::Deselect) )->setToolTip( i18n("Deselect all items in the active view.") );
	action( "edit_invert_selection" )->setToolTip( i18n("Invert selection in the active view.") );

	action( KStdAction::stdName(KStdAction::Cut) )->setWhatsThis( i18n("Cut current selection from the active view into the clipboard.\n"
				"You can also select the 'Cut' command from the 'Edit' menu.") );
	action( KStdAction::stdName(KStdAction::Copy) )->setWhatsThis( i18n("Copy current selection from the active view into the clipboard.\n"
				"You can also select the 'Copy' command from the 'Edit' menu.") );
	action( KStdAction::stdName(KStdAction::Paste) )->setWhatsThis( i18n("Paste clipboard contents into the active view.\n"
				"You can also select the 'Paste' command from the 'Edit' menu.") );
	action( KStdAction::stdName(KStdAction::Find) )->setWhatsThis( i18n("Open the 'Find' dialog\n"
				"You can also select the 'Find' command from the 'Edit' menu.") );


	action( KStdAction::stdName(KStdAction::Cut) )->setEnabled( false );
	action( KStdAction::stdName(KStdAction::Copy) )->setEnabled( false );
	action( KStdAction::stdName(KStdAction::Paste) )->setEnabled( false );

	connect( QApplication::clipboard(), SIGNAL(dataChanged()), this, SLOT(slotClipboardDataChanged()) );
	// we need to call this first to initialize
	slotClipboardDataChanged();

	connect( actionCollection(), SIGNAL( actionStatusText(const QString &) ),
				SIGNAL( setStatusBarText( const QString& ) ) );
	connect( actionCollection(), SIGNAL( clearStatusText() ), this, SLOT( slotClearStatusBar() ) );

}
//-----------------------------------------------
void KBearFileSysPart::setup(){
	m_progressDelayTimer = new QTimer( this, "progress delay timer" );
	connect( m_progressDelayTimer, SIGNAL( timeout() ), SLOT( slotShowProgress() ));

	////////////////////////////////////////////////////////////////////////
	// Setup the GUI
	QVBoxLayout* m_layout = new QVBoxLayout( widget() );
	m_toolBarUpper = new KToolBar( widget(), "ToolBarUpper", false, false );
	m_toolBarLower = new KToolBar( widget(), "ToolBarLower", false, false );

	m_splitter = new QSplitter( widget() );
	m_splitter->setOpaqueResize( true );
	m_splitter->setSizePolicy( QSizePolicy(QSizePolicy::Expanding,QSizePolicy::Expanding));
	QValueList<int> lst;
	int w = QApplication::desktop()->width( ) / 5;
	lst.append( w<100?100:w );
	m_splitter->setSizes( lst );
	 	
	KURL u = m_connection.url();
	u.setPath( m_connection.remotePath() );
	u.adjustPath( +1 );
	m_leftView = new KBearDirView(  m_splitter, "DirView", m_connection );
	m_leftView->setHomeURL( u );
//	m_splitter->setResizeMode( m_leftView, QSplitter::KeepSize );

	m_toolBarUpper->setEnableContextMenu( false );
	m_toolBarUpper->setMovingEnabled( false );
	m_toolBarLower->setEnableContextMenu( false );
	m_toolBarLower->setMovingEnabled( false );
	m_toolBarUpper->setFullSize(true);
	m_toolBarLower->setFullSize(true);

 	// insert a path label
	QLabel* pathLabel = new QLabel( i18n( "Path: " ), m_toolBarLower );
	m_toolBarLower->insertWidget( 0, 35, pathLabel, 0 );

	m_toolBarLower->insertCombo( m_url.path(1),1,true, SIGNAL( activated( int ) ),
						this, SLOT( slotPathActivated( int ) ),true,i18n("This field displays the current path."), -1,1 );
	m_pathCombo = m_toolBarLower->getCombo( 1 );
	m_pathCombo->setMaxCount( 25 );
	m_pathCombo->setSizeLimit( 10 );
	m_pathCombo->setDuplicatesEnabled( false );
	m_pathCombo->setSizePolicy( QSizePolicy(QSizePolicy::Expanding,QSizePolicy::Fixed));
	QWhatsThis::add( m_pathCombo, i18n("This field displays the current path.\n"
							"You can use this to select a new path.") );

	// insert a filter label
	QLabel* filterLabel = new QLabel( m_toolBarLower );
	filterLabel->setPixmap( BarIcon("filter") );
	m_toolBarLower->insertWidget( 2, 32, filterLabel, 2 );
	// insert a combobox for displaying current filter
	m_filterCombo = new KHistoryCombo(m_toolBarLower, "Filter");
	m_filterCombo->setDuplicatesEnabled( false );
	m_toolBarLower->insertWidget( 3, 120, m_filterCombo, 3 );
	QToolTip::add( m_filterCombo, i18n("This field displays the current filter.") );
	QWhatsThis::add( m_filterCombo, i18n("This field displays the current filter. "
							"By changing the filter you can change what is shown in the view.\n"
							"You can use wildcards here to filter the view") );
	m_filterCombo->setFixedWidth( 120 );
	m_filterCombo->addToHistory( "*" );

	m_toolBarLower->alignItemRight( 2, true );	
	m_toolBarLower->alignItemRight( 3, true );

	m_toolBarLower->setItemAutoSized( 1, true );	
	m_toolBarLower->setStretchableWidget(m_pathCombo);

	m_toolBarUpper->updateRects(true);
	m_toolBarLower->updateRects(true);

		//	plug the actions to the toolbar
	upAction->plug( m_toolBarUpper );
	backAction->plug( m_toolBarUpper );
	forwardAction->plug( m_toolBarUpper );
	homeAction->plug( m_toolBarUpper );
	actionSeparator->plug( m_toolBarUpper );
	reloadAction->plug( m_toolBarUpper );
	actionSeparator->plug( m_toolBarUpper );
	stopAction->plug( m_toolBarUpper );
	actionSeparator->plug( m_toolBarUpper );
	mkdirAction->plug( m_toolBarUpper );
	actionSeparator->plug( m_toolBarUpper );
	treeAction->plug( m_toolBarUpper );
	actionSeparator->plug( m_toolBarUpper );
	shortAction->plug( m_toolBarUpper );
	detailedAction->plug( m_toolBarUpper );

	m_layout->addWidget( m_toolBarUpper );			
	m_layout->addWidget( m_toolBarLower );			
	m_layout->addWidget( m_splitter,10 );			

	m_statusBarUpper = new QLabel( widget(), "StatusLabel" );
	QWhatsThis::add( m_statusBarUpper,
					i18n("This field displays current state of the connection as well as some other useful information.") );
	m_layout->addWidget( m_statusBarUpper );			
	m_statusBarUpper->setSizePolicy( QSizePolicy(QSizePolicy::Preferred,QSizePolicy::Preferred) );
	m_statusBarUpper->setFrameStyle ( QFrame::StyledPanel | QFrame::Sunken );

	m_statusBarLower = new QHBox( widget(), "StatusBarLower"  );
	m_statusBarLower->setFrameStyle ( QFrame::StyledPanel | QFrame::Sunken );
	// insert a progress bar
	m_progress = new KProgress( m_statusBarLower, "progress" );
	m_speedLabel = new QLabel( m_statusBarLower, "SpeedLabel");
	m_speedLabel->setAlignment( Qt::AlignRight | Qt::AlignVCenter | Qt::ExpandTabs );
	m_speedLabel->setFrameStyle ( QFrame::StyledPanel | QFrame::Sunken );

	// insert labels for displaying numbers of dirs and files in current path
	m_directoryLabel = new QLabel( m_statusBarLower, "DirectoryLabel");
	QWhatsThis::add( m_directoryLabel, i18n("This field displays the number of sub directories in current directory.") );
	m_directoryLabel->setFrameStyle ( QFrame::StyledPanel | QFrame::Sunken );
	m_directoryLabel->setText( i18n("Directories: ") );

	m_fileLabel = new QLabel( m_statusBarLower, "FileLabel");
	QWhatsThis::add( m_fileLabel, i18n("This field displays the number of files in current directory.") );
	m_fileLabel->setFrameStyle ( QFrame::StyledPanel | QFrame::Sunken );
	m_fileLabel->setText( i18n("Files: ") );

	m_endLabel = new QLabel( m_statusBarLower, "EndLabel");
	m_endLabel->setAlignment( Qt::AlignLeft | Qt::AlignVCenter | Qt::ExpandTabs );
	m_endLabel->setFrameStyle ( QFrame::StyledPanel | QFrame::Sunken );
	m_endLabel->setText( i18n("Connected: %1").arg( i18n("Not connected") ) );
	QTimer* t = new QTimer( this );
	t->start( 1000 );
	connect( t, SIGNAL( timeout() ), this, SLOT( slotUpdateTime() ) );

	m_statusBarLower->setStretchFactor( m_progress, 7 );
	m_statusBarLower->setStretchFactor( m_speedLabel, 10 );
	m_statusBarLower->setStretchFactor( m_directoryLabel, 0 );
	m_statusBarLower->setStretchFactor( m_fileLabel, 0 );
	m_statusBarLower->setStretchFactor( m_endLabel, 10 );
	m_layout->addWidget( m_statusBarLower );

	// add 'wheel' animation
	m_toolBarUpper->insertAnimatedWidget(1, widget(),"",QString("kde"));
	m_toolBarUpper->alignItemRight( 1, true );	
	QWhatsThis::add( m_toolBarUpper->animatedWidget(1),
			i18n("This animation is active during directory loading and will stop as soon as the directory is completely loaded.") );
	
	connect( m_leftView, SIGNAL( folderSelected( const KFileItem* ) ), this,SLOT( slotDirSelected( const KFileItem* ) ) );
	connect( m_leftView, SIGNAL(dropped( QMimeSource*, const QPoint& )),
					this,SIGNAL(dropped( QMimeSource*, const QPoint& )));

	connect( m_filterCombo, SIGNAL( activated( const QString& ) ), this, SLOT( slotFilterChanged(const QString&) ) );
	connect( m_filterCombo, SIGNAL( returnPressed(const QString&) ), m_filterCombo, SLOT( addToHistory(const QString&) ) );

	connect( treeAction, SIGNAL( toggled(bool)), this, SLOT( slotToggleTreeView(bool)));
	connect( showHiddenAction, SIGNAL( toggled( bool ) ), m_leftView, SLOT( slotToggleShowHidden( bool ) ) );

	connect( actionMenu->popupMenu(), SIGNAL( aboutToShow()), this, SLOT( slotSetupPopupMenu()));
}
//-----------------------------------------------
void KBearFileSysPart::slotEnableGUI( const QString& label, bool enable ) {
	if( label == m_dirLister->connection().label() )
		enableGUI( enable );
}
//-----------------------------------------------
void KBearFileSysPart::enableGUI( bool enable ) {
	setActionsEnabled( enable );
	stopAction->setEnabled( enable );
	forwardAction->setEnabled( enable );
	backAction->setEnabled( enable );
	homeAction->setEnabled( enable );
	upAction->setEnabled( enable );
	reloadAction->setEnabled( enable );
	treeAction->setEnabled( enable );
	m_guiEnabled = enable;
}
//-----------------------------------------------
void KBearFileSysPart::setActionsEnabled( bool enable ) {
	// to prevent multiple connections we first disconnect all
/*
	QObject::disconnect( m_leftView, SIGNAL( folderSelected( const KFileItem* ) ),
							this,SLOT( slotDirSelected( const KFileItem* ) ) );
	QObject::disconnect( showHiddenAction, SIGNAL( toggled( bool ) ),
							m_leftView, SLOT( slotToggleShowHidden( bool ) ) );

	QObject::disconnect( m_filterCombo, SIGNAL( activated( const QString& ) ),
							this, SLOT( slotFilterChanged(const QString&) ) );
	QObject::disconnect( m_filterCombo, SIGNAL( returnPressed(const QString&) ),
							m_filterCombo, SLOT( addToHistory(const QString&) ) );
	if( m_fileView ) {
		QObject::disconnect(m_fileView->signaler(), SIGNAL( dirActivated(const KFileItem *) ),
								this, SLOT( slotDirSelected(const KFileItem*) ) );
		QObject::disconnect(m_fileView->signaler(), SIGNAL( fileSelected(const KFileItem *) ),
								this, SLOT( slotFileSelected(const KFileItem*) ) );
	}
	if( enable ) {
*/
		m_leftView->setEnabled( enable );
		m_leftView->blockSignals( ! enable );
		m_filterCombo->blockSignals( ! enable );
		if( m_fileView )  
			m_fileView->signaler()->blockSignals( ! enable );
/*
		connect( m_leftView, SIGNAL( folderSelected( const KFileItem* ) ), this,SLOT( slotDirSelected( const KFileItem* ) ) );
		connect( m_filterCombo, SIGNAL( activated( const QString& ) ), this, SLOT( slotFilterChanged(const QString&) ) );
		connect( m_filterCombo, SIGNAL( returnPressed(const QString&) ), m_filterCombo, SLOT( addToHistory(const QString&) ) );
		if( ! m_partViewerWidget ) {
			connect( showHiddenAction, SIGNAL( toggled( bool ) ), m_leftView, SLOT( slotToggleShowHidden( bool ) ) );

			if( m_fileView ) {
				connect(m_fileView->signaler(), SIGNAL( dirActivated(const KFileItem *) ),
			          	  this, SLOT( slotDirSelected(const KFileItem*) ) );
				connect(m_fileView->signaler(), SIGNAL( fileSelected(const KFileItem *) ),
			     	       this, SLOT( slotFileSelected(const KFileItem*) ) );
			}
		}
	}
*/
	forwardAction->setEnabled( enable && ! m_forwardStack.isEmpty() && ! m_partViewerWidget );
	backAction->setEnabled( enable && ! m_backStack.isEmpty() );
	homeAction->setEnabled( enable );
	upAction->setEnabled( enable && ! isRoot() );
	reloadAction->setEnabled( enable );
	mkdirAction->setEnabled( enable && ! m_partViewerWidget);
	shortAction->setEnabled( enable && ! m_partViewerWidget);
	detailedAction->setEnabled( enable && ! m_partViewerWidget);
	treeAction->setEnabled( enable );
	m_pathCombo->setEnabled( enable );
	m_filterCombo->setEnabled( enable && ! m_partViewerWidget);
	if( enable && m_partViewerWidget ){
		backAction->setEnabled( true );
		homeAction->setEnabled( true );
		upAction->setEnabled( true );
	}
}
//-----------------------------------------------
void KBearFileSysPart::slotFilterChanged( const QString& nameFilter ) {
	kdDebug()<<"KBearFileSysPart::slotFilterChanged() new filter="<<nameFilter<<endl;
	m_dirLister->setNameFilter( nameFilter );
	reload();
}
//-----------------------------------------------
void KBearFileSysPart::slotSpeed( KIO::Job*, unsigned long speed ) {
	slotSpeed( (int) speed );
}
//-----------------------------------------------
void KBearFileSysPart::slotSpeed( int speed ){
	QString s;
	if( speed == 0)
		s = i18n("Stalled");
	else
		s = ( i18n( "%1/s" ).arg( KIO::convertSize( (KIO::filesize_t)speed ) ) );
	m_speedLabel->setText( s );
}
//-----------------------------------------------
void KBearFileSysPart::slotPathActivated( int item ) {
	KURL tmp( m_url );
	QString p = m_pathCombo->text( item );
	if( ! m_connection.isLocal() )
		p = m_codec->fromUnicode( p );
	tmp.setPath( p );
	if( 	m_pathCombo->text( item ) == QString::null ) {
		m_pathCombo->removeItem( item );
	}
	kdDebug()<<"KBearFileSysPart::slotPathActivated() url="<<tmp.prettyURL()<<endl;
	m_dirLister->statURL( tmp );
}
//-----------------------------------------------
KFileView* KBearFileSysPart::createView( QWidget* parent, KFile::FileView view ) {
	KFileView* newView = 0L;
	if((view & KFile::Simple) == KFile::Simple ) {
		newView = new KBearIconView( parent, "simple view", m_connection);
		connect( static_cast<KBearIconView*>(newView), SIGNAL(dropped( QMimeSource*, const QPoint& )),
					this,SIGNAL(dropped(QMimeSource*, const QPoint&)));
		connect( static_cast<KBearIconView*>(newView), SIGNAL(selectionChanged()),
					this,SLOT(slotSelectionChanged()));
	}
	else {
		newView = new KBearDetailView( parent, "detail view", m_connection);
		connect( static_cast<KBearDetailView*>(newView), SIGNAL(dropped( QMimeSource*, const QPoint& )),
					this,SIGNAL(dropped(QMimeSource*, const QPoint&)));
		connect( static_cast<KBearDetailView*>(newView), SIGNAL(selectionChanged()),
					this,SLOT(slotSelectionChanged()));
	}
//QSplitter::FollowSizeHint
//	m_splitter->setResizeMode( newView->widget(), QSplitter::KeepSize );
	return newView;
}
//-----------------------------------------------
void KBearFileSysPart::slotSelectionChanged() {
//	if( m_fileView->selectedItems()->count() > 0 ) {
		unsigned int count = m_fileView->selectedItems()->count();
		action( KStdAction::stdName(KStdAction::Cut) )->setEnabled( count );
		action( KStdAction::stdName(KStdAction::Copy) )->setEnabled( count );
		deleteAction->setEnabled( count );
		shredAction->setEnabled( m_url.isLocalFile() && count );
/*
	}
	else {
		action( KStdAction::stdName(KStdAction::Cut) )->setEnabled( false );
		action( KStdAction::stdName(KStdAction::Copy) )->setEnabled( false );
		deleteAction->setEnabled( false );
		shredAction->setEnabled( false );
	}
*/
}
//-----------------------------------------------
void KBearFileSysPart::slotDeleteItem( KFileItem *item ) {
//	kdDebug()<<"KBearFileSysPart::slotDeleteItem() url="<<item->url().prettyURL()<<endl;
    m_fileView->removeItem( item );
    slotUpdateDirsAndFiles(m_fileView->numDirs(), m_fileView->numFiles());
}
//-----------------------------------------------
void KBearFileSysPart::slotDetailedView() {
//	kdDebug()<<"Switching to Detailed view"<<endl;
    KFile::FileView view = static_cast<KFile::FileView>( (m_currentViewKind & ~KFile::Simple) | KFile::Detail );
    setView( view );
}
//-----------------------------------------------
void KBearFileSysPart::slotSimpleView() {
//	kdDebug()<<"Switching to Simple view"<<endl;
    KFile::FileView view = static_cast<KFile::FileView>( (m_currentViewKind & ~KFile::Detail) | KFile::Simple );
    setView( view );
}
//-----------------------------------------------
void KBearFileSysPart::slotClearStatusBar() {
	emit setStatusBarText( QString::null );
}
//-----------------------------------------------
void KBearFileSysPart::mkdir() {
	setActionsEnabled( false );
    // Modal widget asking the user the name of a new directory
    //
    KDialogBase *lMakeDir;
    QLabel *label;
    KLineEdit *ed;
    QVBox *vbox;

    // Create widgets, and display using geometry management
    //
    lMakeDir = new KDialogBase( m_fileView->widget(),
				"MakeDir Dialog", true, i18n("New Directory"),
				KDialogBase::Ok | KDialogBase::Cancel );
    vbox = new QVBox( lMakeDir );
    vbox->setSpacing( KDialog::spacingHint() );
    lMakeDir->setMainWidget( vbox );
    label = new QLabel( vbox );
    label->setAlignment( AlignLeft | AlignVCenter );
    label->setText(i18n("Create new directory in: ") +
		   QString::fromLatin1( "\n" ) + /* don't break i18n now*/
		   url().prettyURL() );
    ed= new KLineEdit( vbox );
    ed->setText( i18n("New Directory") );
    ed->selectAll();
    connect(ed, SIGNAL(returnPressed()), lMakeDir, SLOT(accept()) );

    connect( lMakeDir->actionButton( KDialogBase::Ok ), SIGNAL(clicked()),
	     lMakeDir, SLOT(accept()) );
    connect( lMakeDir->actionButton( KDialogBase::Cancel ), SIGNAL(clicked()),
	     lMakeDir, SLOT(reject()) );


    // If the users presses enter (not escape) then create the dir
    // and insert it into the ListBox
    lMakeDir->setMinimumSize( 300, 120); // default size
    ed->grabKeyboard();
    if ( lMakeDir->exec() == QDialog::Accepted && !ed->text().isEmpty() ) {
		KURL tmp = m_url;
		tmp.setPath( m_url.path( +1 ) + ed->text() );
		m_dirLister->mkdir( tmp );
	}

    delete lMakeDir;
	setActionsEnabled( true );
}
//-----------------------------------------------
void KBearFileSysPart::setView( KFile::FileView view ) {
    m_currentViewKind = static_cast<int>(view);

    KFileView* newView = createView( m_splitter, view );

    widget()->setFocusProxy(newView->widget());
    newView->setSorting( m_mySorting );
    newView->setOnlyDoubleClickSelectsFiles( false );
    connectView(newView ); // also deletes the old view
}
//-----------------------------------------------
void KBearFileSysPart::connectView(KFileView *view ) {
    bool listDir = true;

	view->setViewMode(KFileView::All);
	view->setSelectionMode( KFile::Extended );
	QValueList<int> sizes = m_splitter->sizes(); // save splitter sizes so we can restore it for the new view

    if(m_fileView) {
        // transfer the state from old view to new view
        view->clear();
        view->addItemList( *m_fileView->items() );
        listDir = false;

        if ( m_fileView->widget()->hasFocus() )
            view->widget()->setFocus();

        KFileItem *oldCurrentItem = m_fileView->currentFileItem();
        if ( oldCurrentItem ) {
            view->setCurrentItem( oldCurrentItem );
            view->setSelected( oldCurrentItem, false );
	    view->ensureItemVisible( oldCurrentItem );
        }

        const KFileItemList *oldSelected = m_fileView->selectedItems();
        if ( !oldSelected->isEmpty() ) {
            KFileItemListIterator it( *oldSelected );
            for ( ; it.current(); ++it )
                view->setSelected( it.current(), true );
        }

        m_fileView->widget()->hide();
		dynamic_cast<QObject*>(m_fileView)->deleteLater();
    }

    m_fileView = view;
    m_viewActionCollection = 0;
    KFileViewSignaler *sig = view->signaler();

    connect(sig, SIGNAL( activatedMenu(const KFileItem *, const QPoint& ) ),
            this, SLOT( activatedMenu(const KFileItem *, const QPoint& )));
    connect(sig, SIGNAL( dirActivated(const KFileItem *) ),
            this, SLOT( slotDirSelected(const KFileItem*) ) );
    connect(sig, SIGNAL( fileSelected(const KFileItem *) ),
            this, SLOT( slotFileSelected(const KFileItem*) ) );
    connect(sig, SIGNAL( sortingChanged( QDir::SortSpec ) ),
            this, SLOT( slotViewSortingChanged( QDir::SortSpec )));

    if ( reverseAction->isChecked() != m_fileView->isReversed() )
        slotSortReversed();

    updateViewActions();
//    m_fileView->widget()->resize(widget()->size());
    m_fileView->widget()->show();
	m_splitter->setSizes( sizes ); // restore splitter sizes

/*
    if ( listDir ) {
        pathChanged();
        m_dirLister->statURL( m_url );
    }
    else
*/
	view->listingCompleted();
}
//-----------------------------------------------
void KBearFileSysPart::slotOpenFile( const KFileItem* item ) {
	slotStatusMessage( i18n( "Opening file: %1").arg(item->url().prettyURL() ) );


	if( m_partViewerWidget && m_partViewer ) {
		m_partViewer->manager()->replacePart( m_partViewer, this );
		delete m_partViewerWidget;
		m_partViewerWidget = 0L;
		m_partViewer = 0L;
		m_leftView->setDNDEnabled( true ); //  It's safe to use DND now
		m_splitter->setSizes( m_splitterSizes );
	}
	m_splitterSizes = m_splitter->sizes();
	m_splitterSizes.append( m_splitterSizes[1] );
	m_partViewerWidget = new QWidget( m_splitter, "PartViewerWidget");
	// request a preview part from dirlister
	m_lastURL = m_url;
	m_url = item->url();
	m_url.adjustPath( -1 );
	m_dirLister->getPreviewPart( item->url(), m_partViewerWidget );
}
//-----------------------------------------------
void KBearFileSysPart::slotGotPreviewPart( KParts::ReadOnlyPart* part, const KURL& realURL) {
	if( part ) {
		m_partViewer = part;
		kdDebug()<<"KBearFileSysPart::slotGotPreviewPart received part="<<m_partViewer<<endl;

		setActionsEnabled( false );
		m_leftView->setDNDEnabled( false ); // we have to turn off DND when in preview mode
//		addToHistory( m_url.path( -1 ) );
		addToPathHistory( m_url.path( -1 ) );
		m_partViewer->widget()->setCursor( Qt::WaitCursor );
		m_partViewer->widget()->setFocusPolicy( QWidget::WheelFocus );
		manager()->replacePart( this, m_partViewer );

		m_partViewer->setProgressInfoEnabled( false );
		KParts::BrowserExtension* browserExtension = KParts::BrowserExtension::childObject( m_partViewer );
		if( browserExtension ) {
			KParts::URLArgs args = browserExtension->urlArgs();
			args.metaData() = m_connection.metaData();
			browserExtension->setURLArgs( args );
			connect( browserExtension, SIGNAL( speedProgress( int ) ),
						this, SLOT( slotSpeed( int ) ) );
			connect( browserExtension, SIGNAL( loadingProgress( int ) ),
						this, SLOT( slotProgress( int ) ) );
		}
		connect( m_partViewer, SIGNAL( started( KIO::Job* ) ), this, SLOT( slotJobStarted( KIO::Job* ) ) );
		connect( m_partViewer, SIGNAL( completed() ), this, SLOT( slotFinishedLoading() ) );
		connect( m_partViewer, SIGNAL( completed() ), m_dirLister, SLOT( slotDisconnectSlaveInfo() ) );
		connect( m_partViewer, SIGNAL( canceled( const QString& ) ), this, SLOT( slotFinishedLoading() ) );
		connect( m_partViewer, SIGNAL( canceled( const QString& ) ), m_dirLister, SLOT( slotDisconnectSlaveInfo() ) );
		connect( m_partViewer, SIGNAL( setWindowCaption( const QString& ) ),
					this, SLOT( slotSetWindowCaption( const QString& ) ) );
		connect( m_partViewer, SIGNAL( setStatusBarText( const QString& ) ),
					this, SLOT( slotInfoMessage( const QString& ) ) );

		m_fileView->widget()->hide();
		m_partViewer->openURL( realURL );
		slotSetWindowCaption( m_url.prettyURL() );
		m_leftView->slotSetURL( m_url.upURL() );
		m_splitter->setResizeMode( m_partViewerWidget, QSplitter::Stretch );
		QVBoxLayout* layout = new QVBoxLayout( m_partViewerWidget );
		layout->addWidget( m_partViewer->widget() );
		m_partViewerWidget->show();

		m_splitter->setSizes( m_splitterSizes );
	}
	else {
		kdDebug()<<"KBearFileSysPart::slotOpenFile No part !!!!!!!!!!"<<endl;
		delete m_partViewerWidget;  // no use for this
		m_partViewerWidget = 0L;
		m_partViewer = 0L;
//		const_cast<KFileItem*>(item)->run();
		KRun::displayOpenWithDialog( m_url );
		slotStatusMessage( i18n( "No Operation in Progress" ) );
		m_backStack.pop();	// we don't want this in history since we didn't open it internally
		m_url = m_lastURL; // since we couldn't open it we set it back to old URL
	}
}
//-----------------------------------------------
void KBearFileSysPart::slotFileSelected( const KFileItem* item ) {
	if( ! item )
		return;

	Window root;
	Window child;
	int root_x, root_y, win_x, win_y;
	uint keybstate;
	XQueryPointer( qt_xdisplay(), qt_xrootwin(), &root, &child, &root_x, &root_y, &win_x, &win_y, &keybstate );
	//Don't open the file if control or shift was pressed
	//because we want the user to be able to make multi selections with these keys
	if( keybstate & ControlMask || keybstate & ShiftMask )
		return;

	KURL tmp = item->url();
	tmp.adjustPath( -1 ); // we know it's a file here
	if( tmp != m_url )
		addToHistory( m_url );
	slotOpenFile( item );
}
//-----------------------------------------------
void KBearFileSysPart::activatedMenu( const KFileItem *, const QPoint& pos ) {
	if( ! m_isConnected )
		return;
//	slotSetupViewMenu();
	actionMenu->popup( pos );
}
//-----------------------------------------------
void KBearFileSysPart::slotSetupViewMenu() {
    // If we have a new view actionCollection(), insert its actions
    //  into viewActionMenu.
    if( m_fileView && m_viewActionCollection != m_fileView->actionCollection() ) {
        m_viewActionCollection = m_fileView->actionCollection();

        if ( !m_viewActionCollection->isEmpty() ) {
            viewActionMenu->insert( actionSeparator );
            for ( uint i = 0; i < m_viewActionCollection->count(); i++ )
                viewActionMenu->insert( m_viewActionCollection->action( i ));
        }

        connect( m_viewActionCollection, SIGNAL( inserted( KAction * )),
                 SLOT( slotViewActionAdded( KAction * )));
        connect( m_viewActionCollection, SIGNAL( removed( KAction * )),
                 SLOT( slotViewActionRemoved( KAction * )));
    }
}
//-----------------------------------------------
void KBearFileSysPart::cleanUpPartViewer() {
	if( m_partViewerWidget && m_partViewer ) { // are we displaying a file in a view part ?
		// if so we have to delete the view part and make sure  the m_fileView is visible
		m_partViewer->manager()->replacePart( m_partViewer, this );
		delete m_partViewerWidget;
		m_partViewerWidget = 0L;
		m_partViewer = 0L;

		connectDirlister();

		m_splitter->setSizes( m_splitterSizes );

		slotToggleTreeView( treeAction->isChecked() );
		m_leftView->setEnabled( true );
		m_leftView->setDNDEnabled( true ); //  It's safe to use DND now
	}
}
//-----------------------------------------------
void KBearFileSysPart::showFileView() {
	kdDebug()<<"KBearFileSysPart::showFileView() m_partViewerWidget="<<m_partViewerWidget
				<<" m_partViewer="<<m_partViewer<<endl;
	if( m_partViewerWidget && m_partViewer ) { // are we displaying a file in a view part ?
		cleanUpPartViewer();
		if( ! m_url.cmp( m_dirLister->url() ) && ! m_dirLister->isLocal() )
			reload();
	}
	setActionsEnabled( true );
	m_fileView->widget()->show();

}
//-----------------------------------------------
void KBearFileSysPart::updateSortActions() {
    if ( KFile::isSortByName( m_mySorting ) )
        byNameAction->setChecked( true );
    else if ( KFile::isSortByDate( m_mySorting ) )
        byDateAction->setChecked( true );
    else if ( KFile::isSortBySize( m_mySorting ) )
        bySizeAction->setChecked( true );

    dirsFirstAction->setChecked( KFile::isSortDirsFirst( m_mySorting ));
    caseInsensitiveAction->setChecked(KFile::isSortCaseInsensitive(m_mySorting));

    if ( m_fileView )
	reverseAction->setChecked( m_fileView->isReversed() );
}
//-----------------------------------------------
void KBearFileSysPart::slotViewSortingChanged( QDir::SortSpec sort ) {
    m_mySorting = sort;
    updateSortActions();
}
//-----------------------------------------------
void KBearFileSysPart::updateViewActions() {
    KFile::FileView fv = static_cast<KFile::FileView>( m_currentViewKind );
    shortAction->setChecked( KFile::isSimpleView( fv ));
    detailedAction->setChecked( KFile::isDetailView( fv ));
}
//-----------------------------------------------
KURL::List KBearFileSysPart::selectedURLs() {
	KURL::List lst;
	KFileItemListIterator it( * m_fileView->selectedItems() );
	for ( ; it.current(); ++it ){
		lst.append( (*it)->url() );
	}
	return lst;
}
//-----------------------------------------------
void KBearFileSysPart::slotSortByName() {
    int sorting = (m_fileView->sorting()) & ~QDir::SortByMask;
    m_fileView->setSorting( static_cast<QDir::SortSpec>( sorting | QDir::Name ));
    m_mySorting = m_fileView->sorting();
}
//-----------------------------------------------
void KBearFileSysPart::slotSortBySize() {
    int sorting = (m_fileView->sorting()) & ~QDir::SortByMask;
    m_fileView->setSorting( static_cast<QDir::SortSpec>( sorting | QDir::Size ));
    m_mySorting = m_fileView->sorting();
}
//-----------------------------------------------
void KBearFileSysPart::slotSortByDate() {
    int sorting = (m_fileView->sorting()) & ~QDir::SortByMask;
    m_fileView->setSorting( static_cast<QDir::SortSpec>( sorting | QDir::Time ));
    m_mySorting = m_fileView->sorting();
}
//-----------------------------------------------
void KBearFileSysPart::slotSortReversed() {
    if( m_fileView )
        m_fileView->sortReversed();
}
//-----------------------------------------------
KURL KBearFileSysPart::currentDir() {
	if( ! m_partViewerWidget )
		return m_url;
	else {
		KURL url = m_url;
		url.setPath( m_url.directory( false ) );
		return url;
	}
}
//-----------------------------------------------
void KBearFileSysPart::reload() {
	kdDebug()<<"KBearFileSysPart::reload()"<<endl;
	if( ! m_partViewerWidget ) {
		pathChanged();
		m_leftView->slotPrepareForReread( m_url );
		m_dirLister->openURL( m_url, true );
	}
	else
		m_dirLister->statURL( m_url );
}
//-----------------------------------------------
void KBearFileSysPart::home() {
	if( m_partViewer && m_partViewer->inherits("KBearDirSynchPart") )
     	cleanUpPartViewer();
	addToHistory( m_url );
	m_lastURL = m_url;
	if( m_dirLister->isLocal() )
		m_url = QDir::homeDirPath();
	else
     	m_url.setPath( m_connection.remotePath() );

	kdDebug()<<"KBearFileSysPart::home() homeURL="<<m_url.prettyURL()<<endl;
	m_url.adjustPath( +1 );
	setDir( m_url );
}
//-----------------------------------------------
void KBearFileSysPart::cdUp() {
	kdDebug()<<"KBearFileSysPart::cdUp()"<<endl;
	if( ! m_partViewer || (m_partViewer && ! m_partViewer->inherits("KBearDirSynchPart") )) {
		kdDebug()<<"KBearFileSysPart::cdUp() no KBearDirSynchPart"<<endl;
		addToHistory( m_url );
		m_lastURL = m_url;
		m_url = m_url.upURL();
		m_url.adjustPath( +1 ); // we know we will land in a directory
		setDir( m_url );
	}
	else
		showFileView();
}
//-----------------------------------------------
void KBearFileSysPart::back() {
	if( m_backStack.isEmpty() ) {
		backAction->setEnabled( false );
		return;
	}

	if( ! m_partViewer || (m_partViewer && ! m_partViewer->inherits("KBearDirSynchPart") )) {
		m_forwardStack.push( new KURL(m_url) );
		KURL* url = m_backStack.pop();
		// we have to stat the url since we don't know if it's a directory or a file
		kdDebug()<<"KBearFileSysPart::back() new url="<<url->prettyURL()<<endl;
		m_dirLister->statURL( *url );
		delete url;
	}
	else
		showFileView();
}
//-----------------------------------------------
void KBearFileSysPart::forward() {
	if( m_forwardStack.isEmpty() ) {
		forwardAction->setEnabled( false );
		return;
	}

	if( ! m_partViewer || (m_partViewer && ! m_partViewer->inherits("KBearDirSynchPart") )) {
		m_backStack.push(new KURL(m_url));
		KURL* url = m_forwardStack.pop();
		// we have to stat the url since we don't know if it's a directory or a file
		kdDebug()<<"KBearFileSysPart::forward() new url="<<url->prettyURL()<<endl;
		m_dirLister->statURL( *url );
		delete url;
	}
	else
		showFileView();
}
//-----------------------------------------------
void KBearFileSysPart::stop() {
	kdDebug()<<"KBearFileSysPart::stop()"<<endl;
	m_dirLister->stop();
	if( m_partViewerWidget ) {
		QObject::disconnect( m_partViewer, SIGNAL( canceled( const QString& ) ), this, 0 );
		m_partViewer->closeURL();
	}	
}
//-----------------------------------------------
void KBearFileSysPart::readProperties( KConfig* config, const QString& group ) {
    if ( ! config )
        return;
    const QString oldGroup = config->group();
    if ( !group.isEmpty() )
        config->setGroup( group );

    m_currentViewKind = 0;
    int sorting = 0;

    QString viewStyle = config->readEntry( QString::fromLatin1("View Style"),
                                       QString::fromLatin1("Detail") );
    if ( viewStyle == QString::fromLatin1("Detail") )
        m_currentViewKind |= KFile::Detail;
    else
        m_currentViewKind |= KFile::Simple;

    if ( config->readBoolEntry( QString::fromLatin1("Sort case insensitively"),
                            true ) )
        sorting |= QDir::IgnoreCase;
    if ( config->readBoolEntry( QString::fromLatin1("Sort directories first"),
                            true ) )
        sorting |= QDir::DirsFirst;


    QString name = QString::fromLatin1("Name");
    QString sortBy = config->readEntry( QString::fromLatin1("Sort by"), name );
    if ( sortBy == name )
        sorting |= QDir::Name;
    else if ( sortBy == QString::fromLatin1("Size") )
        sorting |= QDir::Size;
    else if ( sortBy == QString::fromLatin1("Date") )
        sorting |= QDir::Time;

    m_mySorting = static_cast<QDir::SortSpec>( sorting );
    setSorting( m_mySorting );


    if ( config->readBoolEntry( QString::fromLatin1("Show hidden files"),false ) ) {
         showHiddenAction->setChecked( true );
//         m_dirLister->setShowingDotFiles( true );
    }
    if ( config->readBoolEntry( QString::fromLatin1("Sort reversed"), false ) )
        reverseAction->setChecked( true );

	treeAction->setChecked( config->readBoolEntry( "Show Tree View", true ) );
	slotToggleTreeView( treeAction->isChecked() );
	config->setGroup( oldGroup );
}
//-----------------------------------------------
void KBearFileSysPart::slotViewActionAdded( KAction *action ) {
	if( viewActionMenu->popupMenu()->count() == 6 ) // need to add a separator
		viewActionMenu->insert( actionSeparator );

	viewActionMenu->insert( action );
}
//-----------------------------------------------
void KBearFileSysPart::slotViewActionRemoved( KAction *action ) {
	viewActionMenu->remove( action );

	if( viewActionMenu->popupMenu()->count() == 7 ) // remove the separator
		viewActionMenu->remove( actionSeparator );
}
//-----------------------------------------------
void KBearFileSysPart::slotToggleHidden( bool show ) {
	m_dirLister->setShowingDotFiles( show );
	if( ! m_dirLister->isLocal() )
		reload();
	if( m_fileView )
		m_fileView->listingCompleted();
}
//-----------------------------------------------
void KBearFileSysPart::slotToggleTreeView( bool show ){
	if( show )
		m_leftView->show();
	else{
		m_leftView->hide();
	}
}
//-----------------------------------------------
void KBearFileSysPart::saveProperties( KConfig* config, const QString& group ) {
    if ( ! config )
        return;

    const QString oldGroup = config->group();

    if ( !group.isEmpty() )
        config->setGroup( group );

    QString sortBy = QString::fromLatin1("Name");
    if ( KFile::isSortBySize( m_mySorting ) )
        sortBy = QString::fromLatin1("Size");
    else if ( KFile::isSortByDate( m_mySorting ) )
        sortBy = QString::fromLatin1("Date");
    config->writeEntry( QString::fromLatin1("Sort by"), sortBy );

    config->writeEntry( QString::fromLatin1("Sort reversed"),
                    reverseAction->isChecked() );
    config->writeEntry( QString::fromLatin1("Sort case insensitively"),
                    caseInsensitiveAction->isChecked() );
    config->writeEntry( QString::fromLatin1("Sort directories first"),
                    dirsFirstAction->isChecked() );

    config->writeEntry( QString::fromLatin1("Show hidden files"),
                    showHiddenAction->isChecked() );

    KFile::FileView fv = static_cast<KFile::FileView>( m_currentViewKind );
    QString style;
    if ( KFile::isDetailView( fv ) )
        style = QString::fromLatin1("Detail");
    else if ( KFile::isSimpleView( fv ) )
        style = QString::fromLatin1("Simple");
    config->writeEntry( QString::fromLatin1("View Style"), style );

	m_fileView->writeConfig( config, group );

	config->writeEntry( "Show Tree View", treeAction->isChecked() );
	config->setGroup( oldGroup );
}
//-----------------------------------------------
void KBearFileSysPart::slotFinishedLoading(){
	kdDebug()<<"KBearFileSysPart::slotFinishedLoading()"<<endl;
	m_progressDelayTimer->stop();
	if( ! m_partViewerWidget )
		slotStatusMessage( i18n( "No Operation in Progress" ) );
	emit setStatusBarText( i18n( "Ready." ) );
	stopAction->setEnabled( false );
	m_toolBarUpper->animatedWidget(1)->stop(); // stop wheel animation
	m_leftView->slotStopAnimation();
	m_speedLabel->setText( QString::null );
	slotProgress( 100 );
	m_progress->hide();
	resetCursor();

	if( m_fileView )
		m_fileView->listingCompleted();
	
	setActionsEnabled( true );
	m_isLoading = false;
}
//-----------------------------------------------
void KBearFileSysPart::slotStartLoading(){
	if( m_isLoading )
		return;
	kdDebug()<<"KBearFileSysPart::slotStartLoading()"<<endl;
	m_isLoading = true;
	setActionsEnabled( false );
	m_progress->setProgress( 0 );
	// delay showing the progressbar for one second
//	m_progressDelayTimer->start( 1000, true );
	slotShowProgress();
	if( ! m_partViewerWidget )
		slotStatusMessage( i18n( "Reading ..." ) );
	m_toolBarUpper->animatedWidget(1)->start(); // start wheel animation
	m_leftView->slotStartAnimation();
	stopAction->setEnabled( true );
}
//-----------------------------------------------
void KBearFileSysPart::slotShowProgress() {
//    if( m_progress->value() > 0 ) {
		m_progress->raise();
	    m_progress->show();
	    QApplication::flushX();
//	}
}
//-----------------------------------------------
QPopupMenu* KBearFileSysPart::popupMenu() {
	return ( m_isConnected ? actionMenu->popupMenu() : 0L );
}
//-----------------------------------------------
void KBearFileSysPart::slotStatusMessage( const QString& mess){
	kdDebug()<<"KBearFileSysPart::slotStatusMessage="<<mess<<endl;
	m_statusBarUpper->setText( mess );
}
//-----------------------------------------------
void KBearFileSysPart::slotUpdateDirsAndFiles( int dirs, int files ){
	m_directoryLabel->setText( i18n( "Directories: %1").arg(dirs) );
	m_fileLabel->setText( i18n( "Files: %1" ).arg(files) );
}
//-----------------------------------------------
void KBearFileSysPart::slotCopy(){
	doCutCopy( false );
}
//-----------------------------------------------
void KBearFileSysPart::slotCut(){
	doCutCopy( true );
}
//-----------------------------------------------
void KBearFileSysPart::doCutCopy( bool cut ){
	KURL::List urls = selectedURLs();
     if( urls.count() == 0 )
		return;
	QMap<QString,QString> metaData;
	KURLDrag* dragObject = KURLDrag::newDrag( urls, m_connection.metaData() );
	dragObject->metaData().insert("action", cut ? "move" : "copy" );
	dragObject->metaData().insert("sourceName", m_connection.label().utf8() );
	QApplication::clipboard()->setData( dragObject );
	action( KStdAction::stdName(KStdAction::Paste) )->setEnabled( true );
}
//-----------------------------------------------
void KBearFileSysPart::slotPaste() {
	if( KIO::isClipboardEmpty() ) {
		action( KStdAction::stdName(KStdAction::Paste) )->setEnabled( false );
		return;
	}
	KURLDrag* data = static_cast<KURLDrag*>(QApplication::clipboard()->data());
	KURL::List urls;
	KIO::MetaData metaData;
	if( QUriDrag::canDecode( data ) && KURLDrag::decode( data, urls, metaData ) ) {
		KURLDrag* drag = KURLDrag::newDrag( urls, metaData, widget()->parentWidget() );
		emit transfer( drag ); // emit a new transfer
		if( ! metaData.contains("action") || metaData["action"] == "move" )
			QApplication::clipboard()->clear();
	}
	if( KIO::isClipboardEmpty() )
		action( KStdAction::stdName(KStdAction::Paste) )->setEnabled( false );
}
//-----------------------------------------------
void KBearFileSysPart::slotFind(){
	KBearSearchDialog dialog( widget(), "SearchDialog" );
	KFileItem* item = m_fileView->firstFileItem();
	QRegExp regExp;
	regExp.setWildcard( true );
	unsigned int i = 0;
	if( dialog.exec() == QDialog::Rejected )
		return;	
	bool found = false;
	while( i < m_fileView->count() ) {
		regExp.setPattern( dialog.searchEdit->text() );
		regExp.setCaseSensitive( dialog.caseSensitiveCheckBox->isChecked() );
		i++;
		if( regExp.exactMatch( item->name() ) ) {
			found = true;
			m_fileView->clearSelection();
			m_fileView->setSelected( item, true );
			m_fileView->setCurrentItem( item );
			m_fileView->ensureItemVisible( item );
			if( i < m_fileView->count() && dialog.exec() == QDialog::Rejected )
				return;	
		}
		else if( i == m_fileView->count() && ! found) {
			m_fileView->clearSelection();
			KMessageBox::information( widget(), i18n("Could not find any match for: %1").arg(dialog.searchEdit->text()),i18n("Not found...") );
			return;
		}
		if( i == m_fileView->count() ) {
			if(KMessageBox::questionYesNo( widget(), i18n("Whole view searched.\nSearch again ?"),
														i18n("Search") ) == KMessageBox::No ) {
					return;
			}
			i = 0;
			found = false;
			item = m_fileView->firstFileItem();
		}
		else
			item = m_fileView->nextItem( item );
	}
}
//-----------------------------------------------
void KBearFileSysPart::slotClearSelection(){
	m_fileView->clearSelection();
}
//-----------------------------------------------
void KBearFileSysPart::slotRedirected( const KURL& newURL ) {
	kdDebug()<<"KBearFileSysPart::slotRedirected new url="<<newURL.prettyURL()<<endl;
	m_pathCombo->completionObject()->clear();
	m_pathCombo->changeItem( newURL.path(), m_pathCombo->currentItem() );
}
//-----------------------------------------------
void KBearFileSysPart::slotClear(){
	pathChanged();
}
//-----------------------------------------------
void KBearFileSysPart::slotSelectAll(){
	m_fileView->selectAll();
}
//-----------------------------------------------
void KBearFileSysPart::slotInvertSelection(){
	m_fileView->invertSelection();
}
//-----------------------------------------------
void KBearFileSysPart::slotOpen(){
	KURL::List urlList = selectedURLs();
	for( KURL::List::Iterator it = urlList.begin(); it != urlList.end(); ++it )
		(void)new KRun( *it );
}
//-----------------------------------------------
void KBearFileSysPart::slotOpenWith(){
	KRun::displayOpenWithDialog( selectedURLs() );
}
//-----------------------------------------------
void KBearFileSysPart::slotOpenWithService( int id ) {
	QPopupMenu* openMenu = openWithMenu->popupMenu();
	QObject::disconnect( openMenu, SIGNAL(activated(int)), this, SLOT(slotOpenWithService( int ) ) );
	for( unsigned int i = 0; i < m_serviceActionList.count(); i++ ) {
		if( openMenu->text( id ) == m_serviceActionList.at( i )->text() ) {
			KRun::run( m_serviceActionList.at( i )->name(), selectedURLs(),
							m_serviceActionList.at( i )->text(), m_serviceActionList.at( i )->icon() );
		}
	}
}
//-----------------------------------------------
int KBearFileSysPart::setupOpenWithMenu() {
	static const QString& app_str = KGlobal::staticQString("Application");
	QPopupMenu* openMenu;
	openMenu = openWithMenu->popupMenu();
	openMenu->clear();
	m_serviceActionList.clear();  // remove and delete action from list
	//Now everything is cleared so lets see if we can find any services matching our selection
	QValueList<KServiceOffer> offers = KServiceTypeProfile::offers( m_fileView->currentFileItem()->mimetype(), app_str );
	QValueList<KServiceOffer>::Iterator iter;
	for( iter = offers.begin(); iter != offers.end(); ++iter) {
		// If we gets here there are matching services
		KService::Ptr servicePtr = (*iter).service();
		KAction* action = new KAction( servicePtr->name(), servicePtr->icon(),
							KShortcut(), this, servicePtr->exec());
		m_serviceActionList.append( action );
		action->plug( openMenu );
		connect( openMenu, SIGNAL(activated(int)), this, SLOT(slotOpenWithService( int ) ) );
	}
	actionSeparator->plug( openMenu );
	m_privateActionCollection->action("open with")->setText( i18n("&Other...") );
	m_privateActionCollection->action("open with")->plug( openMenu );
	return offers.count();
}
//-----------------------------------------------
void KBearFileSysPart::slotSetupPopupMenu() {
	bool hasSelection = m_fileView && m_fileView->selectedItems()
									&& ! m_fileView->selectedItems()->isEmpty();
	QPopupMenu* popupMenu = actionMenu->popupMenu();
	popupMenu->clear();
	upAction->plug( popupMenu );
	backAction->plug( popupMenu );
	forwardAction->plug( popupMenu );
	homeAction->plug( popupMenu );
	actionSeparator->plug( popupMenu );
	reloadAction->plug( popupMenu );
	stopAction->plug( popupMenu );
	actionSeparator->plug( popupMenu );
	mkdirAction->plug( popupMenu );
	actionSeparator->plug( popupMenu );
	action("synch_new")->plug( popupMenu );
	actionSeparator->plug( popupMenu );
	if( hasSelection ) {
		deleteAction->plug( popupMenu );
		if( m_url.isLocalFile() )
			shredAction->plug( popupMenu );
		actionSeparator->plug( popupMenu );
		action( KStdAction::stdName(KStdAction::Cut) )->plug( popupMenu );
		action( KStdAction::stdName(KStdAction::Copy) )->plug( popupMenu );
	}
	if( ! KIO::isClipboardEmpty() ) {
		action( KStdAction::stdName(KStdAction::Paste) )->setEnabled( true );
		action( KStdAction::stdName(KStdAction::Paste) )->plug( popupMenu );
		actionSeparator->plug( popupMenu );
	}
	action( KStdAction::stdName(KStdAction::Find) )->plug( popupMenu );
	actionSeparator->plug( popupMenu );
	action( KStdAction::stdName(KStdAction::SelectAll) )->plug( popupMenu );
	action( KStdAction::stdName(KStdAction::Deselect) )->plug( popupMenu );
	action( "edit_invert_selection" )->plug( popupMenu );
	actionSeparator->plug( popupMenu );
	if( hasSelection ) {
		if( m_fileView->selectedItems()->count() == 1 && setupOpenWithMenu()>0 ) {
			openWithMenu->plug( popupMenu );
		}
		else {
			m_privateActionCollection->action("open with")->setText( i18n("&Open With...") );
			m_privateActionCollection->action( "open with" )->plug( popupMenu );
		}
		actionSeparator->plug( popupMenu );
	}			
	sortActionMenu->plug( popupMenu );
	actionSeparator->plug( popupMenu );
	viewActionMenu->plug( popupMenu );
	if( hasSelection ) {
		actionSeparator->plug( popupMenu );
		m_privateActionCollection->action( "properties" )->plug( popupMenu );
	}
}
//-----------------------------------------------
void KBearFileSysPart::setSorting( QDir::SortSpec spec ) {
    if ( m_fileView )
        m_fileView->setSorting( spec );
    m_mySorting = spec;
    updateSortActions();
}
//-----------------------------------------------
void KBearFileSysPart::slotProgress( int percent ) {
    m_progress->setProgress( percent );
    // we have to redraw this in as fast as possible
	if( ! m_progress->isHidden() )
		QApplication::flushX();
}
//-----------------------------------------------
void KBearFileSysPart::slotProgress( KIO::Job*, unsigned long percent ) {
    slotProgress( (int)percent );
}
//-----------------------------------------------
void KBearFileSysPart::slotConnectResult( int ERR ){
#warning TAKE A LOOK ON WHICH ERRORS TO RECONNECT UPON
	kdDebug()<<"KBearFileSysPart::slotConnectResult() error="<<ERR<<endl;
	if( ERR ) {
		if( ! m_dirLister->isLocal() && ! m_isConnected ) {
			
			if( m_connection.autoReconnect()
				&& (ERR == KIO::ERR_UNKNOWN_HOST
				|| ERR == KIO::ERR_CANNOT_ENTER_DIRECTORY
				|| ERR == KIO::ERR_CONNECTION_BROKEN
				|| ERR == KIO::ERR_COULD_NOT_CONNECT
				|| ERR == KIO::ERR_UNKNOWN
				|| ERR == KIO::ERR_SERVER_TIMEOUT ))
			{
				++m_numOfConnRetry;
				int time = m_connection.reconnectTime();
				if( m_numOfConnRetry <= m_connection.numOfRetries() ) {
					QTimer::singleShot( time*1000, this, SLOT(reconnect()));
					m_connRetryTime = time;
					slotStatusMessage( i18n( "Reconnection nr: %1 in %2 seconds" ).arg(m_numOfConnRetry).arg( time) );
				}
				else {
					m_numOfConnRetry = 0;
					emit closeMe();
				}
			}
			else {
				m_numOfConnRetry = 0;
				emit closeMe();
			}
		}
		else {
			m_dirLister->statURL( m_url );
		}
	}
	QApplication::restoreOverrideCursor();
}
//-----------------------------------------------
void KBearFileSysPart::slotDelete(){
	if( ! m_fileView )
		return;
	const KFileItemList *items = m_fileView->selectedItems();
	if( items )
		deleteFiles( items, false );
}
//-----------------------------------------------
void KBearFileSysPart::slotShred(){
	if( ! m_fileView )
		return;
	const KFileItemList* items = m_fileView->selectedItems();
	if( items )
		deleteFiles( items, true );
}
//-----------------------------------------------
void KBearFileSysPart::deleteFiles( const KFileItemList* items, bool shred ){
	if( items->isEmpty() ) { // shouldn't happend, but just in case
		KMessageBox::information( widget(), i18n("You didn't select anything to delete."),i18n("Nothing to delete") );
		return;
	}
	KURL::List urls;
	QStringList files;
	KFileItemListIterator it( *items );
	for( ; it; ++it ) {
		urls.append( (*it)->url() );
		if( (*it)->url().isLocalFile() )
			files.append( (*it)->url().path() );
		else
			files.append( (*it)->url().prettyURL() );
	}
	int ret;
	if( items->count() == 1 ) {
		ret = KMessageBox::warningContinueCancel( widget(),
						i18n( "<qt>Do you really want to %1\n <b>'%2'</b>?</qt>" )
						.arg( shred?i18n("shred"):i18n("delete") ).arg( files.first() ),
						i18n("%1 file").arg(shred?i18n("Shred"):i18n("Delete")),
						shred?i18n("Shred"):i18n("Delete") );
	}
	else {
		ret = KMessageBox::warningContinueCancelList( widget(),
						i18n("Do you really want to %1 these items?").arg( shred?i18n("shred"):i18n("delete") ),
						 files, i18n("%1 file").arg(shred?i18n("Shred"):i18n("Delete")),
						shred?i18n("Shred"):i18n("Delete") );
	}
	if(  (ret == KMessageBox::Continue) ) {
		m_dirLister->deleteFiles( urls, shred, true  );
		
		if( shred )
			slotStatusMessage( i18n("Shredding file(s)...") );
		else
			slotStatusMessage( i18n("Deleting file(s)...") );
	}
}
//-----------------------------------------------
void KBearFileSysPart::slotInfoMessage( KIO::Job*, const QString& message ) {
	slotInfoMessage( message );
}
//-----------------------------------------------
void KBearFileSysPart::slotInfoMessage( const QString& message ) {
	if( message.left(4) == "resp"
		|| message.left(7) == "command"
		|| message.left(10) == "multi-line"
		|| message.left(8) == "internal"
		|| m_url.protocol() != "kbearftp" )
	{
		emit logMessage( message );
	}
	else
		slotStatusMessage( message );
}
//-----------------------------------------------
void KBearFileSysPart::slotOpenDir( const KFileItem* item ){
	kdDebug()<<"KBearFileSysPart::slotOpenDir() url="<<item->url().prettyURL()<<endl;
/*  done in setDir
	if( m_partViewerWidget )
		showFileView();
*/
	m_lastURL = m_url;
	// We now know it's a directory so we add a trailing slash
	m_url = item->url();
	m_url.adjustPath( +1 );
	setDir( m_url );
}
//-----------------------------------------------
void KBearFileSysPart::slotDirSelected( const KFileItem* item ){
	if( ! item )
		return;
	Window root;
	Window child;
	int root_x, root_y, win_x, win_y;
	uint keybstate;
	XQueryPointer( qt_xdisplay(), qt_xrootwin(), &root, &child, &root_x, &root_y, &win_x, &win_y, &keybstate );
	//Don't open the file if control or shift was pressed
	//because we want the user to be able to make multi selections with these keys
	if( keybstate & ControlMask || keybstate & ShiftMask )
		return;
	KURL tmp = item->url();
	tmp.adjustPath( +1 ); // we know it's a directory here	
	if( tmp != m_url )
		addToHistory( m_url );
		
	slotOpenDir( item );
}
//-----------------------------------------------
void KBearFileSysPart::slotUpdateTime(){
	QString tString = i18n("Not connected");
	if( ! m_isConnected && m_numOfConnRetry > 0 ) {
		--m_connRetryTime;
		m_connRetryTime = m_connRetryTime<0?0:m_connRetryTime;
		slotStatusMessage( i18n( "Reconnection nr: %1 in %2 seconds" ).
						arg(m_numOfConnRetry).arg( m_connRetryTime) );
	}
	else if( m_isConnected ) {
		int numOfSecs = m_myTime.secsTo( QDateTime::currentDateTime() );
		int mSecs = m_myTime.time().elapsed();
		QTime t;
		int numOfDays = numOfSecs/int(60*60*24);
		if( numOfDays > 0 ){
			tString = QString("%1 days ").arg( numOfDays );
		}
		else {
			tString = QString::null;
		}
		tString += t.addMSecs( mSecs ).toString();
	}
	m_endLabel->setText( i18n("Connected: %1").arg( tString ) );
}
//-----------------------------------------------
void KBearFileSysPart::slotToggleDirsFirst() {
    QDir::SortSpec sorting = m_fileView->sorting();
    if ( !KFile::isSortDirsFirst( sorting ) )
        m_fileView->setSorting( static_cast<QDir::SortSpec>( sorting | QDir::DirsFirst ));
    else
        m_fileView->setSorting( static_cast<QDir::SortSpec>( sorting & ~QDir::DirsFirst));
    m_mySorting = m_fileView->sorting();
}
//-----------------------------------------------
void KBearFileSysPart::slotToggleIgnoreCase() {
    QDir::SortSpec sorting = m_fileView->sorting();
    if ( !KFile::isSortCaseInsensitive( sorting ) )
        m_fileView->setSorting( static_cast<QDir::SortSpec>( sorting | QDir::IgnoreCase ));
    else
        m_fileView->setSorting( static_cast<QDir::SortSpec>( sorting & ~QDir::IgnoreCase));
    m_mySorting = m_fileView->sorting();
}
//-----------------------------------------------
void KBearFileSysPart::slotJobStarted( KIO::Job* job ) {
	kdDebug()<<"KBearFileSysPart::slotJobStarted() job="<<job<<endl;
	if( job ) {
		job->setMetaData( m_connection.metaData() );
/*
		connect( job, SIGNAL( infoMessage( KIO::Job*, const QString& ) ),
					this, SLOT( slotInfoMessage( KIO::Job*, const QString& ) ) );
*/
		connect( job, SIGNAL( percent( KIO::Job*, unsigned long ) ),
					this, SLOT( slotProgress( KIO::Job*, unsigned long ) ) );
		connect( job, SIGNAL( speed( KIO::Job*, unsigned long ) ),
					this, SLOT( slotSpeed( KIO::Job*, unsigned long ) ) );
//		connect( job, SIGNAL( result( KIO::Job* ) ), this, SLOT( slotFinishedLoading() ) );
	}
	m_partViewer->widget()->unsetCursor();
	slotStartLoading();
}
//-----------------------------------------------
void KBearFileSysPart::slotProperties() {
	if ( m_fileView ) {
		const KFileItemList *list = m_fileView->selectedItems();
		if( !list->isEmpty() ) {
			QObject::disconnect( m_dirLister, SIGNAL( connected() ), this, SLOT( slotProperties() ) );
			if( m_dirLister->isLocal() )
				(void) new KPropertiesDialog( *list, widget(), "props dlg", true );
			else if( ! m_dirLister->isConnected() ) {
				QObject::connect( m_dirLister, SIGNAL( connected() ), this, SLOT( slotProperties() ) );
				m_dirLister->openConnection();
			}
			else {
				if( m_url.protocol() == "kbearftp" ) {  // we only use our own properties dialog for kbearftp
					KBearPropertiesDialog p( (unsigned long) m_dirLister, *list, widget(), "props dlg", m_codec );
					QObject::connect( &p, SIGNAL( infoMessage( const QString& ) ),
										this, SLOT( slotInfoMessage( const QString& ) ) );
					p.exec();
				}
				else {
					(void) new KPropertiesDialog( *list, widget(), "props dlg", true );
				}
			}
		}
    }
}
//-----------------------------------------------
void KBearFileSysPart::slotDeleteFinished(){
	kdDebug()<<"KBearFileSysPart::slotDeleteFinished()"<<endl;
	slotStatusMessage(i18n( "No Operation in Progress" ) );
	reload();
}
//-----------------------------------------------
void KBearFileSysPart::slotClearSynch(){
	int ret = KMessageBox::warningContinueCancel( widget(), i18n("Are you sure you want to clear the list ?"),
								i18n("Clear list..."), KGuiItem( i18n("&Clear") ) );
	if( ret == KMessageBox::Cancel )
		return;
	KConfig config("kbearfilesyspartrc");
	config.deleteGroup( normalizeLabel( m_connection.label() ) );
	config.sync();
	updateSynchList();
}
//-----------------------------------------------
void KBearFileSysPart::slotNewSynch() {
	QString localPath = KFileDialog::getExistingDirectory( ":", widget(), i18n( "Select local directory to synchronize" ) );
     if( localPath.isNull() )
		return;
	QString synchString = localPath + " <> " + m_dirLister->url().path(+1); // create string to store in config file
	KConfig config("kbearfilesyspartrc");
	QString normalizedLabel = normalizeLabel( m_connection.label() );
	config.setGroup( normalizedLabel );
	QMap<QString, QString> entries = config.entryMap( normalizedLabel );
	QMap<QString, QString>::Iterator it;
	bool exist = false;
	// check if entry already exist
	for( it = entries.begin(); it != entries.end(); ++it ) {
		if( synchString == it.data() ) {
			exist = true;
			break;
		}
	}
	if( ! exist ) {// didn't exist so lets create it
		config.writeEntry( "synch"+QString::number( entries.count() ), synchString );
		config.sync();
		updateSynchList();
	}
	if( ! localPath.isNull() ) {
		slotSynch( localPath, m_dirLister->url().path(+1) );
	}
}
//-----------------------------------------------
void KBearFileSysPart::slotSynch(){
	QString synch = sender()->name();
	QStringList list = QStringList::split( " <> ", synch );
	slotSynch( list[ 0 ], list[ 1 ] );
}
//-----------------------------------------------
void KBearFileSysPart::slotSynch( const QString& localDir, const QString& remoteDir ){
	if( ! m_url.hasHost() ) // shouldn't happend
		return;

	Connection c = m_connection;
	c.setLocalPath( localDir );
	c.setRemotePath( remoteDir );

	if( m_partViewerWidget && m_partViewer ) { // this shouldn't happend but just in case
		m_partViewer->manager()->replacePart( m_partViewer, this );
		delete m_partViewerWidget;
		m_partViewerWidget = 0L;
		m_partViewer = 0L;
		m_splitter->setSizes( m_splitterSizes );
	}
	m_splitterSizes = m_splitter->sizes();
	m_splitterSizes.append( m_splitterSizes[1] );
	m_partViewerWidget = new QWidget( m_splitter, "PartViewerWidget");
	m_partViewer = KParts::ComponentFactory::createPartInstanceFromLibrary
									<KParts::ReadOnlyPart>("libkbeardirsynchpart",
									m_partViewerWidget, "KBearDirSynchPartWidget", m_partViewerWidget, "KBearDirSynchPart" );

	if( ! m_partViewer ) {
		delete m_partViewerWidget;
		m_partViewerWidget = 0L;
		m_partViewer = 0L;
		return;
	}
	QObject::disconnect( m_dirLister, SIGNAL( newItems( const KFileItemList &) ), this, SLOT( slotNewItems( const KFileItemList &) ) );
	QObject::disconnect( m_dirLister, SIGNAL( newItems( const KFileItemList &) ), m_leftView, SLOT( slotInsertItems( const KFileItemList &) ) );
	QObject::disconnect( m_dirLister, SIGNAL( deleteItem( KFileItem* ) ), this, SLOT( slotDeleteItem( KFileItem* ) ) );
	QObject::disconnect( m_dirLister, SIGNAL( deleteItem( KFileItem* ) ), m_leftView, SLOT( slotDeleteItem( KFileItem* )));
	QObject::disconnect( m_dirLister, SIGNAL( openDir( const KFileItem* ) ), this, SLOT( slotOpenDir( const KFileItem* ) ) );
	QObject::disconnect( m_dirLister, SIGNAL( openFile( const KFileItem* ) ), this, SLOT( slotOpenFile( const KFileItem* ) ) );
	QObject::disconnect( m_dirLister, SIGNAL( redirection( const KURL& ) ), this, SLOT( slotRedirected( const KURL& ) ) );
	QObject::disconnect( m_dirLister, SIGNAL( clear() ), this, SLOT( slotClear() ) );
	QObject::disconnect( m_dirLister, SIGNAL( deleteFinished() ), this, SLOT( slotDeleteFinished() ) );
     QObject::disconnect( m_dirLister, SIGNAL( started() ), this, SLOT( slotStartLoading() ) );
	QObject::disconnect( m_dirLister, SIGNAL( finishedLoading() ), this, SLOT( slotFinishedLoading() ) );
	QObject::disconnect( m_dirLister, SIGNAL( speed(int) ), this, SLOT( slotSpeed(int) ) );

	connect( m_partViewer, SIGNAL( setStatusBarText( const QString& ) ),
				this, SLOT( slotInfoMessage( const QString& ) ) );
	connect( m_partViewer, SIGNAL( started( KIO::Job* ) ), this, SLOT( slotStartLoading() ) );
	connect( m_partViewer, SIGNAL( completed() ), this, SLOT( slotFinishedLoading() ) );
	connect( static_cast<KBearPart*>(m_partViewer), SIGNAL( logMessage( const QString& ) ),
				this, SLOT( slotInfoMessage( const QString& ) ) );

	static_cast<KBearPart*>( m_partViewer )->setDirLister( m_dirLister );
	manager()->replacePart( this, m_partViewer );
	enableGUI( false );
	m_leftView->hide();
	m_fileView->widget()->hide();
	QVBoxLayout* layout = new QVBoxLayout( m_partViewerWidget );
	layout->addWidget( m_partViewer->widget() );
	m_partViewerWidget->show();

	KBearPart* p = static_cast<KBearPart*>( m_partViewer );
	p->openConnection( c );
	connect( p, SIGNAL( closeMe() ), this, SLOT( showFileView() ) );
}
//-----------------------------------------------
void KBearFileSysPart::updateSynchList(){
	QPtrList<KAction> actions;
     KConfig config("kbearfilesyspartrc");

	QMap<QString, QString> entries = config.entryMap( normalizeLabel( m_connection.label() ) );

	unplugActionList("synch_list");
	QMap<QString, QString>::Iterator it;
	for( it = entries.begin(); it != entries.end(); ++it ) {
		kdDebug()<<"KBearFileSysPart::updateSynchList() adding "<<it.data()<<" to synch list"<<endl;
		KAction *action = new KAction( it.data(), 0, this, SLOT(slotSynch()), actionCollection(), it.data().latin1());
		actions.append(action);
	}
	plugActionList("synch_list", actions);
}
//-----------------------------------------------
void KBearFileSysPart::guiActivateEvent( KParts::GUIActivateEvent* event ) {
	if( event->activated() ) {
		if( ! m_url.isEmpty() ) {
			slotSetWindowCaption( m_url.prettyURL() );
		}
		else
			emit setWindowCaption( "" );
	}
	if( event->activated() ) {
		updateSynchList();
	}
}
//-----------------------------------------------
void KBearFileSysPart::slotSetWindowCaption( const QString& str ) {
	QString s = str;
	if( ! m_connection.isLocal() )
		s = m_codec->toUnicode( s );

	emit setWindowCaption( s );
}
//-----------------------------------------------
#ifndef NO_INCLUDE_MOCFILES
#include "kbearfilesyspart.moc"
#endif
