// --------------------------------------------------------------------
// Ipelet for creating regular k-gons
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipelet.h"
#include "ipepath.h"
#include "ipepage.h"
#include "ipevisitor.h"

// --------------------------------------------------------------------

class KGonIpelet : public Ipelet {
public:
  virtual int IpelibVersion() const { return IPELIB_VERSION; }
  virtual const char *Label() const { return "Regular k-gon"; }
  virtual void Run(int, IpePage *page, IpeletHelper *helper);
};

// --------------------------------------------------------------------

void KGonIpelet::Run(int, IpePage *page, IpeletHelper *helper)
{
  IpePage::iterator it = page->PrimarySelection();
  if (it == page->end() || !it->Object()->AsPath() ||
      it->Object()->AsPath()->NumSubPaths() > 1 ||
      it->Object()->AsPath()->SubPath(0)->Type() != IpeSubPath::EEllipse) {
    helper->Message("Primary selection is not a circle");
    return;
  }

  IpeString str;
  if (!helper->GetString("Enter k (number of corners)", str))
    return;
  IpeLex lex(str);
  int k;
  lex >> k;
  if (k < 3 || k > 1000)
    return;

  const IpePath *p = it->Object()->AsPath();
  const IpeEllipse *e = p->SubPath(0)->AsEllipse();
  IpeMatrix m = p->Matrix() * e->Matrix();

  IpeVector center = m.Translation();
  IpeVector v = m * IpeVector(1,0);
  double radius = (v - center).Len();

  IpeSegmentSubPath *sp = new IpeSegmentSubPath;
  double alpha = 2.0 * IpePi / k;
  IpeVector v0 = center + radius * IpeVector(1,0);
  for (int i = 0; i < k; ++i) {
    IpeVector v1 = center + radius * IpeVector(IpeAngle(i * alpha));
    sp->AppendSegment(v0, v1);
    v0 = v1;
  }
  sp->SetClosed(true);
  IpePath *obj = new IpePath(helper->Attributes());
  obj->AddSubPath(sp);
  page->push_back(IpePgObject(IpePgObject::ESecondary,
			      helper->CurrentLayer(), obj));
  helper->Message("Created regular k-gon");
}

// --------------------------------------------------------------------

IPELET_DECLARE Ipelet *NewIpelet()
{
  return new KGonIpelet;
}

// --------------------------------------------------------------------
