/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
//
// image-proc.c
//
// specialized markup processors
//
// Copyright (c) 1995-96 Jim Nelson.  Permission to distribute
// granted by the author.  No warranties are made on the fitness of this
// source code.
//
*/

#include "defs.h"
#include "option.h"

/*
// specialized markup processors
*/

uint ImageProcessor(TASK *task, HTML_MARKUP *htmlMarkup, char **newPlaintext)
{
    const char *imgfilename;
    char str[32];
    IMAGEFILE imageFile;
    DWORD width, height;
    char *imgSource;
    const char *originalSource;
    char *imgFilename;
    const char *imgText;

    UNREF_PARAM(newPlaintext);

    /* try to find ALT text in store */
    /* first: is there already ALT attribute?  if so, skip this step */
    if(IsAttributeInMarkup(htmlMarkup, "ALT") == FALSE)
    {
        /* parse down the image source to just the filename */
        originalSource = MarkupAttributeValue(htmlMarkup, "SRC");
        if(originalSource == NULL)
        {
            HtpMsg(MSG_WARNING, task->infile, "image SRC not specified, skipping");
            return MARKUP_OKAY;
        }

        imgSource = DuplicateString(originalSource);
        if(imgSource == NULL)
        {
            HtpMsg(MSG_ERROR, task->infile, "out of memory processing IMG tag");
            return MARKUP_ERROR;
        }

        /* parse the image name, just find the filename */
        imgFilename = FindFilename(imgSource);

        /* is the image in the ALT text store? */
        if(VariableExists(&altTextVarStore, imgFilename) == TRUE)
        {
            /* add the specified text to the image */
            imgText = GetVariableValue(&altTextVarStore, imgFilename);
            assert(imgText != NULL);
            AddAttributeToMarkup(htmlMarkup, "ALT", imgText, TRUE);
            HtpMsg(MSG_INFO, task->infile, "ALT text \"%s\" added to IMG \"%s\"",
                imgText, imgSource);
        }

        FreeMemory(imgSource);
        imgSource = NULL;
    }

    /* if option is turned off, then just include the markup as-is */
    if(IMGXY == FALSE)
    {
        return MARKUP_OKAY;
    }

    /* if width and/or height are already specified, then include the */
    /* markup as-is with no modifications */
    if(IsAttributeInMarkup(htmlMarkup, "HEIGHT")
        || IsAttributeInMarkup(htmlMarkup, "WIDTH"))
    {
        return MARKUP_OKAY;
    }

    /* get the filename of the image */
    if((imgfilename = MarkupAttributeValue(htmlMarkup, "SRC")) == NULL)
    {
        HtpMsg(MSG_ERROR, task->infile, "unable to retrieve image source name");
        return MARKUP_ERROR;
    }

    /* open the image file, get its dimensions, and close the file */
    if(OpenImageFile(imgfilename, &imageFile) == FALSE)
    {
        HtpMsg(MSG_WARNING, task->infile, "unable to open image file \"%s\"",
            imgfilename);
        return MARKUP_OKAY;
    }

    if(GetImageDimensions(&imageFile, &width, &height) == FALSE)
    {
        HtpMsg(MSG_WARNING, task->infile, "unable to determine image file \"%s\" dimensions",
            imgfilename);
    }
    else
    {
        /* add the width and height specifier for the image */
        sprintf(str, "%lu", width);
        AddAttributeToMarkup(htmlMarkup, "WIDTH", str, TRUE);
        sprintf(str, "%lu", height);
        AddAttributeToMarkup(htmlMarkup, "HEIGHT", str, TRUE);

        /* print out an informational message to the user */
        HtpMsg(MSG_INFO, task->outfile, "image file \"%s\" dimensions (%u x %u) added",
               imgfilename, width, height);
    }

    CloseImageFile(&imageFile);

    /* include the markup in the final output */
    return MARKUP_OKAY;
}   


uint AltTextProcessor(TASK *task, HTML_MARKUP *htmlMarkup, char **newPlaintext)
{
    const char *imgName, *imgText;
    char *imgNameCopy, *imgTextCopy;

    UNREF_PARAM(newPlaintext);

    /* requires at least a NAME parameter */
    if(IsAttributeInMarkup(htmlMarkup, "NAME") == FALSE)
    {
        HtpMsg(MSG_ERROR, task->infile, "ALTTEXT requires a NAME attribute");
        return MARKUP_ERROR;
    }

    /* get the relevant information */
    imgName = MarkupAttributeValue(htmlMarkup, "NAME");
    if(imgName == NULL)
    {
        HtpMsg(MSG_ERROR, task->infile, "NAME must be specified with a filename");
        return MARKUP_ERROR;
    }
    imgText = MarkupAttributeValue(htmlMarkup, "TEXT");

    /* try to find the graphic name in the ALTTEXT store */
    if (VariableExists(&altTextVarStore, imgName) == TRUE)
    {
        /* if no name specified, delete it from the store */
        if(imgText == NULL)
        {
            RemoveVariable(&altTextVarStore, imgName);
            HtpMsg(MSG_INFO, task->infile, "ALT text for image \"%s\" removed",
                imgName);
            return DISCARD_MARKUP;
        }

        /* since it exists, simply re-storing the value will delete the */
        /* old one and replace with new one */
    }
    else if(imgText == NULL)
    {
        /* tried to delete an image not already in the store */
        /* just post a warning */
        HtpMsg(MSG_WARNING, task->infile, "attempted to delete image text not already defined");
        return DISCARD_MARKUP;
    }

    if ((imgNameCopy = DuplicateString(imgName)) == NULL
        || (imgTextCopy = DuplicateString(imgText)) == NULL) {
        HtpMsg(MSG_ERROR, task->infile,
               "Can't duplicate text, out of Memory.");
        if (imgNameCopy != NULL)
            FreeMemory(imgNameCopy);
        return MARKUP_ERROR;
    }

    StoreVariable(&altTextVarStore, imgNameCopy, imgTextCopy, 
                  VAR_TYPE_ALTTEXT, 
                  VAR_FLAG_DEALLOC_NAME | VAR_FLAG_DEALLOC_VALUE, 
                  NULL, NULL);

    HtpMsg(MSG_INFO, task->infile, "ALT text for image \"%s\" set to \"%s\"",
        imgName, imgText);

    return DISCARD_MARKUP;
}

