/***************************************************************************
                          codeparser.h  -  description
                             -------------------
    begin                : Die Jul 9 2002
    copyright            : (C) 2002 by Andre Simon
    email                : andre.simon1@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef CODEPARSER_H
#define CODEPARSER_H

#include <iostream>
#include <sstream>
#include <string>
#include <iomanip>
#include <cctype>

#include "languagedefinition.h"
#include "documentstyle.h"
#include "ASFormatter.h"
#include "preformatter.h"
#include "enums.h"


#define NUMBER_BUILTIN_STYLES 10
#define LINE_NUMBER_WIDTH     5
#define MAX_LINE__WIDTH       80

#define OUTPUT_FLAG_LN    1
#define OUTPUT_FLAG_LNZ   2
#define OUTPUT_FLAG_FRAG  4

/** The highlight namespace contains all classes and data structures
    needed for parsing input data.
*/
namespace highlight {

/** \brief Base class for parsing. Works similar to a Turing machine.

    The virtual class provides source code parsing functioality, based on
    information stored in language definitions.<br>
    Deriving classes have to define the output format.<br>
    Codegenerator is a singleton class.

* @author Andre Simon
*/

class CodeGenerator
  {

  public:

    virtual ~CodeGenerator();

    /**
      Get appropriate Codegenerator instance
      \param type Output file type (HTML, XHTML, RTF, LATEX, TEX, XSLFO, ANSI)
      \param styleInfoPath Path to formatting style information
      \param styleInPath Path to style definition input file (to be included in styleOutPath)
      \param styleOutPath Path to style definition output file (CSS path for HTML output)
      \param includeStyle Switch to include style information in output file (only XHTML, HTML)
      \param attachAnchors Switch to attach anchors to line numbers (only XHTML, HTML)
      \param replaceQuotes Switch to replace quotes by \dq{} (only LATEX)
      \param fopCompatible Switch to generate FO for Apache FOP  (only XSLFO)
      \param ln Set true if line numbers should be printed
      \param lnz Set true if leading space of line numbers should be filled with 0's
      \param fragment Set true if document header and footer should be omitted
      \param numSpaces Number of spaces which replace a tab
      \param lineWrappingMode Line wrapping mode
    */
    static CodeGenerator* getInstance(OutputType type,
                                      const string& styleInfoPath,
                                      const string& styleInPath,
                                      const string& styleOutPath,
                                      bool includeStyle,
                                      bool attachAnchors,
                                      bool replaceQuotes,
                                      bool fopCompatible,
                                      int numSpaces,
                                      WrapMode lineWrappingMode,
                                      bool ln,
                                      bool lnz,
                                      bool fragment );

    /** Deletes the singleton CodeGenerator instance.
        Call this method if getInstance was already called, or if you want to
        free the momory after usage.*/
    static void deleteInstance();

    /**
     Generates output
     \param inFileName Path of input file (if empty use stdin)
     \param outFileName Path of input file (if empty use stdout)

     \return ParseError
    */
    ParseError printOutput(const string &inFileName, const string &outFileName);

    /** \return True if document style was found */
    bool styleFound();

    /** \return True if reformatting of current input is disabled */
    bool formattingDisabled();

    /** \return True if reformatting of current input is possible */
    bool formattingIsPossible();

    /** \param langDefPath Absolute path to language definition
        \return  Failure: LOAD_FAILED; Reload necessary: LOAD_NEW,
                 no reload necessary:  LOAD_NONE */
    LoadResult initLanguage(const string& langDefPath);

    /** \return Language definition*/
    LanguageDefinition &getLanguage();

    /** tell parser to output line numbers
       \param  flag true if line numbers should be printed
    */
    void setPrintLineNumbers(bool flag);

    /** \return line number flag */
    bool getPrintLineNumbers();


    /** tell parser to output line numbers filled with zeroes
        \param  flag true if zeroes should be printed
    */
    void setPrintZeroes(bool flag);

    /** \return print zeroes flag */
    bool getPrintZeroes();

    /** tell parser to omit document header and footer
       \param  flag true if output should be fragmented
    */
    void setFragmentCode(bool flag);

    /** \return fragment flag */
    bool getFragmentCode();

    /** tell parser the style name
       \param s path to style definition
    */
    void setStyleName(const string& s);

    /** \return style path */
    const string& getStyleName();

    /** tell parser the wrapping mode
       \param lineWrappingStyle wrapping style
       \param lineLength max line length
       \param numberSpaces number of spaces which replace a tab
    */
    void setPreformatting(WrapMode lineWrappingStyle, unsigned int lineLength,int numberSpaces);

    /** \return wrapping style */
    WrapMode getLineWrapping();

   /** Print style definitions to external file
     \param outFile Path of external style definition
     */
    virtual bool printExternalStyle(const string &outFile);

   /** Print index file with all input file names
      \param fileList List of output file names
      \param outPath Output path
    */
    virtual bool printIndexFile(const vector<string> & fileList,
                                const string &outPath);

   /** initialize source code indentation
       \param indentSchemePath Path of indentation scheme
       \return true id successfull
    */
    bool initIndentationScheme(const string&indentSchemePath);

protected:

    CodeGenerator();

    //! CodeGenerator Constructor
    /**
        \param colourTheme Name of coloring style being used
    */
    CodeGenerator(const string &colourTheme);

    /** \param c Character to be masked
        \return Escape sequence of output format */
    virtual string maskCharacter(unsigned char c) = 0;

    /** \param s string
       \return Copy of s with all escaped characters */
    string maskString(const string &s ) ;

    /** \param s Symbol string
        \param searchPos Position where search starts
        \return Found state (integer value)  */
    State getState(const string &s, unsigned int searchPos);

    /** \return Next identifier in current line of code */
    string getIdentifier();

    /** \return Next number in current line of code */
    string getNumber();

    /** Insert line number at the beginning of current output line */
    virtual void insertLineNumber(bool insertNewLine=true);

    /** Prints document footer*/
    virtual string getFooter() = 0;

    /** Prints document body*/
    virtual void printBody() = 0;

    /** prints document header
       \param  title Title of the document
    */
    virtual string getHeader(const string &title) = 0;

    /** Get current line number
      \return line number  */
    unsigned int getLineNumber();


    /** Tag Delimiters for every colour style*/
    vector <string> styleTagOpen, styleTagClose;

    /** Description of document colour style*/
    DocumentStyle docStyle;

    /** Language definition*/
    LanguageDefinition langInfo;

    /** Tag for inserting line feeds*/
    string newLineTag;

    /** String that represents a white space in output */
    string spacer;

    /** file input*/
    istream *in;

    /** file output*/
    ostream *out;

    /** Tags which enclose white space indentation blocks */
    string maskWsBegin, maskWsEnd;

    /** Test if maskWsBegin and maskWsEnd should be applied */
    bool maskWs;

    /** Test if whitespace sould always be separated from enclosing tokens */
    bool excludeWs;

    /** Test if header and footer should be omitted */
    bool fragmentOutput;

    /** Test if line numbers should be printed */
    bool showLineNumbers;

    /** Test if leading spyce of line number should be filled with zeroes*/
    bool lineNumberFillZeroes;

    /** Current line of input file*/
    string line;

    /** Current line number */
    unsigned int lineNumber;

    // Zeigt den aktuellen Zustand an
    // wird nicht in getCurrentState gesetzt, da nur Zustände interessant
    // sind, die als Index auf die styleCloseTags und styleOpenTags verwendet
    // werden knnen
    /** Current state*/
    State currentState;

    /** keyword class id, used to apply the corresponding keyword style*/
    unsigned int currentKeywordClass;

    /** Processes origin state */
    void processRootState();

    /** return line break sequence */
    virtual string getNewLine();

    /**
       \param s current state
       \return Index of style tag corresponding to the states
    */
    unsigned int getStyleID(State s, unsigned int kwClassID = 0);

    /** \return line index */
    unsigned int  getLineIndex();

private:

    CodeGenerator(const CodeGenerator&){}
    CodeGenerator& operator=(CodeGenerator&){ return *this;}

    static CodeGenerator* generator;

    /** return matching open and close tags of the given state */
    virtual string getMatchingOpenTag(unsigned int) = 0;
    virtual string getMatchingCloseTag(unsigned int) = 0;

   /** open a new tag, set current state to s*/
    void openTag(State s);

    /** close opened tag, clear current state */
    void closeTag(State s);


    void closeTag(unsigned int styleID);

    void openTag(unsigned int styleID);

    /** print all remaining white space*/
    void flushWs();

    // path to sytle definition file
    string  stylePath;

    // contains current position in line
    unsigned int lineIndex;


    /**last character of the last line*/
    unsigned char terminatingChar;

    /** Class for reformatting */
    astyle::ASFormatter *formatter;

    /** Class for line wrapping and tab replacement*/
    PreFormatter *preFormatter;

    /** Flag to test if formatting is enabled with current input document*/
    bool formattingEnabled;

    /** Flag to test if formatting is possible with current input document*/
    bool formattingPossible;

    /** contains the current token*/
    string token;

    /** contains white space, which will be printed after a closing tag */
    string wsBuffer;

    /** Resets parser to origin state, call this after every file conversion */
    void reset();

    /** read new line from in stream */
    bool readNewLine(string &newLine);

    /** return next character from in stream */
    unsigned char getInputChar();

   /** return new state */
    State getCurrentState ( bool lastStateWasNumber=false);

    /** Methods that represent a parsing state */
    bool processKeywordState(State myState) ;
    bool processNumberState() ;
    bool processMultiLineCommentState();
    bool processSingleLineCommentState();
    bool processStringState(State oldState);
    bool processEscapeCharState();
    bool processDirectiveState();
    bool processTagState();
    bool processSymbolState();
    void processWsState();

    /** gibt true zurck, falls c ein erlaubter Character innerhalb von Keyword
        oder Typbezeichner ist */
    bool isAllowedChar(char c) ;

    /** returns true if curret token is the first in line and no whitespace */
    bool isFirstNonWsChar() ;

    /** print escaped token and clears it */
    void printMaskedToken(bool flushWhiteSpace=true);

    /** print escape sequence */
    void skipEscapeSequence();

    void closeKWTag(unsigned int styleID);
    void openKWTag(unsigned int styleID);

  };
}

#endif
