/*  
  Copyright 2002, Andreas Rottmann

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This library is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public  License
  along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

// Much of this code was taken from the PostgreSQL DB terminal psql.
// See README for more information

#include <unistd.h>
#include <sys/ioctl.h>

#include <glib.h>

#include "gql++/statement.h"
#include "gql++/result-set.h"
#include "gql++/result-set-metadata.h"

#include "print.h"

#define gettext(s) s

#define DEFAULT_PAGER "more"

namespace GQLShell
{

namespace
{

void print_unaligned_text(const char *title, 
                          const char *const * headers,
                          const char *const * cells, 
                          const char *const * footers,
                          const char *opt_fieldsep, 
                          const char *opt_recordsep, 
                          bool opt_barebones,
                          FILE *fout)
{
  unsigned col_count = 0;
  unsigned i;
  const char *const * ptr;
  bool		need_recordsep = false;

  if (!opt_fieldsep)
    opt_fieldsep = "";
  if (!opt_recordsep)
    opt_recordsep = "";

  /* print title */
  if (!opt_barebones && title)
    fprintf(fout, "%s%s", title, opt_recordsep);

  /* print headers and count columns */
  for (ptr = headers; *ptr; ptr++)
  {
    col_count++;
    if (!opt_barebones)
    {
      if (col_count > 1)
        fputs(opt_fieldsep, fout);
      fputs(*ptr, fout);
    }
  }
  if (!opt_barebones)
    need_recordsep = true;

  /* print cells */
  i = 0;
  for (ptr = cells; *ptr; ptr++)
  {
    if (need_recordsep)
    {
      fputs(opt_recordsep, fout);
      need_recordsep = false;
    }
    fputs(*ptr, fout);
    if ((i + 1) % col_count)
      fputs(opt_fieldsep, fout);
    else
      need_recordsep = true;
    i++;
  }

  /* print footers */

  if (!opt_barebones && footers)
    for (ptr = footers; *ptr; ptr++)
    {
      if (need_recordsep)
      {
        fputs(opt_recordsep, fout);
        need_recordsep = false;
      }
      fputs(*ptr, fout);
      need_recordsep = true;
    }

  /* the last record needs to be concluded with a newline */
  if (need_recordsep)
    fputc('\n', fout);
}



void print_unaligned_vertical(const char *title, 
                              const char *const * headers,
                              const char *const * cells, 
                              const char *const * footers,
                              const char *opt_fieldsep, 
                              const char *opt_recordsep, 
                              bool opt_barebones,
                              FILE *fout)
{
  unsigned col_count = 0;
  unsigned i;
  const char *const * ptr;

  if (!opt_fieldsep)
    opt_fieldsep = "";
  if (!opt_recordsep)
    opt_recordsep = "";

  /* print title */
  if (!opt_barebones && title)
    fputs(title, fout);

  /* count columns */
  for (ptr = headers; *ptr; ptr++)
    col_count++;

  /* print records */
  for (i = 0, ptr = cells; *ptr; i++, ptr++)
  {
    if (i != 0 || (!opt_barebones && title))
    {
      fputs(opt_recordsep, fout);
      if (i % col_count == 0)
        fputs(opt_recordsep, fout);		/* another one */
    }

    fputs(headers[i % col_count], fout);
    fputs(opt_fieldsep, fout);
    fputs(*ptr, fout);
  }

  /* print footers */
  if (!opt_barebones && footers && *footers)
  {
    fputs(opt_recordsep, fout);
    for (ptr = footers; *ptr; ptr++)
    {
      fputs(opt_recordsep, fout);
      fputs(*ptr, fout);
    }
  }

  fputc('\n', fout);
}



/********************/
/* Aligned text		*/
/********************/


/* draw "line" */
void _print_horizontal_line(const unsigned col_count, 
                            const unsigned *widths, 
                            unsigned short border, FILE *fout)
{
  unsigned i, j;
  
  if (border == 1)
    fputc('-', fout);
  else if (border == 2)
    fputs("+-", fout);

  for (i = 0; i < col_count; i++)
  {
    for (j = 0; j < widths[i]; j++)
      fputc('-', fout);

    if (i < col_count - 1)
    {
      if (border == 0)
        fputc(' ', fout);
      else
        fputs("-+-", fout);
    }
  }

  if (border == 2)
    fputs("-+", fout);
  else if (border == 1)
    fputc('-', fout);

  fputc('\n', fout);
}



void print_aligned_text(const char *title, 
                        const char *const * headers,
                        const char *const * cells, 
                        const char *const * footers,
                        const char *opt_align, bool opt_barebones,
                        unsigned short opt_border,
                        FILE *fout)
{
  unsigned col_count = 0;

  unsigned i,tmp;
  unsigned *widths, total_w;
  const char *const * ptr;

  /* count columns */
  for (ptr = headers; *ptr; ptr++)
    col_count++;

  widths = g_new0(unsigned int, col_count);

  /* calc column widths */
  for (i = 0; i < col_count; i++)
  {
    if ((tmp = strlen(headers[i])) > widths[i])
      widths[i] = tmp;
  }

  for (i = 0, ptr = cells; *ptr; ptr++, i++)
  {
    if ((tmp = strlen(*ptr)) > widths[i % col_count])
      widths[i % col_count] = tmp;
  }
  if (opt_border == 0)
    total_w = col_count - 1;
  else if (opt_border == 1)
    total_w = col_count * 3 - 1;
  else
    total_w = col_count * 3 + 1;

  for (i = 0; i < col_count; i++)
    total_w += widths[i];

  /* print title */
  if (title && !opt_barebones)
  {
    int tlen;

    if ((tlen = strlen(title)) >= (int)total_w)
      fprintf(fout, "%s\n", title);
    else
      fprintf(fout, "%-*s%s\n", (int) (total_w - tlen) / 2, "", title);
  }

  /* print headers */
  if (!opt_barebones)
  {
    if (opt_border == 2)
      _print_horizontal_line(col_count, widths, opt_border, fout);

    if (opt_border == 2)
      fputs("| ", fout);
    else if (opt_border == 1)
      fputc(' ', fout);

    for (i = 0; i < col_count; i++)
    {
      int nbspace;

      nbspace = widths[i] - strlen(headers[i]);

      /* centered */
      fprintf(fout, "%-*s%s%-*s",
              nbspace / 2, "", headers[i], (nbspace + 1) / 2, "");

      if (i < col_count - 1)
      {
        if (opt_border == 0)
          fputc(' ', fout);
        else
          fputs(" | ", fout);
      }
    }

    if (opt_border == 2)
      fputs(" |", fout);
    else if (opt_border == 1)
      fputc(' ', fout);;
    fputc('\n', fout);

    _print_horizontal_line(col_count, widths, opt_border, fout);
  }

  /* print cells */
  for (i = 0, ptr = cells; *ptr; i++, ptr++)
  {
    /* beginning of line */
    if (i % col_count == 0)
    {
      if (opt_border == 2)
        fputs("| ", fout);
      else if (opt_border == 1)
        fputc(' ', fout);
    }

    /* content */
    if (opt_align[(i) % col_count] == 'r')
    {
      fprintf(fout, "%*s", widths[i % col_count], cells[i]);
    }
    else
    {
      if ((i + 1) % col_count == 0 && opt_border != 2)
        fputs(cells[i], fout);
      else
      {
        fprintf(fout, "%-*s", widths[i % col_count], cells[i]);
      }
    }

    /* divider */
    if ((i + 1) % col_count)
    {
      if (opt_border == 0)
        fputc(' ', fout);
      else
        fputs(" | ", fout);
    }
    /* end of line */
    else
    {
      if (opt_border == 2)
        fputs(" |", fout);
      fputc('\n', fout);
    }
  }

  if (opt_border == 2)
    _print_horizontal_line(col_count, widths, opt_border, fout);

  /* print footers */
  if (footers && !opt_barebones)
    for (ptr = footers; *ptr; ptr++)
      fprintf(fout, "%s\n", *ptr);

  fputc('\n', fout);

  /* clean up */
  free(widths);
}

void print_aligned_vertical(const char *title, 
                            const char *const * headers,
                            const char *const * cells, 
                            const char *const * footers,
                            bool opt_barebones, 
                            unsigned short opt_border,
                            FILE *fout)
{
  unsigned col_count = 0;
  unsigned record = 1;
  const char *const * ptr;
  unsigned i, tmp = 0, hwidth = 0, dwidth = 0;
  char	   *divider;

#ifdef MULTIBYTE
  unsigned cell_count = 0;
  unsigned *cell_w, *head_w;
#endif

  if (cells[0] == NULL)
  {
    puts(gettext("(No rows)\n"));
    return;
  }

#ifdef MULTIBYTE
  /* pre-count headers */
  for (ptr = headers; *ptr; ptr++)
    col_count++;
  head_w = g_new0(unsigned, col_count);
  
  for (i = 0; i < col_count; i++)
  {
    if ((tmp = pg_wcswidth((unsigned char *) headers[i], strlen(headers[i]))) > hwidth)
      hwidth = tmp;
    head_w[i] = tmp;
  }
  for (ptr = cells; *ptr; ptr++)
    cell_count++;

  if (cell_count > 0)
  {
    cell_w = calloc(cell_count, sizeof(*cell_w));
    if (!cell_w)
    {
      perror("calloc");
      exit(EXIT_FAILURE);
    }
  }
  else
    cell_w = 0;

  /* find longest data cell */
  for (i = 0, ptr = cells; *ptr; ptr++, i++)
  {
    if ((tmp = pg_wcswidth((unsigned char *) *ptr, strlen(*ptr))) > dwidth)
      dwidth = tmp;
    cell_w[i] = tmp;
  }
#else
  /* count columns and find longest header */
  for (ptr = headers; *ptr; ptr++)
  {
    col_count++;
    if ((tmp = strlen(*ptr)) > hwidth)
      hwidth = tmp;		/* don't wanna call strlen twice */
  }

  /* find longest data cell */
  for (ptr = cells; *ptr; ptr++)
  {
    if ((tmp = strlen(*ptr)) > dwidth)
      dwidth = tmp;
  }
#endif

  /* print title */
  if (!opt_barebones && title)
    fprintf(fout, "%s\n", title);

  /* make horizontal border */
  divider = g_new(char, hwidth + dwidth + 10);
  divider[0] = '\0';
  if (opt_border == 2)
    strcat(divider, "+-");
  for (i = 0; i < hwidth; i++)
    strcat(divider, opt_border > 0 ? "-" : " ");
  if (opt_border > 0)
    strcat(divider, "-+-");
  else
    strcat(divider, " ");
  for (i = 0; i < dwidth; i++)
    strcat(divider, opt_border > 0 ? "-" : " ");
  if (opt_border == 2)
    strcat(divider, "-+");


  /* print records */
  for (i = 0, ptr = cells; *ptr; i++, ptr++)
  {
    if (i % col_count == 0)
    {
      if (!opt_barebones)
      {
        char *record_str = g_new(char, 32);
        size_t record_str_len;

        if (opt_border == 0)
          sprintf(record_str, "* Record %d", record++);
        else
          sprintf(record_str, "[ RECORD %d ]", record++);
        record_str_len = strlen(record_str);

        if (record_str_len + opt_border > strlen(divider))
          fprintf(fout, "%.*s%s\n", opt_border, divider, record_str);
        else
        {
          char *div_copy = g_strdup(divider);

          strncpy(div_copy + opt_border, record_str, record_str_len);
          fprintf(fout, "%s\n", div_copy);
          g_free(div_copy);
        }
        g_free(record_str);
      }
      else if (i != 0 || opt_border == 2)
        fprintf(fout, "%s\n", divider);
    }

    if (opt_border == 2)
      fputs("| ", fout);
#if MULTIBYTE
    fprintf(fout, "%-s%*s", headers[i % col_count],
            hwidth - head_w[i % col_count], "");
#else
    fprintf(fout, "%-*s", hwidth, headers[i % col_count]);
#endif

    if (opt_border > 0)
      fputs(" | ", fout);
    else
      fputs(" ", fout);

    if (opt_border < 2)
      fprintf(fout, "%s\n", *ptr);
    else
    {
#ifdef MULTIBYTE
      fprintf(fout, "%-s%*s |\n", *ptr, dwidth - cell_w[i], "");
#else
      fprintf(fout, "%-*s |\n", dwidth, *ptr);
#endif
    }
  }

  if (opt_border == 2)
    fprintf(fout, "%s\n", divider);


  /* print footers */

  if (!opt_barebones && footers && *footers)
  {
    if (opt_border < 2)
      fputc('\n', fout);
    for (ptr = footers; *ptr; ptr++)
      fprintf(fout, "%s\n", *ptr);
  }

  fputc('\n', fout);
  g_free(divider);

#ifdef MULTIBYTE
  g_free(cell_w);
#endif
}





/**********************/
/* HTML printing ******/
/**********************/


void html_escaped_print(const char *in, FILE *fout)
{
  const char *p;

  for (p = in; *p; p++)
    switch (*p)
    {
      case '&':
        fputs("&amp;", fout);
        break;
      case '<':
        fputs("&lt;", fout);
        break;
      case '>':
        fputs("&gt;", fout);
        break;
      case '\n':
        fputs("<br>", fout);
        break;
      default:
        fputc(*p, fout);
    }
}



void print_html_text(const char *title, 
                     const char *const * headers,
                     const char *const * cells, 
                     const char *const * footers,
                     const char *opt_align, 
                     bool opt_barebones, 
                     unsigned short opt_border,
                     const char *opt_table_attr,
                     FILE *fout)
{
  unsigned col_count = 0;
  unsigned i;
  const char *const * ptr;

  fprintf(fout, "<table border=%d", opt_border);
  if (opt_table_attr)
    fprintf(fout, " %s", opt_table_attr);
  fputs(">\n", fout);

  /* print title */
  if (!opt_barebones && title)
  {
    fputs("  <caption>", fout);
    html_escaped_print(title, fout);
    fputs("</caption>\n", fout);
  }

  /* print headers and count columns */
  if (!opt_barebones)
    fputs("  <tr>\n", fout);
  for (i = 0, ptr = headers; *ptr; i++, ptr++)
  {
    col_count++;
    if (!opt_barebones)
    {
      fputs("    <th align=center>", fout);
      html_escaped_print(*ptr, fout);
      fputs("</th>\n", fout);
    }
  }
  if (!opt_barebones)
    fputs("  </tr>\n", fout);

  /* print cells */
  for (i = 0, ptr = cells; *ptr; i++, ptr++)
  {
    if (i % col_count == 0)
      fputs("  <tr valign=top>\n", fout);

    fprintf(fout, "    <td align=%s>", opt_align[(i) % col_count] == 'r' ? "right" : "left");
    if ((*ptr)[strspn(*ptr, " \t")] == '\0')		/* is string only
                                                         * whitespace? */
      fputs("&nbsp;", fout);
    else
      html_escaped_print(*ptr, fout);
    fputs("</td>\n", fout);

    if ((i + 1) % col_count == 0)
      fputs("  </tr>\n", fout);
  }

  fputs("</table>\n", fout);

  /* print footers */

  if (footers && !opt_barebones)
    for (ptr = footers; *ptr; ptr++)
    {
      html_escaped_print(*ptr, fout);
      fputs("<br>\n", fout);
    }

  fputc('\n', fout);
}



void print_html_vertical(const char *title, 
                         const char *const * headers,
                         const char *const * cells, 
                         const char *const * footers,
                         const char *opt_align, 
                         bool opt_barebones, 
                         unsigned short opt_border,
                         const char *opt_table_attr,
                         FILE *fout)
{
  unsigned col_count = 0;
  unsigned i;
  unsigned record = 1;
  const char *const * ptr;

  fprintf(fout, "<table border=%d", opt_border);
  if (opt_table_attr)
    fprintf(fout, " %s", opt_table_attr);
  fputs(">\n", fout);

  /* print title */
  if (!opt_barebones && title)
  {
    fputs("  <caption>", fout);
    html_escaped_print(title, fout);
    fputs("</caption>\n", fout);
  }

  /* count columns */
  for (ptr = headers; *ptr; ptr++)
    col_count++;

  /* print records */
  for (i = 0, ptr = cells; *ptr; i++, ptr++)
  {
    if (i % col_count == 0)
    {
      if (!opt_barebones)
        fprintf(fout, "\n  <tr><td colspan=2 align=center>Record %d</td></tr>\n", record++);
      else
        fputs("\n  <tr><td colspan=2>&nbsp;</td></tr>\n", fout);
    }
    fputs("  <tr valign=top>\n"
          "    <th>", fout);
    html_escaped_print(headers[i % col_count], fout);
    fputs("</th>\n", fout);

    fprintf(fout, "    <td align=%s>", opt_align[i % col_count] == 'r' ? "right" : "left");
    if ((*ptr)[strspn(*ptr, " \t")] == '\0')		/* is string only
                                                         * whitespace? */
      fputs("&nbsp;", fout);
    else
      html_escaped_print(*ptr, fout);
    fputs("</td>\n  </tr>\n", fout);
  }

  fputs("</table>\n", fout);

  /* print footers */
  if (footers && !opt_barebones)
    for (ptr = footers; *ptr; ptr++)
    {
      html_escaped_print(*ptr, fout);
      fputs("<br>\n", fout);
    }

  fputc('\n', fout);
}



/*************************/
/* LaTeX				 */
/*************************/


void latex_escaped_print(const char *in, FILE *fout)
{
  const char *p;

  for (p = in; *p; p++)
    switch (*p)
    {
      case '&':
        fputs("\\&", fout);
        break;
      case '%':
        fputs("\\%", fout);
        break;
      case '$':
        fputs("\\$", fout);
        break;
      case '{':
        fputs("\\{", fout);
        break;
      case '}':
        fputs("\\}", fout);
        break;
      case '\\':
        fputs("\\backslash", fout);
        break;
      case '\n':
        fputs("\\\\", fout);
        break;
      default:
        fputc(*p, fout);
    }
}



void print_latex_text(const char *title, 
                      const char *const * headers,
                      const char *const * cells, 
                      const char *const * footers,
                      const char *opt_align, 
                      bool opt_barebones, 
                      unsigned short opt_border,
                      FILE *fout)
{
  unsigned col_count = 0;
  unsigned i;
  const char *cp;
  const char *const * ptr;


  /* print title */
  if (!opt_barebones && title)
  {
    fputs("\\begin{center}\n", fout);
    latex_escaped_print(title, fout);
    fputs("\n\\end{center}\n\n", fout);
  }

  /* begin environment and set alignments and borders */
  fputs("\\begin{tabular}{", fout);
  if (opt_border == 0)
    fputs(opt_align, fout);
  else if (opt_border == 1)
  {
    for (cp = opt_align; *cp; cp++)
    {
      if (cp != opt_align)
        fputc('|', fout);
      fputc(*cp, fout);
    }
  }
  else if (opt_border == 2)
  {
    for (cp = opt_align; *cp; cp++)
    {
      fputc('|', fout);
      fputc(*cp, fout);
    }
    fputc('|', fout);
  }
  fputs("}\n", fout);

  if (!opt_barebones && opt_border == 2)
    fputs("\\hline\n", fout);

  /* print headers and count columns */
  for (i = 0, ptr = headers; *ptr; i++, ptr++)
  {
    col_count++;
    if (!opt_barebones)
    {
      if (i != 0)
        fputs(" & ", fout);
      latex_escaped_print(*ptr, fout);
    }
  }

  if (!opt_barebones)
  {
    fputs(" \\\\\n", fout);
    fputs("\\hline\n", fout);
  }

  /* print cells */
  for (i = 0, ptr = cells; *ptr; i++, ptr++)
  {
    latex_escaped_print(*ptr, fout);

    if ((i + 1) % col_count == 0)
      fputs(" \\\\\n", fout);
    else
      fputs(" & ", fout);
  }

  if (opt_border == 2)
    fputs("\\hline\n", fout);

  fputs("\\end{tabular}\n\n", fout);


  /* print footers */

  if (footers && !opt_barebones)
    for (ptr = footers; *ptr; ptr++)
    {
      latex_escaped_print(*ptr, fout);
      fputs(" \\\\\n", fout);
    }

  fputc('\n', fout);
}



void print_latex_vertical(const char *title, 
                          const char *const * headers,
                          const char *const * cells, 
                          const char *const * footers,
                          const char *opt_align, 
                          bool opt_barebones, 
                          unsigned short opt_border,
                          FILE *fout)
{
  unsigned col_count = 0;
  unsigned i;
  const char *const * ptr;
  unsigned record = 1;

  (void) opt_align;			/* currently unused parameter */

  /* print title */
  if (!opt_barebones && title)
  {
    fputs("\\begin{center}\n", fout);
    latex_escaped_print(title, fout);
    fputs("\n\\end{center}\n\n", fout);
  }

  /* begin environment and set alignments and borders */
  fputs("\\begin{tabular}{", fout);
  if (opt_border == 0)
    fputs("cl", fout);
  else if (opt_border == 1)
    fputs("c|l", fout);
  else if (opt_border == 2)
    fputs("|c|l|", fout);
  fputs("}\n", fout);


  /* count columns */
  for (ptr = headers; *ptr; ptr++)
    col_count++;


  /* print records */
  for (i = 0, ptr = cells; *ptr; i++, ptr++)
  {
    /* new record */
    if (i % col_count == 0)
    {
      if (!opt_barebones)
      {
        if (opt_border == 2)
          fputs("\\hline\n", fout);
        fprintf(fout, "\\multicolumn{2}{c}{Record %d} \\\\\n", record++);
      }
      if (opt_border >= 1)
        fputs("\\hline\n", fout);
    }

    latex_escaped_print(headers[i % col_count], fout);
    fputs(" & ", fout);
    latex_escaped_print(*ptr, fout);
    fputs(" \\\\\n", fout);
  }

  if (opt_border == 2)
    fputs("\\hline\n", fout);

  fputs("\\end{tabular}\n\n", fout);


  /* print footers */

  if (footers && !opt_barebones)
    for (ptr = footers; *ptr; ptr++)
    {
      latex_escaped_print(*ptr, fout);
      fputs(" \\\\\n", fout);
    }

  fputc('\n', fout);
}






/********************************/
/* Public functions		*/
/********************************/


TablePrinter::TablePrinter(PrintFormat fmt, 
                           bool expanded, 
                           bool tuples_only,
                           bool pager,
                           unsigned short border,
                           const std::string& field_sep, 
                           const std::string& record_sep,
                           const std::string& table_attr)
{
  format_ = fmt;
  expanded_ = expanded;
  tuples_only_ = tuples_only;
  pager_ = pager;
  border_ = border;
  field_sep_ = field_sep;
  record_sep_ = record_sep;
  table_attr_ = table_attr; 
}

void TablePrinter::print(const char *title,
                         const char *const * headers,
                         const char *const * cells,
                         const char *const * footers,
                         const char *align,
                         FILE *fout)
{
  const char *default_footer[] = {NULL};
  unsigned short border = border_;
  
  FILE *pager = NULL, *output;


  if (format_ == PRINT_NOTHING)
    return;

  if (!footers)
    footers = default_footer;

  if (format_ != PRINT_HTML && border > 2)
    border = 2;


  /* check whether we need / can / are supposed to use pager */
  if (fout == stdout && pager_
#ifndef WIN32
      &&
      isatty(fileno(stdin)) &&
      isatty(fileno(stdout))
#endif
      )
  {
    const char *pagerprog;

#ifdef TIOCGWINSZ
    unsigned col_count = 0,
      row_count = 0,
      lines;
    const char *const * ptr;
    int			result;
    struct winsize screen_size;

    /* rough estimate of columns and rows */
    if (headers)
      for (ptr = headers; *ptr; ptr++)
        col_count++;
    if (cells)
      for (ptr = cells; *ptr; ptr++)
        row_count++;
    row_count /= col_count;

    if (expanded_)
      lines = (col_count + 1) * row_count;
    else
      lines = row_count + 1;
    if (!tuples_only_)
      lines += 5;

    result = ioctl(fileno(stdout), TIOCGWINSZ, &screen_size);
    if (result == -1 || lines > screen_size.ws_row)
    {
#endif
      pagerprog = getenv("PAGER");
      if (!pagerprog)
        pagerprog = DEFAULT_PAGER;
      pager = popen(pagerprog, "w");
#ifdef TIOCGWINSZ
    }
#endif
  }

  if (pager)
  {
    output = pager;
#if 0
#ifndef WIN32
    pqsignal(SIGPIPE, SIG_IGN);
#endif
#endif
  }
  else
    output = fout;


  /* print the stuff */

  switch (format_)
  {
    case PRINT_UNALIGNED:
      if (expanded_)
        print_unaligned_vertical(title, headers, cells, footers, 
                                 field_sep_.c_str(), record_sep_.c_str(), 
                                 tuples_only_, output);
      else
        print_unaligned_text(title, headers, cells, footers, 
                             field_sep_.c_str(), record_sep_.c_str(), 
                             tuples_only_, output);
      break;
    case PRINT_ALIGNED:
      if (expanded_)
        print_aligned_vertical(title, headers, cells, footers, tuples_only_, border, output);
      else
        print_aligned_text(title, headers, cells, footers, align, tuples_only_, border, output);
      break;
    case PRINT_HTML:
      if (expanded_)
        print_html_vertical(title, headers, cells, footers, align, 
                            tuples_only_, border, table_attr_.c_str(), output);
      else
        print_html_text(title, headers, cells, footers, align, 
                        tuples_only_, border, table_attr_.c_str(), output);
      break;
    case PRINT_LATEX:
      if (expanded_)
        print_latex_vertical(title, headers, cells, footers, align, tuples_only_, border, output);
      else
        print_latex_text(title, headers, cells, footers, align, tuples_only_, border, output);
      break;
    default:
      g_assert_not_reached();
  }

  if (pager)
  {
    pclose(pager);
#if 0
#ifndef WIN32
    pqsignal(SIGPIPE, SIG_DFL);
#endif
#endif
  }
}


}

ResultSetPrinter::ResultSetPrinter(TablePrinter& tbl_printer, 
                                   const std::string& title,
                                   const std::string& null_print, 
                                   bool quote, 
                                   const char *const *footers,
                                   bool default_footer)
    : tbl_printer_(&tbl_printer), title_(title), null_print_(null_print)
{
  quote_ = quote;
  footers_ = footers;
  default_footer_ = default_footer;
}

void ResultSetPrinter::print(GQL::ResultSet *rs, FILE *fout)
{
  using GQL::SQLType;
  
  int	nfields, nrows;
  const char **headers;
  GPtrArray *cells;
  char	  **footers;
  char	   *align;
  const GQL::ResultSetMetaData *rsmd;
  GQL::SQLObject *obj = rs->get_connection()->create_object();
  GStringChunk *str_chunk = g_string_chunk_new(1024);
  
  /* extract headers */
  rsmd = rs->get_meta_data();
  nfields = rsmd->column_count();

  headers = g_new0(const char *, nfields + 1);
  
  for (int i = 0; i < nfields; i++)
  {
    headers[i] = g_string_chunk_insert_const(
            str_chunk, rsmd->get_column_label(i).c_str());
  }
  
  /* set cells */

  cells = g_ptr_array_sized_new(nfields);

  for (nrows = 0; rs->next(); nrows++)
  {
    for (int i = 0; i < nfields; i++)
    {
      rs->get(i, obj);
      if (obj->is_null())
        g_ptr_array_add(cells, const_cast<char *>(""));
      else
      {
        const char *cp = 
          g_string_chunk_insert_const(str_chunk, obj->to_string().c_str());
        g_ptr_array_add(cells, const_cast<char *>(cp));
      }
    }
  }
  g_ptr_array_add(cells, NULL);
  
  /* set footers */

  if (!footers_ && !tbl_printer_->expanded() && default_footer_)
  {
    footers = g_new0(char *, 2);
    footers[0] = g_new(char, 100);
    
    if (nrows == 1)
      snprintf(footers[0], 100, gettext("(1 row)"));
    else
      snprintf(footers[0], 100, gettext("(%d rows)"), nrows);
  }
  else
    footers = NULL;

  /* set alignment */

  align = g_new0(char, nfields + 1);

  for (int i = 0; i < nfields; i++)
  {
    switch (rsmd->get_column_type(i).typecode())
    {
      case SQLType::FLOAT:
      case SQLType::SMALLINT:
      case SQLType::INTEGER:
      case SQLType::NUMERIC:
        align[i] = 'r';
        break;
      default:
        align[i] = 'l';
    }
  }
  
  /* call table printer */

  tbl_printer_->print(title_.c_str(), headers, (const char **)cells->pdata,
                      footers ? (const char *const *)footers : 
                      (const char *const *)footers_,
                      align, fout);

  g_free(headers);
  g_ptr_array_free(cells, FALSE);
  if (footers)
  {
    g_free(footers[0]);
    g_free(footers);
  }
  g_free(align);
  g_string_chunk_free(str_chunk);
}

}

