/*  
  Copyright 2002, Andreas Rottmann

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This library is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public  License
  along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <algorithm>

#include "gql++/database-metadata.h"

#include "app.h"

namespace GQLShell
{

using namespace std;
using namespace GQL;

class KeyReference
{
  public:
    KeyReference(const string& kname, int s, 
                 const string& refd_table, const string& refdkey)
        : keyname(kname), seqnr(s), table(refd_table), column(refdkey) { }
    
    string keyname;
    int seqnr;
    string table;
    string column;

    bool operator<(const KeyReference& r) const { 
      return keyname < r.keyname && seqnr < r.seqnr; 
    }
};

typedef vector<KeyReference> RefList;
typedef map<string, RefList> RefMap;

namespace
{

void insert(RefMap& ref_map, const string& key, const KeyReference& ref)
{
  RefMap::iterator it = ref_map.find(key);
  if (it != ref_map.end())
    it->second.push_back(ref);
  else
  {
    RefList reflist;
    reflist.push_back(ref);
    ref_map.insert(make_pair(key, reflist));
  }
}

void populate_by_keyname(RefMap& key_map, RefMap& ref_map)
{
  for (RefMap::iterator refs_it = ref_map.begin(); 
       refs_it != ref_map.end(); ++refs_it)
  {
    RefList& refs = refs_it->second;
    sort(refs.begin(), refs.end());
    for (RefList::iterator it = refs.begin(); it != refs.end(); ++it)
    {
      KeyReference ref = *it;
      
      if (!ref.keyname.empty())
      {
        // store column name in ref.keyname and insert the ref into
        // the per-foreign key map (key_map)
        string keyname = ref.keyname;
        ref.keyname = refs_it->first;
        insert(key_map, keyname, ref);
      }
    }
  }
}


bool handle_simple_key(const KeyReference& ref, RefMap& key_map)
{
  RefMap::iterator key_it = key_map.find(ref.keyname);
  if (key_it != key_map.end())
  {
    RefList& key_refs = key_it->second;
    
    // if we have more only one entry, we can handle the key in
    // the modifier line
    if (key_refs.size() <= 1)
    {
      key_map.erase(key_it);
      key_it = key_map.end();
    }
  }
  return key_it == key_map.end();
}

}

bool App::describe_table_details(const std::string& name, bool verbose)
{
  DatabaseMetaData *dbmd = conn_->get_meta_data();
  ResultSet *rs;
  SQLObject *obj = conn_->create_object();
  gchar *headers[4];
  gchar *str;
  GPtrArray *cells = g_ptr_array_new();
  GStringChunk *strchunk = g_string_chunk_new(1024);
  bool is_index = false;

  vector<string> types;
  types.push_back("TABLE");
  types.push_back("INDEX");
  rs = dbmd->get_tables("null", "null", name, types);
  if (!rs->next())
  {
    delete rs;
    error("no such relation '%s'\n", name.c_str());
    return false;
  }
  else
  {
    if (rs->get(3, obj)->to_string() == "INDEX")
      is_index = true;
  }
  delete rs;
  
  headers[0] = "Column";
  headers[1] = "Type";
  if (is_index)
    headers[2] = 0;
  else
  {
    headers[2] = "Modifiers";
    headers[3] = 0;
  }
  
  list<string> pkey_list;
  string pkey_name;
  RefMap ref_map, key_map;
  RefMap unique_map, unique_key_map;
  
  if (!is_index)
  {
    rs = dbmd->get_primary_keys("null", "null", name);
    while (rs->next())
    {
      pkey_list.push_back(rs->get(3, obj)->to_string());
    }
    delete rs;
    
    rs = dbmd->get_imported_keys("null", "null", name);
    while (rs->next())
    {
      string our_col = rs->get(7, obj)->to_string();
      KeyReference ref(rs->get(11, obj)->to_string(),
                       rs->get(8, obj)->to_int(),
                       rs->get(2, obj)->to_string(),
                       rs->get(3, obj)->to_string());
      insert(ref_map, our_col, ref);
    }
    delete rs;
    
    // populate the key_map
    populate_by_keyname(key_map, ref_map);
    
    rs = dbmd->get_index_info("null", "null", name, true, true);
    while (rs->next())
    {
      if (rs->get(6, obj)->to_int() == DatabaseMetaData::TableIndexStatistic)
        continue;
      string colname = rs->get(8, obj)->to_string();
      KeyReference ref(rs->get(5, obj)->to_string(),
                       rs->get(7, obj)->to_int(),
                       std::string(),
                       std::string());
      insert(unique_map, colname, ref);
    }
    delete rs;
    
    populate_by_keyname(unique_key_map, unique_map);
  
    if (pkey_list.size() == 1)
      pkey_name = *pkey_list.begin();
  }

  rs = dbmd->get_columns("null", "null", name, "%");
  while (rs->next())
  {
    string colname = rs->get(3, obj)->to_string();
    str = g_string_chunk_insert_const(strchunk, colname.c_str());
    g_ptr_array_add(cells, str);
    rs->get(4, obj);
    str = g_string_chunk_insert_const(strchunk, obj->to_string().c_str());
    g_ptr_array_add(cells, str);

    if (!is_index)
    {
      bool is_pkey = false;
    
      // now collect modifiers
      string modifiers;
      rs->get(7, obj);
      if (!obj->is_null())
        modifiers += "default " + obj->to_string() + " ";
      if (!pkey_name.empty() && pkey_name == colname)
      {
        modifiers += "primary key ";
        is_pkey = true;
      }
      else if (rs->get(5, obj)->to_int() == DatabaseMetaData::ColumnNoNulls)
        modifiers += "not null ";
    
      // check through the references for entries that we handle
      // in the modifier line
      
      // first unique indices
      RefMap::iterator refs_it = unique_map.find(colname);
      if (refs_it != unique_map.end())
      {
        RefList& refs = refs_it->second;
        for (RefList::iterator it = refs.begin(); it != refs.end(); ++it)
        {
          if (handle_simple_key(*it, unique_key_map) && !is_pkey)
            modifiers += "unique ";
        }
      }
      
      // then foreign keys
      refs_it = ref_map.find(colname);
      if (refs_it != ref_map.end())
      {
        RefList& refs = refs_it->second;
        for (RefList::iterator it = refs.begin(); it != refs.end(); ++it)
        {
          if (handle_simple_key(*it, key_map))
            modifiers += "references " + it->table + " (" + it->column + ") ";
        }
      }
    
      str = g_string_chunk_insert_const(strchunk, modifiers.c_str());
      g_ptr_array_add(cells, str);
    }
  }
  g_ptr_array_add(cells, NULL);
  
  delete rs;
  delete obj;
  
  // create footer
  GPtrArray *footer = g_ptr_array_new();

  // primary key line
  if (pkey_list.size() > 1)
  {
    string pkey_line = "PRIMARY KEY (";
    for (list<string>::iterator it = pkey_list.begin(); it != pkey_list.end();
         ++it)
    {
      pkey_line += *it + ", ";
    }
    pkey_line.resize(pkey_line.size() - 2);
    pkey_line += ")";
    str = g_string_chunk_insert_const(strchunk, pkey_line.c_str());
    g_ptr_array_add(footer, str);
  }

  // add unique constraints to the footer
  for (RefMap::iterator key_it = unique_key_map.begin(); 
       key_it != unique_key_map.end();  ++key_it)
  {
    RefList& refs = key_it->second;

    string unique_line = "UNIQUE (";
    for (RefList::iterator it = refs.begin(); it != refs.end(); ++it)
      unique_line += it->keyname + ", ";

    unique_line.resize(unique_line.length() - 2); // erase last comma
    unique_line += ")";
    
    str = g_string_chunk_insert_const(strchunk, unique_line.c_str());
    g_ptr_array_add(footer, str);
  }
  
  // add foreign key lines to the footer
  for (RefMap::iterator key_it = key_map.begin(); key_it != key_map.end(); 
       ++key_it)
  {
    RefList& refs = key_it->second;

    string fkey_part1 = "FOREIGN KEY (";
    string fkey_part2 = " REFERENCES " + refs.begin()->table + " (";
    for (RefList::iterator it = refs.begin(); it != refs.end(); ++it)
    {
      fkey_part1 += it->keyname + ", ";
      fkey_part2 += it->column + ", ";
    }
    // erase last commas
    fkey_part1.resize(fkey_part1.length() - 2);
    fkey_part2.resize(fkey_part2.length() - 2);
    
    fkey_part1 += ")";
    fkey_part2 += ")";
    fkey_part1 += fkey_part2;
    
    str = g_string_chunk_insert_const(strchunk, fkey_part1.c_str());
    g_ptr_array_add(footer, str);
  }
  g_ptr_array_add(footer, NULL);
  
  tbl_printer_.print(("Table \"" + name + "\"").c_str(), headers, 
                     (gchar **)cells->pdata,
                     (gchar **)footer->pdata, "lll", stdout);
  
  g_string_chunk_free(strchunk);
  g_ptr_array_free(cells, TRUE);
  g_ptr_array_free(footer, TRUE);
  
  return true;
}

/*
 * list_tables()
 *
 * handler for \d, \dt, etc.
 *
 * The infotype is an array of characters, specifying what info is desired:
 * t - tables
 * i - indexes
 * v - views
 * (any order of the above is fine)
 *
 */
bool App::list_tables(const string& infotypes, const string& name, bool desc)
{
  vector<string> types;

  if (infotypes.find('t') != string::npos)
    types.push_back("TABLE");
  if (infotypes.find('v') != string::npos)
    types.push_back("VIEW");
  if (infotypes.find('s') != string::npos)
    types.push_back("SEQUENCE");
  if (infotypes.find('i') != string::npos)
    types.push_back("INDEX");
  
  DatabaseMetaData *dbmd = conn_->get_meta_data();
  ResultSet *rs = dbmd->get_tables("null", "null", "%", types);
  SQLObject *obj = conn_->create_object();
  gchar *headers[4];
  GPtrArray *cells = g_ptr_array_new();
  GStringChunk *strchunk = g_string_chunk_new(1024);
  char *str;
  const char *align = desc ? "lll" : "ll";
  
  headers[0] = "Name";
  headers[1] = "Type";
  if (desc)
  {
    headers[2] = "Description";
    headers[3] = 0;
  }
  else
    headers[2] = 0;
  
  while (rs->next())
  {
    rs->get(2, obj);
    str = g_string_chunk_insert_const(strchunk, obj->to_string().c_str());
    g_ptr_array_add(cells, str);
    rs->get(3, obj);
    str = g_string_chunk_insert_const(strchunk, obj->to_string().c_str());
    g_ptr_array_add(cells, str);
    if (desc)
    {
      rs->get(4, obj);
      str = g_string_chunk_insert_const(strchunk, obj->to_string().c_str());
      g_ptr_array_add(cells, str);
    }
  }
  g_ptr_array_add(cells, NULL);
  
  if (cells->len == 1)
    printf("No relations found\n");
  else
    tbl_printer_.print("List of relations", headers, (gchar **)cells->pdata,
                       NULL, align, stdout);
  
  g_string_chunk_free(strchunk);
  g_ptr_array_free(cells, TRUE);
  
  delete rs;
  delete obj;
  
  return true;
}


}
