/* $Id: pipedtop.cc,v 1.7 2002/03/10 18:37:10 bergo Exp $ */

/*

    GPS - Graphical Process Statistics
    Copyright (C) 1999-2002 Felipe Paulo Guazzi Bergo
    bergo@seul.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <gtk/gtk.h>
#include "transient.h"

#include "polling.h"
#include "pipedtop.h"
#include "tstring.h"

#ifdef HAVETOP

/*********************************************************


 PiperAtTheGatesOfDawn


 *********************************************************/

/* opens a top slave process */
PiperAtTheGatesOfDawn::PiperAtTheGatesOfDawn() {
  FILE *f;
  char x;

  /* should look for top in a better way */
  f=popen("which top","r");
  fgets(toppath,128,f);
  pclose(f);
  if (toppath[strlen(toppath)-1]=='\n')
    toppath[strlen(toppath)-1]=0;

  pipe(pipe_to_top);
  pipe(pipe_from_top);

  toppid=fork();
  if (toppid==0) {
    close(0);
    dup(pipe_to_top[0]);
    close(pipe_to_top[0]);
    close(1);
    dup(pipe_from_top[1]);
    close(pipe_from_top[1]);
    signal(SIGPIPE,SIG_IGN); // shall we ?
    execlp(toppath,toppath,"-i","-dall","-s3600","infinity","-S",NULL);
    return;
  }

  fcntl(pipe_from_top[0],F_SETFL,O_NONBLOCK);
  while(read(pipe_from_top[0],&x,1)!=(-1)) ; // read first output
  fcntl(pipe_from_top[0],F_SETFL,0);
}

/* closes the top slave */
void PiperAtTheGatesOfDawn::top_meets_elvis() {
  char x[16];
  strcpy(x,"q\n");
  write(pipe_to_top[1],x,2); /* tell top to die on its own */
  kill(toppid,SIGHUP);
  waitpid(toppid,NULL,0);
  close(pipe_to_top[0]);
  close(pipe_to_top[1]);
  close(pipe_from_top[0]);
  close(pipe_from_top[1]);
}

void PiperAtTheGatesOfDawn::read_dead_output() {
  char x;
  fcntl(pipe_from_top[0],F_SETFL,O_NONBLOCK);
  while(read(pipe_from_top[0],&x,1)!=(-1)) ; // read lost data from last polls
  fcntl(pipe_from_top[0],F_SETFL,0);
}

char * PiperAtTheGatesOfDawn::read_top_line(char *buffer,int size) {
  return(myfgets(buffer,size,pipe_from_top[0]));
}

void PiperAtTheGatesOfDawn::request_refresh() {
  char y[4];
  strcpy(y," ");
  write(pipe_to_top[1],y,strlen(y)); // ask for refresh
  usleep(10*1000); // wait a little for refresh ?
}

/* fgets for file descriptors... */
char * PiperAtTheGatesOfDawn::myfgets(char *buffer,int bsize,int descriptor) {
  int i;
  memset(buffer,0,bsize);
  for(i=0;;i++) {
    if (read(descriptor,&buffer[i],1)==-1)
      if (i==0)
	return NULL;
      else
	return(buffer);
    if (buffer[i]=='\n')
      return(buffer);
  }
  return NULL; // <<-- should never get here
}

/***************************************************************


  PipedTopListPoller


 ***************************************************************/

PipedTopListPoller::PipedTopListPoller() 
  : ProcessListPoller(), PiperAtTheGatesOfDawn()
{
  /* top process is opened by PiperAtTheGatesofDawn's constructor */
}

void PipedTopListPoller::terminate() {
  this->top_meets_elvis(); /* kill top */
}

void PipedTopListPoller::poll() {
  ProcessListItem *pli;
  char buffer[1024],b2[512],b3[256];
  int nprocesses,i;
  char *xp;
  TString *t;

  reset();

  read_dead_output(); /* flush lost top output */
  request_refresh();  /* request new top poll (with Space) */

  if (read_top_line(buffer,1024)==NULL) return;
  if (read_top_line(buffer,1024)==NULL) return;

  t=new TString(1024);
  t->set(buffer);
  nprocesses=atoi(t->token(" \t\n"));

  while(1) {
    if (read_top_line(buffer,1024)==NULL) { delete t; return; }
    t->set(buffer);
    if (t->token(" \n\t")==NULL) break;
  }

  if (read_top_line(buffer,1024)==NULL) { delete t; return; }

  for(i=0;i<nprocesses;i++) {
    pli=new ProcessListItem();
    pli->setField(MACHINE,ProcessListPoller::get_local_hostname());

    read_top_line(buffer,1024);
    t->set(buffer);
    
    xp=t->token(" \n\t");
    if (xp==NULL) {
      delete pli;
      break;
    }

    /* PID */
    strcpy(b2,xp);
    pli->setField(PID,b2);

    /* OWNER */
    strcpy(b2,t->token(" \n\t"));
    pli->setField(OWNER,b2);

    /* PRIORITY */
    strcpy(b2,t->token(" \n\t"));
    pli->setField(PRIORITY,b2);

    /* NICE */
    strcpy(b2,t->token(" \n\t"));
    pli->setField(NICE,b2);

    /* SIZE */
    strcpy(b2,t->token(" \n\t"));
    pli->setField(SIZE,b2);

    /* RSS */
    strcpy(b2,t->token(" \n\t"));
    pli->setField(RSS,b2);

    /* STATE */
    strcpy(b2,t->token(" \n\t"));
    if (strcmp(b2,"run")==0)
      pli->setField(STATE,"R");
    if (strcmp(b2,"sleep")==0)
      pli->setField(STATE,"S");
    if (strcmp(b2,"stop")==0)
      pli->setField(STATE,"S");
    if (strcmp(b2,"idl")==0)
      pli->setField(STATE,"S");
    if (strcmp(b2,"WAIT")==0)
      pli->setField(STATE,"S");
    if (strcmp(b2,"zomb")==0)
      pli->setField(STATE,"Z");
   
    /* CPU TIME? */
    strcpy(b2,t->token(" \n\t"));

    /* CPU (WCPU in top) */
    strcpy(b2,t->token(" %\t\n"));
    snprintf(b3,256,"%.1f",atof(b2));
    b3[255]=0;
    pli->setField(CPU,b3);    

    strcpy(b2,t->token(" \n\t"));
    strcpy(b2,t->token(" \n\t"));

    pli->setField(NAME,b2);

    /* not parsed */
    pli->setField(START,"<not available>");

    process_list=g_list_append(process_list,(gpointer)pli);
  }

  read_dead_output();
  delete t;
}

/******************************************************************


  PipedTopDetailsPoller


 ******************************************************************/

void PipedTopDetailsPoller::terminate() {
  this->top_meets_elvis(); /* kill top */
}

/* currently this is much like the list poller */
void PipedTopDetailsPoller::poll(int whichpid) {
  char buffer[1024],b2[512],b3[256];
  int nprocesses,i;
  char *xp;
  TString *t;

  reset();

  read_dead_output(); /* flush lost top output */
  request_refresh();  /* request new top poll (with Space) */

  if (read_top_line(buffer,1024)==NULL) return;
  if (read_top_line(buffer,1024)==NULL) return;

  t=new TString(1024);
  t->set(buffer);
  nprocesses=atoi(t->token(" \n\t"));

  while(1) {
    if (read_top_line(buffer,1024)==NULL) { delete t; return; }
    t->set(buffer);
    if (t->token(" \n\t")==NULL) break;
  }

  if (read_top_line(buffer,1024)==NULL) { delete t; return; }

  for(i=0;i<nprocesses;i++) {
    read_top_line(buffer,1024);

    t->set(buffer);
    xp=t->token(" \n\t");
    if (xp==NULL)
      break;

    /* PID */
    strcpy(b2,xp);

    if (atoi(b2)!=whichpid)
      continue;
    
    item=new ProcessItem();
    item->setField(PID,b2);
    item->setField(MACHINE,ProcessDetailsPoller::get_local_hostname());
    
    /* OWNER */
    strcpy(b2,t->token(" \n\t"));
    item->setField(OWNER,b2);
    
    /* PRIORITY */
    strcpy(b2,t->token(" \n\t"));
    item->setField(PRIORITY,b2);
    
    /* NICE */
    strcpy(b2,t->token(" \n\t"));
    item->setField(NICE,b2);

    /* SIZE */
    strcpy(b2,t->token(" \n\t"));
    item->setField(SIZE,b2);

    /* RSS */
    strcpy(b2,t->token(" \n\t"));
    item->setField(RSS,b2);

    /* STATE */
    strcpy(b2,t->token(" \n\t"));
    if (strcmp(b2,"run")==0)
      item->setField(STATE,"R");
    if (strcmp(b2,"sleep")==0)
      item->setField(STATE,"S");
    if (strcmp(b2,"stop")==0)
      item->setField(STATE,"S");
    if (strcmp(b2,"idl")==0)
      item->setField(STATE,"S");
    if (strcmp(b2,"WAIT")==0)
      item->setField(STATE,"S");
    if (strcmp(b2,"zomb")==0)
      item->setField(STATE,"Z");
   
    /* CPU TIME? */
    strcpy(b2,t->token(" \n\t"));
    
    /* CPU (WCPU in top) */
    strcpy(b2,t->token(" %\t\n"));
    snprintf(b3,256,"%.1f",atof(b2));
    b3[255]=0;
    item->setField(CPU,b3);    
    
    strcpy(b2,t->token(" \n\t"));
    
    strcpy(b2,t->token(" \n\t"));

    item->setField(NAME,b2);

    /* not parsed */
    item->setField(START,"<not available from top>");
    item->setField(LONGNAME,"<not available from top>");
    item->setField(PPID,"<can't stat>");
    item->setField(PGID,"<can't stat>");
    item->setField(SID,"<can't stat>");
    item->setField(TTY,"<not available from top>");
    item->setField(TPGID,"<can't stat>");
    item->setField(FLAGS,"<not available from top>");

    item->setField(SIGPENDING,"<not available from top>");
    item->setField(SIGBLOCKED,"<not available from top>");
    item->setField(SIGIGNORED,"<not available from top>");
    item->setField(SIGCAUGHT,"<not available from top>");

    item->setField(RSSLIM,"<not available from top>");
    item->setField(MINFLT,"<not available from top>");
    item->setField(CMINFLT,"<not available from top>");
    item->setField(MAJFLT,"<not available from top>");
    item->setField(CMAJFLT,"<not available from top>");

    item->setField(USRJIFFIES,"<not available from top>");
    item->setField(KRNJIFFIES,"<not available from top>");
    item->setField(CUSRJIFFIES,"<not available from top>");
    item->setField(CKRNJIFFIES,"<not available from top>");
    
    break;
  }
  read_dead_output();
  delete t;
}

/****************************************************************

 
  PipedTopInfoProvider


 ****************************************************************/

PipedTopInfoProvider::PipedTopInfoProvider() 
  : SystemInfoProvider(), PiperAtTheGatesOfDawn()
{
  memory_total=1;
  memory_used=0;
  memory_free=1;
  swap_total=1;
  swap_free=1;
  swap_used=0;

  usleep(70*1000); /* wait some top setup ? */
  request_refresh(); /* get first refresh */
}

/*

 The next method assumes that top gives me a header with these lines
 being the third and fourth:

 CPU states: 0.4% user, 2.0% nice, 2.4% system, 97.2% idle
 Memory: 90M used, 3616K free, 27M shd, 35M buf  Swap: 216K used, 94M free

 Memory free is free+buf. top does not provide "cached", so at least on
 Linux this is wrong (should be free+buf+cached)

 */

void PipedTopInfoProvider::update() {
  /* should fill public variables from SystemInfoProvider with
     memory and overall CPU usage */

  char buffer[1024],b2[512];
  int i,j;
  char *x;
  long m[6];
  TString *t;

  do {
    read_dead_output(); /* flush lost top output */
    request_refresh();  /* request new top poll (with Space) */  
    
    if (read_top_line(buffer,1024)==NULL) continue;  
    if (read_top_line(buffer,1024)==NULL) continue;  
    if (read_top_line(buffer,1024)==NULL) continue;
    break;
  } while(1);    

  /* CPU = 100%-(idle) */

  t=new TString(1024);
  t->set(buffer);

  x=t->token(" \t\n");
  for(i=0;i<7;i++)
    t->token(" \t\n");

  x=t->token(" %\t\n");
  cpu_usage=(100.0-atof(x))/100.0;

  SMP_faker();

  /* MEMORY */
  
  if (read_top_line(buffer,1024)==NULL) {
    memory_total=1;
    memory_used=0;
    memory_free=1;
    swap_total=1;
    swap_free=1;
    swap_used=0;
    delete t;
    return;
  }

  t->set(buffer);
  x=t->token(" \t\n");
  x=t->token(" \t\n"); // used
  memset(b2,0,512);
  memcpy(b2,x,strlen(x)-1);
  j=strlen(x)-1;
  m[0]=atol(b2);
  if (x[j]=='K') m[0]<<=10;
  if (x[j]=='M') m[0]<<=20;
  if (x[j]=='G') m[0]<<=30;

  x=t->token(" \t\n");
  x=t->token(" \t\n"); // free
  memset(b2,0,512);
  memcpy(b2,x,strlen(x)-1);
  j=strlen(x)-1;
  m[1]=atol(b2);
  if (x[j]=='K') m[1]<<=10;
  if (x[j]=='M') m[1]<<=20;
  if (x[j]=='G') m[1]<<=30;

  x=t->token(" \t\n");
  x=t->token(" \t\n"); // shared
  memset(b2,0,512);
  memcpy(b2,x,strlen(x)-1);
  j=strlen(x)-1;
  m[2]=atol(b2);
  if (x[j]=='K') m[2]<<=10;
  if (x[j]=='M') m[2]<<=20;
  if (x[j]=='G') m[2]<<=30;

  x=t->token(" \t\n");
  x=t->token(" \t\n"); // buf
  memset(b2,0,512);
  memcpy(b2,x,strlen(x)-1);
  j=strlen(x)-1;
  m[3]=atol(b2);
  if (x[j]=='K') m[3]<<=10;
  if (x[j]=='M') m[3]<<=20;
  if (x[j]=='G') m[3]<<=30;

  x=t->token(" \t\n");
  x=t->token(" \t\n");
  x=t->token(" \t\n"); // swap used
  memset(b2,0,512);
  memcpy(b2,x,strlen(x)-1);
  j=strlen(x)-1;
  m[4]=atol(b2);
  if (x[j]=='K') m[4]<<=10;
  if (x[j]=='M') m[4]<<=20;
  if (x[j]=='G') m[4]<<=30;

  x=t->token(" \t\n");
  x=t->token(" \t\n"); // swap free
  memset(b2,0,512);
  memcpy(b2,x,strlen(x)-1);
  j=strlen(x)-1;
  m[5]=atol(b2);
  if (x[j]=='K') m[5]<<=10;
  if (x[j]=='M') m[5]<<=20;
  if (x[j]=='G') m[5]<<=30;

  memory_total=m[0]+m[1];
  memory_used=m[0]-m[3];
  memory_free=memory_total-memory_used;

  swap_free=m[5];
  swap_used=m[4];
  swap_total=m[4]+m[5];

  read_dead_output(); /* flush lost top output */
  delete t;
}

void PipedTopInfoProvider::terminate() {
  this->top_meets_elvis();
}

#endif
