/*
 * GPG Keys 
 * cdlgproperties.cpp
 * (c) 2001 Peter Mathiasson <peter@mathiasson.nu>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation
 */

#include <qstring.h>
#include <qtabbar.h>
#include <qtabwidget.h>
#include <qvbox.h>
#include <qwidget.h>
#include <qlistbox.h>
#include <qsplitter.h>
#include <qgroupbox.h>
#include <qlayout.h>
#include <qlineedit.h>
#include <qtextedit.h>
#include <qwidgetstack.h>
#include <qcheckbox.h>
#include <qlabel.h>
#include <qspinbox.h>
#include <qpushbutton.h>
#include <qsizepolicy.h>
#include <qfont.h>
#include <qfontdialog.h>
#include <qcombobox.h>

#include "cdlgproperties.h"
#include "functions.h"
#include "config.h"

/* Tabs */
struct {
    int optionCount;
    const char *name;
} tabs[] = {
    { 5, QT_TR_NOOP( "&General" ) },
    { 3, QT_TR_NOOP( "&Keyserver" ) },
    { 2, QT_TR_NOOP( "&Enviroment" ) },
    { -1, 0 }
};

/* Option Types */
enum {
    OPT_LINEEDIT,
    OPT_CHECKBOX,
    OPT_INTEGER,
    OPT_FONT,
    OPT_COMBO
};

/* Option Table */
struct {
    int optionType;
    const char *config;
    QWidget *widget;
    const char *name;
    const char *description;
    int integer;
    int intMin;
    int intMax;
    const char *string;
} options[] = {

    /* General Tab */
    {
        OPT_COMBO,                  // Option type
        "language",                 // Option variable
        0,
        QT_TR_NOOP("Language"),     // Item text
        QT_TR_NOOP(
            "Language used to display user visible text."
        ),
        0,                          // Combobox Id
        0, 0,
        LANGUAGE                    // Default
    },
    {
        OPT_FONT,
        "stdfont",
        0,
        QT_TR_NOOP("Standard font"),
        QT_TR_NOOP(
            "Font used to display all visiable text, except listings where fixed "
            "width font is used."
        ),
        STD_SIZE, STD_WEIGHT,       // Default Values
        STD_ITALIC, STD_FAMILY      // ...
    },
    {
        OPT_FONT,
        "fixedfont",
        0,
        QT_TR_NOOP("Fixed font"),
        QT_TR_NOOP(
            "Font used to display keylistings. This one should preferable be of fixed "
            "width."
        ),
        FIX_SIZE, FIX_WEIGHT,       // Default Values
        FIX_ITALIC, FIX_FAMILY      // ...
    },
    {
        OPT_CHECKBOX,               // Option type
        "mainsigs",                 // Option variable
        0,
        QT_TR_NOOP("List signatures"),
        QT_TR_NOOP(
            "Enable this if you would like to list signatures in the main keylist\n"
            "Disabling this will increase the list speed, and might be a good idea "
            "if you have a large keyring."
        ),
        MAINSIGS,                   // Default value
        0, 0,
        QT_TR_NOOP("Enable")        // Item text
    },    
    {
        OPT_CHECKBOX,               // Option type
        "utf8userid",               // Option variable
        0,
        QT_TR_NOOP("Unicode User IDs"),
        QT_TR_NOOP(
            "Enable this to unicode convert User IDs."
        ),
        UTF8_USERID,                // Default value
        0, 0,
        QT_TR_NOOP("Enable")        // Item text
    },    

    /* Keyserver Tab */
    {
        OPT_LINEEDIT,               // Option type
        "keyserver",                // Option variable
        0,
        QT_TR_NOOP("Keyserver"),
        QT_TR_NOOP(
            "Keyserver to use when for uploading, downloading or searching for keys."
        ),
        0, 0, 0,
        KEYSERVER                   // Default
    },
    {
        OPT_INTEGER,                // Option type
        "serverport",               // Option variable
        0,
        QT_TR_NOOP("Server port"),
        QT_TR_NOOP(
            "Port to connect to."
        ),
        SERVERPORT,                 // Default value
        1,                          // Min Value
        65535,                      // Max Value
        0
    },    
    {
        OPT_CHECKBOX,               // Option type
        "serversigs",               // Option variable
        0,
        QT_TR_NOOP("List signatures"),
        QT_TR_NOOP(
            "Enable this if you would like to list signatures in the server search "
            "dialog. This does not work with all servers, eg. keyserver.pgp.com."
        ),
        SERVERSIGS,                 // Default value
        0, 0,
        QT_TR_NOOP("Enable")        // Item text
    },    

    /* Enviroment Tab */
    {
        OPT_LINEEDIT,               // Option type
        "gpgpath",                  // Option variable
        0,
        QT_TR_NOOP("GPG executeable"),
        QT_TR_NOOP(
            "GPG executeable, eg. /usr/bin/gpg."
        ),
        0, 0, 0,
        GPGPATH                     // Default
    },    
    {
        OPT_LINEEDIT,               // Option type
        "terminal",                 // Option variable
        0,
        QT_TR_NOOP("Terminal application"),
        QT_TR_NOOP(
            "X-Windows terminal application to use. The GPG executeable parameter, "
            "including additional parameters will be appended to this.\n"
            "Eg. /usr/bin/X11/xterm -e"
        ),
        0, 0, 0,
        TERMINAL                    // Default
    },

    { -1, 0, 0, 0, 0, 0, 0, 0, 0 }
};

CDlgProperties::CDlgProperties( QWidget* parent ) : QDialog( parent, "CDlgProperties", TRUE )
{
    setCaption( tr( "Options" ) );
    resize( 600, 380 );
    
    /* Set Fonts */
    setFont( readConfigFont( "stdfont", STD_FONT ) );

    /* Create Dialog Layout */
    QVBoxLayout *mainLayout = new QVBoxLayout( this );
    mainLayout->setSpacing( 6 );
    mainLayout->setMargin( 11 );

    /* Build Tab Page Widget */
    QWidget *mainWidget = new QWidget( this );
    QVBoxLayout *mainWidgetLayout = new QVBoxLayout( mainWidget );
    mainWidgetLayout->setSpacing( 6 );
    mainWidgetLayout->setMargin( 11 );
    QSplitter *splitter = new QSplitter( mainWidget );
    splitter->setOrientation( QSplitter::Horizontal );
    listBox = new QListBox( splitter );
    connect( listBox, SIGNAL(selectionChanged()), this, SLOT(selectionChanged()) );
    QGroupBox *groupBox = new QGroupBox( splitter );
    groupBox->setTitle( "" );
    groupBox->setColumnLayout( 0, Qt::Vertical );
    groupBox->layout()->setSpacing( 0 );
    groupBox->layout()->setMargin( 0 );
    QVBoxLayout *boxLayout = new QVBoxLayout( groupBox->layout() );
    boxLayout->setAlignment( Qt::AlignTop );
    boxLayout->setSpacing( 6 );
    boxLayout->setMargin( 11 );
    widgetStack = new QWidgetStack( groupBox );
    for ( int i = 0; options[i].optionType != -1; i++ )
        createWidget( i );
    boxLayout->addWidget( widgetStack );
    textEdit = new QTextEdit( groupBox );
    textEdit->setReadOnly( TRUE );
    boxLayout->addWidget( textEdit );
    mainWidgetLayout->addWidget( splitter );
            
    /* Build Tab Widget */
    tabWidget = new QTabWidget( this );
    connect( tabWidget, SIGNAL(currentChanged(QWidget*)), this, SLOT(currentChanged(QWidget*)) );
    for ( int i = 0; tabs[i].optionCount != -1; i++ )
        tabWidget->addTab( mainWidget, new QTab( tr( tabs[i].name ) ) );
    mainLayout->addWidget( tabWidget );

    /* Build Buttons */
    QHBoxLayout *btnLayout = new QHBoxLayout;
    btnLayout->setSpacing( 6 );
    btnLayout->setMargin( 0 );
    QSpacerItem *spacer = new QSpacerItem( 20, 20, QSizePolicy::Expanding, QSizePolicy::Minimum );
    btnLayout->addItem( spacer );
    QPushButton *okButton = new QPushButton( tr( "&OK" ), this );
    okButton->setMinimumSize( QSize( 75, 0 ) );
    okButton->setDefault( TRUE );
    connect( okButton, SIGNAL(clicked()), this, SLOT(saveSettings()) );
    btnLayout->addWidget( okButton );
    QPushButton *cancelButton = new QPushButton( tr( "&Cancel" ), this );
    cancelButton->setMinimumSize( QSize( 75, 0 ) );
    connect( cancelButton, SIGNAL(clicked()), this, SLOT(reject()) );
    btnLayout->addWidget( cancelButton );
    mainLayout->addLayout( btnLayout );

}

/* select Font Button */
void CDlgProperties::selectFont()
{
    int currentOption = getStartOption() + listBox->currentItem();
    QLineEdit *lineEdit = (QLineEdit*)options[currentOption].widget;

    bool ok = FALSE;
    QFont font( lineEdit->font() );
    font = QFontDialog::getFont( &ok, font, this );
    if ( ok ) {
        lineEdit->setText( QString( tr( "%1, %2 pts" ) ).arg(font.family()).arg(font.pointSize()) );
        lineEdit->setFont( font );        
    }        
}

/* tabBar signal currentChanged */
void CDlgProperties::currentChanged( QWidget* )
{
    int startOption = getStartOption();
    int currentPage = tabWidget->currentPageIndex();
    listBox->clear();
    for ( int j = startOption; j < ( tabs[currentPage].optionCount + startOption ); j++ )
        listBox->insertItem( tr( options[j].name ) );    
    listBox->setCurrentItem( 0 );
}

/* listBox signal selectionChanged */
void CDlgProperties::selectionChanged()
{
    int currentOption = getStartOption() + listBox->currentItem();
    widgetStack->raiseWidget( currentOption );
    textEdit->setText( tr( options[currentOption].description ) );
}

/* Get first option in listbox */
int CDlgProperties::getStartOption()
{
    int startOption = 0;
    for ( int i = 0; i < tabWidget->currentPageIndex(); i++ )
        startOption += tabs[i].optionCount;
    return startOption;
}

void CDlgProperties::createWidget( int id )
{
    QLineEdit *lineEdit;
    QCheckBox *checkBox;
    QSpinBox *spinBox;
    QPushButton *pushButton;
    QHBoxLayout *hboxLayout;
    QComboBox *comboBox;
    QFont font;

    QWidget *widget = new QWidget( widgetStack );
    QVBoxLayout *widgetLayout = new QVBoxLayout( widget );
    widgetLayout->setSpacing( 0 );
    widgetLayout->setMargin( 0 );

    switch ( options[id].optionType ) {
        
    case OPT_LINEEDIT:
        lineEdit = new QLineEdit( widget );
        lineEdit->setText( readConfigString( options[id].config, options[id].string ) );
        widgetLayout->addWidget( lineEdit );
        options[id].widget = lineEdit;
        break;
        
    case OPT_CHECKBOX:
        checkBox = new QCheckBox( tr( options[id].string ), widget );
        checkBox->setChecked( readConfigInt( options[id].config, options[id].integer ) );
        widgetLayout->addWidget( checkBox );
        options[id].widget = checkBox;
        break;

    case OPT_INTEGER:
        spinBox = new QSpinBox( widget );
        spinBox->setRange( options[id].intMin, options[id].intMax );
        spinBox->setValue( readConfigInt( options[id].config, options[id].integer ) );
        widgetLayout->addWidget( spinBox );
        options[id].widget = spinBox;
        break;

    case OPT_FONT:
        hboxLayout = new QHBoxLayout;
        hboxLayout->setSpacing( 6 );
        hboxLayout->setMargin( 0 );
        lineEdit = new QLineEdit( widget );
        lineEdit->setReadOnly( TRUE );
        font = readConfigFont( options[id].config, options[id].string, options[id].integer,
                               options[id].intMin, options[id].intMax );
        lineEdit->setText( QString( tr( "%1, %2 pts" ) ).arg(font.family()).arg(font.pointSize()) );
        lineEdit->setFont( font );
        hboxLayout->addWidget( lineEdit );
        pushButton = new QPushButton( tr( "Select" ), widget );
        pushButton->setMinimumSize( QSize( 75, 0 ) );
        connect( pushButton, SIGNAL(clicked()), this, SLOT(selectFont()) );
        hboxLayout->addWidget( pushButton );
        widgetLayout->addLayout( hboxLayout );
        options[id].widget = lineEdit;
        break;

    case OPT_COMBO:
        comboBox = new QComboBox( widget );
        fillComboBoxes( comboBox, options[id].integer );
        setComboBoxText( comboBox, options[id].string );
        widgetLayout->addWidget( comboBox );
        options[id].widget = comboBox;
        break;        
        
    default:
        break;

    }

    widgetStack->addWidget( widget, id );
}

void CDlgProperties::fillComboBoxes( QComboBox *comboBox, int id )
{
    switch( id ) {
    
    case 0:     // Language Combo
        comboBox->insertItem( "English" );
        break;
    
    default:
        break;
    }
}

void CDlgProperties::saveSettings()
{
    QLineEdit *lineEdit;
    QCheckBox *checkBox;
    QSpinBox *spinBox;
    QComboBox  *comboBox;

    for ( int i = 0; options[i].optionType != -1; i++ ) {
        switch ( options[i].optionType ) {
        
        case OPT_LINEEDIT:
            lineEdit = (QLineEdit*)options[i].widget;
            writeConfigString( options[i].config, lineEdit->text() );
            break;

        case OPT_INTEGER:
            spinBox = (QSpinBox*)options[i].widget;
            writeConfigInt( options[i].config, spinBox->value() );
            break;
        
        case OPT_CHECKBOX:
            checkBox = (QCheckBox*)options[i].widget;
            writeConfigInt( options[i].config, checkBox->isChecked() );
            break;

        case OPT_COMBO:
            comboBox = (QComboBox*)options[i].widget;
            writeConfigString( options[i].config, comboBox->currentText() );
            break;

        case OPT_FONT:
            lineEdit = (QLineEdit*)options[i].widget;
            writeConfigFont( options[i].config, lineEdit->font() );
            break;

        default:
            break;

        }
    }
    accept();
}

