/*
 * Copyright (C) 2002,2003 Pascal Haakmat.
 * Licensed under the GNU GPL.
 * Absolutely no warranty.
 */

#include <errno.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <audiofile.h>
#include <gnome.h>
#include "pref.h"
#include <config.h>
#include "mem.h"
#include "mixer.h"
#include "muxers.h"
#include "muxertab.h"

/*
 * Returns true when the channel plays solo.
 */
int
mixer_is_source_solo(mixer *m,
                     unsigned int source_channel) {
    if(source_channel >= m->source_channels) {
        FAIL("cannot get source solo: source_channel %d out of bounds\n",
             source_channel);
        return 0;
    }
    return m->source_is_solo[source_channel] ? 1 : 0;
}

/*
 * Returns true when the channel is muted.
 */
int
mixer_is_source_mute(mixer *m,
                     unsigned int source_channel) {
    if(source_channel >= m->source_channels) {
        FAIL("cannot get source solo: source_channel %d out of bounds\n",
             source_channel);
        return 0;
    }
    return m->source_is_mute[source_channel] ? 0 : 1;
}

/*
 * Toggles solo on a channel.
 */

void
mixer_toggle_source_solo(mixer *m,
                         unsigned int source_channel) {
    if(source_channel >= m->source_channels) {
        FAIL("cannot set source solo: source_channel %d out of bounds\n",
             source_channel);
        return;
    }
    
    m->source_is_solo[source_channel] = ~m->source_is_solo[source_channel];
    DEBUG("source_channel: %d, new solo mask: %x\n", 
          source_channel, m->source_is_solo[source_channel]);
    m->num_solos += (m->source_is_solo[source_channel] ? 1 : -1);
}

/*
 * Mutes/unmutes a channel.
 */
void
mixer_toggle_source_mute(mixer *m,
                         unsigned int source_channel) {
    if(source_channel >= m->source_channels) {
        FAIL("cannot set source mute: source_channel %d out of bounds\n",
             source_channel);
        return;
    }
    DEBUG("source_channel: %d, new mute mask: %x\n", 
          source_channel, m->source_is_mute[source_channel]);
    m->source_is_mute[source_channel] = ~m->source_is_mute[source_channel];
}

/*
 * Interleaves samples from the fb_sources buffer array onto the
 * fb_target buffer. The number of channels in the fb_sources and
 * fb_target channels is determined by the mixer target_channels
 * parameter. This is really a special case of the more general
 * mixer_mux() function below. Used in file save.
 */
void
mixer_unity_mux_generic(mixer *m,
                        frame_bits_t fb_target,
                        frame_bits_t *fb_sources,
                        int frame_width,
                        AFframecount frame_count) {
    register int i, j, mul;
    int8_t *fb8t = (int8_t *)fb_target, **fb8s = (int8_t **)fb_sources;
    int16_t *fb16t = (int16_t *)fb_target, **fb16s = (int16_t **)fb_sources;
    int32_t *fb32t = (int32_t *)fb_target, **fb32s = (int32_t **)fb_sources;

    switch(frame_width) {
    case 1:
        for(i = 0; i < frame_count; i++) {
            mul = i * m->target_channels;
            for(j = 0; j < m->target_channels; j++) 
                fb8t[mul + j] = fb8s[j][i];
        }
        break;
    case 2:
        for(i = 0; i < frame_count; i++) {
            mul = i * m->target_channels;
            for(j = 0; j < m->target_channels; j++) 
                fb16t[mul + j] = fb16s[j][i];
        }
        break;
    case 4:
        for(i = 0; i < frame_count; i++) {
            mul = i * m->target_channels;
            for(j = 0; j < m->target_channels; j++) 
                fb32t[mul + j] = fb32s[j][i];
            break;
        }
    }
}

void
mixer_unity_mux(mixer *m,
                frame_bits_t fb_target,
                frame_bits_t *fb_sources,
                int frame_width,
                AFframecount frame_count) {
    if(m->source_channels <= NUM_UNITY_MUXERS && 
       m->target_channels <= NUM_UNITY_MUXERS &&
       m->source_channels > 0 && 
       m->target_channels > 0) {
        switch(frame_width) {
        case 1:
            muxtable_unity_8[m->source_channels-1].
                muxer(m, fb_target, fb_sources, frame_width, frame_count);
            break;
            
        case 2:
            muxtable_unity_16[m->source_channels-1].
                muxer(m, fb_target, fb_sources, frame_width, frame_count);
            break;

        case 4:
            muxtable_unity_32[m->source_channels-1].
                muxer(m, fb_target, fb_sources, frame_width, frame_count);
            break;
        }
        return;
    }
    mixer_unity_mux_generic(m, fb_target, fb_sources, frame_width, frame_count);
}

/*
 * Mixes samples from the fb_sources buffer array onto the fb_target
 * according to the mixer matrix. Used in playback and mixdown.
 */
inline void
mixer_mux_generic(mixer *m,
                  frame_bits_t fb_target,
                  frame_bits_t *fb_sources,
                  int frame_width,
                  AFframecount frame_count) {
    int i, j, k;
    int8_t *fb8t = (int8_t *)fb_target, **fb8s = (int8_t **)fb_sources;
    int16_t *fb16t = (int16_t *)fb_target, **fb16s = (int16_t **)fb_sources;
    int32_t *fb32t = (int32_t *)fb_target, **fb32s = (int32_t **)fb_sources;

    switch(frame_width) {
    case 1:
        for(i = 0; i < frame_count; i++) {
            for(j = 0; j < m->target_channels; j++) {
                for(k = 0; k < m->source_channels; k++) 
                    fb8t[(i * m->target_channels) + j] +=
                        (float)(fb8s[k][i] & 
                                (m->num_solos ? 
                                 m->source_is_solo[k] : 
                                 m->source_is_mute[k])) *
                        m->mixtable[j][k];
            }
        }
        break;
    case 2:
        for(i = 0; i < frame_count; i++) {
            for(j = 0; j < m->target_channels; j++) {
                for(k = 0; k < m->source_channels; k++) 
                    fb16t[(i * m->target_channels) + j] +=
                        (float)(fb16s[k][i] & 
                                (m->num_solos ? 
                                 m->source_is_solo[k] : 
                                 m->source_is_mute[k])) *
                        m->mixtable[j][k];
            }
        }
        break;
    case 4:
        for(i = 0; i < frame_count; i++) {
            for(j = 0; j < m->target_channels; j++) {
                for(k = 0; k < m->source_channels; k++) 
                    fb32t[(i * m->target_channels) + j] +=
                        (float)(fb32s[k][i] & 
                                (m->num_solos ? 
                                 m->source_is_solo[k] : 
                                 m->source_is_mute[k])) *
                        m->mixtable[j][k];
            }
        }
        break;
    }
}

/*
 * Mixes samples from the fb_sources buffer array onto the fb_target
 * according to the mixer matrix. Used in playback and mixdown.
 */
inline void
mixer_mux(mixer *m,
          frame_bits_t fb_target,
          frame_bits_t *fb_sources,
          int frame_width,
          AFframecount frame_count) {

    if(m->is_unity) {
        mixer_unity_mux(m, fb_target, fb_sources, frame_width, frame_count);
        return;
    }
    
    /*
     * Try to use a loop-unrolled version of the required muxer. This
     * can be a lot faster:
     *
     *             | generic_muxer  | loop_unrolled | % diff
     * ------------+----------------+-----------------+-------
     * 1 to 1      |     0.148664s. |    0.084907s. | 75
     * 2 to 1      |     0.247643s. |    0.187732s. | 31
     * 3 to 1      |     0.343074s. |    0.276960s. | 23
     * 4 to 1      |     0.440596s. |    0.365875s. | 20
     * 5 to 1      |     0.538369s. |    0.459118s. | 17
     *
     * Similar figures are obtained for 1 to 2, 2 to 2, etc 
     * (tests to obtain these results are in tests.c). After 7 or
     * 8 channels the gains flatten off rapidly due to the
     * decrease in the number of branches saved.
     *
     * What these tests don't consider is how the increased code size
     * affects the interaction of this code with the surrounding
     * code. It is conceivable that in some situations this code
     * results in a net negative effect.
     *
     */
    if(m->source_channels <= NUM_MUXERS && 
       m->target_channels <= NUM_MUXERS &&
       m->source_channels > 0 && 
       m->target_channels > 0) {
        switch(frame_width) {
        case 1:
            muxtable_8[m->source_channels-1][m->target_channels-1].
                muxer(m, fb_target, fb_sources, frame_width, frame_count);
            break;

        case 2:
            muxtable_16[m->source_channels-1][m->target_channels-1].
                muxer(m, fb_target, fb_sources, frame_width, frame_count);
            break;

        case 4:
            muxtable_32[m->source_channels-1][m->target_channels-1].
                muxer(m, fb_target, fb_sources, frame_width, frame_count);
            break;
        }
        return;
    }

    /* Fall back on generic muxer. */

    mixer_mux_generic(m, fb_target, fb_sources, frame_width, frame_count);
}

/*
 * The opposite of mixer_mux(), this function extracts a channel 
 * from the buffer fb_target and copies it to the buffer fb_source.
 *
 * @param fb_target the target buffer.
 * @param fb_source the source buffer containing a number of
 * interleaved channels.
 * @param channel the channel to extract.
 * @param channel_count the number of channels in fb_source.
 * @param frame_width bytes per sample.
 */
void
mixer_demux(frame_bits_t fb_target,
            frame_bits_t fb_source,
            int channel,
            int channel_count,
            int frame_width,
            AFframecount frame_count) {
    register int i, src_index;
    int8_t *fb8t = (int8_t *)fb_target, *fb8s = (int8_t *)fb_source;
    int16_t *fb16t = (int16_t *)fb_target, *fb16s = (int16_t *)fb_source;
    int32_t *fb32t = (int32_t *)fb_target, *fb32s = (int32_t *)fb_source;

    src_index = channel;
    switch(frame_width) {
    case 1:
        for(i = 0; i < frame_count - 8; i += 8) {
            fb8t[i] = fb8s[src_index];
            src_index += channel_count;
            fb8t[i+1] = fb8s[src_index];
            src_index += channel_count;
            fb8t[i+2] = fb8s[src_index];
            src_index += channel_count;
            fb8t[i+3] = fb8s[src_index];
            src_index += channel_count;
            fb8t[i+4] = fb8s[src_index];
            src_index += channel_count;
            fb8t[i+5] = fb8s[src_index];
            src_index += channel_count;
            fb8t[i+6] = fb8s[src_index];
            src_index += channel_count;
            fb8t[i+7] = fb8s[src_index];
            src_index += channel_count;
        }
        for(; i < frame_count; i++, src_index += channel_count) 
            fb8t[i] = fb8s[src_index];
        break;
    case 2:
        for(i = 0; i < frame_count - 8; i += 8) {
            fb16t[i] = fb16s[src_index];
            src_index += channel_count;
            fb16t[i+1] = fb16s[src_index];
            src_index += channel_count;
            fb16t[i+2] = fb16s[src_index];
            src_index += channel_count;
            fb16t[i+3] = fb16s[src_index];
            src_index += channel_count;
            fb16t[i+4] = fb16s[src_index];
            src_index += channel_count;
            fb16t[i+5] = fb16s[src_index];
            src_index += channel_count;
            fb16t[i+6] = fb16s[src_index];
            src_index += channel_count;
            fb16t[i+7] = fb16s[src_index];
            src_index += channel_count;
        }
        for(; i < frame_count; i++, src_index += channel_count) 
            fb16t[i] = fb16s[src_index];
        break;
    case 4:
        for(i = 0; i < frame_count - 8; i += 8) {
            fb32t[i] = fb32s[src_index];
            src_index += channel_count;
            fb32t[i+1] = fb32s[src_index];
            src_index += channel_count;
            fb32t[i+2] = fb32s[src_index];
            src_index += channel_count;
            fb32t[i+3] = fb32s[src_index];
            src_index += channel_count;
            fb32t[i+4] = fb32s[src_index];
            src_index += channel_count;
            fb32t[i+5] = fb32s[src_index];
            src_index += channel_count;
            fb32t[i+6] = fb32s[src_index];
            src_index += channel_count;
            fb32t[i+7] = fb32s[src_index];
            src_index += channel_count;
        }
        for(; i < frame_count; i++, src_index += channel_count) 
            fb32t[i] = fb32s[src_index];
        break;
    }

}

float **
mixer_mixtable_new(int target_channels,
                   int source_channels) {
    int i;
    float **mt = NULL;
    
    mt = (float **) mem_alloc(target_channels * sizeof(float *));
    if(!mt) {
        FAIL("could not allocate mix table (%"CONVSPEC_SIZE_T" bytes)\n", 
             target_channels * sizeof(float));
            return NULL;
    }
    //    DEBUG("allocated %d bytes for mixtable\n",
    //          target_channels * sizeof(float));
    for(i = 0; i < target_channels; i++) {
        mt[i] = mem_alloc(source_channels * sizeof(float));
        if(!mt[i]) {
            for(i--; i + 1 > 0; i--)
                mem_free(mt[i]);
            mem_free(mt);
            FAIL("could not allocate mix table rows\n");
            return NULL;
        }
    }
    return mt;
}

void
mixer_mixtable_destroy(float **mt,
                       int target_channels) {
    int i;
    for(i = 0; i < target_channels; i++) 
        mem_free(mt[i]);
    mem_free(mt);
    //    DEBUG("freed %d channel mixtable\n", target_channels);
}

void
mixer_mute(mixer *m,
           int source_channel) {
}

void 
mixer_dump(mixer *m) {
    int i, j;
    DEBUG("target_channels: %d, source_channels: %d\n",
          m->target_channels, m->source_channels);
    for(i = 0; i < m->target_channels; i++) {
        INFO("%2i:", i);
        for(j = 0; j < m->source_channels; j++) 
            INFO(" %.3f (mute: %x)", m->mixtable[i][j], m->source_is_mute[j]);
        INFO("\n");
    }
    
}

void
mixer_init(mixer *m) {
    int i, j;
    if(!m->target_channels)
        return;
    for(i = 0; i < m->target_channels; i++) 
        for(j = 0; j < m->source_channels; j++) 
            m->mixtable[i][j] = 0;
}

void
mixer_configure(mixer *m,
                int target_channels,
                int source_channels) {
    int i, j;

    if(target_channels > pref_get_as_int("invariant_max_tracks") ||
       source_channels > pref_get_as_int("invariant_max_tracks")) {
        FAIL("illegal configure request, target_channels: %d, source_channels: %d\n",
             target_channels, source_channels);
        abort();
    }

    //    DEBUG("m->target_channels: %d, target_channels: %d, m->source_channels: %d, source_channels: %d\n",
    //          m->target_channels, target_channels, m->source_channels, source_channels);
    for(i = 0; i < target_channels; i++) {
        for(j = m->source_channels; j < source_channels; j++) {
            //            DEBUG("[%d][%d] = 0\n", i, j);
            m->mixtable[i][j] = i == j ? 1 : 0;
        }
    }
    m->target_channels = target_channels;
    m->source_channels = source_channels;
    //    mixer_dump(m);

//    DEBUG("target_channels: %d, source_channels: %d\n", 
//          m->target_channels, m->source_channels);
}

void
mixer_destroy(mixer *m) {
    DEBUG("destroying mixer\n");
    mixer_mixtable_destroy(m->mixtable, pref_get_as_int("invariant_max_tracks"));
    mem_free(m->source_is_mute);
    mem_free(m);
    
}

mixer *
mixer_new(int target_channels,
          int source_channels) {
    int32_t *ptr;
    int i;
    mixer *m = mem_calloc(sizeof(mixer), 1);
    if(!m) {
        FAIL("could not allocate mixer object.\n");
        return NULL;
    }
    
    /* Allocate once for both source_is_solo and source_is_mute
       arrays, make source_is_mute point at the beginning. */

    ptr = mem_alloc(2 * sizeof(int32_t) *
                    pref_get_as_int("invariant_max_tracks"));
    if(!ptr) {
        FAIL("could not allocate source mute & solo tables.\n");
        free(m);
        return NULL;
    }

    m->mixtable = mixer_mixtable_new(pref_get_as_int("invariant_max_tracks"), 
                                     pref_get_as_int("invariant_max_tracks"));

    if(!m->mixtable) {
        FAIL("could not allocate mixtable.\n");
        free(m);
        free(ptr);
        return NULL;
    }

    m->source_is_mute = ptr;
    m->source_is_solo = &ptr[pref_get_as_int("invariant_max_tracks")];
    for(i = 0; i < pref_get_as_int("invariant_max_tracks"); i++) {
        /* Unmute all source channels. */
        m->source_is_mute[i] = 0xFFFFFFFF;
        /* Unsolo all source channels. */
        m->source_is_solo[i] = 0;
    }
    m->num_solos = 0;
    
    mixer_configure(m, target_channels, source_channels);
    mixer_init(m);
    
    return m;
}

void 
mixer_buffers_free(int tracks,
                   frame_bits_t fb_downmix,
                   frame_bits_t *fb_sources) {
    int i;
    for(i = 0; i < tracks; i++) {
        if(fb_sources[i])
            mem_free(fb_sources[i]);
        fb_sources[i] = NULL;
    }
    DEBUG("freed mixer buffers %p\n", fb_downmix);
    if(fb_downmix)
        mem_free(fb_downmix);
}

frame_bits_t
mixer_buffers_alloc(int frame_width,
                    int tracks,
                    frame_bits_t *fb_downmix,
                    frame_bits_t *fb_sources,
                    AFframecount frame_count) {
    int i;
    for(i = 0; i < tracks; i++)
        fb_sources[i] = NULL;
    *fb_downmix = mem_alloc(frame_width * tracks * frame_count);
    if(!*fb_downmix) {
        FAIL("could not get memory for downmix buffer (%d tracks).\n", tracks);
        return NULL;
    }
    for(i = 0; i < tracks; i++) {
        fb_sources[i] = mem_alloc(frame_width * frame_count);
        if(fb_sources[i] == NULL) {
            FAIL("could not get memory for track buffer %d\n", i);
            for(i--; i + 1 > 0; i--) {
                mem_free(fb_sources[i]);
                fb_sources[i] = NULL;
            }
            mem_free(*fb_downmix);
            *fb_downmix = NULL;
            return NULL;
        }
    }
    DEBUG("mixer buffers allocated: muxbuf: %ld bytes@%p, srcbufs: %ld bytes (* %d)\n",
          frame_width * tracks * frame_count, *fb_downmix, frame_width * frame_count, tracks);

    return *fb_downmix;
}


int
mixer_save(mixer *m,
           const char *path) {
    int i, j, argc = m->source_channels;
    char key[512], *argv[argc];
    
    for(i = 0; i < argc; i++) {
        argv[i] = mem_alloc(32);
        if(!argv[i]) { 
            FAIL("not enough memory to save mixer, could not allocate 32 bytes for argv %d.\n", i);
            for(i--; i; i--)
                mem_free(argv[i]);
            return 1;
        }
    }

    for(i = 0; i < m->target_channels; i++) {
        snprintf(key, 512, "=%s=/Mixer/%d", path, i);
        for(j = 0; j < m->source_channels; j++) 
            snprintf(argv[j], 32, "%f", m->mixtable[i][j]);
        gnome_config_set_vector(key, m->source_channels, 
                                (const char * const *) argv);
    }
    
    snprintf(key, 512, "=%s=/Mixer/Source Is Mute", path);
    for(j = 0; j < m->source_channels; j++) 
        snprintf(argv[j], 32, "%d", m->source_is_mute[j] ? 0 : 1);
    gnome_config_set_vector(key, m->source_channels, 
                            (const char * const *) argv);

    snprintf(key, 512, "=%s=/Mixer/Source Is Solo", path);
    for(j = 0; j < m->source_channels; j++) 
        snprintf(argv[j], 32, "%d", m->source_is_solo[j] ? 1 : 0);
    gnome_config_set_vector(key, m->source_channels, 
                            (const char * const *) argv);
    for(i = 0; i < argc; i++)
        if(argv[i])
            mem_free(argv[i]);

    return 0;
}

int
mixer_load(mixer *m,
           const char *path) {
    unsigned int i, j, argc, r = 0;
    char **argv;
    char tmp[512];
    mixer_init(m);
    for(i = 0; i < m->target_channels; i++) {
        snprintf(tmp, 512, "=%s=/Mixer/%i", path, i);
        gnome_config_get_vector(tmp, &argc, &argv);
        for(j = 0; j < argc; j++) {
            if(j < m->source_channels) 
                m->mixtable[i][j] = atof(argv[j]);
            else {
                r = 1;
                FAIL("source channel %d (value: %f) out of bounds.\n",
                     j, atof(argv[j]));
            }
        }
        for(j = 0; j < argc; j++) 
            g_free(argv[j]);
        g_free(argv);
    }
    
    /* "Source Is Mute" settings are stored on disk basically opposite
       from how they are stored in memory: a 1 on disk means a 0 in
       memory and a 0 on disk means 0xFFFFFFFF in memory. */

    snprintf(tmp, 512, "=%s=/Mixer/Source Is Mute", path);
    gnome_config_get_vector(tmp, &argc, &argv);
    for(j = 0; j < argc; j++) {
        if(j < m->source_channels) 
            m->source_is_mute[j] = (atoi(argv[j]) ? 0 : 0xFFFFFFFF);
        else {
            r = 1;
            FAIL("is muted channel %d (value: %d) out of bounds.\n",
                 j, atoi(argv[j]));
        }
    }

    for(j = 0; j < argc; j++) 
        g_free(argv[j]);
    g_free(argv);

    /* For "Source Is Solo" it's the same story as for "Source Is
       Mute" above. */

    snprintf(tmp, 512, "=%s=/Mixer/Source Is Solo", path);
    gnome_config_get_vector(tmp, &argc, &argv);
    for(j = 0; j < argc; j++) {
        if(j < m->source_channels) 
            m->source_is_solo[j] = (atoi(argv[j]) ? 0xFFFFFFFF : 0);
        else {
            r = 1;
            FAIL("is muted channel %d (value: %d) out of bounds.\n",
                 j, atoi(argv[j]));
        }
    }

    for(j = 0; j < argc; j++) 
        g_free(argv[j]);
    g_free(argv);
    
    return r;
}

/***********************************************************************/
/* Below code deprecated in favor of gnome_config_* (see file.c).      */
/***********************************************************************/

int
mixer_save_compat(mixer *m,
                  const char *path) {
    int i, j;
    FILE *out;
    
    out = fopen(path, "w");
    if(out == NULL) {
        FAIL("could not open %s for writing\n", path);
        return 1;
    }

    for(i = 0; i < m->target_channels; i++) {
        fprintf(out, "%2i:", i);
        for(j = 0; j < m->source_channels; j++) 
            fprintf(out, " %.3f", m->mixtable[i][j]);
        fprintf(out, "\n");
    }

    if(fclose(out)) {
        FAIL("error closing %s\n", path);
        return 1;
    }

    return 0;
}

/*
 * Parses a target specification line:
 * [0-9]+:( [0-9]+.[0-9]+)*
 * \____/ \_______________/
 *   |            |
 *   |            +-> source channel amplification factor(s)
 *   +-> target channel
 * where : and . stand for themselves, ( and ) denote grouping.
 * @return zero on succes.
 */
int
mixer_parse_line(mixer *m,
                 char *line) {
    char *token, *endptr;
    int target, source;
    double amp;
    while(*line != '\0' && isspace(*line)) 
        line++;
    token = strsep(&line, ":");
    //    DEBUG("token: %s, remaining: %s\n", token, line);
    if(token == NULL) {
        FAIL("parse error: expected [0-9]+: on line %s\n",
             line);
        return 1;
    }

    target = (int)strtol(token, &endptr, 10);
    //    DEBUG("target: %d\n", target);
    if(!(*token != '\0' && *endptr == '\0')) {
        FAIL("invalid token %s, expected integer\n", token);
        return 1;
    }

    if(target > m->target_channels || target < 0) {
        FAIL("target specification out of range: %d\n", target);
        return 1;
    }

    /* Now the source channels. */

    source = 0;
    while(*line != '\0' && isspace(*line)) 
        line++;
    for(token = strsep(&line, " "); token; token = strsep(&line, " "), source++) {
        //        DEBUG("token: %s, remaining: %s\n", token, line);
        amp = strtod(token, &endptr);
        if(!(*token != '\0' && *endptr == '\0')) {
            FAIL("invalid token %s, expected double\n", token);
            return 1;
        }
        while(line && *line != '\0' && isspace(*line)) 
            line++;
        if(amp > 1 || amp < 0) {
            FAIL("amplification factor out of range: %f\n", amp);
            continue;
        }
        
        //        DEBUG("target: %d, source: %d, amp: %f\n",
        //              target, source, amp);

        m->mixtable[target][source] = amp;
    }
    return 0;
}

/* 
 * @return zero on success.
 */
int
mixer_load_compat(mixer *m,
                  const char *path) {
    int fd, r;
    char *buf, *line;
    struct stat stats;

    r = stat(path, &stats);

    if(r == -1) {
        FAIL("unable to stat %s: %s\n", path, strerror(errno));
        return 1;
    }
    
    buf = mem_alloc(stats.st_size + 1);    
    if(!buf) {
        FAIL("unable to allocate %ld bytes for reading %s\n", 
             stats.st_size + 1, path);
        return 1;
    }

    fd = open(path, O_RDONLY);
    if(fd == -1) {
        FAIL("unable to open %s: %s\n", path, strerror(errno));
        mem_free(buf);
        return 1;
    }
    
    r = read(fd, buf, stats.st_size);
    if(r == -1) {
        FAIL("error while reading %s: %s\n", path, strerror(errno));
        mem_free(buf);
        close(fd);
        return 1;
    }
    close(fd);

    if(r != stats.st_size) 
        FAIL("while reading %s: expected %ld bytes, got %d bytes\n", 
             path, stats.st_size, r);

    buf[r] = '\0';
    line = strtok(buf, "\n\r");
    while(line) {
        //        DEBUG("read line: %s\n", line);
        if(line[0] == '#') {
            //            DEBUG("skipping comment\n");
            line = strtok(NULL, "\n\r");
            continue;
        }
        
        r = mixer_parse_line(m, line);
        if(r) {
            mem_free(buf);
            return r;
        }
        line = strtok(NULL, "\n\r");
    }        
    
    mem_free(buf);
    return 0;
}

