/*
** Copyright (C) 10 Feb 1999 Jonas Munsin <jmunsin@iki.fi>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <glib.h>
#include <gdk/gdkkeysyms.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>

#include "common_gtk.h"
#include "filepicker.h"
#include "vector_commands.h"
#include "clist_common.h"
#include "optimize_usage.h"
#include "modify_file_set.h"
#include "command.h"
#include "linebuffer.h"
#include "mainwindow.h"
#include "globals.h"

static GtkWidget *whole_path, *user_mod_path;
static int last_row_selected = -1;

GtkWidget *total_size, *exclude_pattern_combo, *clist, *exclude_dir;
GtkWidget *mkisofs_follow_symlinks, *mkisofs_include_all, *mkisofs_include_dirname;
GtkWidget *path_default_level_sb, *path_default_from_end;
GList *exclude_pattern_list = NULL;

long long int totalsize, sectorsize;
int selected_files[MAX_ROWS];

void mark_size_inaccurate(void) {
	gtk_widget_set_sensitive(total_size, 0);
}

static void toggle_fsymlink(GtkWidget *widget, gpointer data) {
	mark_size_inaccurate();
}

static void toggle_dirname(GtkWidget *widget, gpointer data) {
	mark_size_inaccurate();
}

static void toggle_incall(GtkWidget *widget, gpointer data) {
	mark_size_inaccurate();
}

static void select_all(GtkWidget *widget, gpointer data) {
	gtk_clist_select_all(GTK_CLIST(clist));
}

static void unselect_all(GtkWidget *widget, gpointer data) {
	gtk_clist_unselect_all(GTK_CLIST(clist));
}

static void selection_made(GtkWidget *widget, gint row, gint column,
		GdkEventButton *event, gint *data) {
	file_data *selected_row_info;

	selected_row_info = gtk_clist_get_row_data(GTK_CLIST(clist), row);
	g_assert(NULL != selected_row_info);
	gtk_label_set(GTK_LABEL(whole_path), selected_row_info->realpath);
	gtk_entry_set_text(GTK_ENTRY(user_mod_path), selected_row_info->userpath);

	if (row >= MAX_ROWS)
		g_warning("filepicker.c::selection_made: MAX_ROWS too small!");
	else
		selected_files[row] = 1;
	last_row_selected = row;
}

static void unselection_made(GtkWidget *widget, gint row, gint column,
		GdkEventButton *event, gint *data) {
	if (row >= MAX_ROWS)
		g_warning("filepicker.c::unselection_made: MAX_ROWS too small!");
	else
		selected_files[row] = 0;
	gtk_label_set(GTK_LABEL(whole_path), "");
	gtk_entry_set_text(GTK_ENTRY(user_mod_path), "");
	last_row_selected = -1;
}

static void connect_selsigs(GtkWidget *cl) {
	gtk_signal_connect(GTK_OBJECT(cl), "select_row",
			GTK_SIGNAL_FUNC(selection_made), NULL);
	gtk_signal_connect(GTK_OBJECT(cl), "unselect_row",
			GTK_SIGNAL_FUNC(unselection_made), NULL);
}

static GtkWidget *create_toolbar(void) {
	GtkWidget *toolbar;

	toolbar = gtk_toolbar_new(GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_TEXT);
	gtk_toolbar_append_item(GTK_TOOLBAR(toolbar), _("Add directory"),
			_("Opens file selector to let you add a file or directory. If you "
				"want to add an entry starting with . (dot), write a . in the "
				"name selection field of the file selector and press TAB"), "Private",
			NULL, GTK_SIGNAL_FUNC(choose_file), (gpointer) selected_file);

	gtk_toolbar_append_item(GTK_TOOLBAR(toolbar), _("Del selected"),
			_("Deletes the selected files/directories"), "Private",
			NULL, GTK_SIGNAL_FUNC(del_selected_rows), NULL);

	gtk_toolbar_append_item(GTK_TOOLBAR(toolbar), _("Select all"),
			_("Selects all files/directories"), "Private",
			NULL, GTK_SIGNAL_FUNC(select_all), NULL);

	gtk_toolbar_append_item(GTK_TOOLBAR(toolbar), _("Unselect all"),
			_("Unselects all files/directories"), "Private",
			NULL, GTK_SIGNAL_FUNC(unselect_all), NULL);

	gtk_widget_show_all(toolbar);

	return toolbar;
}


static void clist_keypress(GtkWidget *widget, GdkEventKey *evt, gpointer user) {
	switch(evt->keyval)
	{       
		case GDK_a:
			if(evt->state & GDK_CONTROL_MASK)
				select_all(NULL, NULL);
			break;
		case GDK_A:
			if(evt->state & GDK_CONTROL_MASK)
				unselect_all(NULL, NULL);
			break;
		case GDK_Delete:
			del_selected_rows(widget, NULL);
			break;
	}
}
  
static void create_filelist(void) {
	int i;
	clist = gtk_clist_new(2);
	gtk_widget_show(clist);

	gtk_clist_set_column_title(GTK_CLIST(clist), 0, _("File"));
	gtk_clist_set_column_title(GTK_CLIST(clist), 1, _("Size (MB)"));
/*	gtk_clist_set_column_width(GTK_CLIST(clist), 0, 400);
	gtk_clist_set_column_width(GTK_CLIST(clist), 1, 20); */
	gtk_clist_column_titles_passive(GTK_CLIST(clist));
	gtk_clist_column_titles_show(GTK_CLIST(clist));
	gtk_clist_set_column_justification(GTK_CLIST(clist), 0, GTK_JUSTIFY_LEFT);
	gtk_clist_set_column_justification(GTK_CLIST(clist), 1, GTK_JUSTIFY_RIGHT); 

	for (i = 0; i < MAX_ROWS; i++) {
		selected_files[i] = 0;
	}

	connect_selsigs(clist);
	gtk_signal_connect(GTK_OBJECT(clist), "key_press_event",
			GTK_SIGNAL_FUNC(clist_keypress), NULL);

	gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 0, TRUE);

	gtk_clist_set_selection_mode(GTK_CLIST(clist), GTK_SELECTION_MULTIPLE);
}

static GtkWidget *create_button_area(config_cdr_data *cdr_info) {
	GtkWidget *vbox, *recalc, *button_box;
	GtkWidget *hbox, *label, *radiob;
	GtkObject *spinb_adj;
	GSList *rgroup = NULL;


	vbox = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(vbox);

	total_size = gtk_label_new(_("Sectors:      0\nSize:   0.0 MB"));
	gtk_widget_show(total_size);
	recalc = gtk_button_new();
	gtk_widget_show(recalc);
	gtk_signal_connect(GTK_OBJECT(recalc), "clicked", GTK_SIGNAL_FUNC(recalc_size), cdr_info);
	gtk_tooltips_set_tip(tooltips, recalc, _("Update size estimate"), NULL);

	button_box = gtk_hbox_new(FALSE, 4);
	gtk_widget_show(button_box);
	gtk_container_add(GTK_CONTAINER(recalc), button_box);
	gtk_box_pack_start(GTK_BOX(button_box), total_size, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), recalc, FALSE, FALSE, 0);

	mkisofs_follow_symlinks = gtk_check_button_new_with_label(_("Follow syml"));
	gtk_widget_show(mkisofs_follow_symlinks);
	gtk_signal_connect(GTK_OBJECT(mkisofs_follow_symlinks), "clicked",
			GTK_SIGNAL_FUNC(toggle_fsymlink), 0);
	gtk_tooltips_set_tip(tooltips, mkisofs_follow_symlinks,
			_("Follow symbolic links when creating the image"), NULL);
	gtk_box_pack_start(GTK_BOX(vbox), mkisofs_follow_symlinks, FALSE, FALSE, 0);

	gtk_widget_set_sensitive(mkisofs_follow_symlinks, TRUE);
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(mkisofs_follow_symlinks), TRUE);
	gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(mkisofs_follow_symlinks), TRUE);

	mkisofs_include_dirname = gtk_check_button_new_with_label(_("Include dir name"));
	gtk_widget_show(mkisofs_include_dirname);
	gtk_signal_connect(GTK_OBJECT(mkisofs_include_dirname), "clicked",
			GTK_SIGNAL_FUNC(toggle_dirname), 0);
	gtk_tooltips_set_tip(tooltips, mkisofs_include_dirname,
			_("Include the directory names on the top of the image "
				"(if this option is disabled the _contents_ of the "
				"directories added will appear at the root). NOTE: this "
				"option currently has no effect when making sessions in "
				"multi session mode."), NULL);
	gtk_box_pack_start(GTK_BOX(vbox), mkisofs_include_dirname, FALSE, FALSE, 0);

	gtk_widget_set_sensitive(mkisofs_include_dirname, TRUE);
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(mkisofs_include_dirname), TRUE);
	gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(mkisofs_include_dirname), TRUE);

	mkisofs_include_all = gtk_check_button_new_with_label(_("Include all"));
	gtk_widget_show(mkisofs_include_all);
	gtk_signal_connect(GTK_OBJECT(mkisofs_include_all), "clicked",
			GTK_SIGNAL_FUNC(toggle_incall), 0);
	gtk_tooltips_set_tip(tooltips, mkisofs_include_all, _("Include "
			"all files on the ISO 9660 file system; if turned off, filenames "
			"containing ~ or # or ending with .bak are excluded"), NULL);
	gtk_box_pack_start(GTK_BOX(vbox), mkisofs_include_all, FALSE, FALSE, 0);

	gtk_widget_set_sensitive(mkisofs_include_all, TRUE);
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON (mkisofs_include_all), TRUE);
	gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON (mkisofs_include_all), TRUE);

	optimize_attach(vbox);

	label = gtk_label_new(_("Default pathlevel:"));
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	spinb_adj = gtk_adjustment_new(1, 0, 1000, 1, 10, 10);
	path_default_level_sb = gtk_spin_button_new(GTK_ADJUSTMENT(spinb_adj), 1, 0);
	gtk_box_pack_start(GTK_BOX(hbox), path_default_level_sb, FALSE, FALSE, 0);
	gtk_tooltips_set_tip(tooltips, path_default_level_sb, _("Number of pathcomponents to "
				"ignore/include by default (depends on radiobutton setting bellow). "
				"Note that this option takes effect only at the time you add "
				"files/directories (e.g., add a file, change it to 2 and add another "
				"file and you will see the difference)."),
			NULL);

	label = gtk_label_new(_(" levels"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);


	gtk_widget_show_all(hbox);

	radiob = gtk_radio_button_new_with_label(rgroup, _("Ignore from start"));
	rgroup = gtk_radio_button_group(GTK_RADIO_BUTTON(radiob));
	gtk_box_pack_start(GTK_BOX(vbox), radiob, FALSE, FALSE, 0);
	gtk_tooltips_set_tip(tooltips, radiob, _("Strip N leading pathcomponents "
				"from the paths added"), NULL);
	gtk_widget_show_all(radiob);

	radiob = gtk_radio_button_new_with_label(rgroup, _("Include from end"));
	rgroup = gtk_radio_button_group(GTK_RADIO_BUTTON(radiob));
	gtk_box_pack_start(GTK_BOX(vbox), radiob, FALSE, FALSE, 0);
	gtk_tooltips_set_tip(tooltips, radiob, _("Include N pathcomponents from "
				"the paths added (1 is the default behaviour, 0 "
				"means include all)"), NULL);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radiob), TRUE);
	gtk_widget_show_all(radiob);
	path_default_from_end = radiob;

	return vbox;
}

static void add_pattern_clicked(GtkWidget *widget, gpointer data) {
	gchar *text;

	text = gtk_entry_get_text(data);
	if (strlen(text) == 0)
		return;
	exclude_pattern_list = g_list_append(exclude_pattern_list, g_strdup(text));
	gtk_combo_set_popdown_strings(GTK_COMBO(exclude_pattern_combo), exclude_pattern_list);
	gtk_entry_set_text(data, "");
	mark_size_inaccurate();
}

/* TODO: this could be merged with the similar routine in extra_options.c */
static void del_pattern_clicked(GtkWidget *widget, gpointer data) {
	gchar *text;
	int i, l;
	GList *tmp;

	text = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(exclude_pattern_combo)->entry));
	if ((strlen(text) == 0) || (exclude_pattern_list == NULL))
		return;

	l = g_list_length(exclude_pattern_list);
	tmp = exclude_pattern_list;
	for(i = 0; i < l; i++) {
		if (strcmp(tmp->data, text) == 0) {
			exclude_pattern_list = g_list_remove(exclude_pattern_list, tmp->data);
			g_free(tmp->data);
			l--;
			break;
		}
		tmp = tmp->next;
	}

	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(exclude_pattern_combo)->entry), "");

	if (exclude_pattern_list == NULL) {
		exclude_pattern_list = g_list_append(exclude_pattern_list, g_strdup(""));
		gtk_combo_set_popdown_strings(GTK_COMBO(exclude_pattern_combo), exclude_pattern_list);
		g_free(exclude_pattern_list->data);
		exclude_pattern_list = NULL;
	} else
		gtk_combo_set_popdown_strings(GTK_COMBO(exclude_pattern_combo), exclude_pattern_list);
	mark_size_inaccurate();
}

static GtkWidget *create_exclude_pattern(void) {
	GtkWidget *hbox, *add_pattern, *pattern_to_be_added, *del_pattern;

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_widget_show(hbox);

	exclude_pattern_combo = gtk_combo_new();

	gtk_widget_show(exclude_pattern_combo);
	gtk_box_pack_start(GTK_BOX(hbox), exclude_pattern_combo, FALSE, FALSE, 0);

	pattern_to_be_added = gtk_entry_new();
	add_pattern = gtk_button_new_with_label(_("Add pattern:"));
	gtk_widget_show(add_pattern);
	gtk_box_pack_start(GTK_BOX(hbox), add_pattern, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(add_pattern), "clicked",
			(GtkSignalFunc) add_pattern_clicked, pattern_to_be_added);
	gtk_tooltips_set_tip(tooltips, add_pattern,
			_("Add a pattern to be excluded (*.o)") , NULL);

	gtk_widget_show(pattern_to_be_added);
	gtk_box_pack_start(GTK_BOX(hbox), pattern_to_be_added, FALSE, FALSE, 0);

	del_pattern = gtk_button_new_with_label(_("Delete pattern"));
	gtk_widget_show(del_pattern);
	gtk_box_pack_start(GTK_BOX(hbox), del_pattern, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(del_pattern),
			"clicked", (GtkSignalFunc) del_pattern_clicked, del_pattern);

	return hbox;
}

static void user_renamed_file(GtkWidget *widget, gpointer *data) {
	gchar *newpath;
	file_data *selected_row_info;

	if (-1 == last_row_selected)
		return;
	selected_row_info = gtk_clist_get_row_data(GTK_CLIST(clist), last_row_selected);
	g_assert(NULL != selected_row_info);

	newpath = gtk_entry_get_text(GTK_ENTRY(user_mod_path));
	g_free(selected_row_info->userpath);
	selected_row_info->userpath = g_strdup(newpath);

	gtk_clist_set_text(GTK_CLIST(clist), last_row_selected, 0, newpath);
}

/* DnD support is pretty much copied from ghex (c) by Jaka Mocnik */
static void dropped_data(GtkWidget *widget, GdkDragContext *context, gint x, gint y,
		GtkSelectionData *selection_data, guint info, guint time, gpointer data) {
	if (0 == info) {
		/* Gnome (gmc) doesn't implement xdnd correctly, it doesn't utf-8
		 * encode the uris. KDE (konqueror) is a little less buggy, so it
		 * UTF8 encodes the strings. None of the implementations care to
		 * set the non-optional hostname.
		 *
		 * Here we try to guess the source based on the GdkDragAction type
		 * (fortunately, for some unlogical reason they are different for
		 * gmc/konqueror), so that the uri strings can be decoded
		 * correctly.
		 *
		 * FIXME: The hostname isn't removed as it should be. It doesn't
		 * seem to be wildy implemented (rox-filer implements it). If
		 * you're reading this, feel free to fix it. I've spent enough time
		 * fixing DnD, and I don't even use it.
		 *
		 * Oh, xdnd is well specsed at
		 * http://www.newplanetsoftware.com/xdnd/, but what would life be
		 * if all programmers chose to follow specs (or, indeed, chose to
		 * break the specs in the same friggin way?)?
		 */

		GList *list, *names;

		switch (context->action) {
			case GDK_ACTION_COPY:
				list = names = gnome_uri_list_extract_filenames((gchar *)selection_data->data, 0);
				break;
			case GDK_ACTION_MOVE:
				list = names = gnome_uri_list_extract_filenames((gchar *)selection_data->data, 1);
				break;
			default:
				/* Does GdkDragContext really need this many slightly different and completely
				 * undocumented GdkDragActions?!? */
				g_warning("%s::%i: Unrecognized GdkDragAction (please report to author): %i %i %i",
						__FILE__, __LINE__, context->actions, context->suggested_action,
						context->action);
				return;
		}

		cursor_wait();
		mainwindow_disable_tabs();
		while (names) {
			add_path_to_files((gchar *)names->data);
			names = names->next;
			while (gtk_events_pending()) gtk_main_iteration_do(0);
		}
		mainwindow_enable_tabs();
		gnome_uri_list_free_strings(list);
		cursor_reset();
	} else {
		g_message("%s %i: unknown type dropped, should not happen!",
				__FILE__, __LINE__);
	}
}

void create_filepicker(GtkWidget *tab, config_cdr_data *cdr_info) {
	GtkWidget *vbox, *toolbar, *hbox, *right_box, *widget, *fill, *swin, *rename;
	GtkTargetEntry drop_types[] = {
		{ "text/uri-list", 0, 0}
	};
	int n_drop_types = sizeof (drop_types) / sizeof(drop_types[0]);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(vbox);
	gtk_container_add(GTK_CONTAINER(tab), vbox);

	toolbar = create_toolbar();
	gtk_box_pack_start(GTK_BOX(vbox), toolbar, FALSE, TRUE, 0);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_widget_show(hbox);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);

	create_filelist();
	swin = gtk_scrolled_window_new(NULL, NULL);
	gtk_widget_show(swin);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(swin),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

	gtk_container_add(GTK_CONTAINER(swin), clist);
	gtk_box_pack_start(GTK_BOX(hbox), swin, TRUE, TRUE, 0);

	gtk_drag_dest_set(GTK_WIDGET(swin),
			GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_HIGHLIGHT | GTK_DEST_DEFAULT_DROP,
			drop_types, n_drop_types, GDK_ACTION_COPY | GDK_ACTION_MOVE);
	gtk_signal_connect(GTK_OBJECT(swin), "drag_data_received",
			GTK_SIGNAL_FUNC(dropped_data), NULL);

	right_box = create_button_area(cdr_info);
	gtk_box_pack_start(GTK_BOX(hbox), right_box, FALSE, TRUE, 0);

/* gtk_label_set_justify doesn't seem to work for some odd reason,
 * and this hack needs whole_path to be non-empty to begin with to stop
 * another gtk bug from showing...
 */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_widget_show(hbox);

	whole_path = gtk_label_new(" ");
	gtk_label_set_justify(GTK_LABEL(whole_path), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(hbox), whole_path, FALSE, FALSE, 0);
	gtk_widget_show(whole_path);

	fill = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(fill), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(hbox), fill, TRUE, FALSE, 0);
	gtk_widget_show(fill);

	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_widget_show(hbox);

	user_mod_path = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(hbox), user_mod_path, TRUE, TRUE, 0);
	gtk_widget_show(user_mod_path);

	rename = gtk_button_new_with_label(_("Rename"));
	gtk_box_pack_start(GTK_BOX(hbox), rename, FALSE, FALSE, 0);
	gtk_widget_show(rename);

	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	gtk_signal_connect(GTK_OBJECT(rename),
			"clicked", (GtkSignalFunc) user_renamed_file, NULL);

	widget = create_exclude_pattern();
	gtk_box_pack_start(GTK_BOX(vbox), widget, FALSE, FALSE, 0);

	gtk_widget_set_sensitive(total_size, 1);
}
