#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/ioctl.h>
#include <linux/slab.h>
#include <linux/vmalloc.h>
#include <asm/uaccess.h>

#include "mpdev.h"

#define MP_BLOCK_SIZE 1024    
#define MP_SECTOR_SIZE 512    
#define MP_HEADS   16
#define MP_SECTORS 32
#define MP_CYLINDERS 4
#define MP_DISK_SIZE ((MP_HEADS*MP_SECTORS*MP_CYLINDERS)*MP_SECTOR_SIZE)
#define MP_MAX_NUM_DISKS 3     

/* Defines that we need to do in advance of including blk.h */                
#define MAJOR_NR            MPDEV_MAJOR
#define DEVICE_REQUEST      mpdev_do_request
#define DEVICE_NAME         "mpa"        

#define MP_MINOR_SHIFT_BITS 3 
#define MAX_PARTITIONS 8        /* partition 0 + 7 more possible due to 3 bit partition number field */
#define TOTAL_MP_PARTITIONS  ((MP_MAX_NUM_DISKS<<MP_MINOR_SHIFT_BITS)+MAX_PARTITIONS)
#define DEVICE_NR(device)   (MINOR(device)>>MP_MINOR_SHIFT_BITS)
#define DEVICE_PT(device)   (MINOR(device)&MAX_PARTITIONS)

#define DEVICE_NO_RANDOM    
#define DEVICE_OFF(device)
#define DEVICE_ON(device)

#include <linux/blk.h>
#include <linux/blkdev.h>
#include <linux/hdreg.h>
#include <linux/kdev_t.h>
#include <linux/genhd.h>     /* needed for supporting partitions */


/* a few typdefs just for setting up a blank msdos partition table */
typedef struct _Partition_Record {
	char record[16];
} Partition_Record;

typedef struct _Master_Boot_Record {
	char                Reserved[446];
	Partition_Record    Partition_Table[4];
	u_int16_t           Signature;
} Master_Boot_Record;

#define MSDOS_DISKMAGIC    0xAA55

void *mp_disk_addr = NULL;

int device_status[MP_MAX_NUM_DISKS];

/*    
   information about a particular ram disk.
  */
typedef struct Ramdisk {
	int       device_major;			 /* block device major number            */
	int       device_unit;			 /* block device unit number             */
	int       sectors;			 /* number of sectors on device          */
	char     *device_name;			 /* device name ... /dev/pdv             */
	void     *DiskStartAddrs;		 /* starting memory address of ramdisk   */
	void     *DiskEndAddrs;			 /* ending     "       "    "    "       */
} RAMDISK;

static RAMDISK *pRD[MP_MAX_NUM_DISKS];

static int     mpdev_open( struct inode *inode, struct file *pfile);
static int     mpdev_release( struct inode *inode, struct file *pfile);
static int     mpdev_ioctl ( struct inode *pinode, struct file *pfile, unsigned int cmd, unsigned long arg );
static void    mpdev_do_request(request_queue_t *rq);
static int     mpdev_revalidate(kdev_t rdev);
static void *  GetDisk(int size, int unit_number);
static void    FreeDisk(int unit_number);
static int     CheckDisk(int unit_number, int minor_number);
static void    SetupBlockDeviceParmTable(void);
static void    RemoveBlockDeviceParmTable(void);
static int     BuildDiskPartitionStructs(struct gendisk *gd);
static int         mpdev_init (void);
static void        mpdev_cleanup (void);

static struct block_device_operations blkops = {
	open:       mpdev_open,
	release:    mpdev_release,
	ioctl:      mpdev_ioctl,
};

static int     Major = MAJOR_NR;
static void   *DiskStartAddrs[MP_MAX_NUM_DISKS];          
static void   *DiskEndAddrs[MP_MAX_NUM_DISKS];
static int     ReadOnlyFlags[MP_MAX_NUM_DISKS];

/* Module parms used to setup the block device driver */                                             
static int     Disk_Size           = MP_DISK_SIZE;
static int     Sector_Size         = MP_SECTOR_SIZE;
static int     Readahead_Size      = MP_SECTOR_SIZE * 8;
static int     Disk_Size_Array[(MP_MAX_NUM_DISKS<<MP_MINOR_SHIFT_BITS)+MAX_PARTITIONS];
static int     Block_Size_Array[(MP_MAX_NUM_DISKS<<MP_MINOR_SHIFT_BITS)+MAX_PARTITIONS];
static int     Sector_Size_Array[(MP_MAX_NUM_DISKS<<MP_MINOR_SHIFT_BITS)+MAX_PARTITIONS];
static int     Readahead_Size_Array[(MP_MAX_NUM_DISKS<<MP_MINOR_SHIFT_BITS)+MAX_PARTITIONS];


/*
   This array of structs describe each possible partition on 
   all physical devices because it is indexed by device Minor
   which is the unit + partition number.
*/
static struct hd_struct hd[MP_MAX_NUM_DISKS<<MP_MINOR_SHIFT_BITS];

/*
   Gendisk struct is the Linux Generic HD support structure, 
   used by block device drivers that support partitions.  The
   structure describes all physical devices managed by the block
   device driver. You need just 1 of these to describe all your
   physical devices you will support.   
*/
static char genhd_flags = 0;	//wonder if I should try removeable media ... GENHD_FL_REMOVABLE;
static struct gendisk gd = {
	MAJOR_NR,		/* device major number */
	DEVICE_NAME,		/* device name /dev/pdv  */
	MP_MINOR_SHIFT_BITS,	/* bits to shift to get real from Minor */
	MAX_PARTITIONS,		/* max number of partitions per real  */
	&hd[0],			/* array of hd_structs */
	&Disk_Size_Array[0],	/* block sizes */
	MP_MAX_NUM_DISKS,	/* number of real devices */
	NULL,			/* internal use only */
	NULL,			/* next ptr */
	&blkops,		/* blk dev opts ptr */
	NULL,			/* devfs handle */
	&genhd_flags		/* flags */
};


MODULE_SUPPORTED_DEVICE( DEVICE_NAME );
MODULE_PARM(Major, "i");
MODULE_PARM(Disk_Size, "i");


/*======================================================*/
/*                 BLOCK DEVICE ROUTINES               =*/
/*======================================================*/

static int mpdev_open (struct inode *pinode, struct file *pfile)
{
	int unit_number = DEVICE_NR(pinode->i_rdev);
	int minor_number = 0;
	int err;

	printk(KERN_INFO "mpdev: open called, i_rdev= 0x%04X\n", MINOR(pinode->i_rdev));

	/*
	   Our checkdisk routine will look to see if we own the 
	   minor and have successfully allocated memory for it.        
	   
	   If we get an error that says the device memory isn't 
	   allocated yet then we'll try and allocate the ram disk
	   because we dont allocate ram disks untill we see 
	   someone trying to use one!! 
	*/
	err=CheckDisk(unit_number, minor_number);
	if (err) {
		if (err==-ENOMEM) { /* try and allocate memory for this ram disk */
			if ( GetDisk(Disk_Size, unit_number) == NULL) return -ENOMEM;
			else printk(KERN_INFO "mpdev: open exit rc= ENOMEM\n");
		} else { /* must be bad device number */
			printk(KERN_INFO "mpdev: open exit rc= bad minor field in device number\n");
			return err;
		}
	}

	printk(KERN_INFO "mpdev open check disk says Ok\n");

	/* only look at the f_mode field in the file struct! */
	if (pfile != NULL) ReadOnlyFlags[unit_number] = pfile->f_mode;

	printk(KERN_INFO "mpdev: open exit rc=0\n");

	MOD_INC_USE_COUNT;
	return 0;
}


static int mpdev_release (struct inode *pinode, struct file *pfile)
{
	MOD_DEC_USE_COUNT;
	printk(KERN_INFO "mpdev: release called\n");
	return 0;
}


static int mpdev_ioctl ( struct inode *pinode, struct file *pfile, unsigned int cmd, unsigned long arg )
{
	int unit_number;
	int part_number;
	int minor;
	int size;
	int err;
	mpdev_cmd_t     mpdev_cmd;

	if (!pinode || !pinode->i_rdev)	 return -EINVAL;

	/* get the unit and partition numbers */
	unit_number = MINOR(pinode->i_rdev) >> MP_MINOR_SHIFT_BITS; 
	part_number = MINOR(pinode->i_rdev) & MP_MINOR_SHIFT_BITS;
	minor       = MINOR(pinode->i_rdev) & MP_MINOR_SHIFT_BITS;

	printk(KERN_INFO "mpdev: IOCTL  unit= %d partition= %d\n", unit_number, part_number);

	/* Validate that the device Minor is Ok */
	err = CheckDisk(unit_number, minor);
	if (err) {
		printk(KERN_INFO " ... checkdisk error\n");
		return err;
	}

	switch (cmd) {
	case BLKGETSIZE:  /* return device size ... in sectors */
		{
			size = MP_DISK_SIZE; //gd.part[minor].nr_sects;
			printk( KERN_INFO "BLKGETSIZE size= %d\n", size); //gd.part[minor].nr_sects);
			if (!arg) return -EINVAL;
			err = verify_area(VERIFY_WRITE, (long *)arg, sizeof(long));
			if (err) return err;
			put_user(size, (long *) arg); 
			return 0;
		}
	case BLKGETSIZE64:
		{
			uint64_t size =  MP_DISK_SIZE; //Disk_Size_Array[unit_number]*2; //gd.part[minor].nr_sects;
			printk( KERN_INFO "BLKGETSIZE64 size= %lld\n", size);           
			if (!arg) return -EINVAL;
			err = verify_area(VERIFY_WRITE, (long *)arg, sizeof(uint64_t));
			if (err) return err;

			put_user(size, (uint64_t *) arg); 
			return 0;
		}

	case BLKFLSBUF:	  /* flush */
		printk( KERN_INFO "BLKFLSBUF\n");
		return 0 ;

	case BLKRAGET:	  /* return the readahead value */
		printk( KERN_INFO "BLKRAGET readahead= %d\n", read_ahead[Major]);
		if (!arg) return -EINVAL;
		err = verify_area(VERIFY_WRITE, (long *)arg, sizeof(long));
		if (err) return err;
		put_user(read_ahead[Major], (long *) arg);
		return 0;

	case BLKRASET:	  /* set the readahead value */
		printk( KERN_INFO "BLKRASET size= %ld\n", arg );
		if (!suser()) return -EACCES;	/* check privilage */
		if (arg > 0xff)	return -EINVAL;	 /* keep readahead low */
		read_ahead[Major] = arg;
		return 0;

	case BLKROSET:	  /* set the read-only flags */
		printk(KERN_INFO "BLKROSET flag= %ld\n", arg );
		if (!suser()) return -EACCES;	/* check privilage */
		if ((arg != FMODE_READ) && 
		    (arg != FMODE_WRITE) &&
		    (arg != FMODE_READ+FMODE_WRITE)) return -EINVAL; /* validate mode */
		ReadOnlyFlags[unit_number] = arg; 
		return 0;

	case BLKROGET:	  /* get the read-only flags */
		printk( KERN_INFO "BLKROGET \n");
		if (!arg) return -EINVAL;
		err = verify_area(VERIFY_WRITE, (long *)arg, sizeof(long));
		if (err) return err;
		put_user(ReadOnlyFlags[unit_number], (long *)arg);  
		return 0;       


	case BLKSECTGET:  /* get max sectors that can be read per request */
		printk(KERN_INFO "BLKSECTGET max sectors= %d\n", Readahead_Size/Sector_Size );
		if (!arg) return -EINVAL;
		err = verify_area(VERIFY_WRITE, (long *)arg, sizeof(long));
		if (err) return err;
		put_user((Readahead_Size/Sector_Size), (long *) arg);
		return 0;

	case HDIO_GETGEO: /* return the disk geometry */
		{
			struct hd_geometry *loc = (struct hd_geometry *) arg;

			unsigned short bios_cyl  = (unsigned short) MP_CYLINDERS;
			unsigned char  bios_head = (unsigned char) MP_HEADS;
			unsigned char  bios_sect = (unsigned char) MP_SECTORS;

			printk(KERN_INFO "HDIO_GETGEO cmd\n");

			if (!loc) return -EINVAL;

			if (put_user(bios_head, (unsigned char *) &loc->heads))	return -EFAULT;
			if (put_user(bios_sect, (unsigned char *) &loc->sectors)) return -EFAULT;
			if (put_user(bios_cyl, (unsigned short *) &loc->cylinders)) return -EFAULT;

			if (put_user(hd[minor].start_sect,(unsigned long *) &loc->start)) return -EFAULT;

			return 0;
		}



	case HDIO_GETGEO_BIG:
	case HDIO_GETGEO_BIG_RAW:
		{               
			struct hd_big_geometry *loc = (struct hd_big_geometry *) arg;

			unsigned int   bios_cyl  = (unsigned int) MP_CYLINDERS;
			unsigned char  bios_head = (unsigned char) MP_HEADS;
			unsigned char  bios_sect = (unsigned char) MP_SECTORS;

			printk(KERN_INFO "HDIO_GETGEO_BIG cmd\n");

			if (!loc) return -EINVAL;

			if (put_user(bios_head, (unsigned char *) &loc->heads))	return -EFAULT;
			if (put_user(bios_sect, (unsigned char *) &loc->sectors)) return -EFAULT;
			if (put_user(bios_cyl, (unsigned int *) &loc->cylinders)) return -EFAULT;

			if (put_user(hd[minor].start_sect, (unsigned long *) &loc->start)) return -EFAULT;

			return 0;
		}


	case BLKRRPART:	  /* re-read the partition table */
		printk( KERN_INFO " BLKRRPART cmd\n");        
		err = mpdev_revalidate(pinode->i_rdev);
		printk(KERN_INFO "rc = %d\n", err);
		return err;

	case BLKSECTSET:  /* set max sectors that can be read per request */
		printk(KERN_INFO " ERROR ... BLKSECTSET not supported\n");
		return -EINVAL;

	case BLKSSZGET:
		printk(KERN_INFO "BLKSSZGET cmd\n");        
		err = verify_area(VERIFY_WRITE, (long *)arg, sizeof(long));
		if (err) return err;
		if (put_user( 512, (long *) arg)) return -EFAULT;
		return 0;

	case BLKBSZGET:
		printk(KERN_INFO "BLKBSZGET cmd\n");    
		err = verify_area(VERIFY_WRITE, (long *)arg, sizeof(long));
		if (err) return -EFAULT;
		if (put_user( 1024, (long *) arg)) return -EFAULT;
		return 0;

	case MPDEV_CMD:  

		printk(KERN_INFO "mpdev: ioctl from mp\n");

		if ( copy_from_user(&mpdev_cmd, ( mpdev_cmd_t *)arg, sizeof(mpdev_cmd_t)) ) return -EFAULT;

		if (unit_number >= 0  && unit_number <= MP_MAX_NUM_DISKS) {
			device_status[unit_number] = mpdev_cmd.cmd;
			if (mpdev_cmd.cmd==1) {
				printk(KERN_INFO "mpdev: failing unit device %d\n", unit_number);
			} else {
				printk(KERN_INFO "mpdev: recovering unit device %d\n", unit_number);
			}
			err = 0;
		} else {
			err = EINVAL;
		}                    

		printk(KERN_INFO "mpdev: rc = %d\n", err );
		mpdev_cmd.status = err;
		copy_to_user((mpdev_cmd_t *)arg, &mpdev_cmd, sizeof(mpdev_cmd));
		return 0;


	default: 
		printk(KERN_INFO " ERROR ... default ... cmd= %d or 0x%x\n", cmd, cmd); 
		return -EINVAL;
	}
	return -EINVAL;
}


static int request_should_fail( int dev )
{
	if (dev >= 0 && dev < MP_MAX_NUM_DISKS) {
		return device_status[dev];
	}
	return 0;
}


static void mpdev_do_request(request_queue_t *rq)
{
	void *pDisk=NULL;    // *pEndOfDisk;
	void *vPtr ;
	int size;
	int unit;
	int minor;
	int err;
	u_int64_t lsn;
	u_int64_t lba;
	u_int64_t nr_sectors;

	while (1) {
		INIT_REQUEST;  /* verify request ... return to kernel if Q is empty */

		unit     = MINOR( CURRENT->rq_dev ) >> MP_MINOR_SHIFT_BITS;
		minor    = MINOR( CURRENT->rq_dev ) & MP_MINOR_SHIFT_BITS;
		lsn      = CURRENT->sector;
		nr_sectors  = CURRENT->current_nr_sectors;
/*
       if (CURRENT->cmd == READ)
	   printk(KERN_INFO "mpdev: read  request, Unit= 0x%04X  Minor= 0x%04X \n", unit, minor);
       else
	   printk(KERN_INFO "mpdev: write request, Unit= 0x%04X  Minor= 0x%04X \n", unit, minor);
*/
		err = CheckDisk(unit,minor);
		if (err) {
			printk(KERN_INFO "mpdev: CheckDisk error = %d\n", err);
			end_request(0);	 /* dequeue request and print error msg to log */
			continue;
		}

		lba = gd.part[minor].start_sect + lsn;

		/*   Get pointer to ramdisk memory */
		pDisk      = pRD[unit]->DiskStartAddrs;

		/* calculate sector address for start of I/O and the number of bytes to move */
		vPtr  = (void *) (pDisk + Sector_Size*(hd[minor].start_sect + lsn));
		size = nr_sectors * Sector_Size;

		/* make sure we dont run off the end of the Disk */
		if ( lsn + nr_sectors > gd.part[minor].nr_sects) {
 			end_request(0);	 /* dequeue request and print error msg to log */
			continue;
		}

		if ( request_should_fail( unit ) ) {
			if (CURRENT->cmd==READ)
				printk(KERN_INFO "mpdev: read(%d) lsn= %lld  count= %lld (FAILURE)\n", minor,lsn,nr_sectors);
			else
				printk(KERN_INFO "mpdev: write(%d) lsn= %lld  count= %lld (FAILURE)\n", minor,lsn,nr_sectors);
			end_request(0);
			continue;
		}

		/* execute the command */
		switch (CURRENT->cmd) {
		case READ:
			memcpy(CURRENT->buffer, vPtr, size);
			break;

		case WRITE:
			memcpy(vPtr, CURRENT->buffer, size);
			break;

		default:
			printk(KERN_INFO "mpdev: io request failed cuz we dont support cmd = %d\n", CURRENT->cmd );
			end_request(0);	/* dequeue request and print error msg to log */
			continue;
		}

		if (CURRENT->cmd==READ)
			printk(KERN_INFO "mpdev: read(%d) lsn= %lld  count= %lld (SUCCESS)\n", minor,lsn,nr_sectors);
		else
			printk(KERN_INFO "mpdev: write(%d) lsn= %lld  count= %lld (SUCCESS)\n", minor,lsn,nr_sectors);

		end_request(1);	 /* dequeue request */
	}
}


static void   DisplayHD()
{
	int i;

	for (i=0; i< (MP_MAX_NUM_DISKS<<MP_MINOR_SHIFT_BITS); i++) {
		printk(KERN_INFO "\nHD[%d].nr_sects       = %ld\n",i, hd[i].nr_sects );
		printk(KERN_INFO "HD[%d].start sector   = %ld\n",i, hd[i].start_sect );
		printk(KERN_INFO "Disk_Size_Array[%d]   = %d\n",i, Disk_Size_Array[i]);
		printk(KERN_INFO "Block_Size_Array[%d]  = %d\n",i, Block_Size_Array[i]);
		printk(KERN_INFO "Sector_Size_Array[%d] = %d\n",i, Sector_Size_Array[i]);
	}


}

static int     mpdev_revalidate(kdev_t rdev)
{
	/* get Minor number for 1st partition and then get the total number of partitions */
	int part1 = (DEVICE_NR(rdev)<<MP_MINOR_SHIFT_BITS)+1;
	int partn = (1<<MP_MINOR_SHIFT_BITS)-1;

	/* clear out old partition information ... dont clear out partition 0 ! */
	memset(gd.sizes+part1, 0, partn*sizeof(int));
	memset(gd.part+part1,  0, partn*sizeof(struct hd_struct));

	printk(KERN_INFO "Going to grok ... \n");
	DisplayHD();

	/* tell kernel it can fill in new partition info now  */
	grok_partitions( &gd, DEVICE_NR(rdev), partn, MP_DISK_SIZE/MP_SECTOR_SIZE );

	printk(KERN_INFO "After grok    ... \n");
	DisplayHD();

	return 0;
}


/*======================================================*/
/*                 MISC SUPPORT ROUTINES               =*/
/*======================================================*/


/* called to allocate a ram disk for the specified Minor */
static void *  GetDisk(int size, int unit_number )
{
	char *endaddr;

	if (mp_disk_addr == NULL) {
		mp_disk_addr = vmalloc(size);
		if (mp_disk_addr != NULL) {
			memset(mp_disk_addr, 0, size);  
			//((Master_Boot_Record *)mp_disk_addr)->Signature = MSDOS_DISKMAGIC;
		}
	}

	printk(KERN_INFO "mpdev: getdisk called, size= %d bytes  unit number= %d\n", size, unit_number );

	if (mp_disk_addr != NULL) {
		endaddr = (char *)mp_disk_addr;
		pRD[unit_number]->DiskStartAddrs = mp_disk_addr;
		pRD[unit_number]->DiskEndAddrs   = (void *) (endaddr+size);
	}

	printk(KERN_INFO "mpdev: getdisk returning addr= %p\n", mp_disk_addr);
	return mp_disk_addr;
}

/* called to free memory that was allocated for the specified ram disk */
static void    FreeDisk(int unit_number)
{
	int i;
	printk(KERN_INFO "mpdev: -----> FREEDISK CALLED <----- unit number= %d\n", unit_number);

	if ((unit_number >= 0) && (unit_number < MP_MAX_NUM_DISKS)) {
		if (mp_disk_addr != NULL) {
			vfree(mp_disk_addr);	     /* free ramdisk memory */
			mp_disk_addr = NULL;
			for (i=0;i<MP_MAX_NUM_DISKS; i++) {
				pRD[i] = NULL;
			}

		}
	}

}


static int     CheckDisk(int unit_number, int minor_number)
{
	if ( unit_number<0 || unit_number>MP_MAX_NUM_DISKS )	    return -EIO;
	if ( minor_number<0 || minor_number > MAX_PARTITIONS ) return -EIO;

	if ( pRD[unit_number] == NULL )				    return -ENOMEM;
	if ( pRD[unit_number]->DiskStartAddrs == NULL )	    return -ENOMEM;
	if ( gd.part[minor_number].nr_sects == 0)		   return -EIO;

	return 0;
}


static void    SetupBlockDeviceParmTable()
{
	int i,j, minor;


	/* zero fill data structures */
	memset(&Disk_Size_Array, 0, sizeof(int)*((MP_MAX_NUM_DISKS<<MP_MINOR_SHIFT_BITS)+MAX_PARTITIONS));
	memset(&Block_Size_Array, 0, sizeof(int)*((MP_MAX_NUM_DISKS<<MP_MINOR_SHIFT_BITS)+MAX_PARTITIONS));
	memset(&Sector_Size_Array, 0, sizeof(int)*((MP_MAX_NUM_DISKS<<MP_MINOR_SHIFT_BITS)+MAX_PARTITIONS));
	memset(&Readahead_Size_Array, 0, sizeof(int)*((MP_MAX_NUM_DISKS<<MP_MINOR_SHIFT_BITS)+MAX_PARTITIONS));

	/*
	   Now initialize for all our possible partitions. If you only init for partition 0, which
	   by convention is the entire drive, you won't be able to get any logical partitions defined
	   later when you call grok_partitions().
	*/   
	for (i=0; i<MP_MAX_NUM_DISKS; i++) {
		for (j=0;j<=MAX_PARTITIONS;j++) {
			minor = (i<<MP_MINOR_SHIFT_BITS)+j;

			Disk_Size_Array[minor]      = MP_DISK_SIZE/1024; /* kbytes */
			Block_Size_Array[minor]     = MP_BLOCK_SIZE;
			Sector_Size_Array[minor]    = MP_SECTOR_SIZE; 
			Readahead_Size_Array[minor] = Readahead_Size;
		}
	}

	/* point kernel to our data structs */
	blk_size[Major]         = &Disk_Size_Array[0];
	blksize_size[Major]     = &Block_Size_Array[0];
	hardsect_size[Major]    = &Sector_Size_Array[0];
	max_readahead[Major]    = &Readahead_Size_Array[0];


}

/* called to cleanup the kernels block device parm table for our device */
static void    RemoveBlockDeviceParmTable()
{
	/* this will tell kernel there is no device for our Major */
	blk_size[Major]         = NULL;
	blksize_size[Major]     = NULL;
	hardsect_size[Major]    = NULL;
	max_readahead[Major]    = NULL;
}


static int     BuildDiskPartitionStructs( struct gendisk *gd )
{
	int    i;

	/* parm validity checking */
	if (!gd)  return -EINVAL;

	/* clear the partition entries */
	for (i=0; i < MP_MAX_NUM_DISKS<<MP_MINOR_SHIFT_BITS; i++)
		memset( &hd[i], 0, sizeof(struct hd_struct));

	/* init partition 0 of each device so the kernel can do I/O to the drive */
	for ( i=0; i < MP_MAX_NUM_DISKS; i++) {
		hd[i<<MP_MINOR_SHIFT_BITS].nr_sects   = (MP_DISK_SIZE/MP_SECTOR_SIZE);
		hd[i<<MP_MINOR_SHIFT_BITS].start_sect = 0;
	}

	gd->part = &hd[0];

	return 0;
}


static int mpdev_init ()
{
	int rc, i,j;

	printk(KERN_INFO "mpdev: init called\n");


	for (i=0;i<MP_MAX_NUM_DISKS;i++) {
		device_status[i] = 0; // 
	}

	/* allocate the ramdisks */
	for (i=0;i<MP_MAX_NUM_DISKS;i++) {

		pRD[i] = kmalloc(sizeof(struct Ramdisk), GFP_KERNEL);
		if (!pRD[i]) {
			printk(KERN_INFO "mpdev: failed to malloc ramdisk structs\n");
			return -ENOMEM;
		} else {
			/* fill in the new ramdisk structure */
			memset(pRD[i], 0, sizeof(struct Ramdisk));
			pRD[i]->device_major   = Major;  
			pRD[i]->device_unit    = i<<MP_MINOR_SHIFT_BITS;
			pRD[i]->sectors        = (MP_DISK_SIZE/MP_SECTOR_SIZE);
		}
		DiskStartAddrs[i] = DiskEndAddrs[i] = NULL;
		ReadOnlyFlags[i]  = FMODE_READ + FMODE_WRITE; 

		if (GetDisk(Disk_Size,i) == NULL) {
			printk(KERN_INFO "mpdev: failed GetDisk call\n");
			for (j=0; j<=i; j++) {
				//kfree(pRD[j]);
			}
			return -ENOMEM;
		}
	}    

	/* register our device driver with kernel */
	rc = register_blkdev( Major, DEVICE_NAME, &blkops);
	if (rc) {
		printk(KERN_INFO "mpdev: register_blkdev failed with rc= %d\n", rc);
		for (i=0;i<MP_MAX_NUM_DISKS;i++) FreeDisk(i);
		return -EIO;
	}

	/* build our block device parameter arrays in the kernel */
	SetupBlockDeviceParmTable();

	/* 
	   Tell the kernel we wish to use the standard blk request 
	   handling. This means requests will be sorted and merged
	   for us. By this call we also tell the kernel which of 
	   our routines will handle requests when they are available
	   for processing on our queue.
	*/
	blk_init_queue(BLK_DEFAULT_QUEUE(MAJOR_NR), mpdev_do_request );

	printk(KERN_INFO "mpdev: about to register partitions\n"); 

	rc = BuildDiskPartitionStructs(&gd);
	if (rc) {
		printk(KERN_INFO "mpdev: failed to detect and register partitions\n");
		unregister_blkdev( Major, DEVICE_NAME );

		for (i=0; i<MP_MAX_NUM_DISKS; i++) {
			FreeDisk(i);
			if (pRD[i] != NULL) kfree(pRD[i]);
		}

		return rc;
	}

	add_gendisk(&gd);

	printk(KERN_INFO "mpdev: init completed Ok\n");
	return 0;
}



static void mpdev_cleanup ()
{
	int i;

	printk(KERN_INFO "mpdev: cleanup called\n");

	/* unregister our block device Major */
	unregister_blkdev( Major, DEVICE_NAME ); 

	/* remove our block device parameter arrays in the kernel */
	RemoveBlockDeviceParmTable(); 

	/* unlink our gendisk struct */
	del_gendisk(&gd);

	blk_cleanup_queue(BLK_DEFAULT_QUEUE(MAJOR_NR)); 

	/* free our ram disk memory */
	for (i=0; i<MP_MAX_NUM_DISKS; i++) {
		FreeDisk(i);
		if (pRD[i] != NULL) kfree(pRD[i]);
	}

}


module_init(mpdev_init);
module_exit(mpdev_cleanup);
#ifdef MODULE_LICENSE
MODULE_LICENSE("GPL");
#endif
