/*  $Header: /cvsroot/dvipdfmx/src/ttf.c,v 1.22 2004/03/17 11:04:47 hirata Exp $
    
    This is dvipdfmx, an eXtended version of dvipdfm by Mark A. Wicks.

    Copyright (C) 2002 by Jin-Hwan Cho and Shunsaku Hirata,
    the dvipdfmx project team <dvipdfmx@project.ktug.or.kr>
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
    
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
*/

/*
 * OpenType GSUB support is not available yet.
 *
 *  Required features: onum (oldstyle digits), c2sc/smcp, and more.
 *
 */

#if HAVE_CONFIG_H
#include "config.h"
#endif

#include <ctype.h>
#include "system.h"

#include "mem.h"
#include "error.h"
#include "mfileio.h"

#include "numbers.h"
#include "dpxutil.h"

#include "pdfobj.h"
#include "pdfresource.h"
#include "pdffont.h"

#include "encodings.h"
#include "unicode.h"
#include "agl.h"

/* TrueType */
#include "sfnt.h"
#include "tt_cmap.h"
#include "tt_table.h"
#include "tt_build.h"
#include "tt_aux.h"

#include "ttf.h"

#define TTFONT_DEBUG_STR "TTFont"
#define TTFONT_DEBUG     3

static void
validate_name (char **name, int len)
{
  char *tmp;
  int i, pos;

  if (len > PDF_NAME_LEN_MAX)
    ERROR("Name string length too large");

  /* eg, 0x5b -> #5b */
  tmp = NEW(3*len + 1, char);
  pos = 0;
  for (i = 0; i < len; i++) {
    if (*(*name+i) == 0)
      continue;
    else if (*(*name+i) < '!' || *(*name+i) > '~' ||
	     /*         ^ `space' is here */
	     strchr("/()[]<>{}#", *(*name+i)) != NULL) {
      sprintf(tmp+pos, "#%02x", (unsigned char) *(*name+i));
      pos += 3;
    } else {
      *(tmp+pos) = *(*name+i);
      pos++;
    }
  }

  if (pos != len)
    *name = RENEW(*name, pos+1, char);
  memmove(*name, tmp, pos);
  *(*name+pos) = '\0';

  RELEASE(tmp);

  if (strlen(*name) == 0)
    ERROR("No valid character found in name string.");

  return;
}

int
TTFont_match (pdf_font *font, const char *name,
	      int encoding_id, double font_scale, double design_size, int embedding)
{
  ASSERT(font);

  if (name && font->ident && !strcmp(font->ident, name) &&
      encoding_id == font->encoding_id &&
      embedding == font->embedding)
    return 1;

  return 0;
}

int
TTFont_open (pdf_font *font, const char *name,
	     int encoding_id, double font_scale, double design_size, int embedding)
{
  sfnt   *sfont;
  char   *fullname;
  char   *fontname;
  int     namelen;

  ASSERT(font);

  fullname = kpse_find_file(name, kpse_truetype_format, 0);
  if (!fullname)
    return -1;
  if ((sfont = sfnt_open(fullname)) == NULL ||
      sfont->type != SFNT_TYPE_TRUETYPE     ||
      sfnt_read_table_directory(sfont, 0) < 0)
    return -1;

  font->ident = NEW(strlen(name)+1, char);
  strcpy(font->ident, name);
  font->filename = NEW(strlen(fullname)+1, char);
  strcpy(font->filename, fullname);

  font->encoding_id = encoding_id;
  font->embedding   = embedding;

  font->descriptor  = tt_get_fontdesc(sfont, &(font->embedding), 1);
  if (!font->descriptor)
    ERROR("Could not obtain neccesary font info.");

  if (!font->embedding && font->encoding_id >= 0)
    ERROR("Custum encoding not allowed for non-embedded TrueType font.");

  fontname = NEW(3*PDF_NAME_LEN_MAX+1, char);
  memset(fontname, 0, PDF_NAME_LEN_MAX);
  namelen = tt_get_ps_fontname(sfont, fontname, PDF_NAME_LEN_MAX-7);
  if (namelen == 0) {
    namelen = MIN(strlen(name), PDF_NAME_LEN_MAX-7);
    strncpy(fontname, name, namelen);
    fontname[namelen] = '\0';
  }
  validate_name(&fontname, namelen);
  strcpy(font->fontname, fontname);

  if (font->embedding) {
    pdf_font_make_tag(font->uniqueID);
    sprintf(fontname, "%6s+%s", font->uniqueID, font->fontname);
  } else {
    strcpy(fontname, font->fontname);
  }

  sfnt_close(sfont);

  font->fontdict = pdf_new_dict();
  pdf_add_dict(font->fontdict,
	       pdf_new_name("Type"), pdf_new_name("Font"));
  pdf_add_dict(font->fontdict,
	       pdf_new_name("Subtype"), pdf_new_name("TrueType"));
  pdf_add_dict(font->fontdict,
	       pdf_new_name("BaseFont"), pdf_new_name(fontname));
  pdf_add_dict(font->descriptor,
	       pdf_new_name("FontName"), pdf_new_name(fontname));

  RELEASE(fontname);

  /*
   * We use MacRoman as "default" encoding.
   */
  if (encoding_id < 0)
    pdf_add_dict(font->fontdict,
		 pdf_new_name("Encoding"),
		 pdf_new_name("MacRomanEncoding"));
  else {
    Encoding *encoding = Encoding_cache_get(encoding_id);
    if (Encoding_is_predefined(encoding))
      pdf_add_dict(font->fontdict,
		   pdf_new_name("Encoding"),
		   pdf_new_name(Encoding_get_name(encoding)));
  }

  return 0;
}

/*
 * The 'name' table should be preserved since it contains copyright
 * information, but it might cause problem when there are invalid
 * table entries (wrongly encoded text which is often the case in
 * CJK fonts). Acrobat does not use 'name' table. Unicode TrueType
 * fonts may have 10K bytes 'name' table...
 *
 * We preserve the 'OS/2' table too, since it contains the license
 * information. PDF applications should use this table to decide
 * whether the font is embedded only for the purpose of preview &
 * printing. Otherwise, we must encrypt the document. Acrobat does
 * not use 'OS/2' table, though...
 */
static struct
{
  const char *name;
  int   must_exist;
} required_table[] = {
  {"OS/2", 1}, {"head", 1}, {"hhea", 1}, {"loca", 1}, {"maxp", 1},
  {"name", 1}, {"glyf", 1}, {"hmtx", 1}, {"fpgm", 0}, {"cvt ", 0},
  {"prep", 0}, {"cmap", 1}, {NULL, 0}
};

static void
do_widths (pdf_font *font, double *widths)
{
  int code, firstchar, lastchar;

  firstchar = 255; lastchar = 0;
  for (code = 0; code < 256; code++) {
    if (font->used_chars[code]) {
      if (code < firstchar) firstchar = code;
      if (code > lastchar)  lastchar  = code;
    }
  }
  {
    pdf_obj *tmp;

    tmp = pdf_new_array();
    for (code = firstchar; code <= lastchar; code++) {
      if (font->used_chars[code])
	pdf_add_array(tmp, pdf_new_number(ROUND(widths[code], 1)));
      else
	pdf_add_array(tmp, pdf_new_number(0.0));
    }
    pdf_add_dict(font->fontdict, pdf_new_name("Widths"), pdf_ref_obj(tmp));
    pdf_release_obj(tmp);
  }
  pdf_add_dict(font->fontdict,
	       pdf_new_name("FirstChar"), pdf_new_number(firstchar));
  pdf_add_dict(font->fontdict,
	       pdf_new_name("LastChar"),  pdf_new_number(lastchar));
}

static int verbose = 0;

#define PDFUNIT(v) ((double) (ROUND(1000.0*(v)/(glyphs->emsize), 1)))

/*
 * Default encoding "Mac-Roman" is used.
 *
 * BUG:
 *  This uses cmap format 0 (byte encoding table).
 *  It does not work with encodings that uses full 256 range since
 *  GID = 0 is reserved for .notdef. GID = 256 is not accessible.
 */
static void
do_builtin_encoding (pdf_font *font, sfnt *sfont)
{
  struct tt_glyphs *glyphs;
  char    *cmap_table;
  tt_cmap *ttcm;
  int      code, count;

  ttcm = tt_cmap_read(sfont, TT_MAC, TT_MAC_ROMAN);
  if (!ttcm)
    ERROR("Cannot read Mac-Roman TrueType cmap table");

  cmap_table = NEW(274, char);
  memset(cmap_table, 0, 274);
  sfnt_put_ushort(cmap_table,    0);            /* Version  */
  sfnt_put_ushort(cmap_table+2,  1);            /* Number of subtables */
  sfnt_put_ushort(cmap_table+4,  TT_MAC);       /* Platform ID */
  sfnt_put_ushort(cmap_table+6,  TT_MAC_ROMAN); /* Encoding ID */
  sfnt_put_ulong (cmap_table+8,  12);           /* Offset   */
  sfnt_put_ushort(cmap_table+12, 0);            /* Format   */
  sfnt_put_ushort(cmap_table+14, 262);          /* Length   */
  sfnt_put_ushort(cmap_table+16, 0);            /* Language */

  glyphs = tt_build_init();

  if (verbose > 2)
    MESG("[glyphs:/.notdef");

  count = 1; /* .notdef */
  for (code = 0; code < 256; code++) {
    unsigned short new_gid, gid = 0;
    if (!font->used_chars[code])
      continue;
    if (verbose > 2) MESG("/.c0x%02x", code);
    gid = tt_cmap_lookup(ttcm, (unsigned short) code);
    if (gid == 0) {
      WARN("Character 0x%02x missing in font \"%s\".", code, font->ident);
      new_gid = 0;
    } else {
      if ((new_gid = tt_find_glyph(glyphs, gid)) == 0) {
	new_gid = tt_add_glyph(glyphs, gid, count); /* count returned. */
      }
    }
    cmap_table[18+code] = new_gid;
    count++;
  }
  tt_cmap_release(ttcm);

  if (verbose > 2)
    MESG("]");

  if (tt_build_tables(sfont, glyphs) < 0)
    ERROR("Could not created FontFile stream.");

  {
    double widths[256];

    for (code = 0; code < 256; code++) {
      if (font->used_chars[code]) {
	USHORT idx;
	idx = tt_get_index(glyphs, (USHORT) cmap_table[18+code]);
	widths[code] = PDFUNIT(glyphs->gd[idx].advw);
      } else {
	widths[code] = 0.0;
      }
    }
    do_widths(font, widths);
  }

  if (verbose > 1) 
    MESG("[%d glyphs]", glyphs->num_glyphs);

  tt_build_finish(glyphs);

  sfnt_set_table(sfont, "cmap", cmap_table, 274);

}

static void
do_custum_encoding (pdf_font *font, sfnt *sfont)
{
  struct tt_glyphs *glyphs;
  struct tt_glyph_names *ttgn;
  char     *cmap_table;
  Encoding *encoding;
  int       code, aglm_id, count;
  tt_cmap  *ttcm;
  AGLmap   *aglm;
  char    **enc_vec;

  encoding = Encoding_cache_get(font->encoding_id);
  enc_vec  = Encoding_get_encoding(encoding);
  ttgn     = tt_get_glyph_names(sfont);
  ttcm     = tt_cmap_read(sfont, TT_WIN, TT_WIN_UNICODE);
  aglm_id  = AGLmap_cache_find(AGLMAP_DEFAULT_GLYPHLIST);
  aglm     = (aglm_id >= 0) ? AGLmap_cache_get(aglm_id) : NULL;

  ASSERT(enc_vec);
  if (!ttgn && (!aglm || !ttcm))
    WARN("PostScript glyph name list nor Unicode cmap table not available.");

  cmap_table = NEW(274, char);
  memset(cmap_table, 0, 274);
  sfnt_put_ushort(cmap_table,    0);            /* Version  */
  sfnt_put_ushort(cmap_table+2,  1);            /* Number of subtables */
  sfnt_put_ushort(cmap_table+4,  TT_MAC);       /* Platform ID */
  sfnt_put_ushort(cmap_table+6,  TT_MAC_ROMAN); /* Encoding ID */
  sfnt_put_ulong (cmap_table+8,  12);           /* Offset   */
  sfnt_put_ushort(cmap_table+12, 0);            /* Format   */
  sfnt_put_ushort(cmap_table+14, 262);          /* Length   */
  sfnt_put_ushort(cmap_table+16, 0);            /* Language */

  glyphs = tt_build_init();

  if (verbose > 2)
    MESG("[glyphs:/.notdef");

  count = 1; /* +1 for .notdef */
  for (code = 0; code < 256; code++) {
    USHORT new_gid, gid = 0;
    if (!font->used_chars[code])
      continue;
    if (!enc_vec[code] || !strcmp(enc_vec[code], ".notdef")) {
      WARN("%s: Character mapped to .notdef used. (char: 0x%02X, font: %s, encoding: %s)",
	   TTFONT_DEBUG_STR, code, font->ident, Encoding_get_name(encoding));
      WARN("%s: Maybe incorrect encoding specified.", TTFONT_DEBUG_STR);
      new_gid = 0;
    } else {
      if (verbose > 2)
	MESG("/%s", enc_vec[code]);
      /*
       * First we try glyph name to GID mapping using post table if post table
       * is available. If post table is not available or glyph is not listed 
       * in the post table, then we try Unicode if Windows-Unicode TrueType
       * cmap is available.
       */
      if (ttgn)
	gid = tt_glyph_lookup(ttgn, enc_vec[code]);
      /*
       * UCS-4 not supported yet.
       */
      if (gid == 0 && ttcm) {
	if (AGName_is_unicode(enc_vec[code]))
	  gid = tt_cmap_lookup(ttcm, (USHORT) AGName_convert_uni(enc_vec[code]));
	else if (aglm) {
	  AGList *agl;
	  agl = AGLmap_lookup(aglm, enc_vec[code], NULL); /* FIXME: suffix */
	  /*
	   * Try all alternatives.
	   */
	  while (agl) {
	    /*
	     * We can't support decomposed form.
	     */
	    if (!AGList_is_composite(agl)) {
	      gid = tt_cmap_lookup(ttcm, (USHORT) AGList_get_code(agl)); /* TODO: UCS4 */
	      if (gid > 0) {
		if (verbose > 3)
		  MESG("(U+%04X)", AGList_get_code(agl));
		break;
	      }
	    }
	    agl = AGList_next_alternative(agl);
	  }
	}
      }
      /*
       * Older versions of gs had problem with glyphs (other than .notdef)
       * mapped to gid = 0.
       */
      if (gid == 0 && enc_vec[code])
	WARN("Glyph \"%s\" missing in font \"%s\".", enc_vec[code], font->ident);
      if ((new_gid = tt_find_glyph(glyphs, gid)) == 0) {
	new_gid = tt_add_glyph(glyphs, gid, count); /* count returned. */
	count++;
      }
    }
    cmap_table[18+code] = new_gid;
  }
  tt_cmap_release(ttcm);
  tt_release_glyph_names(ttgn);

  if (verbose > 2)
    MESG("]");

  if (tt_build_tables(sfont, glyphs) < 0)
    ERROR("Could not created FontFile stream.");

  {
    double widths[256];

    for (code = 0; code < 256; code++) {
      if (font->used_chars[code]) {
	USHORT idx;
	idx = tt_get_index(glyphs, (USHORT) cmap_table[18+code]);
	widths[code] = PDFUNIT(glyphs->gd[idx].advw);
      } else {
	widths[code] = 0.0;
      }
    }
    do_widths(font, widths);
  }

  if (verbose > 1) 
    MESG("[%d glyphs]", glyphs->num_glyphs);

  tt_build_finish(glyphs);

  sfnt_set_table(sfont, "cmap", cmap_table, 274);

}

void
TTFont_dofont (pdf_font *font)
{
  char    *stream_data;
  pdf_obj *stream_dict;
  long     stream_len;
  sfnt    *sfont;

  ASSERT(font);

  if (!font->indirect)
    return;

  pdf_add_dict(font->fontdict, 
	       pdf_new_name("FontDescriptor"),
	       pdf_ref_obj (font->descriptor));
  if (font->encoding_id >= 0)
    UC_add_ToUnicode_CMap(font->fontdict, Encoding_cache_get(font->encoding_id));

  if (!font->embedding)
    return;

  verbose = pdf_font_get_verbose();

  sfont = sfnt_open(font->filename);
  if (!sfont)
    ERROR("%s: Unable to open file (%s)", TTFONT_DEBUG_STR, font->filename);
  if (sfont->type != SFNT_TYPE_TRUETYPE || 
      sfnt_read_table_directory(sfont, 0) < 0)
    ERROR("%s: Not TrueType font ?", TTFONT_DEBUG_STR);

  if (!font->used_chars)
    ERROR("%s: Unexpected error.", TTFONT_DEBUG_STR);

  /*
   * Create new TrueType cmap table with MacRoman encoding.
   */
  if (font->encoding_id < 0)
    do_builtin_encoding(font, sfont);
  else
    do_custum_encoding(font, sfont);

  /*
   * TODO: post table?
   */

  {
    int i;
    for (i = 0; required_table[i].name != NULL; i++) {
      if (sfnt_require_table(sfont,
			     required_table[i].name,
			     required_table[i].must_exist) < 0)
	ERROR("%s: TrueType table \"%s\" does not exist.",
	      TTFONT_DEBUG_STR, required_table[i].name);
    }
  }

  stream_len  = sfnt_get_size(sfont);
  stream_data = NEW(stream_len, char);
  if (sfnt_build_font(sfont, stream_data, stream_len) == NULL)
    ERROR("%s: Could not created FontFile stream.", TTFONT_DEBUG_STR);
  sfnt_close(sfont);

  /*
   * FontFile2
   */
  {
    pdf_obj *fontfile;

    fontfile = pdf_new_stream(STREAM_COMPRESS);
    pdf_add_dict(font->descriptor,
		 pdf_new_name("FontFile2"), pdf_ref_obj(fontfile));
    stream_dict = pdf_stream_dict(fontfile);
    pdf_add_dict(stream_dict, pdf_new_name("Length1"), pdf_new_number(stream_len));
    pdf_add_stream(fontfile, stream_data, stream_len);
    pdf_release_obj(fontfile);
    RELEASE(stream_data);
  }

  if (verbose > 1)
    MESG("[%ld bytes]", stream_len);

  return;
}
