/*
 * Dynamic Encapsulation Gateway Client for Linux.
 * Protocol information sourced from the MFNOS implementation
 * by Barry Siegfried, K2MF
 *
 * Copyright (c) 1999, Terry Dawson, VK2KTJ.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "dgclient.h"
#include "daemon.h"

#include <errno.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <netinet/in.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <signal.h>

static char *
banner="Dynamic Gateway Client";

static char *
copyright="Copyright (c) 1999, Terry Dawson, VK2KTJ";

static char *
usage="\nUsage:\n\tdgclient [-t msgtimer] [-k password] [-p port] [-d] server network/mask\n\n";

char *key=NULL,
	*server=NULL,
	*network=NULL;

char *msg;

int msglen=0,
	daemonise=1,
	port=IPPORT_REMOTE,
	msgtimer=DEFTIMER;

int opt;

struct sockaddr_in servsa;
struct hostent *hp;
int sockfd;

int main(int argc, char **argv)
{

	/*
	 * Print welcome banner and copyright
	 */
	 fprintf(stderr,"\n%s %s\n%s\n\n",banner,VERSION,copyright);

	/*
	 * Process command line arguments
	 *
	 * command line must contain at least two arguments.
	 */
	if (argc<2) {
		fprintf(stderr,usage);
		exit(1);
	}

	while ((opt=getopt(argc, argv, "dk:p:t:")) != EOF) {
		switch(opt) {
			case 'd':
				daemonise=(daemonise ? 0 : 1);
				break;

			case 'k':
				key=optarg;
				break;

			case 'p':
				port=atoi(optarg);
				break;

			case 't':
				msgtimer=atoi(optarg);
				break;

			default:
				perror(usage);
				exit(-2);
		}
	}

	server=argv[argc-2];
	network=argv[argc-1];

	/*
	 * Open UDP socket
	 */

	/* Ensure the server host IP address is resolvable. */
	if ((hp=gethostbyname(server))==NULL) {
		perror("Server IP address could not be resolved");
		exit(-3);
	}
	servsa.sin_family = hp->h_addrtype;
	memcpy(&servsa.sin_addr, hp->h_addr, hp->h_length);

	/* Set UDP port to use */
	servsa.sin_port = htons(port);

	/* Open socket */
	if ((sockfd=socket(AF_INET,SOCK_DGRAM,0))==-1) {
		perror("Opening Socket failed");
		exit(-4);
	}

	/*
	 * Prepare message to send
	 */

	asprintf(&msg, "%c%c%s%s", ROUTE_ME, strlen(network), network, key);
	msglen=strlen(msg);

	/*
	 * Daemonise if necessary
	 */

	if (daemonise) {
		if (! daemon_start(TRUE)) {
			perror("dgclient: attempt to daemonise failed");
			exit(-5);
		}
	}


	/*
	 * Ensure that any caught signal causes us to unroute properly
	 */
	signal(SIGHUP,unroute);
	signal(SIGINT,unroute);
	signal(SIGQUIT,unroute);
	signal(SIGABRT,unroute);
	signal(SIGUSR1,unroute);
	signal(SIGUSR2,unroute);
	signal(SIGTERM,unroute);

	/*
	 * Loop, periodically transmitting message.
	 */

	while(TRUE) {
		sendto(sockfd, msg, msglen, 0,
			(struct sockaddr *)&servsa, sizeof(servsa));
		sleep(msgtimer);
	}

	exit(0);
}

void
unroute(int signum)
{
	int repeat=3;

	asprintf(&msg, "%c%c%s%s", UNROUTE_ME, strlen(network), network, key);
	msglen=strlen(msg);

	while(repeat) {
		sendto(sockfd, msg, msglen, 0,
			(struct sockaddr *)&servsa, sizeof(servsa));
		sleep(1);
		repeat--;
	}

	exit(0);
}

