%  Copyright (C) 2002-2004 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\begin{code}
module DarcsArguments ( DarcsFlag( .. ), isa,
                        fix_flag, fix_filepath, unfix_filepaths,
                        DarcsOption( .. ), option_from_darcsoption,
                        help, verbose, list_options, list_files,
                        quiet, any_verbosity, disable,
                        notest, test, working_repo_dir,
                        leave_test_dir,
                        possibly_remote_repo_dir, get_repodir,
                        list_registered_files, list_unregistered_files,
                        author, get_author,
                        patchname_option, distname_option,
                        logfile, rmlogfile, from_opt,
                        target, cc, get_cc, output, recursive,
                        edit_file, askdeps, ignoretimes, lookforadds,
                        ask_long_comment,
                        sign, verify, edit_description,
                        reponame, tagname, creatorhash,
                        apply_conflict_options, reply,
                        use_external_merge, want_external_merge,
                        no_deps, nocompress, compress, uncompress,
                        uncompress_nocompress,
                        options_latex,
                        noskip_boring, allow_caseonly,
                        applyas, human_readable,
                        changes_reverse,
                        changes_format, match_one_context, match_one_nontag,
                        send_to_context,
                        pipe_interactive, all_interactive,
                        gui_pipe_interactive, gui_interactive,
                        all_gui_pipe_interactive, all_gui_interactive,
                        summary, unified, tokens,
                        checkpoint, partial, partial_check,
                        diffflags, unidiff, xmloutput,
                        force_replace, dry_run,
                        print_dry_run_message_and_exit,
                        match_one, match_several, match_range,
                        happy_forwarding,
                        set_default,
                        fancy_move_add, pristine_tree,
                        set_scripts_executable,
                        (///),
                        sibling, flagsToSiblings, relink
                      ) where
import System.Console.GetOpt
import Directory hiding ( getCurrentDirectory )
import List
import System
import Maybe ( catMaybes )
import Monad ( liftM, mplus, when )
import Char ( isDigit )

import Autoconf ( path_separator )
import Patch ( Patch, patch_summary, patch2patchinfo )
import PatchInfo ( human_friendly )
import DarcsUtils ( askUser, catchall, ortryrunning )
import DarcsURL ( is_relative )
import RepoPrefs ( boring_file_filter, get_preflist, get_global )
import FileName ( fn2fp, fp2fn, norm_path )
import PatchMatchData ( patch_match )
import DarcsFlags ( DarcsFlag(..) )
import Repository ( slurp_pending, slurp_all_but_darcs )
import SlurpDirectory ( list_slurpy )
import Printer ( putDocLn, text, vsep, ($$) )
#include "impossible.h"
\end{code}

\begin{code}
get_content :: DarcsFlag -> Maybe String
get_content (SignAs s) = Just s
get_content (SignSSL s) = Just s
get_content (Toks s) = Just s
get_content (Target s) = Just s
get_content (Output s) = Just s
get_content (Author s) = Just s
get_content (PatchName s) = Just s
get_content (DistName s) = Just s
get_content (Verify s) = Just s
get_content (VerifySSL s) = Just s
get_content (ApplyAs s) = Just s
get_content (RepoName s) = Just s
get_content (TagName s) = Just s
get_content (DiffFlags s) = Just s
get_content (ExternalMerge s) = Just s
get_content (RepoDir s) = Just s
get_content _ = Nothing

isa :: DarcsFlag -> (String -> DarcsFlag) -> Bool
a `isa` b = case get_content a of
            Nothing -> False
            Just s -> a == b s

data DarcsOption
    = DarcsArgOption [Char] [String] (String->DarcsFlag) String String
    | DarcsNoArgOption [Char] [String] DarcsFlag String
    | DarcsMultipleChoiceOption [DarcsOption]
option_from_darcsoption :: DarcsOption -> [OptDescr DarcsFlag]
option_from_darcsoption (DarcsNoArgOption a b c h)
    = [Option a b (NoArg c) h]
option_from_darcsoption (DarcsArgOption a b c n h)
    = [Option a b (ReqArg c n) h]
option_from_darcsoption (DarcsMultipleChoiceOption os)
    = concat $ map option_from_darcsoption os
\end{code}

\begin{code}
fix_flag :: FilePath -> DarcsFlag -> DarcsFlag
fix_flag fix (Output s) = Output $ fix_maybe_absolute fix s
fix_flag fix (Verify s) = Verify $ fix_maybe_absolute fix s
fix_flag fix (LogFile s) = LogFile $ fix_maybe_absolute fix s
fix_flag fix (VerifySSL s) = VerifySSL $ fix_maybe_absolute fix s
fix_flag _ (Context "") = Context ""
fix_flag fix (Context s) = Context $ fix_maybe_absolute fix s
fix_flag _ f = f

fix_maybe_absolute :: FilePath -> FilePath -> FilePath
fix_maybe_absolute fix pat = fma $ map cleanup pat
    where fma p@('/':_) = p
          fma p = fix /// p
          cleanup '\\' | path_separator == '\\' = '/'
          cleanup c = c

fix_filepath :: [DarcsFlag] -> FilePath -> FilePath
fix_filepath [] f = f
fix_filepath (FixFilePath fix:_) f = fix_maybe_absolute fix f
fix_filepath (_:fs) f = fix_filepath fs f

unfix_filepaths :: [DarcsFlag] -> [FilePath] -> [FilePath]
unfix_filepaths [] f = f
unfix_filepaths (FixFilePath fix:_) f = drop_paths fix f
unfix_filepaths (_:fs) f = unfix_filepaths fs f

(///) :: FilePath -> FilePath -> FilePath
""///a = do_norm a
a///b = do_norm $ a ++ "/" ++ b

do_norm :: FilePath -> FilePath
do_norm f = fn2fp $ norm_path $ fp2fn f

drop_paths :: String -> [String] -> [String]
drop_paths "" ps = ps
drop_paths fix ps = catMaybes $ map drop_path ps
  where drop_path ('.':'/':p) = drop_path $ dropWhile (=='/') p
        drop_path p = if take (length fix) p == fix
                      then Just $ dropWhile (=='/') $ drop (length fix) p
                      else if is_relative p
                           then Nothing
                           else Just p
\end{code}

\begin{code}
list_options :: DarcsOption
list_options = DarcsNoArgOption [] ["list-options"] ListOptions
               "simply list the command's arguments"

verbose :: DarcsOption
quiet :: DarcsOption
any_verbosity :: DarcsOption
reponame :: DarcsOption
tagname :: DarcsOption
no_deps :: DarcsOption
checkpoint :: DarcsOption
partial :: DarcsOption
partial_check :: DarcsOption
tokens :: DarcsOption
working_repo_dir :: DarcsOption
possibly_remote_repo_dir :: DarcsOption
disable :: DarcsOption

gui_interactive, pipe_interactive, gui_pipe_interactive, all_gui_interactive,
  all_gui_pipe_interactive, all_interactive, all_patches, interactive, pipe,
  human_readable, diffflags, allow_caseonly, noskip_boring,
  ask_long_comment, match_one_nontag, changes_reverse, creatorhash,
  changes_format, match_one_context, happy_forwarding, send_to_context,
  use_external_merge, target, cc, apply_conflict_options, reply, xmloutput,
  distname_option, patchname_option, edit_description, output, unidiff,
  unified, summary, compress, uncompress, uncompress_nocompress, nocompress,
  author, askdeps, lookforadds, ignoretimes, test, notest, help, force_replace,
  match_one, match_range, match_several, dry_run, fancy_move_add,
#ifdef HAVEWX
  gui,
#endif
  logfile, rmlogfile, leave_test_dir, from_opt, set_default, pristine_tree

      :: DarcsOption

recursive :: String -> DarcsOption

sign, applyas, verify :: DarcsOption
\end{code}

\section{Common options to darcs commands}

\begin{options}
--help
\end{options}
Every {\tt COMMAND} accepts \verb!--help! as an argument, which tells it to
provide a bit of help.  Among other things, this help always provides an
accurate listing of the options available with that command, and is
guaranteed never to be out of sync with the version of darcs you actually
have installed (unlike this manual, which could be for an entirely
different version of darcs).
\begin{verbatim}
% darcs COMMAND --help
\end{verbatim}
\begin{code}
help = DarcsNoArgOption ['h'] ["help"] Help
       "shows brief description of command and its arguments"
\end{code}

\begin{options}
--disable
\end{options}
Every {\tt COMMAND} accepts the \verb!--disable! option, which can be used in
\verb!_darcs/prefs/defaults! to disable some commands in the repository. This
can be helpful if you want to protect the repository from accidential use of
advanced commands like unpull, unrecord or amend-record.
\begin{code}
disable = DarcsNoArgOption [] ["disable"] Disable
        "disable this command"
\end{code}

\begin{options}
--verbose
\end{options}
Most commands also accept the \verb!--verbose! option, which tells darcs to
provide additional output.  The amount of verbosity varies from command to
command.
\begin{code}
verbose = DarcsMultipleChoiceOption
          [DarcsNoArgOption ['v'] ["verbose"] Verbose
           "give verbose output",
           DarcsNoArgOption [] ["standard-verbosity"] NormalVerbosity
           "don't give verbose output"]
quiet = DarcsMultipleChoiceOption
        [DarcsNoArgOption ['q'] ["quiet"] Quiet
         "suppress informational output",
         DarcsNoArgOption [] ["standard-verbosity"] NormalVerbosity
         "normal informational output"]
any_verbosity = DarcsMultipleChoiceOption
                [DarcsNoArgOption ['v'] ["verbose"] Verbose
                 "give verbose output",
                 DarcsNoArgOption ['q'] ["quiet"] Quiet
                 "suppress informational output",
                 DarcsNoArgOption [] ["standard-verbosity"] NormalVerbosity
                 "neither verbose nor quiet output"]
\end{code}

\begin{options}
--repodir
\end{options}
Another common option is the \verb!--repodir! option, which allows you to
specify the directory of the repository in which to perform the command.
This option is used with commands, such as whatsnew, that ordinarily would
be performed within a repository directory, and allows you to use those
commands without actually being in the repo directory when calling the
command.  This is useful when running darcs as a pipe, as might be the case
when running apply from a mailer.

\begin{code}
working_repo_dir = DarcsArgOption [] ["repodir"] WorkDir "DIRECTORY"
             "specify the repository directory in which to run"
possibly_remote_repo_dir = DarcsArgOption [] ["repo"] RepoDir "URL"
             "specify the repository URL"
get_repodir :: [DarcsFlag] -> String
get_repodir [] = "."
get_repodir (WorkDir r:_) = r
get_repodir (RepoDir r:_) = r
get_repodir (_:fs) = get_repodir fs
\end{code}

\input{Match.lhs}
\input{PatchMatch.lhs}

\begin{code}
patchname_option = DarcsArgOption ['m'] ["patch-name"] PatchName "PATCHNAME"
                   "name of patch"

send_to_context = DarcsArgOption [] ["context"] Context "FILENAME"
                  "send to context stored in FILENAME"

match_one_context =
    DarcsMultipleChoiceOption
    [DarcsArgOption [] ["to-match"] mp "PATTERN"
     "select changes up to a patch matching PATTERN",
     DarcsArgOption [] ["to-patch"] OnePatch "REGEXP"
     "select changes up to a patch matching REGEXP",
     DarcsArgOption [] ["tag"] OneTag "REGEXP"
     "select tag matching REGEXP",
     DarcsArgOption [] ["context"] Context "FILENAME"
     "version specified by the context in FILENAME"
    ]
    where mp s = OnePattern (patch_match s)
match_one = DarcsMultipleChoiceOption
            [DarcsArgOption [] ["match"] mp "PATTERN"
             "select patch matching PATTERN",
             DarcsArgOption ['p'] ["patch"] OnePatch "REGEXP"
             "select patch matching REGEXP",
             DarcsArgOption ['t'] ["tag"] OneTag "REGEXP"
             "select tag matching REGEXP"]
    where mp s = OnePattern (patch_match s)
match_one_nontag = DarcsMultipleChoiceOption
                   [DarcsArgOption [] ["match"] mp "PATTERN"
                    "select patch matching PATTERN",
                    DarcsArgOption ['p'] ["patch"] OnePatch "REGEXP"
                    "select patch matching REGEXP"]
    where mp s = OnePattern (patch_match s)
match_several = DarcsMultipleChoiceOption
                [DarcsArgOption [] ["matches"] mp "PATTERN"
                 "select patches matching PATTERN",
                 DarcsArgOption ['p'] ["patches"] SeveralPatch "REGEXP"
                 "select patches matching REGEXP",
                 DarcsArgOption ['t'] ["tags"] OneTag "REGEXP"
                 "select tags matching REGEXP"]
    where mp s = SeveralPattern (patch_match s)
match_range = DarcsMultipleChoiceOption
              [DarcsArgOption [] ["to-match"] uptop "PATTERN"
               "select changes up to a patch matching PATTERN",
               DarcsArgOption [] ["to-patch"] UpToPatch "REGEXP"
               "select changes up to a patch matching REGEXP",
               DarcsArgOption [] ["to-tag"] UpToTag "REGEXP"
               "select changes up to a tag matching REGEXP",
               DarcsArgOption [] ["from-match"] fromp "PATTERN"
               "select changes starting with a patch matching PATTERN",
               DarcsArgOption [] ["from-patch"] AfterPatch "REGEXP"
               "select changes starting with a patch matching REGEXP",
               DarcsArgOption [] ["from-tag"] AfterTag "REGEXP"
               "select changes starting with a tag matching REGEXP",
               DarcsArgOption [] ["match"] onep "PATTERN"
               "select a single patch matching PATTERN",
               DarcsArgOption ['p'] ["patch"] OnePatch "REGEXP"
               "select a single patch matching REGEXP",
               DarcsArgOption [] ["last"] lastn "NUMBER"
               "select the last NUMBER patches"]
    where uptop s = UpToPattern (patch_match s)
          fromp s = AfterPattern (patch_match s)
          onep s = OnePattern (patch_match s)
          lastn s = if and (map isDigit s)
                    then LastN (read s)
                    else LastN (-1)
\end{code}

\begin{code}
notest = DarcsMultipleChoiceOption
         [DarcsNoArgOption [] ["no-test"] NoTest "don't run the test script",
          DarcsNoArgOption [] ["test"] Test "run the test script"]
test = DarcsMultipleChoiceOption
          [DarcsNoArgOption [] ["test"] Test "run the test script",
           DarcsNoArgOption [] ["no-test"] NoTest "don't run the test script"]
leave_test_dir = DarcsMultipleChoiceOption
                 [DarcsNoArgOption [] ["leave-test-directory"]
                  LeaveTestDir "don't remove the test directory",
                  DarcsNoArgOption [] ["remove-test-directory"]
                  NoLeaveTestDir "remove the test directory"]
\end{code}

\begin{options}
--ignore-times
\end{options}
Darcs optimizes its operations by keeping track of the modification times
of your files.  This dramatically speeds up commands such as
\verb!whatsnew! and \verb!record! which would otherwise require reading
every file in the repo and comparing it with a reference version.  However,
there are times when this can cause problems, such as when running a series
of darcs commands from a script, in which case often a file will be
modified twice in the same second, which can lead to the second
modification going unnoticed.  The solution to such predicaments is the
\verb!--ignore-times! option, which instructs darcs not to trust the file
modification times, but instead to check each file's contents explicitly.
\begin{code}
ignoretimes = DarcsNoArgOption [] ["ignore-times"] IgnoreTimes
              "don't trust the file modification times"
lookforadds =
    DarcsMultipleChoiceOption
    [DarcsNoArgOption ['l'] ["look-for-adds"] LookForAdds
     "Add any new files or directories in the working dir",
     DarcsNoArgOption [] ["dont-look-for-adds"] NoLookForAdds
     "Don't add any files or directories automatically"]

fancy_move_add =
    DarcsMultipleChoiceOption
    [DarcsNoArgOption [] ["date-trick"] FancyMoveAdd
     "add files with date appended to avoid conflict. [EXPERIMENTAL] ",
     DarcsNoArgOption [] ["no-date-trick"] NoFancyMoveAdd
     "don't use experimental date appending trick. [DEFAULT]"]

pristine_tree =
    DarcsMultipleChoiceOption
    [DarcsNoArgOption [] ["plain-pristine-tree"] PristinePlain
     "Use a plain pristine tree [DEFAULT]",
     DarcsNoArgOption [] ["no-pristine-tree"] PristineNone
     "Use no pristine tree"]

\end{code}

\begin{code}
askdeps =
    DarcsMultipleChoiceOption
    [DarcsNoArgOption [] ["ask-deps"] AskDeps
     "ask about dependencies",
     DarcsNoArgOption [] ["no-ask-deps"] NoAskDeps
     "don't ask about dependencies"]

ask_long_comment =
    DarcsMultipleChoiceOption
    [DarcsNoArgOption [] ["edit-long-comment"] EditLongComment
     "Edit the long comment by default",
     DarcsNoArgOption [] ["skip-long-comment"] NoEditLongComment
     "Don't give a long comment",
     DarcsNoArgOption [] ["prompt-long-comment"] PromptLongComment
     "Prompt for whether to edit the long comment"]
\end{code}

\begin{options}
--author
\end{options}
Several commands need to be able to identify you.  Conventionally, you
provide an email address for this purpose, which can include comments,
e.g.\ ``David Roundy <droundy@abridgegame.org>''.  The easiest way to do
this is
to define an environment variable \verb!EMAIL! or \verb!DARCS_EMAIL! (with
the latter overriding the former).  You can also override this using the
\verb!--author! flag to any command.  Alternatively, you could set your
email address on a per-repository basis using the ``defaults'' mechanism
for ``ALL'' commands, as described in Appendix~\ref{repository_format}.
Or, you could specify the author on a per-repository basis using the
\verb!_darcs/prefs/author! file as described in section~\ref{author_prefs}.

Also, a global author file can be created with the name
\verb!.darcs/author! in your home directory.  This file overrides the
contents of the environment variables, but a repository-specific author
file overrides the global author file.

\begin{code}
logfile = DarcsArgOption [] ["logfile"] LogFile "FILE"
          "give patch name and comment in file"

rmlogfile = DarcsNoArgOption [] ["delete-logfile"] RmLogFile
            "delete the logfile when done"

author = DarcsArgOption ['A'] ["author"] Author "EMAIL" "specify author id"
from_opt = DarcsArgOption [] ["from"] Author "EMAIL" "specify email address"

get_author :: [DarcsFlag] -> IO String
get_author (Author a:_) = return a
get_author (Pipe:_) = do askUser "Who is the author? "
get_author (_:flags) = get_author flags
get_author [] = do
  pref_author <- author_pref
  easy_author <- try_author [getEnv "DARCS_EMAIL",
                             getEnv "EMAIL"]
  case pref_author `mplus` easy_author of
    Just a -> return a
    Nothing -> do
      aminrepo <- doesDirectoryExist "_darcs/prefs"
      if aminrepo then do
          putStr "Darcs needs to know what name (conventionally an email "
          putStr "address) to use as the\npatch author, e.g. 'Fred Bloggs "
          putStr "<fred@bloggs.invalid>'.  If you provide one\nnow "
          putStr "it will be stored in the file '_darcs/prefs/author' and "
          putStr "used as a default\nin the future.  To change your preferred "
          putStr "author address, simply delete or edit\nthis file.\n\n"
          add <- askUser "What is your email address? "
          writeFile "_darcs/prefs/author" add
          return add
        else do askUser "What is your email address? "
    where try_author (g:gs) = (liftM Just g) `catchall` try_author gs
          try_author [] = return Nothing
          author_pref = do au <- get_preflist "author"
                           augl <- get_global "author"
                           case au++augl of [] -> return Nothing
                                            (a:_) -> return $ Just a
\end{code}

\begin{options}
--dont-compress, --compress
\end{options}
By default, darcs commands that write patches to disk will compress the
patch files.  If you don't want this, you can choose the
\verb!--dont-compress! option, which causes darcs not to compress the patch
file.

\begin{code}
nocompress
    = DarcsMultipleChoiceOption
      [DarcsNoArgOption [] ["compress"] Compress
       "create compressed patches",
       DarcsNoArgOption [] ["dont-compress"] NoCompress
       "don't create compressed patches"]
uncompress
    = DarcsMultipleChoiceOption
      [DarcsNoArgOption [] ["compress"] Compress
       "create compressed patches",
       DarcsNoArgOption [] ["uncompress"] NoCompress
       "uncompress patches"]
compress
    = DarcsMultipleChoiceOption
      [DarcsNoArgOption [] ["dont-compress"] NoCompress
       "don't create compressed patches",
       DarcsNoArgOption [] ["compress"] Compress
       "create compressed patches"]

uncompress_nocompress
    = DarcsMultipleChoiceOption
      [DarcsNoArgOption [] ["compress"] Compress
       "create compressed patches",
       DarcsNoArgOption [] ["dont-compress"] NoCompress
       "don't create compressed patches",
       DarcsNoArgOption [] ["uncompress"] UnCompress
       "uncompress patches"]
\end{code}

\begin{options}
--gui
\end{options}
Certain commands may have an optional graphical user interface.  If such
commands are supported, you can activate the graphical user interface by
calling darcs with the \verb!--gui! flag.

\begin{code}
#ifdef HAVEWX
gui = DarcsNoArgOption ['g'] ["gui"] Gui "use graphical interface"
#endif
summary = DarcsMultipleChoiceOption
          [DarcsNoArgOption ['s'] ["summary"] Summary "summarize changes",
           DarcsNoArgOption [] ["no-summary"] NoSummary "don't summarize changes"]
unified = DarcsNoArgOption ['u'] ["unified"] Unified
          "output patch in a darcs-specific format similar to diff -u"
unidiff = DarcsNoArgOption ['u'] ["unified"] Unified
          "pass -u option to diff"
\end{code}

\begin{code}
target = DarcsArgOption [] ["to"] Target "EMAIL" "specify destination email"
cc = DarcsArgOption [] ["cc"] Cc "EMAIL" "specify email address to cc"
get_cc :: [DarcsFlag] -> String
get_cc fs = lt $ catMaybes $ map whatcc fs
            where whatcc (Cc t) = Just t
                  whatcc _ = Nothing
                  lt [t] = t
                  lt [t,""] = t
                  lt (t:ts) = t++" , "++lt ts
                  lt [] = ""
\end{code}

\begin{code}
output = DarcsArgOption ['o'] ["output"] Output "FILE"
         "specify output filename"
\end{code}

\begin{code}
edit_description = DarcsNoArgOption [] ["edit-description"] EditDescription
                  "edit the patch bundle description"
\end{code}

\begin{code}
distname_option = DarcsArgOption ['d'] ["dist-name"] DistName "DISTNAME"
                  "name of version"
\end{code}

\begin{code}
recursive h
    = DarcsMultipleChoiceOption
      [DarcsNoArgOption ['r'] ["recursive"] Recursive h,
       DarcsNoArgOption [] ["not-recursive"] NoRecursive ("don't "++h)]
\end{code}

\begin{code}
xmloutput = DarcsNoArgOption [] ["xml-output"] XMLOutput
        "generate XML formatted output"
\end{code}

\begin{code}
creatorhash = DarcsArgOption [] ["creator-hash"] CreatorHash "HASH"
              "specify hash of creator patch (see docs)"
\end{code}

\begin{code}
sign = DarcsMultipleChoiceOption
       [DarcsNoArgOption [] ["sign"] Sign
        "sign the patch with your gpg key",
        DarcsArgOption [] ["sign-as"] SignAs "KEYID"
        "sign the patch with a given keyid",
        DarcsArgOption [] ["sign-ssl"] SignSSL "IDFILE"
        "sign the patch using openssl with a given private key",
        DarcsNoArgOption [] ["dont-sign"] NoSign
        "do not sign the patch"]
applyas = DarcsMultipleChoiceOption
           [DarcsArgOption [] ["apply-as"] ApplyAs "USERNAME"
            "apply patch as another user using sudo",
            DarcsNoArgOption [] ["apply-as-myself"] NonApply
            "don't use sudo to apply as another user [DEFAULT]"]
happy_forwarding = DarcsNoArgOption [] ["happy-forwarding"] HappyForwarding
                   "forward unsigned messages without extra header"
set_default = DarcsMultipleChoiceOption
              [DarcsNoArgOption [] ["set-default"] SetDefault
               "set default repository [DEFAULT]",
               DarcsNoArgOption [] ["no-set-default"] NoSetDefault
               "don't set default repository"]
\end{code}

\begin{code}
verify = DarcsMultipleChoiceOption
         [DarcsArgOption [] ["verify"] Verify "PUBRING"
          "verify that the patch was signed by a key in PUBRING",
          DarcsArgOption [] ["verify-ssl"] VerifySSL "KEYS"
          "verify using openSSL with authorized keys from file 'KEYS'",
          DarcsNoArgOption [] ["no-verify"] NonVerify
          "don't verify patch signature"]
\end{code}

\begin{code}
reponame = DarcsArgOption [] ["repo-name"] RepoName "DIRECTORY"
           "path of output directory"
tagname = DarcsArgOption ['t'] ["tag-name"] TagName "TAGNAME"
          "name of version to pull"
no_deps = DarcsNoArgOption [] ["no-deps"] DontGrabDeps
                       "don't automatically fulfill dependencies"
checkpoint = DarcsNoArgOption [] ["checkpoint"] CheckPoint
             "create a checkpoint file"
partial = DarcsMultipleChoiceOption
          [DarcsNoArgOption [] ["partial"] Partial
           "get partial repository using checkpoint",
           DarcsNoArgOption [] ["complete"] Complete
           "get a complete copy of the repository"]
partial_check = DarcsMultipleChoiceOption
                [DarcsNoArgOption [] ["complete"] Complete
                 "check the entire repository",
                 DarcsNoArgOption [] ["partial"] Partial
                 "check patches since latest checkpoint"]
tokens = DarcsArgOption [] ["token-chars"] Toks "\"[CHARS]\""
         "define token to contain these characters"
\end{code}

\begin{code}
force_replace = DarcsMultipleChoiceOption
                [DarcsNoArgOption [] ["force"] ForceReplace
                 "proceed with replace even if 'new' token already exists",
                 DarcsNoArgOption [] ["no-force"]
                 NonForce "don't force the replace if it looks scary"]
\end{code}

\begin{code}
reply = DarcsArgOption [] ["reply"] Reply "FROM" "send email response"
apply_conflict_options
    = DarcsMultipleChoiceOption
      [DarcsNoArgOption [] ["mark-conflicts"]
       MarkConflicts "mark conflicts",
       DarcsNoArgOption [] ["allow-conflicts"]
       AllowConflicts "allow conflicts, but don't mark them",
       DarcsArgOption [] ["external-merge"]
       ExternalMerge "COMMAND" "Use external tool to merge conflicts",
       DarcsNoArgOption [] ["no-resolve-conflicts"] NoAllowConflicts
       "equivalent to --dont-allow-conflicts, for backwards compatibility",
       DarcsNoArgOption [] ["dont-allow-conflicts"]
       NoAllowConflicts "fail on patches that create conflicts [DEFAULT]"]
use_external_merge = DarcsArgOption [] ["external-merge"]
                     ExternalMerge "COMMAND" "Use external tool to merge conflicts"

want_external_merge :: [DarcsFlag] -> Maybe String
want_external_merge [] = Nothing
want_external_merge (ExternalMerge c:_) = Just c
want_external_merge (_:fs) = want_external_merge fs
\end{code}

\begin{options}
--dry-run
\end{options}
The \verb!--dry-run! option will cause darcs to not actually take the specified
action.  but only print what would have happened.  Not all commands accept
\verb!--dry-run!, but those that do should accept the \verb!--summary!  option.

\begin{options}
--summary, --no-summary
\end{options}
The \verb!--summary! show a summary of the patches that would have been
pulled/pushed/whatever. The format is similar to the output format of
\verb!cvs update! and looks like this:

\begin{verbatim}
A  ./added_but_not_recorded.c
A! ./added_but_not_recorded_conflicts.c
a  ./would_be_added_if_look_for_adds_option_was_used.h

M  ./modified.t -1 +1
M! ./modified_conflicts.t -1 +1

R  ./removed_but_not_recorded.c
R! ./removed_but_not_recorded_conflicts.c

\end{verbatim}

You can probably guess what the flags mean from the clever file names.

\verb!A!  is for files that have been added but not recorded yet. \verb!a!  is
for files found using the \verb!--look-for-adds! option available for
\verb!whatsnew! and \verb!record!. They have not been added yet, but would be
added automatically if \verb!--look-for-adds! were used with the next
\verb!record!  command.

\verb!M! is for files that have been modified in the working directory but not
recorded yet. The number of added and subtracted lines is also shown.

\verb!R!  is for files that have been removed, but the removal is not
recorded yet.

An explanation mark appears next to any option that has a conflict.

\begin{code}
dry_run = DarcsNoArgOption [] ["dry-run"] DryRun
          "don't actually take the action"

print_dry_run_message_and_exit :: String -> [DarcsFlag] -> [Patch] -> IO ()
print_dry_run_message_and_exit action opts patches =
  when (DryRun `elem` opts) $ do
    let showpatch p = if Verbose `elem` opts || Summary `elem` opts
                      then (human_friendly $ fromJust $ patch2patchinfo p)
                        $$ patch_summary p
                      else human_friendly $ fromJust $ patch2patchinfo p
    putDocLn $ text ("Would " ++ action ++ " the following changes:")
            $$ (vsep $ map showpatch patches)
    putStrLn ""
    putStrLn "Making no changes:  this is a dry run."
    exitWith ExitSuccess
\end{code}

\input{Resolution.lhs}

\begin{code}
noskip_boring = DarcsNoArgOption [] ["boring"]
                Boring "don't skip boring files"
allow_caseonly = DarcsNoArgOption [] ["case-ok"]
                 AllowCaseOnly "don't refuse to add files differing only in case"
diffflags = DarcsArgOption [] ["diff-opts"]
            DiffFlags "OPTIONS" "options to pass to diff"
\end{code}

\begin{code}
changes_format = DarcsMultipleChoiceOption
                 [DarcsNoArgOption [] ["context"]
                  (Context "") "give output suitable for get --context",
                  xmloutput,
                  human_readable
                 ]
changes_reverse = DarcsNoArgOption [] ["reverse"] Reverse
                  "show changes in reverse order"

human_readable = DarcsNoArgOption [] ["human-readable"]
                 HumanReadable "give human-readable output"
pipe = DarcsNoArgOption [] ["pipe"] Pipe "Expect to receive input from a pipe"

interactive =
    DarcsNoArgOption [] ["interactive"] Interactive
                         "Prompt user interactively"
all_patches = DarcsNoArgOption ['a'] ["all"] All "answer yes to all patches"

all_interactive = DarcsMultipleChoiceOption [all_patches, interactive]

all_gui_pipe_interactive
#ifdef HAVEWX
    = DarcsMultipleChoiceOption [all_patches,gui,pipe,interactive]
#else
    = DarcsMultipleChoiceOption [all_patches,pipe,interactive]
#endif

all_gui_interactive
#ifdef HAVEWX
    = DarcsMultipleChoiceOption [all_patches,gui,interactive]
#else
    = all_interactive
#endif

gui_pipe_interactive =
#ifdef HAVEWX
    DarcsMultipleChoiceOption [gui, pipe, interactive]
#else
    pipe_interactive
#endif

pipe_interactive =
    DarcsMultipleChoiceOption [pipe, interactive]

gui_interactive =
#ifdef HAVEWX
    DarcsMultipleChoiceOption [gui, interactive]
#else
    DarcsMultipleChoiceOption []
#endif
\end{code}

\begin{code}
edit_file :: String -> IO ExitCode
edit_file f = do
  ed <- get_editor
  system (ed++" "++f)
             `ortryrunning` system ("emacs "++f)
             `ortryrunning` system ("emacs -nw "++f)
             `ortryrunning` system ("nano "++f)
get_editor :: IO String
get_editor = getEnv "DARCS_EDITOR" `catchall`
             getEnv "DARCSEDITOR" `catchall`
             getEnv "VISUAL" `catchall`
             getEnv "EDITOR" `catchall` return "vi"
\end{code}

\begin{code}
list_files :: IO [String]
list_files = do s <- slurp_all_but_darcs "."
                skip_boring <- boring_file_filter
                return (skip_boring $ list_slurpy s)

list_unregistered_files :: IO [String]
list_unregistered_files =
    do s <- slurp_all_but_darcs "."
       skip_boring <- boring_file_filter
       regs <- slurp_pending "."
       return $ (skip_boring $ list_slurpy s) \\ (list_slurpy regs)

list_registered_files :: IO [String]
list_registered_files = list_slurpy `liftM` slurp_pending "."
\end{code}

\begin{code}
options_latex :: [DarcsOption] -> String
options_latex opts = "\\begin{tabular}{lll}\n"++
                     unlines (map option_latex opts)++
                     "\\end{tabular}\n"

latex_help :: String -> String
latex_help h
    = "\\begin{minipage}{7cm}\n\\raggedright\n" ++ h ++ "\\end{minipage}\n"

option_latex :: DarcsOption -> String
option_latex (DarcsNoArgOption a b _ h) =
    show_short_options a ++ show_long_options b ++ latex_help h ++ "\\\\"
option_latex (DarcsArgOption a b _ arg h) =
    show_short_options a ++
    show_long_options (map (++(" "++arg)) b) ++ latex_help h ++ "\\\\"
option_latex (DarcsMultipleChoiceOption os) =
    unlines (map option_latex os)

show_short_options :: [Char] -> String
show_short_options [] = "&"
show_short_options [c] = "\\verb!-"++[c]++"! &"
show_short_options (c:cs) = "\\verb!-"++[c]++"!,"++show_short_options cs

show_long_options :: [String] -> String
show_long_options [] = " &"
show_long_options [s] = "\\verb!--" ++ s ++ "! &"
show_long_options (s:ss)
    = "\\verb!--" ++ s ++ "!,"++ show_long_options ss

set_scripts_executable :: DarcsOption
set_scripts_executable = DarcsMultipleChoiceOption
                              [DarcsNoArgOption [] ["set-scripts-executable"] SetScriptsExecutable
                               "make scripts executable",
                               DarcsNoArgOption [] ["dont-set-scripts-executable"] DontSetScriptsExecutable
                               "don't make scripts executable"]

\end{code}

\begin{code}
relink, sibling :: DarcsOption
relink = DarcsNoArgOption [] ["relink"] Relink
         "Relink random internal data to a sibling"

sibling = DarcsArgOption [] ["sibling"] Sibling "URL"
          "specify a sibling directory"

flagsToSiblings :: [DarcsFlag] -> [String]
flagsToSiblings ((Sibling s) : l) = s : (flagsToSiblings l)
flagsToSiblings (_ : l) = flagsToSiblings l
flagsToSiblings [] = []
\end{code}

