/*
 * generate_map2.c
 *
 * crafted - a pud editor for the freecraft project.
 *
 * Copyright (C) 2001-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <gtk/gtk.h>

#include "crafted.h"

#include "tile.h"
#include "mini_map.h"

#include "craftedintl.h"



#define TABLE_SIZE 64
#define WEIGHT(T) ((2.0*fabs(T)-3.0)*(T)*(T)+1.0)

#ifndef RAND_MAX
#define G_MAXRAND G_MAXINT
#else
#define G_MAXRAND RAND_MAX
#endif

typedef struct _GimpVector2 GimpVector2;

struct _GimpVector2
{
  gdouble x, y;
};

gint seed;
static gdouble     offset, factor;
static gint        perm_tab[TABLE_SIZE];
static GimpVector2 grad_tab[TABLE_SIZE];
static gdouble water_percentage;
static gdouble dark_water_percentage;
static gdouble coast_percentage;

/*************************************************************************
*  solid_noise_init
**************************************************************************/
static void solid_noise_init(void)
{
  gint    i, j, k, t;
  gdouble m;

  /*  Force sane parameters  */
  seed = 1;

  /*  Define the pseudo-random number generator seed  */
  seed = time(NULL);
  srand (seed);

  offset=0.94;
  factor=0.526;

  /*  Initialize the permutation table  */
  for (i = 0; i < TABLE_SIZE; i++)
    perm_tab[i] = i;
  for (i = 0; i < (TABLE_SIZE >> 1); i++)
  {
    j = rand () % TABLE_SIZE;
    k = rand () % TABLE_SIZE;
    t = perm_tab[j];
    perm_tab[j] = perm_tab[k];
    perm_tab[k] = t;
  }

  /*  Initialize the gradient table  */
  for (i = 0; i < TABLE_SIZE; i++)
  {
    do
    {
      grad_tab[i].x = (double)(rand () - (G_MAXRAND >> 1)) / (G_MAXRAND >> 1);
      grad_tab[i].y = (double)(rand () - (G_MAXRAND >> 1)) / (G_MAXRAND >> 1);
      m = grad_tab[i].x * grad_tab[i].x + grad_tab[i].y * grad_tab[i].y;
    } while (m == 0.0 || m > 1.0);
    m = 1.0 / sqrt(m);
    grad_tab[i].x *= m;
    grad_tab[i].y *= m;
  }
}

/*************************************************************************
*  plain_noise
**************************************************************************/
static gdouble plain_noise(gdouble x,
                           gdouble y,
                           guint   s)
{
  GimpVector2 v;
  gint        a, b, i, j, n;
  gdouble     sum;

  sum = 0.0;
  x *= s;
  y *= s;
  a = (int) floor (x);
  b = (int) floor (y);

  for (i = 0; i < 2; i++)
    for (j = 0; j < 2; j++)
    {
      n = perm_tab[(a + i + perm_tab[(b + j) % TABLE_SIZE]) % TABLE_SIZE];
      v.x = x - a - i;
      v.y = y - b - j;
      sum += WEIGHT(v.x) * WEIGHT(v.y) * (grad_tab[n].x * v.x + grad_tab[n].y * v.y);
    }
  return sum / s;
}

/*************************************************************************
*  noise
**************************************************************************/
static gdouble noise(gdouble x, gdouble y)
{
  gint i;
  guint s;
  gdouble sum;

  s = 1;
  sum = 0.0;
  x *= 4.0;
  y *= 4.0;

  sum += plain_noise (x, y, s);
  s <<= 1;

  return (sum+offset)*factor;
}

/*************************************************************************
*  solid_noise
**************************************************************************/
static void solid_noise(void)
{
  gint    col, row, idx, tool;
  gdouble val;

  solid_noise_init();
  progressbar_init();
  for (row = 0 ; row<Pud.height ; row++)
  {
    for (col = 0 ; col<Pud.width ; col++)
    {
      val = noise ((double) col / Pud.width, (double) row / Pud.height);
      if (val<water_percentage)
      { /* we are in water... */
        if (val<dark_water_percentage*water_percentage)
          tool = 2;
        else
          tool = 1;
        idx = get_solid_index(tool);
        affect_exact_tile(col, row, (tool<<4)+idx);
        check_around(col, row, tool);
      } else
      {
        if (val<water_percentage+(1-water_percentage)*coast_percentage)
        {
          tool = 3;
          idx = get_solid_index(tool);
          affect_exact_tile(col, row, (tool<<4)+idx);
          check_around(col, row, tool);
        }
      }
    }
    progressbar_update((gdouble) row / (gdouble)Pud.height, "Solid noise generation");
  }
  progressbar_finish();
}

/*************************************************************************
*  generate_map_do
**************************************************************************/
static void generate_map_do(void)
{
  gint i, idx, tool, x, y;
  g_print("Generating map!\n");

  /* first pass: all grass! */
  for (y=0 ; y<Pud.height ; y++)
    for (x=0 ; x<Pud.width ; x++)
    {
      idx = get_solid_index(5);
      affect_exact_tile(x,y, 0x50+idx);
    }
  /* second pass: adding some water */
  solid_noise();
  update_minimap();
}

void double_adjustment_update(GtkAdjustment *adjustment,
                              gpointer       data)
{
  gdouble *val;

  val = (gdouble *)data;
  *val = adjustment->value;
}

/*************************************************************************
*  generate_map2
**************************************************************************/
void generate_map2(void)
{
  static GtkWidget      *gm_dialog = NULL;
  GtkWidget     *hbbox, *button, *vbox, *label, *scale;
  GtkAdjustment *adj;

  if (gm_dialog)
  {
    if (!GTK_WIDGET_VISIBLE(gm_dialog))
    {
      gtk_widget_show(gm_dialog);
    } else
    {
      gdk_window_raise(gm_dialog->window);
    }
    return;
  }

  gm_dialog = crafted_dialog_new(_("Auto-map settinfs"),
                                 "crafted",
                                 GTK_WIN_POS_MOUSE,
                                 FALSE, FALSE, TRUE,
                                 _("Apply"), generate_map_do,
                                 NULL, NULL, NULL, TRUE, FALSE,
                                 _("Close"), gtk_widget_destroy,
                                 NULL, 1, NULL, FALSE, TRUE, NULL);
  gtk_signal_connect(GTK_OBJECT(gm_dialog), "destroy",
                     (GtkSignalFunc)gtk_widget_destroyed,
                     &gm_dialog);

  /* The main part */
  vbox = gtk_vbox_new(FALSE, 4);
  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(gm_dialog)->vbox), vbox);

  /* water percentage */
  label = gtk_label_new(_("water percentage"));
  gtk_box_pack_start_defaults(GTK_BOX(vbox), label);
  water_percentage = 0.50;
  adj = (GtkAdjustment *)gtk_adjustment_new(water_percentage,
                                            0.0, 1.0, 0.01, 0.01, 0.01);
  scale = gtk_hscale_new(adj);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), scale);
  gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                     (GtkSignalFunc)double_adjustment_update,
                     &water_percentage);

  /* percentage of dark water in water */
  label = gtk_label_new(_("dark water percentage"));
  gtk_box_pack_start_defaults(GTK_BOX(vbox), label);
  dark_water_percentage = 0.80;
  adj = (GtkAdjustment *)gtk_adjustment_new(dark_water_percentage,
                                            0.0, 1.0, 0.01, 0.01, 0.01);
  scale = gtk_hscale_new(adj);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), scale);
  gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                     (GtkSignalFunc)double_adjustment_update,
                     &dark_water_percentage);
  
  /* percentage of coast in ground */
  label = gtk_label_new(_("coast percentage"));
  gtk_box_pack_start_defaults(GTK_BOX(vbox), label);
  coast_percentage = 0.30;
  adj = (GtkAdjustment *)gtk_adjustment_new(coast_percentage,
                                            0.0, 1.0, 0.01, 0.01, 0.01);
  scale = gtk_hscale_new(adj);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), scale);
  gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                     (GtkSignalFunc)double_adjustment_update,
                     &coast_percentage);

  gtk_widget_show_all(gm_dialog);
}
