/*
 *  Copyright (C) 2002,2003,2004  Mattia Dongili<dongili@supereva.it>
 *                                George Staikos <staikos@0wned.org>
 *
 *  2004.01.27
 *  - fixed program matching routines by plasmagunman@users.sourceforge.net
 *
 *  2004.01.04
 *  - fixed pidfile permission on create by Michael Scherer
 *
 *  2004.01.04
 *  - fixed PPC problem in strcmp return value conversion 
 *    by Dominik 'Aeneas' Schnitzer
 *    
 *  2003.16.08
 *  - added support for cpu monitoring, base code by Dietz Proepper and minor
 *    fixes by Mattia Dongili
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "cpufreqd.h"
#include "main.h"
#include "config_parser.h"
#include "sys_check.h"

/* flags */
unsigned int                       force_exit; /* force exit from the main loop by signal handlers */
unsigned int                       force_reinit; /* issue a reinitalization */
unsigned int                       log_opened; /* syslog already opened */

struct string_list *            running_programs;
rule *                          active_rule;
general *                       configuration;
int                             delay_rule;
int                             log_level;
void *                          libsys;   /* handle for the loaded shared library  */
int (*libsys_scan_system_info)  (sys_info *); /* function pointer */
int (*libsys_get_cpu)           (void); /* function pointer */

int numeric_entry(const struct dirent *d);
  
/******
 *
 * MAIN 
 *
 ******/
int main (int argc, char *argv[]) {

  struct sigaction signal_action;

  /* init global vars */
  running_programs  = string_list_new();
  active_rule       = 0L;
  force_exit        = 0;
  force_reinit      = 0;
  
  configuration = (general *) malloc(sizeof(general));
  memset(configuration, 0, sizeof(general));
  /* set defaults */
  snprintf(configuration->config_file, 512,  CPUFREQD_CONFIG);
  snprintf(configuration->pidfile, 512, CPUFREQD_PIDFILE);
  snprintf(configuration->pm_plugin, 512, DEFAULT_PMPLUGIN);
  configuration->log_level = DEFAULT_VERBOSITY; /* LOG_WARNING */

  /* parse command line (done later than parse_config in order to 
   * be able to overwrite common parameters)
   */
  if (read_args(argc,argv)<0)
    return 1;
  
  /* check cpufreq interface */
  if (find_cpufreq_interface(configuration)<0) {
    cp_log(LOG_CRIT, "Unable to find a cpufreq interface, please ensure to have cpufreq enabled in the running kernel.\nExiting.\n");
    return 1;
  }

  /* get number of cpus */
  if (get_cpu_num(configuration)<0) {
    cp_log(LOG_CRIT, "Unable to find any CPUs, please check your procfs.\nExiting.\n");
    return 1;
  }
 
  /* read configuration */
  if (parse_config(configuration) < 0) {
    cp_log(LOG_CRIT, "Failed to parse configuration file %s.\nExiting.\n", configuration->config_file);
    return 1;
  }
  
  /* check perms */
  if (geteuid() != 0) {
    cp_log(LOG_CRIT, "%s: must be run as root.\n", argv[0]);
    return 1;
  }

  cp_log(LOG_NOTICE, "Starting operations.\n");

  /* setup signal handlers */
  sigemptyset(&signal_action.sa_mask);
  sigaddset(&signal_action.sa_mask, SIGTERM);
  sigaddset(&signal_action.sa_mask, SIGINT);
  sigaddset(&signal_action.sa_mask, SIGHUP);
  signal_action.sa_flags = 0;
                             
  signal_action.sa_handler = term_handler;
  sigaction(SIGTERM, &signal_action, 0);

  signal_action.sa_handler = int_handler;
  sigaction(SIGINT, &signal_action, 0);

  signal_action.sa_handler = hup_handler;
  sigaction(SIGHUP, &signal_action, 0);
  
  /* open libraries */
  if (load_libraries() < 0) {
    cp_log(LOG_CRIT, "Cannot open shared libraries.\nExiting.\n");
    return 1;
  }

  /* go background and setup stderr, stdout and stdin */
  if (!configuration->no_daemon && daemonize()<0) {
    cp_log(LOG_CRIT, "Unable to go to background.\nExiting.\n");
    return 1;
  }
 
  /* write pidfile */
  if (write_cpufreqd_pid() < 0) {
    cp_log(LOG_CRIT, "Cannot write pidfile %s.\nExiting.\n", configuration->pidfile);
    return 1;
  }
  cp_log(LOG_INFO, "pid: %d\n", getpid());

  main_loop();

  close_libraries();

  if (clear_cpufreqd_pid() < 0) {
    cp_log(LOG_ERR, "Unable to remove %s.\n", configuration->pidfile);
    return 1;
  }
  
  /* free configuration */
  free_config(configuration);
  free(configuration);
  
  /* TODO: this call to close log shound not sit here... */
  closelog();
  return 0;
}

/* int load_libraries(void)
 *
 * Open shared libraries
 */
int load_libraries(void) {
  int init_ok;
  
  void (*libsys_preinit)(void (*fn_ptr)(const int prio, const char *fmt, ...), general *config); /* set_log function pointer */
  int (*libsys_init)(void); /* init function pointer */
  
  char libsysname[512];
  const char* error;    /* pointer to an error message, if any */

  snprintf(libsysname, 512, CPUFREQD_LIBDIR"libsys_%s.so", configuration->pm_plugin);
 
  libsys = dlopen(libsysname, RTLD_LAZY);
  if (!libsys) {
    cp_log(LOG_ERR, "load_libraries(): %s\n", dlerror());
    return -1;
  }

  /* preinit plugin */
  libsys_preinit = (void (*)(void (*fn_ptr)(const int prio, const char *fmt, ...), general *config))dlsym(libsys, "libsys_preinit");
  error = dlerror();
  if (error) {
    cp_log(LOG_ERR, "load_libraries(): %s\n", error);
    return -1;
  }
  (*libsys_preinit)(&cp_log, configuration);
  
  /* init library */
  libsys_init = (int (*)(void))dlsym(libsys, "libsys_init");
  error = dlerror();
  if (error) {
    cp_log(LOG_ERR, "load_libraries(): %s\n", error);
    return -1;
  }
  init_ok = (*libsys_init)();
  if (init_ok < 0) {
    cp_log(LOG_ERR, "load_libraries(): cannot initizalize PM plugin.\n");
    return -1;
  }
  
  /* get function pointers */
  /* scan_sistem_info */
  libsys_scan_system_info = (int (*)(sys_info*))dlsym(libsys, "scan_system_info");
  error = dlerror();
  if (error) {
    cp_log(LOG_ERR, "load_libraries(): %s\n", error);
    return -1;
  }
  /* get_cpu */
  libsys_get_cpu = (int (*)(void))dlsym(libsys, "get_cpu");
  error = dlerror();
  if (error) {
    cp_log(LOG_ERR, "load_libraries(): %s\n", error);
    return -1;
  }
        
  return 0;
              
}

/* void close_libraries(void)
 *
 * Close shared libraries
 */
void close_libraries(void) {
  /* finalize library */
  const char* error;    /* pointer to an error message, if any */
  void (*libsys_fini)(void) = (void (*)(void))dlsym(libsys, "libsys_fini");
  error = dlerror();
  if (error) { 
    cp_log(LOG_ERR, "close_libraries(): %s\n", error);
    return;
  } else
    (*libsys_fini)();
                
  dlclose(libsys);
}

/* void reinit (void)
 *
 * reinit the daemon
 *
 */
void reinit (void) {

  free_config(configuration);
  active_rule = 0L;

  /* close shared library */
  close_libraries();
  
  /* remove pidfile */
  if (clear_cpufreqd_pid() < 0) {
    cp_log(LOG_CRIT, "Cannot remove pidfile %s.\nAborting.\n", configuration->pidfile);
    force_exit = 1;
    return;
  }
    
  /* check cpufreq interface */
  if (find_cpufreq_interface(configuration)<0) {
    cp_log(LOG_CRIT, "Unable to find a cpufreq interface, please ensure to have cpufreq enabled in the running kernel.\nExiting.\n");
    force_exit = 1;
    return;
  }
  
  /* re-read configuration */
  if (parse_config(configuration) < 0) {
    cp_log(LOG_CRIT, "Unable to parse configuration file %s.\nAborting.\n", configuration->config_file);
    force_exit = 1;
    return;
  }
  
  /* load shared lib */
  if (load_libraries() < 0) {
    cp_log(LOG_CRIT, "Cannot open shared libraries.\nAborting.\n");
    force_exit = 1;
    return;
  }

  /* write pidfile */
  if (write_cpufreqd_pid() < 0) {
    cp_log(LOG_CRIT, "Cannot write pidfile %s.\nAborting.\n", configuration->pidfile);
    force_exit = 1;
    return;
  }
  force_reinit = 0;
}

/* void hup_handler(int signo)
 * 
 * SIGUP handler, issues a reinit at the end of the loop.
 *
 */
void hup_handler(int signo) {
  cp_log(LOG_NOTICE, "Caught HUP signal (%s), reloading configuration file.\n", strsignal(signo));

  force_reinit = 1;
}

/* void int_handler(int signo)
 *
 * SIGINT handler.
 */
void int_handler(int signo) {
  cp_log(LOG_NOTICE, "Caught INT signal (%s).\n", strsignal(signo));

  force_exit = 1;
}


/* void term_handler(int signo)
 *
 * SIGTERM handler.
 */
void term_handler(int signo) {
  cp_log(LOG_NOTICE, "Caught TERM signal (%s).\n", strsignal(signo));

  force_exit = 1;
}

/* int main_loop(void)
 *
 * The daemon loop.
 */
int main_loop(void) {
  int profile_changed = 0; /* to avoid writing a new policy if not really necessary */
  rule *r_temp = 0L;
  sys_info *_system = 0L;
  _system = (sys_info *)malloc(sizeof(sys_info)); 
  memset(_system, 0, sizeof(sys_info));
  
  while (!force_exit) {

    /* reset system_info object */
    _system->ac = 0;
    _system->has_battery = 0;
    _system->temp = 0;
    _system->battery_percent = 0;
    _system->battery_time = 0;
    _system->cpu_percent = 0;
    
    /* scan_system_info */
    if ((*libsys_scan_system_info)(_system)<0) {
      cp_log(LOG_ERR, "main_loop(): Can't get system info. Exiting.\n");
      return -1;
    }
    
    /* get_cpu */
    _system->cpu_percent = (*libsys_get_cpu)();
    if (_system->cpu_percent == -1) {
      cp_log(LOG_ERR, "main_loop(): Can't get CPU usage. Exiting.\n");
      return -1;
    }

    /* read running programs */
    if (get_running_programs() <= 0) {
      cp_log(LOG_WARNING, "main_loop(): WARNING: get_running_programs() returned a negative integer.\n");
    }

    if (_system && active_rule) {
      /* check if we need a new rule */
      if (active_rule->hits <= (MATCH_CPU | MATCH_AC | MATCH_BATTERY | MATCH_PROG)) {

        /* find a valid rule */
        r_temp = find_valid_rule(_system, configuration->rules);

        /* if no valid rule has been found leave system untouched and go on */
        if (r_temp == NULL) {
          cp_log(LOG_WARNING, "main_loop(): no valid rule found check your config file against current system state\n");
          continue;
        }
        
        /* check if the new rule is different from the currently active one */
        if (strcmp(r_temp->name, active_rule->name)!=0) {
          /* set profile_changed to avoid writing a new policy if not necessary */
          profile_changed = strcmp(r_temp->prof->name, active_rule->prof->name);
            
          active_rule = r_temp;
          /* set the associated policy */
          /* no need to write it if it is the same profile as before */
          if (profile_changed) {
            if (active_rule && set_policy(active_rule->prof) < 0) {
              return -1;
            }
            cp_log(LOG_NOTICE, "main_loop(): profile set \"%s\" for rule \"%s\".\n",
                active_rule->prof->name, active_rule->name);
          } /* end if profile changed */
          else {
            cp_log(LOG_NOTICE, "main_loop(): profile %s already set, skipping.\n", 
                active_rule->prof->name);
          }
        } /* end if rule changed */
      } /* end if rule no more valid */
    } else {
      /* If it the first time we read system status */
      
      /* find a valid rule */
      active_rule = find_valid_rule(_system, configuration->rules);
      
      /* if no valid rule has been found leave system untouched and go on */
      if (active_rule==NULL) {
        cp_log(LOG_WARNING, "main_loop(): no valid rule found check your config file against current system state\n");
        continue;
      }
      
      /* set the associated policy */
      if (set_policy(active_rule->prof) < 0) {
        cp_log(LOG_ERR, "main_loop(): cannot set current policy. Exinting\n");
        return -1;
      }
    
      cp_log(LOG_NOTICE, "main_loop(): profile set \"%s\" for rule \"%s\".\n",
             active_rule->prof->name, active_rule->name);
    }
    
    if (force_reinit)
      reinit();
    else
      sleep(configuration->poll_interval);
  } /* end loop */

  free(_system);
  return 0;
}

/* int write_cpufreqd_pid(void)
 *
 * Writes the pid file.
 *
 * Returns 0 on success, -1 otherwise.
 *
 */
int write_cpufreqd_pid(void) {
  FILE *pid;
  struct stat sb;
  int rc = 0;
  
  /* check if old pidfile is still there */
  rc = stat(configuration->pidfile, &sb);
  if (rc == 0) {
    char oldpid[10];
    pid = fopen(configuration->pidfile, "r");
    /* see if there is a pid already */
    if (fscanf(pid, "%s", oldpid) == 1) {
      FILE *fd;
      char old_pidfile[256];
      char old_cmdline[256];

      snprintf(old_pidfile, 256, "/proc/%s/cmdline", oldpid);
      fd = fopen(old_pidfile, "r");
      /* if the file exists see if there's another cpufreqd process running */
      if (fd && fscanf(fd, "%s", old_cmdline) == 1 && strstr(old_cmdline,"cpufreqd") != NULL) {
        cp_log(LOG_ERR, "write_cpufreqd_pid(): the daemon is already running.\n");
        fclose(fd);
        fclose(pid);
        return -1;
      }
    }
    fclose(pid);
  }

  /* set permission mask 033 */
  umask( S_IXGRP | S_IXOTH | S_IWOTH | S_IWGRP );

  pid = fopen(configuration->pidfile, "w");
  if (!pid) {
    cp_log(LOG_ERR, "write_cpufreqd_pid(): %s: %s.\n", configuration->pidfile, strerror(errno));
    return -1;
  }

  if (!fprintf(pid, "%d", getpid())) {
    cp_log(LOG_ERR, "write_cpufreqd_pid(): cannot write pid %d.\n", getpid());
    fclose(pid);
    clear_cpufreqd_pid();
    return -1;
  }

  fclose(pid);
  return 0;
}

/* int clear_cpufreqd_pid(void)
 *
 * Removes pid file
 *
 * Returns 0 on success, -1 otherwise.
 */
int clear_cpufreqd_pid(void) {
        
  if (unlink(configuration->pidfile) < 0) {
    cp_log(LOG_ERR, "clear_cpufreqd_pid(): error while removing %s: %s.\n", configuration->pidfile, strerror(errno));
    return -1;
  }

  return 0;
}

/* void set_policy(profile *prof)
 *
 * Sets a new policy through /proc or sysfs interface
 *
 * Returns 0 on success, -1 otherwise
 */
int set_policy(profile *prof) {
  FILE *fp;
  char path[256];
  
  if (!prof)
    return -1;

  /* /sys/devices/system/cpu* */
  if (configuration->has_sysfs) {
    int cpu;

    for(cpu=0; cpu < configuration->cpu_num; cpu++) {
      /* governor */
      snprintf(path, 256, CPUFREQ_SYSFS_INTERFACE_POLICY, cpu);
      fp = fopen(path, "w");
      if (!fp) {
        cp_log(LOG_ERR, "set_policy(): %s: %s\n", path, strerror(errno));
        return -1;
      }
      fprintf(fp, "%s", prof->policy_name);
      fclose(fp);

      /* max speed */
      snprintf(path, 256, CPUFREQ_SYSFS_INTERFACE_MAX, cpu);
      fp = fopen(path, "w");
      if (!fp) {
        cp_log(LOG_ERR, "set_policy(): %s: %s\n", path, strerror(errno));
        return -1;
      }
      fprintf(fp, "%li", prof->max_freq);
      fclose(fp);

      /* min speed */
      snprintf(path, 256, CPUFREQ_SYSFS_INTERFACE_MIN, cpu);
      fp = fopen(path, "w");
      if (!fp) {
        cp_log(LOG_ERR, "set_policy(): %s: %s\n", path, strerror(errno));
        return -1;
      }
      fprintf(fp, "%li", prof->min_freq);
      fclose(fp);

      cp_log(LOG_NOTICE, "set_policy(): %s CPU%i %li %li.\n", prof->policy_name, cpu, prof->min_freq, prof->max_freq);
    }

  } else {

    /* /proc/cpufreq */
    fp = fopen(CPUFREQ_PROC_INTERFACE, "w");
    if (!fp) {
      cp_log(LOG_ERR, "set_policy(): %s: %s\n", CPUFREQ_PROC_INTERFACE, strerror(errno));
      return -1;
    }

    cp_log(LOG_NOTICE, "set_policy(): %li%c%li%c%s - profile name: %s.\n",
                     prof->min_freq, 
                     prof->sep, prof->max_freq,
                     prof->sep, prof->policy_name, 
                     prof->name);

    fprintf(fp, "%li%c%li%c%s", 
                     prof->min_freq, 
                     prof->sep, prof->max_freq,
                     prof->sep, prof->policy_name);

    fclose(fp);
  }
  return 0;
}


/* rule *find_valid_rule(const sys_info *s, rule *r)
 * 
 * Loops through a rule linked list r and invokes is_rule_valid
 * for each rule.
 *
 * Returns a pointer the the first valid rule or NULL if
 * no valid rules are found.
 */
rule *find_valid_rule(const sys_info *s, rule *r) {

  rule *r_iter = 0L;
  rule *ret = 0L;
  int necessary_hits = (MATCH_CPU | MATCH_AC | MATCH_BATTERY | MATCH_PROG);
  int hits_prev = 0L;
  
  if (active_rule != NULL && active_rule->hits > 0)
    necessary_hits = active_rule->hits;

  r_iter = r;
  while (r_iter != NULL) {
    r_iter->hits = get_rule_hits(s, r_iter);
    r_iter = r_iter->next;
  }
  
  while (necessary_hits > 0) {
    r_iter = r;
    
    cp_log(LOG_DEBUG, "find_valid_rule(): necessary hits: %d\n", necessary_hits);

    while (r_iter != NULL) {
      /* TODO: add precedence factor to rules having he same score */
      if (r_iter->hits > hits_prev && r_iter->hits >= necessary_hits) {
        hits_prev = r_iter->hits;
        ret = r_iter;
      }
      r_iter = r_iter->next;
    }
    if (ret) {
      cp_log(LOG_INFO, "find_valid_rule(): name %s, hits: %d, necessary: %d\n", 
                       ret->name, hits_prev, necessary_hits);
      return ret;
    }
    necessary_hits >>= 1;
  }
  
  cp_log(LOG_WARNING, "find_valid_rule(): WARNING: no valid rule found for current"
		  " system state!\n\tbattery: state %s - %d - ac: %s - cpu: %d\n",
                  s->has_battery?"present":"absent", 
                  s->battery_percent, 
                  s->ac?"on-line":"off-line",
                  s->cpu_percent);

  return NULL;
}

/*  inline int intersects (const struct interval *i, const int *val) 
 *
 *  Check if rule's cpu_interval intersects actual cpu usage
 *  weigthing nice_time based on rule's settings
 *
 *  Returns 1 if the interval intersects 0 otherwise
 */
/*
int intersects (const struct interval *i, const int *val) {
  return (*val >= i->min && *val <= i->max);
}
*/

/*
 *
 */
/*
int match_cpu_usage (sys_info *si, cpu_interval *cpu, int nice_scale) {
  int delta_activity = 0, weighted_activity = 0;
  
  weighted_activity = si->cur_sys_activity + si->cur_nice_activity / nice_scale;
  delta_activity = weighted_activity - si->old_weighted_activity;
  old_weighted_activity = weighted_activity;

  cp_log(LOG_DEBUG,
         "get_cpu(): CPU delta_activity=%d delta_time=%d weighted_activity=%d.\n",
         delta_activity, delta_time, weighted_activity);

  weighted_activity = c_user + c_nice / 3 + c_sys;
  
  if ( delta_activity > delta_time || delta_time <= 0) {
    perc = 100;
  } else {
    perc = delta_activity * 100 / delta_time;
  }
  cp_log(LOG_INFO, "get_cpu(): CPU usage = %d.\n", perc);
  return 0;
}
*/

/* int get_rule_hits(const sys_info *si, const rule *r, int necessary_hits)
 * 
 * Check if the rule r is valid against sytem_info si.
 *
 * Precedence is AC, CPU, BATTERY, PROGRAMS
 * 
 * Returns the number of hits the rule matched with, 0 if it doesn't match.
 *
 */
int get_rule_hits(const sys_info *si, const rule *r) {
  int hits = 0, debug_count = 0;
  
  /* check battery */
  if (si->battery_percent >= r->bat->min && si->battery_percent <= r->bat->max) { 
    hits |= MATCH_BATTERY;
  }

  /* check cpu */
  if (si->cpu_percent >= r->cpu->min && si->cpu_percent <= r->cpu->max) {
    hits |= MATCH_CPU;
  }
    
  
  /* check AC */
  if (si->ac == r->ac) {
    hits |= MATCH_AC;
  }
  
  /* check running programs */
  if (r->program_list) {
    struct string_node *pr_iter = 0L, *pr_config = 0L;
    pr_iter = running_programs->first;
    
    /* go through running programs */
    while (pr_iter && !(MATCH_PROG & hits)) {
      pr_config = r->program_list->first;
      /* go through configured programs */
      while (pr_config && !(MATCH_PROG & hits)) {
        /* first match the configured program name with the 
           basename of the running cmdline */
        if (strcmp(pr_config->string, basename(pr_iter->string)) == 0) {
          cp_log(LOG_DEBUG, "get_rule_hits(): %s matched requested program \"%s\" against \"%s\"\n",
                          r->name, pr_config->string,  basename(pr_iter->string));
          hits |= MATCH_PROG;
        } 

        /* try to match configured program name with the running cmdline */
        if (strcmp(pr_config->string, pr_iter->string) == 0) {
          cp_log(LOG_DEBUG, "get_rule_hits(): %s matched requested program \"%s\" against \"%s\"\n",
                          r->name, pr_config->string,  pr_iter->string);
          hits |= MATCH_PROG;
        } 
        pr_config = pr_config->next;
      } /* end configured programs loop */
      pr_iter = pr_iter->next;
      debug_count++;
    } /* end full program list loop */
    cp_log(LOG_DEBUG, "get_rule_hits(): checked %d processes\n", debug_count);
  }
  cp_log(LOG_DEBUG, "get_rule_hits(): hits for %s: %d\n", r->name, hits);
  return hits;
}

/* int numeric_entry(const struct dirent *d) 
 *
 * Select function for scandir()
 *
 */
int numeric_entry(const struct dirent *d) {
  return isdigit(d->d_name[0]);
}

/* int get_running_programs(void)
 *
 * looks for running programs and fills the 
 * global struct running_programs.
 *
 * Returns the length of the newly created list.
 */
int get_running_programs(void) {
  
  struct dirent **namelist;
  int n=0, ret=0;
  FILE *fp;
  char file[256];
  struct string_node *s_temp = 0L;

  n = scandir("/proc", &namelist, numeric_entry, alphasort);
  if (n < 0)
    cp_log(LOG_ERR, "get_running_programs() - scandir: %s\n", strerror(errno));
  else {
    /* reuse the old running_programs string_list */
    s_temp = running_programs->first;
    
    while(n--) {
      snprintf(file, 255, "/proc/%s/cmdline", namelist[n]->d_name);    
      free(namelist[n]);
      fp = fopen(file, "r");
      
      if (!fp) /* probably this process has disappeared while scanning, don't worry */
        continue;

      if (!s_temp) {
        s_temp = string_node_new();
        string_list_append(running_programs, s_temp);
      }
      /* reset string */
      s_temp->string[0] = '\0';
      /* read cmdline */
      if (!fgets(s_temp->string, 255, fp)) {
        fclose(fp);
        /* this entry is not readable, go on */
        continue;
      }
      fclose(fp);
      
      s_temp = s_temp->next;

      ret++;
    }
  }
  free(namelist);
  cp_log(LOG_DEBUG, "get_running_programs(): read %d processes\n", ret);
  cp_log(LOG_DEBUG, "get_running_programs(): freed %d structs\n", string_list_free_sublist(running_programs, s_temp));
  return ret;
}

/*
 * Creates a child and detach from tty
 */
int daemonize (void) {

  cp_log(LOG_INFO, "daemonize(): going background, bye.\n");

  switch (fork ()) {
    case -1:
      cp_log(LOG_ERR, "deamonize(): fork: %s\n", strerror (errno));
      return -1;
    case 0:
      /* child */
      break;
    default:
      /* parent */
      exit (0);
  }

  /* disconnect */
  setsid ();
  umask (0);

  /* set up stdout to log and stderr,stdin to /dev/null */
  if (freopen("/dev/null", "r", stdin) == NULL) {
    cp_log(LOG_ERR, "deamonize(): %s: %s.\n", "/dev/null", strerror(errno));
    return -1;
  }
  
  if (freopen("/dev/null", "w", stdout) == NULL) {
    cp_log(LOG_ERR, "deamonize(): %s: %s.\n", "/dev/null", strerror(errno));
    return -1;
  }
  
  if (freopen("/dev/null", "w", stderr) == NULL) {
    cp_log(LOG_ERR, "deamonize(): %s: %s.\n", "/dev/null", strerror(errno));
    return -1;
  }

  /* get outta the way */
  chdir ("/");
  return 0;
}

/*
 * Reads command line arguments
 *
 */
int read_args (int argc, char *argv[]) {
  
  static struct option long_options[] = {
    { "help", 0, 0, 'h' }, 
    { "version", 0, 0, 'v' }, 
    { "file", 1, 0, 'f' },
    { "no-daemon", 0, 0, 'D' },
    { "verbosity", 1, 0, 'V' },
    { 0, 0, 0, 0 }, 
  };
  int ch,option_index = 0;

  while ((ch = getopt_long(argc, argv, "hvf:DV:", long_options, &option_index)) != -1) {
    switch (ch) {
      case '?':
      case 'h':
        print_help(argv[0]);
        return -1;
      case 'v':
        print_version(argv[0]);
        return -1;
      case 'f':
        if (realpath(optarg, configuration->config_file) == NULL) {
          cp_log(LOG_ERR, "read_args(): %s: %s.\n", optarg, strerror(errno));
          return -1;
        }
        break;
      case 'D':
        configuration->no_daemon = 1;
        break;
      case 'V':
        configuration->log_level = atoi(optarg);
        if (configuration->log_level>7) {
          configuration->log_level = 7;
        } else if (configuration->log_level<0) {
          configuration->log_level = 0;
        }
        configuration->log_level_overridden = 1;
        break;
      default:
        break;
    }
  }
  return 0;
}

/*
 * Prints program version
 */
void print_version(const char *me) {
  printf("%s version "_CPUFREQD_VERSION_".\n", me);
  printf("Copyright 2002,2003,2004 Mattia Dongili <dongili@supereva.it>\n"
	    	 "                         George Staikos <staikos@0wned.org>\n");
}

/*
 * Prints program help
 *
 */
void print_help(const char *me) {
  printf("Usage: %s [OPTION]...\n\n"
"  -h, --help                   display this help and exit\n"
"  -v, --version                display version information and exit\n"
"  -f, --file                   config file (default: "CPUFREQD_CONFIG")\n"
"  -D, --no-daemon              stay in foreground and print log to stdout (used to debug)\n"
"  -V, --verbosity              verbosity level from 0 (less verbose) to 7 (most verbose)\n"
"\n"
"Report bugs to Mattia Dongili <dongili@supereva.it>.\n", me);
}

/* 
 * Logger infrastructure. It reuses the same priorities as 
 * sys/syslog.h because it's easier to manage.
 *
 *	LOG_EMERG	  0	
 *  LOG_ALERT	  1
 *  LOG_CRIT	  2
 *  LOG_ERR		  3
 *  LOG_WARNING	4
 *  LOG_NOTICE	5
 *  LOG_INFO	  6
 *  LOG_DEBUG	  7
 *
 */
void cp_log(int prio, const char *fmt, ...) {
  va_list argp;
  
  /* do we need to write? */
  if (configuration->log_level < prio)
    return;
  
  va_start(argp, fmt);

  if (configuration->no_daemon) {
    if (configuration->log_level <= LOG_ERR) {
      vfprintf(stderr, fmt, argp);
      /* fflush(stderr); */
    } else {
      vfprintf(stdout, fmt, argp);
      /* fflush(stdout); */
    }
  } else {
    if (!log_opened) {
      /* open syslog */
      openlog("cpufreqd", LOG_CONS, LOG_DAEMON);
      log_opened = 1;
    }
    vsyslog(prio, fmt, argp);
    if (configuration->log_level <= LOG_ERR) {
      vfprintf(stderr, fmt, argp);
      /* fflush(stderr); */
    }
  }
  va_end(argp);
}

