;;;; syntax-case.scm - New version of syntax-case - felix
;
; This file is mainly by Felix L. Winkelmann, but contains a good deal of code
; from the portable syntax case (psyntax) distribution.


(declare
 (unit syntax-case)
 (disable-interrupts)
 (number-type fixnum)
 (usual-integrations) 
 (compress-literals 50 ##syncase#init-literals) )

(cond-expand
 [paranoia]
 [else
  (declare
    (no-bound-checks) ) ] )

(declare
  (hide properties putprop remprop getprop srfi-0-def std-defs enable-strict-hl-letrec
	eval-hook expand-install-hook loaded-modules) )


(define ##syncase#current-export-prefix #f)

(define loaded-modules '())

(define ##syncase#visit
  (let ([with-input-from-file with-input-from-file]
	[display display]
	[evalh ##sys#eval-handler]
	[load-verbose load-verbose] )
    (define (walk x)
;      (print "[walk: " x)
      (when (pair? x)
	(let ([hd (car x)])
	  (if (symbol? hd)
	      (case hd
		[(quote ##core#global-ref)]
		[(lambda ##core#loop-lambda) (for-each walk (cddr x))]
		[(##core#named-lambda) (for-each walk (cdddr x))]
		[(##core#require-for-syntax)
		 (for-each (lambda (id) (##sys#require ((evalh) id))) (cdr x)) ]
		[(##core#require-extension)
		 (for-each
		  (lambda (id)
		    (let-values ([(x y) (##sys#do-the-right-thing (cadr id) #f)])
		      (walk x) ) )
		  (cdr x) ) ]		  
		[(##core#elaborationtimeonly ##core#elaborationtimetoo)
		 ((evalh) (cadr x)) ]
		[else (for-each walk (cdr x))] )
	      (for-each walk x) ) ) ) )
    (lambda (filename)
      (when (load-verbose)
	(display "; visiting ")
	(display filename)
	(display " ...\n") )
      (with-input-from-file filename
	(lambda ()
	  (let loop ()
	    (let ([x (read)]) 
	      (unless (eof-object? x)
		(walk (##sys#compiler-toplevel-macroexpand-hook x))
		(loop) ) ) ) ) ) ) ) )

(define visit ##syncase#visit)

(define ##syncase#find-imported-module
  (lambda (id)
    (unless (memq id loaded-modules)
      (let ([fn (##sys#resolve-include-filename (##sys#symbol->string id) #t)])
	(when (##sys#file-info fn)
	  (##syncase#visit fn) ) ) ) ) )

(define properties (make-vector 997 '()))

(define (putprop s k x)
  (let ([props (##sys#hash-table-ref properties s)])
    (if props
	(let ([a (assq props k)])
	  (if a
	      (##sys#setslot a 1 x)
	      (##sys#hash-table-set! properties s (cons (cons k x) props)) ) )
	(##sys#hash-table-set! properties s (list (cons k x))) ) ) )

(define (getprop s k)
  (and-let* ([props (##sys#hash-table-ref properties s)]
	     [a (assq k props)] )
    (cdr a) ) )

(define (remprop s k)
  (and-let* ([props (##sys#hash-table-ref properties s)])
    (let loop ([ps props] [prev #f])
      (unless (null? ps)
	(let ([a (##sys#slot ps 0)]
	      [r (##sys#slot ps 1)] )
	  (if (eq? k (##sys#slot a 0))
	      (if prev
		  (##sys#setslot prev 1 r)
		  (##sys#hash-table-set! properties s r) )
	      (loop r ps) ) ) ) ) ) )

(define ##syncase#getprop getprop)
(define ##syncase#putprop putprop)


(include "psyntax-chicken.pp")


;;; CHICKEN specific macros:

(define srfi-0-def #<<EOF
(define-syntax cond-expand
  (lambda (x)
    (syntax-case x (else not or and)
      [(_)
       (##sys#error
	(##core#immutable '"no matching clause in `cond-expand' form") ) ]
      [(_ (else body ...)) 
       (syntax (begin body ...)) ]
      [(_ ((and) body ...) more ...)
       (syntax (begin body ...)) ]
      [(_ ((and req1 req2 ...) body ...) more ...)
       (syntax (cond-expand
		(req1
		 (cond-expand
		  ((and req2 ...) body ...)
		  more ...))
		more ...) ) ]
      [(_ ((or) body ...) more ...)
       (syntax (cond-expand more ...)) ]
      [(_ ((or req1 req2 ...) body ...) more ...)
       (syntax (cond-expand
		(req1 (begin body ...))
		(else (cond-expand
		       ((or req2 ...) body ...)
		       more ...) ) ) ) ]
      [(_ ((not req) body ...) more ...)
       (syntax (cond-expand
		(req (cond-expand more ...))
		(else body ...) ) ) ]
      [(_ (req body ...) more ...)
       (if (##sys#test-feature (syntax-object->datum (syntax req)))
	   (syntax (begin body ...))
	   (syntax (cond-expand more ...)) ) ] ) ) )
EOF
)

(define std-defs #<<EOF
(begin

(define-syntax with-syntax
   (lambda (x)
      (syntax-case x ()
         ((_ () e1 e2 ...)
          (syntax (begin e1 e2 ...)))
         ((_ ((out in)) e1 e2 ...)
          (syntax (syntax-case in () (out (begin e1 e2 ...)))))
         ((_ ((out in) ...) e1 e2 ...)
          (syntax (syntax-case (list in ...) ()
                     ((out ...) (begin e1 e2 ...))))))))

(define-syntax syntax-rules
  (lambda (x)
    (syntax-case x ()
      ((_ (k ...) ((keyword . pattern) template) ...)
       (syntax (lambda (x)
                (syntax-case x (k ...)
                  ((dummy . pattern) (syntax template))
                  ...)))))))

(define-syntax or
   (lambda (x)
      (syntax-case x ()
         ((_) (syntax #f))
         ((_ e) (syntax e))
         ((_ e1 e2 e3 ...)
          (syntax (let ((t e1)) (if t t (or e2 e3 ...))))))))

(define-syntax and
   (lambda (x)
      (syntax-case x ()
         ((_ e1 e2 e3 ...) (syntax (if e1 (and e2 e3 ...) #f)))
         ((_ e) (syntax e))
         ((_) (syntax #t)))))

(define-syntax let
   (lambda (x)
      (syntax-case x ()
         ((_ ((x v) ...) e1 e2 ...)
          (andmap identifier? (syntax (x ...)))
          (syntax ((lambda (x ...) e1 e2 ...) v ...)))
         ((_ f ((x v) ...) e1 e2 ...)
          (andmap identifier? (syntax (f x ...)))
          (syntax ((letrec ((f (lambda (x ...) e1 e2 ...))) f)
                    v ...))))))

(define-syntax let*
  (lambda (x)
    (syntax-case x ()
      ((let* ((x v) ...) e1 e2 ...)
       (andmap identifier? (syntax (x ...)))
       (let f ((bindings (syntax ((x v)  ...))))
         (if (null? bindings)
             (syntax (let () e1 e2 ...))
             (with-syntax ((body (f (cdr bindings)))
                           (binding (car bindings)))
               (syntax (let (binding) body)))))))))

(define-syntax cond
  (lambda (x)
    (syntax-case x ()
      ((_ m1 m2 ...)
       (let f ((clause (syntax m1)) (clauses (syntax (m2 ...))))
         (if (null? clauses)
             (syntax-case clause (else =>)
               ((else e1 e2 ...) (syntax (begin e1 e2 ...)))
               ((e0) (syntax (let ((t e0)) (if t t))))
               ((e0 => e1) (syntax (let ((t e0)) (if t (e1 t)))))
               ((e0 e1 e2 ...) (syntax (if e0 (begin e1 e2 ...))))
               (_ (syntax-error x)))
             (with-syntax ((rest (f (car clauses) (cdr clauses))))
               (syntax-case clause (else =>)
                 ((e0) (syntax (let ((t e0)) (if t t rest))))
                 ((e0 => e1) (syntax (let ((t e0)) (if t (e1 t) rest))))
                 ((e0 e1 e2 ...) (syntax (if e0 (begin e1 e2 ...) rest)))
                 (_ (syntax-error x))))))))))

(define-syntax do
   (lambda (orig-x)
      (syntax-case orig-x ()
         ((_ ((var init . step) ...) (e0 e1 ...) c ...)
          (with-syntax (((step ...)
                         (map (lambda (v s)
                                 (syntax-case s ()
                                    (() v)
                                    ((e) (syntax e))
                                    (_ (syntax-error orig-x))))
                              (syntax (var ...))
                              (syntax (step ...)))))
             (syntax-case (syntax (e1 ...)) ()
                (() (syntax (let doloop ((var init) ...)
                               (if (not e0)
                                   (begin c ... (doloop step ...))))))
                ((e1 e2 ...)
                 (syntax (let doloop ((var init) ...)
                            (if e0
                                (begin e1 e2 ...)
                                (begin c ... (doloop step ...))))))))))))

(define-syntax quasiquote
   (letrec
     ; these are here because syntax-case uses literal-identifier=?,
     ; and we want the more precise free-identifier=?
      ((isquote? (lambda (x)
                   (and (identifier? x)
                        (free-identifier=? x (syntax quote)))))
       (islist? (lambda (x)
                  (and (identifier? x)
                       (free-identifier=? x (syntax list)))))
       (iscons? (lambda (x)
                  (and (identifier? x)
                       (free-identifier=? x (syntax cons)))))
       (quote-nil? (lambda (x)
                    (syntax-case x ()
                      ((quote? ()) (isquote? (syntax quote?)))
                      (_ #f))))
       (quasilist*
        (lambda (x y)
          (let f ((x x))
            (if (null? x)
                y
                (quasicons (car x) (f (cdr x)))))))
       (quasicons
        (lambda (x y)
          (with-syntax ((x x) (y y))
            (syntax-case (syntax y) ()
              ((quote? dy)
               (isquote? (syntax quote?))
               (syntax-case (syntax x) ()
                 ((quote? dx)
                  (isquote? (syntax quote?))
                  (syntax (quote (dx . dy))))
                 (_ (if (null? (syntax dy))
                        (syntax (list x))
                        (syntax (cons x y))))))
              ((listp . stuff)
               (islist? (syntax listp))
               (syntax (list x . stuff)))
              (else (syntax (cons x y)))))))
       (quasiappend
        (lambda (x y)
          (let ((ls (let f ((x x))
                      (if (null? x)
                          (if (quote-nil? y)
                              '()
                              (list y))
                          (if (quote-nil? (car x))
                              (f (cdr x))
                              (cons (car x) (f (cdr x))))))))
            (cond
              ((null? ls) (syntax (quote ())))
              ((null? (cdr ls)) (car ls))
              (else (with-syntax (((p ...) ls))
                      (syntax (append p ...))))))))
       (quasivector
        (lambda (x)
          (with-syntax ((pat-x x))
            (syntax-case (syntax pat-x) ()
              ((quote? (x ...))
               (isquote? (syntax quote?))
               (syntax (quote #(x ...))))
              (_ (let f ((x x) (k (lambda (ls) (cons (syntax vector) ls))))
                   (syntax-case x ()
                     ((quote? (x ...))
                      (isquote? (syntax quote?))
                      (k (syntax ((quote x) ...))))
                     ((listp x ...)
                      (islist? (syntax listp))
                      (k (syntax (x ...))))
                     ((cons? x y)
                      (iscons? (syntax cons?))
                      (f (syntax y) (lambda (ls) (k (cons (syntax x) ls)))))
                     (else
                      (syntax (list->vector pat-x))))))))))
       (quasi
        (lambda (p lev)
           (syntax-case p (unquote unquote-splicing quasiquote)
              ((unquote p)
               (if (= lev 0)
                   (syntax p)
                   (quasicons (syntax (quote unquote))
                              (quasi (syntax (p)) (- lev 1)))))
              (((unquote p ...) . q)
               (if (= lev 0)
                   (quasilist* (syntax (p ...)) (quasi (syntax q) lev))
                   (quasicons (quasicons (syntax (quote unquote))
                                         (quasi (syntax (p ...)) (- lev 1)))
                              (quasi (syntax q) lev))))
              (((unquote-splicing p ...) . q)
               (if (= lev 0)
                   (quasiappend (syntax (p ...)) (quasi (syntax q) lev))
                   (quasicons (quasicons (syntax (quote unquote-splicing))
                                         (quasi (syntax (p ...)) (- lev 1)))
                              (quasi (syntax q) lev))))
              ((quasiquote p)
               (quasicons (syntax (quote quasiquote))
                          (quasi (syntax (p)) (+ lev 1))))
              ((p . q)
               (quasicons (quasi (syntax p) lev) (quasi (syntax q) lev)))
              (#(x ...) (quasivector (quasi (syntax (x ...)) lev)))
              (p (syntax (quote p)))))))
    (lambda (x)
       (syntax-case x ()
          ((_ e) (quasi (syntax e) 0))))))

(define-syntax unquote
  (lambda (x)
    (syntax-case x ()
      ((_ e ...)
       (syntax-error x
         "expression not valid outside of quasiquote")))))

(define-syntax unquote-splicing
  (lambda (x)
    (syntax-case x ()
      ((_ e ...)
       (syntax-error x
         "expression not valid outside of quasiquote")))))

(define-syntax case
  (lambda (x)
    (syntax-case x ()
      ((_ e m1 m2 ...)
       (with-syntax
         ((body (let f ((clause (syntax m1)) (clauses (syntax (m2 ...))))
                  (if (null? clauses)
                      (syntax-case clause (else)
                        ((else e1 e2 ...) (syntax (begin e1 e2 ...)))
                        (((k ...) e1 e2 ...)
                         (syntax (if (memv t '(k ...)) (begin e1 e2 ...))))
                        (_ (syntax-error x)))
                      (with-syntax ((rest (f (car clauses) (cdr clauses))))
                        (syntax-case clause (else)
                          (((k ...) e1 e2 ...)
                           (syntax (if (memv t '(k ...))
                                       (begin e1 e2 ...)
                                       rest)))
                          (_ (syntax-error x))))))))
         (syntax (let ((t e)) body)))))))

(define-syntax identifier-syntax
  (lambda (x)
    (syntax-case x (set!)
      ((_ e)
       (syntax
         (lambda (x)
           (syntax-case x ()
             (id
              (identifier? (syntax id))
              (syntax e))
             ((_ x (... ...))
              (syntax (e x (... ...))))))))
      ((_ (id exp1) ((set! var val) exp2))
       (and (identifier? (syntax id)) (identifier? (syntax var)))
       (syntax
         (cons 'macro!
           (lambda (x)
             (syntax-case x (set!)
               ((set! var val) (syntax exp2))
               ((id x (... ...)) (syntax (exp1 x (... ...))))
               (id (identifier? (syntax id)) (syntax exp1))))))))))

(define-syntax delay
   (lambda (x)
      (syntax-case x ()
         ((delay exp)
          (syntax (##sys#make-promise (lambda () exp)))))))
)
EOF
)

(define (enable-strict-hl-letrec)
  (sc-expand 
   '(define-syntax (letrec x)
      (syntax-case x ()
	[(_ ([var init] ...) . body)
	 (with-syntax ([(tmp ...) (generate-temporaries #'(var ...))])
	   #'(let ([var #f] ...)
	       (let ([tmp init] ...)
		 (##core#set! var tmp) ...
		 (let () . body) ) ) ) ] ) ) ) )

(define ##syncase#install-macro-defs
  (let ([open-input-string open-input-string]
	[read read]
	[sc-expand sc-expand] )
    (lambda (defstr)
      (let ([in (open-input-string defstr)])
	(sc-expand (read in)) ) ) ) )

(define ##syncase#installed #f)

(define (##syncase#install-macro-package . args)
  (unless ##syncase#installed
    (let-optionals args ([std #f] [srfi0 #f] [sletrec #f] [ffi #f])
      (set! ##syncase#installed #t)
      (##syncase#init-literals)
      (expand-install-hook sc-expand)
      (set! macroexpand (lambda (exp . me) (sc-expand exp)))
      (set! undefine-macro!
	(lambda (name)
	  (##sys#check-symbol name 'undefine-macro!)
	  (let ([a (getprop name '*sc-expander*)])
	    (when (and a (eq? 'macro (car a)))
	      (remprop name '*sc-expander*) ) ) ) )
      (set! macro? 
	(lambda (name)
	  (##sys#check-symbol name 'macro?)
	  (and-let* ([a (getprop name '*sc-expander*)])
	    (eq? 'macro (car a)) ) ) )
      (register-feature! #:hygienic-macros)
      (##syncase#install-macro-defs std-defs)
      (when srfi0 (##syncase#install-macro-defs srfi-0-def))
      (when sletrec (enable-strict-hl-letrec))
      (unless std (load (##sys#resolve-include-filename "chicken-highlevel-macros")))
      (when ffi (load (##sys#resolve-include-filename "chicken-ffi-macros"))) ) ) )

(define eval-hook
  (let ([old (##sys#eval-handler)])
    (lambda (x) 
      (old x) ) ) )

(define expand-install-hook
  (lambda (expand)
    (##sys#eval-handler (lambda (x . env) (eval-hook (expand x))))
    (set! ##sys#macroexpand-hook (lambda (x me) x))
    (set! ##sys#macroexpand-1-hook (lambda (x me) x))
    (set! ##sys#compiler-toplevel-macroexpand-hook expand) ) )

(define (install-highlevel-macro-system . features)
  (let* ([fs (if (null? features) '(r5rs) features)]
	 [r5rs (memq 'r5rs fs)] )
    (##syncase#install-macro-package 
     (or r5rs (not (memq 'extensions fs)))
     (memq 'srfi-0 fs)
     r5rs) ) )


;;; Courtesy of Scott G. Miller:

(define (##syncase#module-exports modname)
  (define (module-record? x)
    (and (pair? x) (eq? (car x) 'module) (vector? (cdr x))))
  (define (module-interface? x)
    (and (vector? x) (= 3 (vector-length x))
         (eq? (vector-ref x 0) 'interface)))
  (let ([symrec (getprop modname '*sc-expander*)])
    (if (not (module-record? symrec))
	(error "not a module" modname) )
    (let ([iface (cdr symrec)])
      (if (not (module-interface? iface))
	  (error "not a recognizable module" modname))
      (map syntax-object->datum
	   (vector->list (vector-ref iface 1))))))
