;;;; batch-driver.scm - Driver procedure for the compiler
;
; Copyright (c) 2000-2004, Felix L. Winkelmann
; All rights reserved.
;
; Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following
; conditions are met:
;
;   Redistributions of source code must retain the above copyright notice, this list of conditions and the following
;     disclaimer. 
;   Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following
;     disclaimer in the documentation and/or other materials provided with the distribution. 
;   Neither the name of the author nor the names of its contributors may be used to endorse or promote
;     products derived from this software without specific prior written permission. 
;
; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS
; OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
; AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
; CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
; SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
; THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
; OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
; POSSIBILITY OF SUCH DAMAGE.
;
; Send bugs, suggestions and ideas to: 
;
; felix@call-with-current-continuation.org
;
; Felix L. Winkelmann
; Unter den Gleichen 1
; 37130 Gleichen
; Germany


(declare (unit driver))


#{compiler
  build-information compiler-arguments process-command-line dump-nodes
  default-standard-bindings default-extended-bindings side-effecting-standard-bindings
  non-foldable-standard-bindings foldable-standard-bindings non-foldable-extended-bindings foldable-extended-bindings
  standard-bindings-that-never-return-false side-effect-free-standard-bindings-that-never-return-false
  installation-home compiler-cleanup-hook
  foreign-type-table-size file-io-only
  unit-name insert-timer-checks used-units
  debugging perform-lambda-lifting! disable-stack-overflow-checking no-c-syntax-checks
  register-ffi-macro ffi-include-path-list
  foreign-declarations emit-trace-info block-compilation analysis-database-size line-number-database-size
  target-heap-size target-stack-size try-harder default-installation-home target-heap-growth target-heap-shrinkage
  default-default-target-heap-size default-default-target-stack-size verbose-mode original-program-size
  target-initial-heap-size split-level default-user-pass-2
  current-program-size line-number-database-2 foreign-lambda-stubs immutable-constants foreign-variables
  rest-parameters-promoted-to-vector inline-table inline-table-used constant-table constants-used mutable-constants
  broken-constant-nodes inline-substitutions-enabled enable-sharp-greater-read-syntax
  emit-profile profile-lambda-list profile-lambda-index profile-info-vector-name
  direct-call-ids foreign-type-table first-analysis optimizers-used
  initialize-compiler canonicalize-expression expand-foreign-lambda update-line-number-database scan-toplevel-assignments
  perform-cps-conversion analyze-expression simplifications perform-high-level-optimizations perform-pre-optimization!
  reorganize-recursive-bindings substitution-table simplify-named-call emit-unsafe-marker
  perform-closure-conversion prepare-for-code-generation compiler-source-file create-foreign-stub expand-foreign-lambda*
  transform-direct-lambdas! source-filename compressed-literals literal-compression-threshold
  debugging-chicken warnings-enabled bomb check-signature posq stringify symbolify flonum? build-lambda-list
  string->c-identifier c-ify-string words check-and-open-input-file close-checked-input-file fold-inner constant?
  collapsable-literal? immediate? canonicalize-begin-body extract-mutable-constants string->expr get get-all
  put! collect! count! get-line get-line-2 find-lambda-container display-analysis-database varnode qnode 
  build-node-graph build-expression-tree fold-boolean inline-lambda-bindings match-node expression-has-side-effects?
  simple-lambda-node? compute-database-statistics print-program-statistics output gen gen-list 
  pprint-expressions-to-file foreign-type-check estimate-foreign-result-size scan-used-variables scan-free-variables
  topological-sort print-version print-usage initialize-analysis-database dump-exported-globals
  copyright compiler-features default-declarations units-used-by-default words-per-flonum default-debugging-declarations
  default-profiling-declarations default-optimization-passes compressed-literals-initializer
  foreign-string-result-reserve parameter-limit eq-inline-operator optimizable-rest-argument-operators
  membership-test-operators membership-unfold-limit valid-compiler-options valid-compiler-options-with-argument
  chop-separator chop-extension display-real-name-table display-line-number-database explicit-use-flag
  generate-code make-variable-list make-argument-list generate-foreign-stubs foreign-type-declaration
  export-list module-export-list compute-module-exports
  foreign-argument-conversion foreign-result-conversion parse-script-header}


(include "tweaks")
(include "parameters")


;;; Compile a complete source file:

(define (compile-source-file filename . options)
  (define (option-arg p)
    (if (null? (cdr p))
	(quit "missing argument to `-~A' option" (car p))
	(cadr p) ) )
  (initialize-compiler)
  (set! explicit-use-flag (memq 'explicit-use options))
  (let ([initforms `((##core#declare
		      ,@(map (lambda (x) `(quote ,x))
			     (append 
			      default-declarations
			      (if explicit-use-flag
				  '()
				  `((uses ,@units-used-by-default)) ) ) ) ) ) ]
        [verbose (memq 'verbose options)]
	[outfile (cond [(memq 'output-file options) 
			=> (lambda (node)
			     (let ([oname (option-arg node)])
			       (if (symbol? oname)
				   (symbol->string oname)
				   oname) ) ) ]
		       [(memq 'to-stdout options) #f]
		       [else (make-pathname #f (if filename (pathname-file filename) "out") "c")] ) ]
        [partitions (cond [(memq 'split options)
                           => (lambda (node)
                                (let ([oname (option-arg node)])
                                  (cond
                                    ((symbol? oname)
                                     (string->number (symbol->string oname)))
                                    ((string? oname)
                                     (string->number oname))
                                    (else oname))))]
                          [else 1])]
	[home (chop-separator (or (getenv "CHICKEN_HOME") default-installation-home ""))]
	[ipath (map chop-separator (string-split (or (getenv "CHICKEN_INCLUDE_PATH") "") ";"))]
	[strict (memq 'strict options)]
	[strict-srfi0 (memq 'strict-srfi-0 options)]
	[strict-letrec (memq 'strict-letrec options)]
	[strictr (memq 'strict-reader options)]
	[opasses default-optimization-passes]
	[time0 #f]
	[time-breakdown #f]
	[forms '()]
	[cleanup-forms '(((##sys#implicit-exit-handler)))]
	[profile (memq 'profile options)]
	[hsize (memq 'heap-size options)]
	[hisize (memq 'heap-initial-size options)]
	[hgrowth (memq 'heap-growth options)]
	[hshrink (memq 'heap-shrinkage options)]
	[kwstyle (memq 'keyword-style options)]
	[lcthreshold (memq 'compress-literals options)]
	[hygienic (or (memq 'hygienic options) (memq 'syntax options))]
	[hygienic-at-rt (or (memq 'hygienic-at-run-time options) (memq 'syntax-at-run-time options))]
	[uunit (memq 'unit options)]
	[a-only (memq 'analyze-only options)]
	[dynamic (memq 'dynamic options)]
	[dumpnodes #f]
	[ffi-mode (memq 'ffi options)]
	[ofilelist #f]
	[quiet (memq 'quiet options)]
	[ssize (or (memq 'nursery options) (memq 'stack-size options))] )

    (define (outfile->list outfile num)
      (cond
       [(not outfile) #f]
       [(= num 1) (list outfile)]
       [else
	(let ([file (pathname-file outfile)])
	  (list-tabulate 
	   num
	   (lambda (i) (make-pathname #f (sprintf "~A~A.c" file i))) ) ) ] ) )

    (define (cputime) (##sys#fudge 6))

    (define (print-header mode dbgmode)
      (when verbose (printf "pass: ~a~%~!" mode))
      (and (memq dbgmode debugging-chicken)
	   (begin
	     (printf "[~a]~%" mode)
	     #t) ) )

    (define (print-node mode dbgmode n)
      (when (print-header mode dbgmode)
	(if dumpnodes
	    (dump-nodes n)
	    (pretty-print (build-expression-tree n)) ) ) )

    (define (print-db mode dbgmode db pass)
      (when (print-header mode dbgmode)
	(printf "(iteration ~s)~%" pass)
	(display-analysis-database db) ) )

    (define (print mode dbgmode xs)
      (when (print-header mode dbgmode)
	(for-each pretty-print xs) ) )

    (define (infohook class data val)
      (let ([data2 ((or ##sys#default-read-info-hook (lambda (a b c) b)) class data val)])
	(when (and (eq? 'list-info class) (symbol? (car data2)))
	  (##sys#hash-table-set!
	   ##sys#line-number-database
	   (car data2)
	   (alist-cons data2 val
		       (or (##sys#hash-table-ref ##sys#line-number-database (car data2))
			   '() ) ) ) )
	data2) )

    (define (arg-val str)
      (let* ((len (string-length str))
	     (len1 (- len 1)) )
	(or (if (< len 2)
		(string->number str)
		(case (string-ref str len1)
		  ((#\m #\M) (* (string->number (substring str 0 len1)) (* 1024 1024)))
		  ((#\k #\K) (* (string->number (substring str 0 len1)) 1024))
		  (else (string->number str)) ) )
	    (quit "invalid numeric argument ~S" str) ) ) )

    (define (collect-options opt)
      (let loop ([opts options])
	(cond [(memq opt opts) => (lambda (p) (cons (option-arg p) (loop (cddr p))))]
	      [else '()] ) ) )

    (define (begin-time)
      (when time-breakdown (set! time0 (cputime))) )

    (define (end-time pass)
      (when time-breakdown
	(printf "milliseconds needed for ~a: ~s~%" pass (- (cputime) time0)) ) )

    (define (install-macros)
      (fluid-let ([##sys#read-line-counter ##sys#read-line-counter])
	(##syncase#install-macro-package (or strict strict-srfi0) (not strict) (or strict strict-srfi0 strict-letrec)) ) )

    (define (strictness s s0)
      (cond [s
	     (set! strict #t)
	     (when verbose (display "Strict mode\n"))
	     (set! ##sys#strict-mode #t)
	     (unregister-feature! #:srfi-8 #:srfi-2 #:srfi-0 #:srfi-16 #:srfi-9 #:srfi-26)
	     (register-feature! #:strict)
	     (##sys#undefine-non-standard-macros '()) ]
	    [s0 
	     (set! strict-srfi0 #t)
	     (when verbose (display "Strict mode with SRFI-0\n"))
	     (set! ##sys#strict-mode #t)
	     (unregister-feature! #:srfi-8 #:srfi-2 #:srfi-16 #:srfi-9 #:srfi-26)
	     (register-feature! #:strict)
	     (##sys#undefine-non-standard-macros '(cond-expand)) ] )
      (if (or s s0 strictr) 
	  (begin
	    (strict-reader #t)
	    (keyword-style #:none)
	    (case-sensitive #f) )
	  (enable-sharp-greater-read-syntax) ) )

    (define (read-form in)
      (if ffi-mode
	  (let ([s (read-string #f in)])
	    (if (string=? s "")
		(end-of-file)
		`(declare 
		   (foreign-declare ,s)
		   (foreign-parse ,s) ) ) )
	  (##sys#read in infohook) ) )

    (when (or (< partitions 1) (not outfile))
        (set! partitions 1))
    (when uunit
      (set! unit-name (string->c-identifier (stringify (option-arg uunit)))) )
    (set! debugging-chicken 
      (append-map
       (lambda (do)
	 (map (lambda (c) (string->symbol (string c)))
	      (string->list do) ) )
       (collect-options 'debug) ) )
    (set! try-harder #t)
    (set! dumpnodes (memq '|D| debugging-chicken))
    (when (memq 't debugging-chicken) (##sys#start-timer))
    (when (memq 'b debugging-chicken) (set! time-breakdown #t))
    (when (memq 'no-warnings options) 
      (when verbose (printf "Warnings are disabled~%~!"))
      (set! warnings-enabled #f)
      (set! ##sys#warnings-enabled #f) )
    (when (memq 'optimize-leaf-routines options) (set! optimize-leaf-routines #t))
    (when (memq 'unsafe options) 
      (set! unsafe #t)
      (##match#set-error-control #:unspecified) )
    (when (memq 'unsafe-libraries options)
      (set! emit-unsafe-marker #t) )
    (when (memq 'disable-interrupts options) (set! insert-timer-checks #f))
    (when (memq 'fixnum-arithmetic options) (set! number-type 'fixnum))
    (when (memq 'block options) (set! block-compilation #t))
    (when (memq 'disable-c-syntax-checks options) (set! no-c-syntax-checks #t))
    (when (memq 'case-insensitive options) 
      (when verbose (printf "Identifiers and symbols are case insensitive~%~!"))
      (register-feature! 'case-insensitive)
      (case-sensitive #f) )
    (and-let* ([slevel (memq 'split-level options)])
      (set! split-level 
	(let ([n (string->number (option-arg slevel))])
	  (if (and n (<= 0 n 2))
	      n
	      (quit "invalid argument to `-split-level' option") ) ) ) )
    (when kwstyle
      (let ([val (option-arg kwstyle)])
	(cond [(string=? "prefix" val) (keyword-style #:prefix)]
	      [(string=? "none" val) (keyword-style #:none)]
	      [(string=? "suffix" val) (keyword-style #:suffix)]
	      [else (quit "invalid argument to `-keyword-style' option")] ) ) )
    (when lcthreshold
      (let ([t (option-arg lcthreshold)])
	(set! literal-compression-threshold 
	  (or (string->number t)
	      (quit "invalid argument to `-compress-literals' option: ~A" t) ) ) ) )
    (set! verbose-mode verbose)
    (set! ##sys#read-error-with-line-number #t)
    (set! ##sys#include-pathnames
      (append (map chop-separator (collect-options 'include-path))
	      ##sys#include-pathnames
	      ipath
	      (if home (list home) '()) ) )
    (set! ofilelist (outfile->list outfile partitions))
    (when (and outfile filename (any (cut string=? <> filename) ofilelist))
      (quit "source- and output-filename are the same") )

    ;; Handle feature options:
    (for-each register-feature! (collect-options 'feature))
    (for-each unregister-feature! (collect-options 'no-feature))

    ;; Handle FFI defines and include-paths:
    (for-each register-ffi-macro (collect-options 'ffi-define))
    (set! ffi-include-path-list (append (collect-options 'ffi-include-path) ffi-include-path-list))

    ;; Load extensions:
    (set! ##sys#features (cons #:compiler-extension ##sys#features))
    (let ([extends (collect-options 'extend)])
      (when verbose
	(printf "Loading compiler extensions...~%~!")
	(load-verbose #t) )
      (for-each (lambda (f) (load (##sys#resolve-include-filename f #t))) extends) )
    (set! ##sys#features (delete #:compiler-extension ##sys#features eq?))

    (set! ##sys#features (cons '#:compiling ##sys#features))
    ;; Install highlevel macros:
    (if hygienic
	(begin
	  (when verbose 
	    (printf "Using hygienic macros~%~!") )
	  (install-macros) )
	(begin
	  (set! ##sys#features (cons #:match (cons #:match-support ##sys#features)))
	  (##sys#provide 'match) 
	  (when (or strict strict-srfi0 strict-letrec) (##sys#enable-strict-letrec) ) ) )

    ;; Loads required extensions:
    (let ([ids (map string->symbol (collect-options 'require-for-syntax))])
      (for-each
       (lambda (id)
	 (when verbose (printf "Requiring ~A ...~%" id))
	 (##sys#require id) )
       ids)
      (let ([rs (##sys#lookup-runtime-requirements ids)])
	(when (pair? rs)
	  (set! initforms
	    (append initforms (map (lambda (r) `(##sys#require ',r)) rs)) ) ) ) )

    (when (memq 'run-time-macros options)
      (set! ##sys#enable-runtime-macros #t) )
    (strictness strict strict-srfi0)
    (set! target-heap-size
      (if hsize
	  (arg-val (option-arg hsize))
	  (and-let* ([hsize default-default-target-heap-size]
		     [(not (zero? hsize))] )
	    hsize) ) )
    (set! target-initial-heap-size (and hisize (arg-val (option-arg hisize))))
    (set! target-heap-growth (and hgrowth (arg-val (option-arg hgrowth))))
    (set! target-heap-shrinkage (and hshrink (arg-val (option-arg hshrink))))
    (set! target-stack-size
      (if ssize
	  (arg-val (option-arg ssize))
	  (and-let* ([ssize default-default-target-stack-size]
		     [(not (zero? ssize))] )
	    ssize) ) )
    (set! emit-trace-info (not (memq 'no-trace options)))
    (set! disable-stack-overflow-checking (memq 'disable-stack-overflow-checks options))
    (when (memq 'm debugging-chicken) (set-gc-report! #t))
    (when (memq 'usual-integrations options)
      (set! standard-bindings default-standard-bindings)
      (set! extended-bindings default-extended-bindings) )
    (when verbose
      (printf "debugging info: ~A~%~!"
	      (if emit-trace-info
		  "stacktrace"
		  "none") ) )
    (when profile
      (set! emit-profile #t)
      (set! initforms (append initforms default-profiling-declarations))
      (when verbose
	(printf "Generating profile~%~!" emit-profile) ) )

    (cond ((memq 'version options)
	   (print-version #t)
	   (newline) )
	  ((or (memq 'help options) (memq '-help options)) (print-usage))
	  ((not filename)
	   (unless quiet
	     (print-version #t)
	     (display "\n\nEnter \"chicken -help\" for information on how to use it.\n") ) )
	  (else

	   ;; Display header:
	   (unless quiet
	     (printf "compiling `~a' ...~%" filename) )
	   (set! source-filename filename)
	   (debugging 'r "options" options)
	   (debugging 'r "debugging options" debugging-chicken)
	   (debugging 'r "home directory" home)
	   (debugging 'r "target heap size" target-heap-size)
	   (debugging 'r "target stack size" target-stack-size)
	   (debugging 'r "compiler features" compiler-features)

	   ;; Read toplevel expressions:
	   (set! ##sys#line-number-database (make-vector line-number-database-size '()))
	   (let ([prelude (collect-options 'prelude)]
		 [postlude (collect-options 'postlude)] 
		 [files (append 
			 (collect-options 'prologue)
			 (list filename)
			 (collect-options 'epilogue) ) ]  )

	     (let ([proc (user-read-pass)])
	       (cond [proc
		      (when verbose (printf "User read pass...~%~!"))
		      (set! forms (proc prelude files postlude)) ]
		     [else
		      (do ([files files (cdr files)])
			  ((null? files)
			   (set! forms
			     (append (map string->expr prelude)
				     (reverse forms)
				     (map string->expr postlude) ) ) )
			(let* ([f (car files)]
			       [in (check-and-open-input-file f)]
			       [c0 (peek-char in)] )
			  (set! ##sys#read-line-counter 1)

			  ;; Check for script header:
			  (cond [ffi-mode]
				[(char=? #\# c0)
				 (read-char in)
				 (if (not (char=? #\! (peek-char in)))
				     (if (string=? "-" f)
					 (quit "source input from pipe may not begin with `#' character if not a script") 
					 (begin
					   (close-input-port in)
					   (set! in (check-and-open-input-file f)) ) )
				     (begin
				       (read-char in)
				       (let ([header (open-output-string)]
					     [cleanup #t] )
					 (do ([c (read-char in) (read-char in)])
					     ((char=? c #\newline)
					      (set! ##sys#read-line-counter (add1 ##sys#read-line-counter)) )
					   (if (eof-object? c)
					       (quit "unexpected end of file - incorrect header")
					       (write-char c header) ) )
					 (when verbose (printf "Compiling UNIX script~%~!"))
					 (case (parse-script-header (get-output-string header) in)
					   [(script)
					    (set! cleanup #f) ]
					   [(chicken)]
					   [(chicken-hygienic) 
					    (install-macros) ]
					   [(r5rs)
					    (install-macros)
					    (strictness #t #f) ]
					   [(srfi0)
					    (install-macros)
					    (strictness #t #t) ]
					   [(#f) (warning "unrecognized script header - might be corrupt")]
					   [else (strictness #t #f)] ) ; r4rs & ieee
					 (when cleanup
					   (set! cleanup-forms
					     (cons '(##sys#script-main '0 '0 main)
						   cleanup-forms) ) ) ) ) ) ]
				[(char=? #\@ c0)
				 (read-line in)
				 (when verbose (printf "Compiling Windows batch file~%~!")) ] )

			  (do ([x (read-form in) (read-form in)])
			      ((eof-object? x) 
			       (close-checked-input-file in f) )
			    (set! forms (cons x forms)) ) ) ) ] ) ) )

	   ;; Start compilation passes:
	   (let ([proc (user-preprocessor-pass)])
	     (when proc
	       (when verbose (printf "User preprocessing pass...~%~!"))
	       (set! forms (cons (first forms) (map proc (cdr forms)))) ) )

	   (print "source" '|1| forms)
	   (begin-time)
	   (let ([us (collect-options 'uses)])
	     (unless (null? us)
	       (set! forms (cons `(declare (uses ,@us)) forms)) ) )
	   (let* ([exps0 (map canonicalize-expression (append initforms forms))]
		  [pvec (gensym)]
		  [plen (length profile-lambda-list)]
		  [exps (append
			 (map (lambda (ic) `(set! ,(cdr ic) ',(car ic))) immutable-constants)
			 (map (lambda (n) `(##core#callunit ,n)) used-units)
			 (if hygienic-at-rt
			     '((##core#callunit "syntax_case")
			       (##syncase#install-macro-package '#t '#t) )
			     '() )
			 (if emit-profile
			     `((set! ,profile-info-vector-name 
				 (##sys#register-profile-info
				  ',plen
				  ',(if unit-name #f "PROFILE") ) ) )
			     '() )
			 (map (lambda (pl)
				`(##core#inline 
				  "C_i_setslot"
				  ,profile-info-vector-name
				  ',(* profile-info-entry-size (car pl)) 
				  ',(cdr pl) ) )
			      profile-lambda-list)
			 (let ([is (fold (lambda (clf r)
					   `(let ([,(gensym) 
						   (set! ,(car clf)
						     (##sys#read-from-string ',(cdr clf)))])
					      ,r) )
					 '(##core#undefined) 
					 compressed-literals) ] )
			   (if compressed-literals-initializer
			       `((##core#set! ,compressed-literals-initializer 
					      (lambda () ,is) ) )
			       (list is) ) )
			 exps0
			 (if hygienic
			     (compute-module-exports module-export-list export-list)
			     '() )
			 (if (and (not unit-name) (not dynamic))
			     cleanup-forms
			     '() ) ) ] )

   	     (when (debugging '|N| "real name table:")
	       (display-real-name-table) )
	     (when (debugging 'n "line number database:")
	       (display-line-number-database) )

	     (when hygienic-at-rt
	       (set! used-units (cons "syntax_case" used-units))
	       (when unit-name
		 (warning "enabling hygienic macros at runtime will apply to all programs using this unit") ) )

	     (when (and block-compilation unit-name)
	       (warning "compilation of library unit in block-mode - globals may not be accessible outside this unit"
			unit-name) )

	     (set! ##sys#line-number-database line-number-database-2)
	     (set! line-number-database-2 #f)

	     (end-time "canonicalization")
	     (print "canonicalized" '|2| exps)

	     (when (memq 'check-syntax options) (exit))

	     (let ([proc (user-pass)])
	       (when proc
		 (when verbose (printf "User pass...~%~!"))
		 (begin-time)
		 (set! exps (map proc exps))
		 (end-time "user pass") ) )

	     (let* ([node0 (make-node
			    'lambda '(())
			    (list (build-node-graph
				   (canonicalize-begin-body exps) ) ) ) ] 
		    [proc (or (user-pass-2) (and optimizers-used default-user-pass-2))] )
	       (when proc
		 (when verbose (printf "Secondary user pass...~%"))
		 (begin-time)
		 (set! first-analysis #f)
		 (let ([db (analyze-expression node0)])
		   (print-db "analysis (u)" '|0| db 0)
		   (end-time "pre-analysis (u)")
		   (begin-time)
		   (proc node0 
			 (lambda (k p) (cut get db <> <>))
			 (lambda (k p x) (cut put! db <> <>)) )
		   (end-time "secondary user pass")
		   (print-node "secondary user pass" '|U| node0) )
		 (set! first-analysis #t) )

	       (when (memq 'lambda-lift options)
		 (begin-time)
		 (set! first-analysis #f)
		 (let ([db (analyze-expression node0)])
		   (print-db "analysis" '|0| db 0)
		   (end-time "pre-analysis")
		   (begin-time)
		   (perform-lambda-lifting! node0 db)
		   (end-time "lambda lifting")
		   (print-node "lambda lifted" '|L| node0) )
		 (set! first-analysis #t) )

	       (set! ##sys#line-number-database #f)
	       (set! constant-table #f)
	       (set! inline-table #f)
	       (when (and try-harder (not unsafe))
		 (scan-toplevel-assignments (first (node-subexpressions node0))) )

	       (begin-time)
	       (let ([node1 (perform-cps-conversion node0)])
		 (end-time "cps conversion")
		 (print-node "cps" '|3| node1)

		 ;; Optimization loop:
		 (let loop ([i 1] [node2 node1] [progress #t])

		   (begin-time)
		   (let ([db (analyze-expression node2)])
		     (set! first-analysis #f)
		     (end-time "analysis")
		     (print-db "analysis" '|4| db i)

		     (when (memq 's debugging-chicken) (print-program-statistics db))

		     (cond [(and progress (or try-harder (<= i default-optimization-passes)))
			    (debugging 'p "optimization pass" i)

			    (begin-time)
			    (receive (node2 progress-flag) (perform-high-level-optimizations node2 db)
			      (end-time "optimization")
			      (print-node "optimized-iteration" '|5| node2)

			      (cond [progress-flag (loop (add1 i) node2 #t)]
				    [(not inline-substitutions-enabled)
				     (debugging 'p "rewritings enabled...")
				     (set! inline-substitutions-enabled #t)
				     (loop (add1 i) node2 #t) ]
				    [optimize-leaf-routines
				     (begin-time)
				     (let ([db (analyze-expression node2)])
				       (end-time "analysis")
				       (begin-time)
				       (let ([progress (transform-direct-lambdas! node2 db)])
					 (end-time "leaf routine optimization")
					 (loop (add1 i) node2 progress) ) ) ]
				    [else (loop (add1 i) node2 #f)] ) ) ]
			   
			   [else
			    (print-node "optimized" '|7| node2)

			    (begin-time)
			    (let ([node3 (perform-closure-conversion node2 db)])
			      (end-time "closure conversion")
			      (print-db "final-analysis" '|8| db i)
			      (when (debugging 'e "exported toplevel variables:")
				(dump-exported-globals db) )
			      (let ([upap (user-post-analysis-pass)])
				(when upap 
				  (upap db
					(lambda (k p) (cut get db <> <>))
					(lambda (k p x) (cut put! db <> <>)) ) ) )
			      (when a-only (exit 0))
			      (print-node "closure-converted" '|9| node3)

                              (when (and verbose
					 outfile
					 (not (> partitions 1)) )
                                (printf "files to be generated: ~A~%" (string-intersperse ofilelist ", ")) )
                              
			      (begin-time)
			      (receive (node literals lambdas) (prepare-for-code-generation node3 db partitions)
				(end-time "preparation")

                                (begin-time)
                                (let loopfile [(lof ofilelist)
                                               (file-partition 0)]
                                  (let* ([outfile (and lof (car lof))]
					 [out (if outfile (open-output-file outfile) (current-output-port))] )
				    (unless quiet
				      (printf "generating `~A' ...~%" outfile) )
                                    (generate-code 
				     literals lambdas out filename dynamic db
				     (if (= partitions 1) #f file-partition))
                                    (when outfile (close-output-port out))
                                    (when (and lof (pair? (cdr lof)))
                                      (loopfile (cdr lof) (+ file-partition 1)))))
                                (end-time "code generation")
                                (when (memq 't debugging-chicken) (##sys#display-times (##sys#stop-timer)))
                                (compiler-cleanup-hook)
                                (when verbose 
                                  (printf "compilation finished.~%~!") ) ) ) ] ) ) ) ) ) ) ) ) ) )


;;; Parse script header:

(define (parse-script-header line in)
  (let ([len (string-length line)])
    (define (cmp-suffix str)
      (let ([index (- len (string-length str))])
	(and (> index 0)
	     (string=? str (substring line index len))
	     (memq (string-ref line (sub1 index)) '(#\space #\/)) ) ) )
    (cond [(cmp-suffix "-script") 'script]
	  [(cmp-suffix "-script-meta")
	   (read-line in)
	   'script]
	  [(cmp-suffix "scheme-r4rs") 'r4rs]
	  [(cmp-suffix "scheme-r5rs") 'r5rs]
	  [(cmp-suffix "scheme-srfi-0") 'srfi0]
	  [(cmp-suffix "scheme-ieee-1178-1990") 'ieee]
	  [else #f] ) ) )
