(**************************************************************************)
(*                   Cameleon                                             *)
(*                                                                        *)
(*      Copyright (C) 2002 Institut National de Recherche en Informatique et   *)
(*      en Automatique. All rights reserved.                              *)
(*                                                                        *)
(*      This program is free software; you can redistribute it and/or modify  *)
(*      it under the terms of the GNU General Public License as published by  *)
(*      the Free Software Foundation; either version 2 of the License, or  *)
(*      any later version.                                                *)
(*                                                                        *)
(*      This program is distributed in the hope that it will be useful,   *)
(*      but WITHOUT ANY WARRANTY; without even the implied warranty of    *)
(*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *)
(*      GNU General Public License for more details.                      *)
(*                                                                        *)
(*      You should have received a copy of the GNU General Public License  *)
(*      along with this program; if not, write to the Free Software       *)
(*      Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA          *)
(*      02111-1307  USA                                                   *)
(*                                                                        *)
(*      Contact: Maxence.Guesdon@inria.fr                                *)
(**************************************************************************)

(** Data class containing the descriptions of GUI elements. *)

open Zog_types

module C = Configwin

(** Create a new entity. *)
let entity () = 
  { ent_name = "" ;
    ent_component = None ;
    ent_params = [] ;
  }

(** Return [true] if the given component will be in the class interface. *)
let in_interface c =
  match c#name with
    "" -> false
  | s -> s.[0] <> '_'

let element_counter = ref 0
  

let blank = "[ \013\009\012]"

(** Return a correct OCaml id from the given string.
   Remove beginning and trailing blanks, replace
   all unauthorized chars by '_', add 'x' at the 
   beginning of the string if it begins with a digit,
   and eventually lowercase the first letter.*)
let get_correct_id s =
  let s2 = Str.global_replace 
      (Str.regexp ("^"^blank^"*")) "" s
  in
  let s3 = Str.global_replace 
      (Str.regexp (blank^"*$")) "" s2
  in
  let l = String.length s3 in
  if l > 1 then
    for i = 1 to l - 1 do
      match s3.[i] with
	'a'..'z' | 'A'..'Z' | '0'..'9' | '\'' | '_' -> ()
      |	_ -> s3.[i] <- '_'
    done;
  let s4 = 
    if l > 0 then
      match s3.[0] with
        'a'..'z' 
      | '_'
        -> s3 
      | 'A'..'Z' -> String.uncapitalize s3
      | '0'..'9' | '\''  ->           "x"^s3
      |	_ -> s3.[0] <- 'x' ; s3
    else
      "x"
  in
  s4

(*
(** Take a [gui_element] and changes its name if it is incorrect,
   that is, if it is not a correct OCaml id. Do the same to the
   children of the element.*)
let rec correct_gui_element ele =
  ele.name <- get_correct_id ele.name ;
  List.iter correct_gui_element ele.children
*)


class data ?(gui=true) file =
  object (self)
    val mutable info = {info_header = "" ; info_entities = []}
    val mutable templates = ([] : Zog_types.entity list)
    val mutable changed = false

    method info = info
    method templates = templates
    method set_changed b = changed <- b
    method changed = changed

    method save =
      try
	(* make sure all ocaml ids are correct *)
(*
	List.iter
	  (fun ent -> 
	    ent.en_name <- get_correct_id ent.en_name ;
	    Zog_misc.apply_opt correct_gui_element ent.en_ele 
	  )
	  entities ;
*)

	Zog_io.write file info ;
(*
	let chanout = open_out_bin file in
	output_value chanout entities ;
	close_out chanout ;
*)
	changed <- false
      with
	Failure s ->
	  if gui then
	    GToolbox.message_box Zog_messages.error s
	  else
	    prerr_endline s

    method private load_file ?(msg_if_error=true) f =
      try
	(* create the file if it doesn't exist *)
	(
	 try ignore (Unix.stat f)
	 with Unix.Unix_error _ -> let oc = open_out f in close_out oc
	);
(*
	List.iter add_missing_properties_in_entity l ;
	List.iter remove_properties_in_entity l ;
*)
	Zog_io.read file
      with
      |	Failure s | Sys_error s ->
	  if msg_if_error then
	    if gui then
	      GToolbox.message_box Zog_messages.error s
	    else
	      prerr_endline s;
	  { info_header = "" ;
	    info_entities = [] ;
	  } 

    method load =
      info <- self#load_file file ;
      changed <- false

    method load_templates =
      let info = self#load_file ~msg_if_error: false Zog_config.templates in
      templates <- info.info_entities

    method file = (file : string)

    method add_entity ent = 
      info.info_entities <- info.info_entities @ [ent] ;
      self#set_changed true

    method remove_entity ent =
      let old = info.info_entities in
      info.info_entities <- List.filter (fun e -> e != ent) info.info_entities ;
      self#set_changed (old <> info.info_entities)

    method up_entity ent =
      let rec f = function
          ent1 :: ent2 :: q -> 
            if ent2 == ent then
              ent2 :: ent1 :: q
            else
              ent1 :: (f (ent2 :: q))
        | l -> l
      in
      let old = info.info_entities in
      info.info_entities <- f info.info_entities ;
      self#set_changed (old <> info.info_entities)

(** {2 Entity manipulation functions} *)

    (** Remove a gui element from the given parent, in the given entity. *)
(*    method entity_remove_from_parent entity component parent_opt =
      match parent_opt with
	None ->
	  (match entity.ent_component with
	    Some c when c == component -> 
	      entity.ent_component <- None ;
	      self#set_changed true 
	  | _ -> ()
	  )
      |	Some p ->
	  p.children <- (List.filter (fun e -> e != ele) p.children) ;
	  self#set_changed true 
*)
(*
    (** Return [true] if an element of the given class can be appended
       to the children of the given element. *)
    method can_append_ele parent cl =
      match parent.classe with
      | Clist
      | Label
      | Hseparator
      | Vseparator
      | Custom_box
      | Color_selection
      | Pixmap 
      | Pixmap_file
      | Pixmap_data
      | Pixmap_code
      | Entry
      | Spin_button
      | Combo
      | Statusbar
      | Tree
      | Text 
      |	Table (* Table : A VOIR ? *)
      |	Progress_bar
      |	HRuler
      |	VRuler
      |	Arrow
      |	Calendar
      |	Drawing_area
      |	Font_selection
	-> false

      |	Menubar ->
	  List.mem cl [ Menu_item; Tearoff_menu_item; Menu_separator]

      |	Menu ->
	  List.mem cl [ Menu_item ; Check_menu_item; Menu_separator ;
			Radio_menu_item ; Tearoff_menu_item]

      |	Menu_item -> 
	  cl = Menu && parent.children = []

      |	Tearoff_menu_item 
      | Check_menu_item
      | Radio_menu_item
      |	Menu_separator ->
	  false

      | Button
      | Toggle_button
      | Check_button
      | Radio_button
      | Frame
      | Aspect_frame
      | Scrolled_window
      | Event_box
      | Handle_box
      | Viewport
      | Fixed
      | Toolbar 
      |	Window ->
	  not (List.mem cl [ Menu_item ; Check_menu_item; Menu_separator ;
			     Radio_menu_item ; Tearoff_menu_item ; Menu]) &&
	  parent.children = []

      | Hpaned
      |	Vpaned ->
	  not (List.mem cl [ Menu_item ; Check_menu_item; Menu_separator ;
			     Radio_menu_item ; Tearoff_menu_item ; Menu]) &&
	  (List.length parent.children) < 2

      |	Notebook
      | Hbox
      | Vbox
      |	Vbutton_box
      | Hbutton_box ->
	  not (List.mem cl [ Menu_item ; Check_menu_item; Menu_separator ;
			     Radio_menu_item ; Tearoff_menu_item ; Menu])
	  

    (** Add a gui element to another one, in a given entity.*)
    method entity_append_in_parent entity ele_paste parent_opt =
      match parent_opt with
	None -> 
	  (
	   match entity.en_ele with
	     None ->
	       entity.en_ele <- Some ele_paste ;
	       self#set_changed true
	   | _ -> 
	       raise (Failure Zog_messages.only_one_root)
	  )
      |	Some p -> 
	  if ele_paste.classe <> Window &
	    self#can_append_ele p ele_paste.classe then
	    (
	     p.children <- p.children @ [ele_paste] ;
	     self#set_changed true 
	    )
	  else
	    raise (Failure (Zog_messages.cant_add_element p.name))

    method up_element ele parent_opt =
      let rec f = function
          ele1 :: ele2 :: q -> 
            if ele2 == ele then
              ele2 :: ele1 :: q
            else
              ele1 :: (f (ele2 :: q))
        | l -> l
      in
      match parent_opt with
        None -> ()
      | Some p_ele ->
          p_ele.children <- f p_ele.children ;
          self#set_changed true 
*)

    initializer
      self#load ;
      self#load_templates
  end
