(**************************************************************************)
(*                   Cameleon                                             *)
(*                                                                        *)
(*      Copyright (C) 2002 Institut National de Recherche en Informatique et   *)
(*      en Automatique. All rights reserved.                              *)
(*                                                                        *)
(*      This program is free software; you can redistribute it and/or modify  *)
(*      it under the terms of the GNU General Public License as published by  *)
(*      the Free Software Foundation; either version 2 of the License, or  *)
(*      any later version.                                                *)
(*                                                                        *)
(*      This program is distributed in the hope that it will be useful,   *)
(*      but WITHOUT ANY WARRANTY; without even the implied warranty of    *)
(*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *)
(*      GNU General Public License for more details.                      *)
(*                                                                        *)
(*      You should have received a copy of the GNU General Public License  *)
(*      along with this program; if not, write to the Free Software       *)
(*      Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA          *)
(*      02111-1307  USA                                                   *)
(*                                                                        *)
(*      Contact: Maxence.Guesdon@inria.fr                                *)
(**************************************************************************)

(** OCaml code generation. *)

open Zog_types


let p = Format.pp_print_string 

let get_prop_label ele prop =
  match prop.prop_kind with
  | Function -> ""
  | Expand -> "expand"
  | Fill -> "fill"
  | Padding -> "padding"
  | Width -> "width"
  | Height -> "height"
  | Border_width -> "border_width"
  | Title -> "title"
  | Allow_shrink -> "allow_shrink"
  | Allow_grow -> "allow_grow"
  | Auto_shrink -> "auto_shrink"
  | X_pos -> "x"
  | Y_pos -> "y"
  | PLabel -> "label"
  | Group -> "group"
  | Orientation -> "orientation"
  | Toolbar_style -> "style"
  | Toolbar_space_size -> "space_size"
  | Toolbar_space_style -> "space_style"
  | Tooltips -> "tooltips"
  | Button_relief_style -> "button_relief"
  | Spacing -> "spacing"
  | Homogeneous -> "homogeneous"
  | Button_box_style -> "layout"
  | Child_width -> "child_width"
  | Child_height -> "child_height"
  | Child_ipadx -> "child_ipadx"
  | Child_ipady -> "child_ipady"
  | Label_xalign -> "label_xalign"
  | Label_yalign -> "label_yalign"
  | Shadow_type -> 
      (
       match ele.classe with
	 Arrow -> "shadow"
       | _ -> "shadow_type"
      )
  | Obey_child -> "obey_child"
  | Ratio -> "ratio"
  | Hscrollbar_policy -> "hpolicy"
  | Vscrollbar_policy -> "vpolicy"
  | Handle_position -> "handle_position" (* A VOIR qd ce sera dans lablgtk *)
  | Snap_edge -> "snap_edge"  (* A VOIR qd ce sera dans lablgtk *)
  | Column_titles -> "titles"
  | Show_titles -> "titles_show"
  | X_align -> "xalign"
  | Y_align -> "yalign"
  | X_pad -> "xpad"
  | Y_pad -> "ypad"
  | PText -> "text"
  | Line_wrap -> "line_wrap"
  | Tab_pos -> "tab_pos"
  | Show_tabs -> "show_tabs"
  | Homogeneous_tabs -> "homogeneous_tabs"
  | Show_border -> "show_border"
  | Scrollable -> "scrollable"
  | Tab_border -> "tab_border"
  | Popup -> "popup"
  | SBUpdate_policy -> "update_policy"
  | Visibility -> "visibility"
  | Editable -> "editable"
  | Use_arrows -> "use_arrows"
  | Case_sensitive -> "case_sensitive"
  | Word_wrap -> "word_wrap"
  | Column_number -> "columns"
  | Draw_indicator -> "draw_indicator"
  | Active -> "active"
  | Placement -> "placement"
  | Selection_mode -> "selection_mode"
  | Justification -> "justify"
  | Max_length -> "max_length"
  | View_mode -> "view_mode"
  | View_lines -> "view_lines"
  | Handle_size -> "handle_size"
  | Modal -> "modal"
  | Tab_label -> ""
	
    (* these properties need post creation code *)
  | Accel_group_name
  | Accel_modifier
  | Accel_flags
  | Accel_keysym
  | Show_toggle
  | Show_indicator
  | Right_justify -> ""
     
  | Arrow_type -> "kind"
  | Calendar_options -> "options"
  | Popdown_strings -> "popdown_strings"
  | Value_in_list -> "value_in_list"
  | Ok_if_empty -> "ok_if_empty"
  | Update_policy -> "update_policy"

(** Remove blanks (space, tabs, \r and \n) from a string. *)
let remove_blanks s = Str.global_replace (Str.regexp "[ \t\n\r]+") "" s

(** Indicate whether a property param must be printed.*)
let must_print prop =
  try
    let (_,_,values_kind,_) =  Zog_types.get_prop_info prop.prop_kind in
    match values_kind with
      Bool -> true
    | PosInt -> 
	(try (int_of_string prop.prop_value) >= 0
	with _ -> false)
    | Float -> 
	(try (float_of_string prop.prop_value) >= 0.
	with _ -> false)
    | Code 
    | Code_list -> prop.prop_value <> ""
    | Enum [] -> false
    | Enum ((s,_) :: _) -> true
    | Enum_list [] -> false
    | Enum_list ((s,_) :: _) -> true
    | Keysym ->
	let v_no_blank = remove_blanks prop.prop_value in
	v_no_blank <> ""
  with
    Failure s ->
      prerr_endline s ;
      false

let print_creation_options_code fmt ele =
  let f prop =
    match prop.prop_kind with
      Function 
    | Tab_label
    | Expand
    | Fill 
    | Padding -> ()

    | Accel_group_name
    | Accel_modifier
    | Accel_flags
    | Accel_keysym
    | Show_toggle
    | Show_indicator
    | Right_justify -> ()

    |  _ ->
	 if must_print prop then
	   p fmt ("\n      ~"^(get_prop_label ele prop)^
		  ": ("^prop.prop_value^")")
  in
  List.iter f ele.props

let print_pack_options_code fmt ele =
  let f prop =
    match prop.prop_kind with
      Expand | Fill | Padding ->
	if must_print prop then
	  p fmt ("~"^(get_prop_label ele prop)^
		 ": ("^prop.prop_value^") ")
    | _ ->
	()
  in
  List.iter f ele.props

let print_pack_code fmt parent ele =
  try
    let pack_met = Zog_types.pack_method_of_ele parent ele in
    (match pack_met with
      No_pack -> ()
    | Insert_page ->
	p fmt ("\n     ~packing: (fun w -> "^parent.name^"#") ;
	p fmt ("append_page ~tab_label: ((GMisc.label ~text:("^
		   (Zog_types.get_prop_value ele.props Tab_label)^
		   ") ())#coerce) w)")
    | _ ->
	p fmt ("\n     ~packing: ("^parent.name^"#") ;
	(match pack_met with
	  Pack -> 
	    p fmt "pack ";
	    print_pack_options_code fmt ele 
	| Add -> p fmt "add"
	| Add1 -> p fmt "add1"
	| Add2 -> p fmt "add2"
	| Add_with_viewport -> p fmt "add_with_viewport"
	| Set_submenu -> p fmt "set_submenu"
	| Insert_page
	| No_pack -> ());
	p fmt ")"
    )
  with
    Failure s ->
      prerr_endline s
      
let print_custom_pack_code fmt parent ele =
  try
    let pack_met = Zog_types.pack_method_of_ele parent ele in
    (match pack_met with
      No_pack -> ()
    | Insert_page ->
	p fmt ("  let _ = "^parent.name^"#") ;
	p fmt ("append_page ~tab_label: ((GMisc.label ~text:("^
	       (Zog_types.get_prop_value ele.props Tab_label)^
	       ") ())#coerce) "^ele.name^"#coerce in\n")
    | _ ->
	p fmt ("  let _ = "^parent.name^"#") ;
	(match pack_met with
	  Pack -> 
	    p fmt "pack ";
	    print_pack_options_code fmt ele 
	| Add -> p fmt "add"
	| Add1 -> p fmt "add1"
	| Add2 -> p fmt "add2"
	| Add_with_viewport -> p fmt "add_with_viewport"
	| Set_submenu -> p fmt "set_submenu"
	| Insert_page
	| No_pack -> ());
	p fmt (" "^ele.name^"#coerce in\n")
    )
  with
    Failure s ->
      prerr_endline s

(** The accel_group variable name for the given Menubar ele. *)
let accel_group_name ele = 
  let accel_group_v = Zog_types.get_prop_value ele.props Accel_group_name in
  let name = 
    match remove_blanks accel_group_v with
      "" -> "accel_"^ele.name
    | s -> s
  in
  name

(** Output the OCaml for the given menu_item (or check or radio)
   and its optional submenu, to perform after creation initializations
   like adding accelerators and fixing some properties. *)
let rec print_post_menu_item_creation_code fmt accel_name ele =
  (
   let keysym_v = Zog_types.get_prop_value ele.props Accel_keysym in
   match remove_blanks keysym_v with
     "" -> ()
   | v ->
       let modifier = 
	 match remove_blanks (Zog_types.get_prop_value ele.props Accel_modifier) with
	   "" -> "[]"
	 | s -> s
       in
       let flags = 
	 match remove_blanks (Zog_types.get_prop_value ele.props Accel_flags) with
	   "" -> "[]"
	 | s -> s
       in
       p fmt ("  let _ = "^ele.name^"#add_accelerator "^
	      "~group: "^accel_name^" "^
	      "~modi: ("^modifier^") "^
	      "~flags: ("^flags^") "^
	      "("^v^") in\n")
  );
  List.iter (print_post_menu_creation_code fmt accel_name) ele.children

(** Output the OCaml for the given menu
   and its items, to perform after creation initializations
   like adding accelerators and fixing some properties. *)
and print_post_menu_creation_code fmt accel_name ele =
  match ele.classe with
    Menu ->
      p fmt ("  let _ = "^ele.name^"#set_accel_group "^accel_name^" in\n");
      List.iter (print_post_menu_item_creation_code fmt accel_name) ele.children
  | _ ->
      ()

(** Output the OCaml for the given element which must be a Menubar. *)
let print_post_menubar_creation_code fmt ele =
  match ele.classe with
    Menubar ->
      let acc_name = accel_group_name ele in
      p fmt ("  let "^acc_name^" = GtkData.AccelGroup.create () in\n");
      List.iter (print_post_menu_item_creation_code fmt acc_name) ele.children
  | _ ->
      ()
(*

  let opt_code p =
    match Zog_types.get_prop_value ele.props p with

  | Accel_group
  | Accel_modifier
  | Accel_flags
  | Accel_keysym
  | Show_toggle
  | Show_indicator
  | Right_justify
*)
let rec print_ele_creations fmt parent_opt previous_opt ele =
  let (_,_,_,fonc) = Zog_types.get_class_info ele.classe in
  let fonc2 = 
    match ele.classe with 
      Custom_box -> "("^(Zog_types.get_prop_value ele.props Function)^")"
    | _ -> fonc
  in
  p fmt ("  let "^ele.name^" = "^fonc2^"") ;
  print_creation_options_code fmt ele ;
  (
   match ele.classe, previous_opt with
     Radio_menu_item, Some e when e.classe = Radio_menu_item ->
       p fmt ("\n      ~group: "^e.name^"#group")
   | _ ->
       ()
  );
  (match parent_opt with
    None -> 
      p fmt ("    () in \n") 
  | Some parent when ele.classe = Custom_box ->
      p fmt ("    () in \n") ;
      print_custom_pack_code  fmt parent ele
  | Some parent when ele.classe <> Custom_box ->
      print_pack_code fmt parent ele ;
      p fmt ("    () in \n") 
  );

  let rec iter prev = function
      [] -> ()
    | e :: q ->
	print_ele_creations fmt (Some ele) prev e ;
	iter (Some e) q
  in
  iter None ele.children ;

  (* some more code is output if it is a menubar *)
  print_post_menubar_creation_code fmt ele

  
let rec print_ele_methods fmt ele =
  p fmt ("    method "^ele.name^" = "^ele.name^"\n");
  if ele.classe = Menubar then
    (
     let accel_name = accel_group_name ele in
     p fmt ("    method "^accel_name^" = "^accel_name^"\n")
    );
  List.iter (print_ele_methods fmt) ele.children

let rec print_ele_vals fmt ele =
  p fmt ("    val "^ele.name^" = "^ele.name^"\n");
  if ele.classe = Menubar then
    (
     let accel_name = accel_group_name ele in
     p fmt ("    val "^accel_name^" = "^accel_name^"\n")
    );
  List.iter (print_ele_vals fmt) ele.children

let print_entity fmt entity =
  let p = p fmt in
  p "class " ;
  p entity.en_name ;
  List.iter (fun par -> p (" "^par)) entity.en_params ;
  p " () =\n";
  (match entity.en_ele with
    None -> ()
  | Some e -> print_ele_creations fmt None None e);
  p "  object\n" ;
  (match entity.en_ele with
    None -> ()
  | Some e -> 
      print_ele_vals fmt e ;
      print_ele_methods fmt e ;
      match e.classe with
	Window -> ()
      |	_ -> p ("    method coerce = "^e.name^"#coerce\n")
  );
  p "  end\n\n"

let gen_code data file =
  try
    let chanout = open_out file in
    let fmt = Format.formatter_of_out_channel chanout in
    p fmt ("(** "^Zog_messages.generated_by^" *)\n\n");
    List.iter (print_entity fmt) data#entities ;
    flush chanout ;
    close_out chanout
  with
    Sys_error s ->
      raise (Failure s)
