(**************************************************************************)
(*                   Cameleon                                             *)
(*                                                                        *)
(*      Copyright (C) 2002 Institut National de Recherche en Informatique et   *)
(*      en Automatique. All rights reserved.                              *)
(*                                                                        *)
(*      This program is free software; you can redistribute it and/or modify  *)
(*      it under the terms of the GNU General Public License as published by  *)
(*      the Free Software Foundation; either version 2 of the License, or  *)
(*      any later version.                                                *)
(*                                                                        *)
(*      This program is distributed in the hope that it will be useful,   *)
(*      but WITHOUT ANY WARRANTY; without even the implied warranty of    *)
(*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *)
(*      GNU General Public License for more details.                      *)
(*                                                                        *)
(*      You should have received a copy of the GNU General Public License  *)
(*      along with this program; if not, write to the Free Software       *)
(*      Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA          *)
(*      02111-1307  USA                                                   *)
(*                                                                        *)
(*      Contact: Maxence.Guesdon@inria.fr                                *)
(**************************************************************************)

(** Analysis of command line arguments. *)

(** The schema file. *)
let file = ref None

(** Generate code for a dbms. *)
let gen_code = ref (None : Dbf_types.Current.dbms option)

(** Generate code of classes for a dbms. *)
let gen_objects_code = ref (None : Dbf_types.Current.dbms option)

(** Optional prefix for output files. *)
let output = ref None

(** Generate debug code or not. *)
let debug = ref false

(** The DBMS to include when html doc. *)
let html_dbms = ref []

(** A flag to generate or not the HTML header of the document. *)
let with_html_header = ref true

(** The title for generated code and documentation. *)
let title = ref None

(** Will define the [db] type with its manifest (e.g. Mysql.dbd). *)
let db_manifest = ref false

(** Will remove the given prefix from table names to get the module names.*)
let remove_table_prefix = ref ""

let add_html_dbms s =
  let dbms_opt =
    match s with
      "mysql" -> Some Dbf_types.Current.Mysql
    | "pg" -> Some Dbf_types.Current.Postgres
    | "odbc" -> Some Dbf_types.Current.Odbc
    | "all" -> 
	html_dbms := [ Dbf_types.Current.Odbc ; Dbf_types.Current.Mysql ;
		       Dbf_types.Current.Postgres ] ;
	None
    | _ ->
	prerr_endline (Dbf_messages.unknown_target_lib s);
	None
  in
  match dbms_opt with
    None -> ()
  | Some d -> html_dbms := !html_dbms @ [d]

let set_html_dbms s =
  match Str.split (Str.regexp_string ",") s with
    [] -> prerr_endline (Dbf_messages.incorrect_target_lib_spec s)
  | l -> List.iter add_html_dbms l

let options  = ref [
  "-debug", Arg.Set debug, Dbf_messages.op_debug ;
  "-odbc", Arg.Unit (fun () -> gen_code := Some Dbf_types.Current.Odbc), Dbf_messages.op_gen_odbc ;
  "-mysql", Arg.Unit (fun () -> gen_code := Some Dbf_types.Current.Mysql), Dbf_messages.op_gen_mysql ;
  "-pg", Arg.Unit (fun () -> gen_code := Some Dbf_types.Current.Postgres), Dbf_messages.op_gen_postgres ;

  "-objodbc", Arg.Unit (fun () -> gen_objects_code := Some Dbf_types.Current.Odbc), Dbf_messages.op_gen_objodbc ;
  "-objmysql", Arg.Unit (fun () -> gen_objects_code := Some Dbf_types.Current.Mysql), Dbf_messages.op_gen_objmysql ;
  "-objpg", Arg.Unit (fun () -> gen_objects_code := Some Dbf_types.Current.Postgres), Dbf_messages.op_gen_objpostgres ;

  "-html", Arg.String set_html_dbms, Dbf_messages.op_gen_html ;
  "-noheader", Arg.Clear with_html_header, Dbf_messages.op_noheader ;
  "-t", Arg.String (fun s -> title := Some s), Dbf_messages.op_title ;
  "-o", Arg.String (fun s -> output := Some s), Dbf_messages.op_output ;
  "-dbmanifest", Arg.Set db_manifest, Dbf_messages.op_dbmanifest ;
  "-remove-table-prefix", Arg.Set_string remove_table_prefix, Dbf_messages.op_remove_table_prefix ;
 ] 

let options_gui  = ref [
 ] 

(** Parse the command line and fill the arguments variables. *)
let parse () =
  try
    let _ = Arg.parse !options
	(fun s -> 
	  match !file with
	    None -> file := Some s;
	  | Some f -> raise (Failure Dbf_messages.usage)
	)
	(Dbf_messages.usage^Dbf_messages.options_are)
    in
    ()
  with
    Failure s ->
      prerr_endline s ;
      exit 1

(** Parse the command line and fill the arguments variables for the GUI. *)
let parse_gui () =
  try
    let _ = Arg.parse !options_gui
	(fun s -> 
	  match !file with
	    None -> file := Some s;
	  | Some f -> raise (Failure Dbf_messages.usage_gui)
	)
	(Dbf_messages.usage_gui^Dbf_messages.options_are)
    in
    ()
  with
    Failure s ->
      prerr_endline s ;
      exit 1
