/***************************************************************************
                          main.cpp  -  description
                             -------------------
    begin                : Sat Dec 28 10:51:34 CST 2002
    copyright            : (C) 2002 by Ken Schenke
    email                : kenschenke at yahoo dot com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   In addition, as a special exception, Ken Schenke gives permission to  *
 *   link the code of this program with the Qt non-commercial edition (or  *
 *   with modified versions of the Qt non-commercial edition that use the  *
 *   same license as the Qt non-commercial edition, and distribute linked  *
 *   combinations including the two.  You must obey the GNU General Public *
 *   License in all respects for all of the code used other than the Qt    *
 *   Non-Commercial edition.  If you modify this file, you may extend this *
 *   exception to your version of the file, but you are not obligated to   *
 *   do so.  If you do not wish to do so, delete this exception statement  *
 *   from your version.                                                    *
 *                                                                         *
 ***************************************************************************/

#ifdef _WIN32
#include "xgetopt.h"
#else
#include <unistd.h>
#endif

#include <qapplication.h>
#include <qfont.h>
#include <qstring.h>
#include <qtextcodec.h>
#include <qtranslator.h>
#include <qmessagebox.h>
#include <qmime.h>

#include "dlgmain.h"
#include "bridgecfg.h"
#include "bkutil.h"
#include "images.h"

/***************************************************************************
 *                                                                         *
 *  Global Variables                                                       *
 *                                                                         *
 ***************************************************************************/

		bool	gNoDates;		// Don't write dates to bookmark tree file
								// this is used for testing purposes only
static	bool	gMergeAction;	// Non-zero if merge requested then exit
static	bool	gReadOnly;		// Don't save bookmark tree (bookmarks.xbel)

		QString	gTreeFile;		// Bookmark tree file from command line
		QString	gConfigFile;	// Config file from command line

/***************************************************************************
 *                                                                         *
 *  Function Prototypes                                                    *
 *                                                                         *
 ***************************************************************************/

static	bool	MergeAction(void);
static	bool	ParseCmdLine(void);

/***************************************************************************
 *                                                                         *
 *   MergeAction()                                                         *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      true if error, false otherwise                                     *
 *   Description:                                                          *
 *      This function performs a merge if requested as a command line      *
 *      option.  It is designed to do everything silently and simply       *
 *      return with an error on failure.                                   *
 *                                                                         *
 ***************************************************************************/

static bool MergeAction(void)
{
	try
	{
		BridgeCfg	cfg;
		BkUtil	util(cfg);

		cfg.readConfig();

		// read BookmarkBridge's copy of the bookmark tree from the XBEL file
		
		util.readBookmarkTree();

		// read the bookmarks from each of the browsers, adding them to the
		// bookmark tree.

		util.readBookmarksFromBrowsers();

		// If any changes were found between a browser's copy of a bookmark
		// and the bookmark tree's copy, they are recorded in the tree in a
		// "difference list".  Each bookmark node in the tree has one of these
		// lists.  This function looks at each bookmark where this is list is
		// not empty and decides which version of the bookmark to keep.  It
		// then moves the changes up to the bookmark node and emptys the list.
		
		util.root()->rollUpDifferences();

		// If we're conducting a regression test (indicated by gNoDates set
		// to non-zero), then sort the bookmarks and folders by title so that
		// the output files are consistent.  Some operations on the bookmark
		// tree alter the order of elements within the bookmark and folder lists.

		if(gNoDates)
			util.root()->sortContents();

		// This function saves the bookmark tree out to each browser, thereby
		// saves changes detected during the merge.
		
		util.saveBookmarksToBrowsers();

		// This function saves BookmarkBridge's copy of the bookmark tree.

		if(gReadOnly == false)
			cfg.writeBookmarksXbel(*util.root());
	}
	catch(BkException)
	{
		return true;
	}

	return false;
}

/***************************************************************************
 *                                                                         *
 *   ParseCmdLine()                                                        *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      true if error, false otherwise                                     *
 *   Description:                                                          *
 *      This function analyzes the command line arguments.                 *
 *                                                                         *
 ***************************************************************************/

static bool ParseCmdLine(void)
{
	extern	char	*optarg;
	auto	int		optionC;

	while((optionC = getopt(qApp->argc(), qApp->argv(), "b:f:mrt")) != EOF)
	{
		switch(optionC)
		{
		case 'b':	// bookmark tree file (bookmarks.xbel)
			gTreeFile = optarg;
			break;
		case 'f':	// config file
			gConfigFile = optarg;
			break;
		case 'm':	// merge then exit
			gMergeAction = true;
			break;
		case 'r':	// read only (don't save to bookmarks.xbel)
			gReadOnly = true;
			break;
		case 't':	// don't save dates to bookmark tree file
			gNoDates = true;
			break;
		default:
			return true;
		}
	}

	return false;
}

/***************************************************************************
 *                                                                         *
 *   main()                                                                *
 *                                                                         *
 *   Parameters:                                                           *
 *      int argc                                                           *
 *      char *argv[]                                                       *
 *   Return:                                                               *
 *      exit code for program                                              *
 *   Description:                                                          *
 *      C++ program entry point.                                           *
 *                                                                         *
 ***************************************************************************/

int main(int argc, char *argv[])
{
	// initialize global variables

	gNoDates = false;
	gMergeAction = false;
	gReadOnly = false;
	gTreeFile = QString::null;
	gConfigFile = QString::null;

	QApplication a(argc, argv);
	a.setFont(QFont("helvetica", 12));
	QTranslator tor( 0 );
	// set the location where your .qm files are in load() below as the last parameter instead of "."
	// for development, use "/" to use the english original as
	// .qm files are stored in the base project directory.
	tor.load( QString("bookmarkbridge.") + QTextCodec::locale(), "." );
	a.installTranslator( &tor );
	/* uncomment the following line, if you want a Windows 95 look*/
	// a.setStyle(WindowsStyle);

	if(	ParseCmdLine()
	 ||	(gReadOnly && gMergeAction==false)
	 ||	(gConfigFile==QString::null && gTreeFile!=QString::null)
	 ||	(gConfigFile!=QString::null && gTreeFile==QString::null))
	{
		QMessageBox::information(NULL, "Command Line Options",
			"Usage:  bookmarkbridge [-b file] [-f file] [-m [-r]] [-t]\n"
			"\n"
			"-b file\n"
			"      Specify bookmark tree file\n"
			"-f file\n"
			"      Specify config file\n"
			"-m\n"
			"      Merge then exit\n"
			"-r\n"
			"      Do not save bookmark tree (read only)\n"
			"-t\n"
			"      Don't Save Dates to Bookmark Tree File\n");
		return 5;
	}

	if(gMergeAction)
	{
		int ret = MergeAction();
#ifdef QT_THREAD_SUPPORT
		if(a.locked())
			a.unlock(false);
#endif
		return ret ? 5 : 0;
	}
	
	// Set up the mime factories for the embedded images
	
	QMimeSourceFactory::defaultFactory()->setImage(
		"bookmarkbridgelogo",
		qembed_findImage("bookmarkbridgelogo"));
	QMimeSourceFactory::defaultFactory()->setImage(
		"osilogo",
		qembed_findImage("osilogo"));

	DlgMain *dlgmain = new DlgMain();
	a.setMainWidget(dlgmain);

	dlgmain->show();

	return a.exec();
}
