/* transform.c -- do various transformations on the parse tree
 * Written by Charles Briscoe-Smith; refer to the file LEGAL for details.
 */

/* Interface definition */

/* Do all the necessary transformations in sequence */
extern void transform(void);


#ifndef SEEN_transform_h

/* Implementation */

#include <stdio.h>
#include "parsetree.h"
#include <string.h>
#include "packagetree.h"
#include "error.h"
#include "stringpool.h"

struct env {
	const char *id;
	struct type *type;
	struct env *rest;
};

static void
spoutname(struct name *n, FILE *f)
{
	if (n==0) return;
	spoutname(n->prefix, f);
	if (n->prefix) putc('.', f);
	fputs(n->id, f);
}

/*
 * The class java.lang.Object has no supertype.  The parse tree gives
 * everything a supertype initially, so we strip java.lang.Object's
 * supertype off again here.
 */

static void
javalangobjecthasnosupertype(struct typedecl *t)
{
	if (strcmp("Object", t->name->id)==0
	    && t->name->prefix
	    && strcmp("lang", t->name->prefix->id)==0
	    && t->name->prefix->prefix
	    && strcmp("java", t->name->prefix->prefix->id)==0
	    && t->name->prefix->prefix->prefix==0) {
		/* This is java.lang.Object -- it has no superclass. */
		t->super=0;
	}
}

/*
 * Create the string constant for the name of each class and method.
 */

static int
namelength(struct name *n)
{
	if (n==0) return 1;
	return namelength(n->prefix)+1+strlen(n->id);
}

static char *
rendername(char *bufferstart, struct name *n)
{
	char *cur;
	const char *idscan;

	if (n==0) return bufferstart;
	cur=rendername(bufferstart, n->prefix);
	if (n->prefix) *cur++='.';
	for (idscan=n->id; *idscan; )
		*cur++=*idscan++;
	return cur;
}

static void
internnames(struct typedecl *t)
{
	struct memberdecl *m;
	char name[namelength(t->name)];
	*(rendername(name, t->name))='\0';
	t->name->data=stringpool_intern(strdup(name));
	for (m=t->body; m; m=m->rest) {
		if (m->tag==Method) {
			stringpool_intern(m->content.meth->sig->name);
		}
	}
}

/*
 * Separate field declarations describing many fields into many
 * declarations describing one field each.  Do the same for local variable
 * declarations.  Extract the initialisation step from for statements,
 * creating an enclosing block to correctly scope declarations.
 *
 * This code was added late in development, so there's a lot of redundant
 * code in other parts of the compiler that could be cleaned up now.
 */

static void
simplifyfield(struct memberdecl *m)
{
	struct fielddecl *f=m->content.field;

	if (f->decls->rest) {
		struct fielddecl *newf;
		newf=newfielddecl(f->modifiers, f->type, f->decls->rest);
		f->decls->rest=0;
		m->rest=newmemberdecl(Field, (union member) newf, m->rest);
	}

	f->type=arrayoftypedepth(f->type, f->decls->adddepth);
	f->decls->adddepth=0;
}

static void
simplifylocalvar(struct stmt *s)
{
	struct fielddecl *f=s->var;

	while (f->decls->rest) {
		struct fielddecl *newf;
		newf=newfielddecl(0,
		                  arrayoftypedepth(f->type, f->decls->adddepth),
		                  f->decls);
		f->decls=f->decls->rest;
		newf->decls->adddepth=0;
		newf->decls->rest=0;
		s->rest=newstmt(Localvar, newf, 0, 0, 0, 0, 0, 0, s->rest);
	}

	f->type=arrayoftypedepth(f->type, f->decls->adddepth);
	f->decls->adddepth=0;
}

static void
simplifyblock(struct stmt *b)
{
	struct catchclause *catch;

	if (b==0) return;

	if (b->tag==For && b->stmt1) {
		struct stmt *s1
		    =newstmt(For, 0, b->expr, 0, b->stmt2, b->stmt3, 0, 0, 0);

		struct stmt *scan=b->stmt1;
		while (scan->rest) scan=scan->rest;
		scan->rest=s1;

		b->tag=Block;
		b->expr=0;
		b->stmt2=b->stmt3=0;
	}

	if (b->tag==Localvar)
		simplifylocalvar(b);

	simplifyblock(b->stmt1);
	simplifyblock(b->stmt2);
	simplifyblock(b->stmt3);
	simplifyblock(b->stmt4);

	for (catch=b->catch; catch; catch=catch->rest) {
		simplifyblock(catch->block);
	}

	simplifyblock(b->rest);
}

static void
simplifytype(struct typedecl *t)
{
	struct memberdecl *m;
	for (m=t->body; m; m=m->rest) {
		switch (m->tag) {
		case Field:
			simplifyfield(m);
			break;
		case Method:
			simplifyblock(m->content.meth->body);
			break;
		case StaticInit:
		case InstanceInit:
			simplifyblock(m->content.init);
			break;
		case Constructor:
			simplifyblock(m->content.constr->body);
			break;
		}
	}
}

/*
 * We must resolve names throughout the program.
 */

static struct name *curtypename;
static struct name *curimports;
static struct name *cursearchpath;
static struct memberdecl *curmember;

static int
lookupprefixforsimpletypename(const char *name, struct name **rv)
{
	struct name tempname;
	struct name *scan;

	/* Look for a matching single type import declaration */
	for (scan=curimports; scan; scan=scan->rest) {
		if (strcmp(scan->id, name)==0) {
			*rv=scan->prefix;
			return 1;
		}
	}

	/* Look for a matching type in the current package */
	tempname.id=name;
	tempname.prefix=curtypename->prefix;
	if (packagetree_findtype(&tempname)) {
		*rv=curtypename->prefix;
		return 1;
	}

	/* Look for a matching type on the type search path */
	for (scan=cursearchpath; scan; scan=scan->rest) {
		tempname.prefix=scan;
		if (packagetree_findtype(&tempname)) {
			*rv=scan;
			return 1;
		}
	}

	return 0;
}

static void
resolvetypename(struct name *n)
{
	/* Do nothing if there's no name to resolve */
	if (n==0) {
		return;
	}

	/* Qualified names are already fully resolved */
	if (n->prefix!=0) {
		return;
	}

	/* Otherwise, look it up as a simple type name */
	if (lookupprefixforsimpletypename(n->id, &n->prefix)==0) {
		/* No type satisfies... */
		fprintf(stderr, "Can't resolve TypeName `%s'\n", n->id);
		if (curmember==0) {
			fputs("which ", stderr);
			spoutname(curtypename, stderr);
			fputs(" inherits from.\n", stderr);
		} else {
			fputs("mentioned in ", stderr);
			switch (curmember->tag) {
			case Field:
				fputs("field initialiser", stderr);
				break;
			case Method:
				fprintf(stderr, "method `%s'",
				        curmember->content.meth->sig->name);
				break;
			case StaticInit:
				fputs("static initialiser", stderr);
				break;
			case InstanceInit:
				fputs("instance initialiser", stderr);
				break;
			case Constructor:
				fprintf(stderr, "constructor `%s'",
				        curmember->content.constr->sig->name);
				break;
			}
			fputs(" in type `", stderr);
			spoutname(curtypename, stderr);
			fputs("'.\n", stderr);
		}
		error("stop");
	}
}

static struct name *
searchtypeforfield(struct name *searchclass, const char *name,
                   struct type **rtype, int *modifiers)
{
	struct typedecl *t=packagetree_findtype(searchclass);
	struct memberdecl *m;
	struct name *i;
	struct name *rv;

	if (t==0) return 0;

	for (m=t->body; m; m=m->rest) {
	    if (m->tag==Field) {
		struct vardecl *d;
		for (d=m->content.field->decls; d; d=d->rest) {
		    if (strcmp(d->name, name)==0) {
			if (rtype) {
			    *rtype=m->content.field->type;
			}
			if (modifiers) {
			    *modifiers=m->content.field->modifiers;
			}
			return searchclass;
		    }
		}
	    }
	}

	rv=searchtypeforfield(t->super, name, rtype, modifiers);
	if (rv) return rv;

	for (i=t->implements; i; i=i->rest) {
		rv=searchtypeforfield(i, name, rtype, modifiers);
		if (rv) return rv;
	}

	return 0;
}

static struct expr *
resolveexprname(struct name *n, struct env *env)
{
	struct expr *prefixexpr;

	/* This case should only apply for unqualified MethodNames.  */
	if (n == 0) {
		return newexpr(This,0,0,0,0,0,0,0,0);
	}

	/* Is it an unqualified name? */
	if (n->prefix==0) {
		/* First, search our environment for local variables
		   and parameters with this name.  */
		while (env) {
			if (strcmp(env->id, n->id)==0) {
				return newexpr(Localacc,0,0,0,env->type,
				               0,0,n->id,0);
			}
			env=env->rest;
		}

		/* Now look for a field in the current class or any of
		   its superclasses, or any of their superinterfaces.  */
		if (searchtypeforfield(curtypename, n->id, 0, 0))
			return newexpr(Primfield,
			               newexpr(This,0,0,0,0,0,0,0,0),
			               0,0,0,0,0,n->id,0);

		/* Look it up as a simple type name.  */
		if (lookupprefixforsimpletypename(n->id, &n->prefix)) {
			return newexpr(Name,0,0,0,0,n,0,0,0);
		}

		/* If all else fails, it must be a top-level package name;
		   our caller must sort out the rest.  */
		return 0;
	}

	/* It's a qualified name.  First, resolve the prefix.  */
	prefixexpr=resolveexprname(n->prefix, env);

	/* If that worked, the prefix is a TypeName or ExpressionName.	*/
	if (prefixexpr) {
		/* So this name is an ExpressionName.  */
		return newexpr(Primfield,prefixexpr,0,0,0,0,0,n->id,0);
	}

	/* Otherwise, see if it's a TypeName.  */
	if (packagetree_findtype(n)) {
		return newexpr(Name,0,0,0,0,n,0,0,0);
	}

	/* If all else fails, it must be a qualified package name; our
	   caller must sort out the rest.  */
	return 0;
}

static void resolveexprs(struct expr *, struct env *);

static void
resolveexpr(struct expr *e, struct env *env)
{
	resolveexprs(e->exp1, env);
	resolveexprs(e->exp2, env);
	resolveexprs(e->exp3, env);
	if (e->type) {
		resolvetypename(e->type->ref);
	}
	/* MethodNames are tricky.  We leave them until the pass where we
	   compute types for each expression.  Luckily, the parser has
	   already separated out the last component of the MethodName,
	   so we only see ExpressionNames here.  */
	if (e->tag == Name) {
		/* ExpressionName: resolve it using the environment.  */
		struct expr *newexpr;
		newexpr=resolveexprname(e->name, env);
		if (newexpr) {
			newexpr->rest=e->rest;
			*e=*newexpr;
		} else {
			fputs("Can't resolve ExpressionName `", stderr);
			spoutname(e->name, stderr);
			fputs("' mentioned in class `", stderr);
			spoutname(curtypename, stderr);
			fputs("'.\n", stderr);
			error("stop");
		}
	} else /* (e->tag == New) */ {
		/* TypeName: resolve it without the environment.  */
		resolvetypename(e->name);
	}
}

static void
resolveexprs(struct expr *e, struct env *env)
{
	while (e) {
		resolveexpr(e, env);
		e=e->rest;
	}
}

static void
resolvefield(struct fielddecl *f, struct env *env)
{
	if (f) {
		struct vardecl *decl;
		resolvetypename(f->type->ref);
		for (decl=f->decls; decl; decl=decl->rest) {
			if (decl->init) {
				resolveexpr(decl->init, env);
			}
		}
	}
}

static void resolveblock(struct stmt *, struct env *);

static void
resolvecatch(struct catchclause *c, struct env *env)
{
	struct env myvar;

	while (c) {
		resolvetypename(c->param->type->ref);
		myvar.id=c->param->name;
		myvar.type=c->param->type;
		myvar.rest=env;
		resolveblock(c->block, &myvar);
		c=c->rest;
	}
}

static void chainlocalvarsandresolveblock(struct type *, struct vardecl *,
                                          struct stmt *, struct env *e);

static void
resolvestmt(struct stmt *s, struct env *env)
{
	if (s) {
		resolvefield(s->var, env);
		resolveblock(s->stmt1, env);
		resolveblock(s->stmt4, env);
		resolvecatch(s->catch, env);
		if (s->tag==For && s->stmt1 && s->stmt1->tag==Localvar) {
			struct stmt ls={Exprstmt,0,s->expr,0,0,0,0,0,0};
			thiscanthappen;
			chainlocalvarsandresolveblock (s->stmt1->var->type,
			        s->stmt1->var->decls, &ls, env);
			chainlocalvarsandresolveblock (s->stmt1->var->type,
			        s->stmt1->var->decls, s->stmt2, env);
			chainlocalvarsandresolveblock (s->stmt1->var->type,
			        s->stmt1->var->decls, s->stmt3, env);
		} else {
			resolveexprs(s->expr, env);
			resolveblock(s->stmt2, env);
			resolveblock(s->stmt3, env);
		}
	}
}

static void
chainlocalvarsandresolveblock(struct type *t, struct vardecl *d,
                              struct stmt *b, struct env *e)
{
	struct env myvar;

	if (d) {
		myvar.id=d->name;
		myvar.type=t;
		myvar.rest=e;
		chainlocalvarsandresolveblock(t, d->rest, b, &myvar);
	} else {
		resolveblock(b, e);
	}
}

static void
resolveblock(struct stmt *b, struct env *e)
{
	if (b) {
		resolvestmt(b, e);
		if (b->tag == Localvar) {
			chainlocalvarsandresolveblock(b->var->type,
			                        b->var->decls, b->rest, e);
		} else {
			resolveblock(b->rest, e);
		}
	}
}

static void
resolvetypenames(struct name *n)
{
	while (n) {
		resolvetypename(n);
		n=n->rest;
	}
}

static void
chainparamsandcall(struct formparlist *p, void (*fun)(void *, struct env *),
                   void *arg, struct env *env)
{
	struct env myvar;

	if (p) {
		resolvetypename(p->type->ref);
		myvar.id=p->name;
		myvar.type=p->type;
		myvar.rest=env;
		chainparamsandcall(p->rest, fun, arg, &myvar);
	} else {
		(*fun)(arg, env);
	}
}

static void
resolvetype(struct typedecl *t)
{
	struct memberdecl *scan;

	curtypename=t->name;
	curimports=t->typeimports;
	cursearchpath=t->typesearchpath;

	resolvetypename(t->super);
	resolvetypenames(t->implements);

	for (scan=t->body; scan; scan=scan->rest) {
		struct methoddecl *meth;
		struct constrdecl *constr;

		curmember=scan;

		switch (scan->tag) {
		case Field:
			resolvefield(scan->content.field, 0);
			break;
		case Method:
			meth=scan->content.meth;
			resolvetypename(meth->type->ref);
			resolvetypenames(meth->throws);
			chainparamsandcall(meth->sig->params,
			    (void (*)(void *, struct env *)) resolveblock,
			    meth->body, 0);
			break;
		case StaticInit:
		case InstanceInit:
			resolveblock(scan->content.init, 0);
			break;
		case Constructor:
			constr=scan->content.constr;
			if (strcmp(t->name->id, constr->sig->name)!=0) {
				fputs("No return type given for `", stderr);
				fputs(constr->sig->name, stderr);
				fputs("' in type `", stderr);
				spoutname(t->name, stderr);
				fputs("'.\n", stderr);
				error("stop");
			}
			resolvetypenames(constr->throws);
			if (constr->cons) {
			    chainparamsandcall(constr->sig->params,
				(void (*)(void *, struct env *)) resolveexprs,
				constr->cons->arglist, 0);
			}
			chainparamsandcall(constr->sig->params,
			    (void (*)(void *, struct env *)) resolveblock,
			    constr->body, 0);
			break;
		}
	}
}

/* Find the type (if any) which contains the non-overriding method which
   this method overrides.  */
static struct name *
overrides(struct name *n, struct methodsig *sig, struct type *ret)
{
	struct typedecl *thistype;
	struct name *over;
	struct memberdecl *m;

	thistype=packagetree_findtype(n);
	if (thistype==0) return 0;

	over=overrides(thistype->super, sig, ret);
	if (over) return over;

	for (m=thistype->body; m; m=m->rest) {
		if (m->tag==Method
		    && equaltypes(ret, m->content.meth->type)
		    && equalsig(sig, m->content.meth->sig)) {
			return thistype->name;
		}
	}

	return 0;
}

/* Work out which methods override which */

static void
overridemethods(struct typedecl *t)
{
	struct memberdecl *m;
	int native=1;

	for (m=t->body; m; m=m->rest) {
		if (m->tag==Method
		    && !(m->content.meth->modifiers & ACC_static)) {
			struct name *over;
			over=overrides(t->super, m->content.meth->sig,
			               m->content.meth->type);
			m->content.meth->overrides=over;
			if (over==0) {
				native=0;
			}
		}
	}

	if (native) {
		t->modifiers|=ACC_native;
	}
}

/*
 * In the following several functions, we determine the type of every
 * expression in the program, and determine the maximum depth of array
 * nesting for each type.
 */

static void
notetype(struct type *t)
{
	struct typedecl *td;

	if (t->dims==0 && t->prim!=Ref) return;

	switch (t->prim) {
	case Boolean:
	case Byte:
	case Short:
	case Int:
	case Long:
	case Char:
	case Float:
	case Double:
		noteprimtype(t->prim, t->dims);
		break;
	case Ref:
		td=packagetree_findtype(t->ref);

		/* Check it exists, for sanity's sake */
		if (td==0) {
			fputs("Nonexistent type `", stderr);
			spoutname(t->ref, stderr);
			fputs("' mentioned in class `", stderr);
			spoutname(curtypename, stderr);
			fputs("'.\n", stderr);
			error("stop");
		}

		if (t->dims > td->arrdepth) {
			td->arrdepth=t->dims;
		}
		break;
	case Void:
	case Null:
		break;
	}
}

static struct type *
unarynumericpromote(struct type *t)
{
	if (t->dims) return 0;
	if (t->prim==Boolean || t->prim==Ref || t->prim==Void) return 0;
	if (t->prim==Double) return &DoubleType;
	if (t->prim==Float) return &FloatType;
	if (t->prim==Long) return &LongType;
	return &IntType;
}

static struct type *
binarynumericpromote(struct type *t1, struct type *t2)
{
	if (t1->dims || t2->dims) return 0;
	if (t1->prim==Boolean || t1->prim==Ref || t1->prim==Void) return 0;
	if (t2->prim==Boolean || t2->prim==Ref || t2->prim==Void) return 0;
	if (t1->prim==Double || t2->prim==Double) return &DoubleType;
	if (t1->prim==Float || t2->prim==Float) return &FloatType;
	if (t1->prim==Long || t2->prim==Long) return &LongType;
	return &IntType;
}

static struct expr *
pushbacktype(struct expr *e, struct type *t)
{
	if (e->type) notetype(e->type);
	notetype(t);

	if (e->tag==Name) {
		e->type=t;
		return e;
	} else if (equaltypes(e->type, t)) {
		return e;
	} else {
		struct expr *cast;
		cast=newexpr(Cast,e,0,0,t,0,0,0,e->rest);
		e->rest=0;
		return cast;
	}
}

static struct expr *
castparamshelp(struct expr *params, struct formparlist *formals)
{
	if (params) {
		params->rest=castparamshelp(params->rest, formals->rest);
		return pushbacktype(params, formals->type);
	} else {
		return 0;
	}
}

static int
canprimitivewiden(struct type *from, struct type *to)
{
	int fp=from->prim, tp=to->prim;

	/* Arrays don't participate in primitive widening.  */
	if (from->dims || to->dims) return 0;

	/* Now, we follow the rules from JLS S5.1.2 */
	if (fp==Byte
	    && (tp==Short || tp==Int || tp==Long || tp==Float || tp==Double))
		return 1;

	if ((fp==Short || fp==Char)
	    && (tp==Int || tp==Long || tp==Float || tp==Double))
		return 1;

	if (fp==Int && (tp==Long || tp==Float || tp==Double))
		return 1;

	if (fp==Long && (tp==Float || tp==Double))
		return 1;

	if (fp==Float && tp==Double)
		return 1;

	return 0;
}

static int
canreferencewiden(struct type *from, struct type *to)
{
	if (from->prim==Null && (to->prim==Ref || to->dims))
		return 1;

	if ((from->prim==Ref || from->dims) && equaltypes(to, &ObjectType))
		return 1;

	if (from->prim!=Ref || to->prim!=Ref)
		return 0;

	if (from->dims != to->dims)
		return 0;

	if (issubclass(from->ref, to->ref))
		return 1;

	if (implements(from->ref, to->ref))
		return 1;

	return 0;
}

/* Decide whether the formal parameters P are applicable for a method
   call with actual parameter expressions A.  */
static int
isapplicable(struct formparlist *p, struct expr *a)
{
	if (p==0 && a==0) return 1;
	if (p==0 || a==0) return 0;
	if (! isapplicable(p->rest, a->rest)) return 0;

	if (equaltypes(a->type, p->type)) return 1;
	if (canprimitivewiden(a->type, p->type)) return 1;
	if (canreferencewiden(a->type, p->type)) return 1;
	return 0;
}

/* Determine whether T is more specific than U.  */
static int
ismorespecific(struct formparlist *t, struct formparlist *u)
{
	if (t==0 && u==0) return 1;
	if (t==0 || u==0) thiscanthappen;
	if (!ismorespecific(t->rest, u->rest)) return 0;

	if (equaltypes(t->type, u->type)) return 1;
	if (canprimitivewiden(t->type, u->type)) return 1;
	if (canreferencewiden(t->type, u->type)) return 1;
	return 0;
}

static struct name *
castparams(struct name *searchclass, const char *name, int constructorcall,
           int ignoreoverrides,
           struct expr **params, struct type **rtype, int *modifiers)
{
	struct typedecl *t=packagetree_findtype(searchclass);
	struct memberdecl *m;
	struct methoddecl *maximalmeth=0;
	struct constrdecl *maximalconstr=0;
	int seenconstructor=0;

	if (t==0) return 0;

	for (m=t->body; m; m=m->rest) {
		if (m->tag==Constructor) {
			seenconstructor=1;
		}
		if (m->tag==Method && !constructorcall) {
			if (strcmp(name, m->content.meth->sig->name)!=0)
			{
				continue; /* to next type member */
			}
			if (!isapplicable(m->content.meth->sig->params,
			                  *params))
			{
				continue;
			}
			if (maximalmeth==0 ||
			    ismorespecific(m->content.meth->sig->params,
			                   maximalmeth->sig->params))
			{
				maximalmeth=m->content.meth;
			}
		} else if (m->tag==Constructor && constructorcall) {
			if(!isapplicable(m->content.constr->sig->params,
			                 *params))
			{
				continue;
			}
			if (maximalconstr==0 ||
			    ismorespecific(m->content.constr->sig->params,
			                   maximalconstr->sig->params))
			{
				maximalconstr=m->content.constr;
			}
		}
	}
	if (maximalmeth) {
		/* Found a maximally specific method.  Set return
		   type and modifiers and push appropriate casts.  */
		*rtype=maximalmeth->type;
		*params=castparamshelp(*params,
		                       maximalmeth->sig->params);
		*modifiers=maximalmeth->modifiers;
		return maximalmeth->overrides && !ignoreoverrides ?
		       maximalmeth->overrides : t->name;
	}
	if (maximalconstr) {
		/* Found a maximally specific constructor.
		   Push appropriate casts.  */
		*params=castparamshelp(*params,
		            maximalconstr->sig->params);
		return t->name;
	}
	if (constructorcall) {
		if (seenconstructor==0 && *params==0) {
			return t->name;
		}
		return 0;
	}

	if (t->modifiers & ACC_interface) {
		struct name *i, *rv;
		for (i=t->implements; i; i=i->rest) {
			rv=castparams(i, name, 0, ignoreoverrides,
			              params, rtype, modifiers);
			if (rv) return rv;
		}
		return 0;
	} else {
		return castparams(t->super, name, 0, ignoreoverrides,
		                  params, rtype, modifiers);
	}
}

static void decorateexprs(struct expr *);

static struct expr *
stringconvert(struct expr *e)
{
	const char *tn;
	enum typeid kind=e->type->prim;
	if (e->type->dims) kind=Ref;

	if (equaltypes(&StringType, e->type)) {
		return newexpr(NullString, e, 0, 0, &StringType, 0, 0, 0, 0);
	}

	switch(kind) {
	case Null:
		return newexpr(Stringlit, 0, 0, 0, &StringType, 0,
		               stringpool_intern("null"), "null", 0);
	case Ref:
		return newexpr(NullString,
		               newexpr(Meth,
		                       newexpr(NullString,
		                               pushbacktype(e, &ObjectType),
		                               0, 0, &ObjectType, 0, 0, 0, 0),
		                       0, 0, &StringType, 0, 0, "toString", 0),
			       0, 0, &StringType, 0, 0, 0, 0);
	case Boolean:	tn="Boolean"; break;
	case Char:	tn="Character"; break;
	case Byte: case Short:
			e=newexpr(Cast, e, 0, 0, &IntType, 0, 0, 0, 0);
	case Int:	tn="Integer"; break;
	case Long:	tn="Long"; break;
	case Float:	tn="Float"; break;
	case Double:	tn="Double"; break;
	default: thiscanthappen;
	}

	return stringconvert
	           (newexpr(New, 0, e, 0,
	                    referencetype(newname(ObjectType.ref->prefix,
	                                          tn, 0, 0)),
	                    0, 0, 0, 0));
}

static void
decorateexpr(struct expr *e)
{
	struct type *promotedtype;
	struct name *searchclass;
	struct name *methclass;
	struct expr **rexp;

	int modifiers;

	if (e->tag!=Arrayinit) decorateexprs(e->exp1);
	decorateexprs(e->exp2);
	decorateexprs(e->exp3);

	switch (e->tag) {

	case Assign:	case Multass:	case Divass:	case Remass:
	case Addass:	case Subass:	case Lshiftass:	case Rshiftass:
	case Urshiftass: case Andass:	case Xorass:	case Orass:
		e->type=e->exp1->type;
		e->exp2=pushbacktype(e->exp2, e->type);
		break;

	case Predec:	case Preinc:	case Postdec:	case Postinc:
		e->type=e->exp1->type;
		break;

	case Cond:
		if (e->exp2->type->prim==Null) {
			e->exp2=pushbacktype(e->exp2, e->exp3->type);
		}
		if (e->exp3->type->prim==Null) {
			e->exp3=pushbacktype(e->exp3, e->exp2->type);
		}
		if (equaltypes(e->exp2->type, e->exp3->type)) {
			e->type=e->exp2->type;
			break;
		}
		error("This kind of Cond not yet decorated");

	case Inst:
		if (e->type->dims) {
			error("Cannot handle instanceof with an array type");
		}
		e->name=e->type->ref;
		e->exp1=pushbacktype(e->exp1, &ObjectType);
		/* and fall through... */
	case Condor: case Condand:
		e->type=&BooleanType;
		break;

	case Or: case Xor: case And:
		if (equaltypes(e->exp1->type, &BooleanType)
		    && equaltypes(e->exp2->type, &BooleanType))
		{
			e->type=&BooleanType;
		} else {
			e->type=binarynumericpromote(e->exp1->type,
			                             e->exp2->type);
		}
		break;

	case Eq: case Neq:
		promotedtype=binarynumericpromote(e->exp1->type, e->exp2->type);
		if (promotedtype) {
			e->exp1=pushbacktype(e->exp1, promotedtype);
			e->exp2=pushbacktype(e->exp2, promotedtype);
		} else if (e->exp1->type->prim==Boolean) {
			e->exp2=pushbacktype(e->exp2, &BooleanType);
		} else {
			e->exp1=pushbacktype(e->exp1, &ObjectType);
			e->exp2=pushbacktype(e->exp2, &ObjectType);
		}
		e->type=&BooleanType;
		break;

	case Lt: case Gt: case Leq: case Geq:
		promotedtype=binarynumericpromote(e->exp1->type, e->exp2->type);
		e->exp1=pushbacktype(e->exp1, promotedtype);
		e->exp2=pushbacktype(e->exp2, promotedtype);
		e->type=&BooleanType;
		break;

	case Lshift: case Rshift: case Urshift:
		e->type=unarynumericpromote(e->exp1->type);
		if (e->type) {
			e->exp1=pushbacktype(e->exp1, e->type);
		}
		promotedtype=unarynumericpromote(e->exp2->type);
		if (promotedtype) {
			e->exp2=pushbacktype(e->exp2, promotedtype);
		}
		break;

	case Add:
		e->type=binarynumericpromote(e->exp1->type, e->exp2->type);
		if (e->type==0) {
			e->type=&StringType;
			e->exp3=stringconvert(e->exp2);
			e->exp2=stringconvert(e->exp1);
			e->exp1=newexpr(Add, e->exp2, e->exp3, 0,
			                &StringType, 0, 0, 0, 0);
			e->tag=NullString;
			e->exp2=e->exp3=0;
		}
		break;

	case Sub: case Mult: case Div: case Rem:
		e->type=binarynumericpromote(e->exp1->type, e->exp2->type);
		break;

	case Cast:
		if (e->exp1->type->prim==Ref && e->type->prim==Ref
		    && e->exp1->type->dims==0 && e->type->dims==0
		    && !implements(e->exp1->type->ref, e->type->ref))
		{
			e->exp1=pushbacktype(e->exp1, &ObjectType);
		}
		break;

	case Localacc:
		/* No action needed */
		break;

	case Not:
		e->type=&BooleanType;
		break;

	case Comp: case Plus: case Minus:
		e->type=unarynumericpromote(e->exp1->type);
		break;

	case Name:
		/* Leave Names alone */
		break;

	case Prim:
		error("Prim not used; this can't happen!");

	case Intlit: e->type=&IntType; break;
	case Longlit: e->type=&LongType; break;
	case Boollit: e->type=&BooleanType; break;
	case Charlit: e->type=&CharType; break;
	case Floatlit: e->type=&FloatType; break;
	case Doublit: e->type=&DoubleType; break;
	case Nulllit: e->type=&NullType; break;
	case Stringlit:
		e->type=&StringType;
		e->integ=stringpool_intern_wide(e->str);
		break;

	case This:
		e->type=referencetype(curtypename);
		break;

	case New:
		if (e->name) {
			e->type=newtype(e->integ, Ref, e->name);
		} else {
			e->type=newtype(e->integ, e->type->prim, e->type->ref);
		}
		if (e->exp1) {
			e->type->dims++;
		} else {
			struct name *methclass;
			methclass=castparams(e->name, 0, 1, 0,
			                     &(e->exp2), 0, 0);
			if (!methclass) {
				fputs("Couldn't find constructor for class `",
				      stderr);
				spoutname(e->name, stderr);
				fputs("' mentioned in class `", stderr);
				spoutname(curtypename, stderr);
				fputs("'.\n", stderr);
				error("stop");
			}
		}
		break;

	case Primfield:
		if (e->exp1->tag==Name) {
			searchclass=e->exp1->name;
		} else {
			searchclass=e->exp1->type->ref;
			if (e->exp1->type->dims
			    && strcmp("length", e->str)==0)
			{
				e->type=&IntType;
				e->integ=0;
				break;
			}
		}
		methclass=searchtypeforfield(searchclass, e->str, &(e->type),
		                             &modifiers);
		e->integ=modifiers;
		if (methclass) {
			e->exp1=pushbacktype(e->exp1, referencetype(methclass));
		} else {
			fprintf(stderr, "Couldn't find field `%s' in class `",
			                e->str);
			spoutname(searchclass, stderr);
			fputs("' mentioned in class `", stderr);
			spoutname(curtypename, stderr);
			fputs("'.\n", stderr);
			error("stop");
		}
		break;

	case Super:
		searchclass=packagetree_findtype(curtypename)->super;
		if (e->integ) {
			/* Superclass field access */
			methclass=searchtypeforfield(searchclass, e->str,
			                             &(e->type), &modifiers);
			e->tag=Primfield;
			e->integ=modifiers;
			if (methclass) {
				e->exp1=newexpr(Cast,
				            newexpr(This, 0, 0, 0,
				                    referencetype(curtypename),
				                    0, 0, 0, 0),
				            0, 0, referencetype(methclass),
				            0, 0, 0, 0);
			} else {
				fprintf(stderr, "Couldn't find field `%s' "
				                "in superclass of `", e->str);
				spoutname(curtypename, stderr);
				fputs("'.\n", stderr);
				error("stop");
			}
		} else {
			/* Superclass method access */
			methclass=castparams(searchclass, e->str, 0, 1,
			                     &(e->exp2),&(e->type),&modifiers);
			e->tag=Meth;
			e->integ=modifiers | ACC_final;
			e->exp1=newexpr(This, 0, 0, 0, 0, 0, 0, 0, 0);
			decorateexpr(e->exp1);
			if (methclass) {
				e->exp1=pushbacktype(e->exp1,
				                     referencetype(methclass));
			} else {
				fprintf(stderr, "Couldn't find an "
				                "appropriate method `%s' "
				                "in superclass of `", e->str);
				spoutname(curtypename, stderr);
				fputs("'.\n", stderr);
				error("stop");
			}
		}
		break;

	case Meth:
		if (e->exp1->tag==Name) {
			searchclass=e->exp1->name;
		} else if (e->exp1->type->dims) {
			searchclass=ObjectType.ref;
		} else {
			searchclass=e->exp1->type->ref;
		}
		methclass=castparams(searchclass, e->str, 0, 0,
		                     &(e->exp2), &(e->type), &modifiers);
		e->integ=modifiers;
		if (modifiers & ACC_static && e->exp1->tag==This) {
			/* Special case: we can call a static method
			   from a context where there is no 'this'.  */
			e->exp1->tag=Name;
		}
		if (methclass) {
			e->exp1=pushbacktype(e->exp1, referencetype(methclass));
		} else {
			fprintf(stderr, "Couldn't find an appropriate "
			                "method `%s' in class `", e->str);
			spoutname(searchclass, stderr);
			fputs("'\n  mentioned in class `", stderr);
			spoutname(curtypename, stderr);
			fputs("'.\n", stderr);
			error("stop");
		}
		break;

	case Array:
		e->exp2=pushbacktype(e->exp2,
		                     unarynumericpromote(e->exp2->type));
		if (e->exp1->type->dims) {
			e->type=newtype(e->exp1->type->dims-1,
			                e->exp1->type->prim,
			                e->exp1->type->ref);
		}
		break;

	case NullString: thiscanthappen;

	case Arrayinit:
		promotedtype=newtype(e->type->dims,e->type->prim,e->type->ref);
		if (--(promotedtype->dims)<0) {
			error("Array initialisers nested too deeply");
		}
		e->integ=0;
		for (rexp=&(e->exp1); *rexp; rexp=&((*rexp)->rest)) {
			if ((*rexp)->tag==Arrayinit) {
				(*rexp)->type=promotedtype;
			}
			decorateexpr(*rexp);
			*rexp=pushbacktype(*rexp, promotedtype);
			e->integ++;
		}
		break;
	}

	if (e->type) notetype(e->type);
}

static void
decorateexprs(struct expr *e)
{
	while (e) {
		decorateexpr(e);
		e=e->rest;
	}
}

static void
decoratedecls(struct vardecl *d, struct type *t)
{
	while (d) {
		if (d->init) {
			if (d->init->tag==Arrayinit) {
				d->init->type=t;
			}
			decorateexprs(d->init);
			d->init=pushbacktype(d->init, t);
		}
		d=d->rest;
	}
}

static void
decoratefield(struct fielddecl *f)
{
	notetype(f->type);
	decoratedecls(f->decls, f->type);
}

static void decorateblock(struct stmt *);

static void
decoratecatchclauses(struct catchclause *c)
{
	while (c) {
		notetype(c->param->type);
		decorateblock(c->block);
		c=c->rest;
	}
}

static void
decorateblock(struct stmt *b)
{
	while (b) {
		if (b->tag==NativeStmt) {
			return;
		}

		if (b->var) decoratefield(b->var);
		if (b->expr) {
			decorateexprs(b->expr);
			if (b->tag==Return) {
				b->expr=pushbacktype(b->expr,
				            curmember->content.meth->type);
			}
			notetype(b->expr->type);
		}
		if (b->stmt1) decorateblock(b->stmt1);
		if (b->stmt2) decorateblock(b->stmt2);
		if (b->stmt3) decorateblock(b->stmt3);
		if (b->stmt4) decorateblock(b->stmt4);
		if (b->catch) decoratecatchclauses(b->catch);
		b=b->rest;
	}
}

static void
decorateparams(struct formparlist *p)
{
	while (p) {
		notetype(p->type);
		p=p->rest;
	}
}

static void
decoratemethod(struct methoddecl *m)
{
	notetype(m->type);
	decorateparams(m->sig->params);
	decorateblock(m->body);
}

static void
decorateconstructor(struct constrdecl *c)
{
	decorateparams(c->sig->params);
	if (c->cons) {
		decorateexprs(c->cons->arglist);
		if (c->cons->arglist) {
			if (castparams(c->cons->super
			              ? packagetree_findtype(curtypename)->super
			                : curtypename,
			              0, 1, 0, &(c->cons->arglist), 0, 0)==0) {
				error("Can't find constructor");
			}
		}
	}
	decorateblock(c->body);
}

static void
decoratetype(struct typedecl *t)
{
	struct memberdecl *m;

	curtypename=t->name;
	for (m=t->body; m; m=m->rest) {
		curmember=m;
		switch (m->tag) {
		case Field:
			decoratefield(m->content.field);
			break;
		case Method:
			decoratemethod(m->content.meth);
			break;
		case StaticInit:
		case InstanceInit:
			decorateblock(m->content.init);
			break;
		case Constructor:
			decorateconstructor(m->content.constr);
			break;
		}
	}
}

/*
 * Program transformations which are to be done before we start generating
 * code.
 */

void
transform(void)
{
	packagetree_foralltypes(javalangobjecthasnosupertype);
	packagetree_foralltypes(internnames);
	packagetree_foralltypes(simplifytype);
	packagetree_foralltypes(resolvetype);
	packagetree_foralltypes(overridemethods);
	packagetree_foralltypes(decoratetype);
}

#endif /* SEEN_transform_h */
