/***************************************************************************
 *   Copyright (C) 2003 by Sbastien Laot                                 *
 *   sebastien.laout@tuxfamily.org                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qlabel.h>
#include <kurl.h>
#include <qlayout.h>
#include <kiconloader.h>
#include <qcursor.h>
#include <klocale.h>
#include <qpushbutton.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <kcolorbutton.h>
#include <qhgroupbox.h>
//#include <qhbox.h>
//#include <kglobalsettings.h>

#include "linklabel.h"
#include "variouswidgets.h"

#include "basket.h" // For textToHTML()
#include "item.h" // For LinkLabel::enterEvent() to not make hover effect during edit

/** LinkLook */

LinkLook *LinkLook::soundLook = 0L;
LinkLook *LinkLook::fileLook  = 0L;
LinkLook *LinkLook::localLook = 0L;
LinkLook *LinkLook::urlLook   = 0L;
LinkLook *LinkLook::noUrlLook = 0L;

LinkLook::LinkLook(bool underline, bool italic, bool bold, bool hoverUnder,
                   QColor color, QColor hoverColor,
                   bool showIcon, int iconSize, bool onTopIcon)
{
	setLook( underline, italic, bold, hoverUnder, color, hoverColor, showIcon, iconSize, onTopIcon );
}

LinkLook::LinkLook(const LinkLook &other)
{
	setLook( other.underline(), other.italic(), other.bold(), other.hoverUnder(),
	         other.color(), other.hoverColor(), other.showIcon(), other.iconSize(), other.onTopIcon() );
}

void LinkLook::setLook(bool underline, bool italic, bool bold, bool hoverUnder,
                       QColor color, QColor hoverColor,
                       bool showIcon, int iconSize, bool onTopIcon)
{
	m_underline  = underline;
	m_italic     = italic;
	m_bold       = bold;
	m_hoverUnder = hoverUnder;
	m_color      = color;
	m_hoverColor = hoverColor;
	m_showIcon   = showIcon;
	m_iconSize   = iconSize;
	m_onTopIcon  = onTopIcon;
}

LinkLook* LinkLook::lookForURL(const KURL &url)
{
	return url.isLocalFile() ? localLook : urlLook;
}

QString LinkLook::toCSS(const QString &cssClass)
{
	// Set the link class:
	QString css = QString("      .%1 a { display: block; width: 100%;").arg(cssClass);
	if (m_underline == true)
		css += " text-decoration: underline;";
	else
		css += " text-decoration: none;";
	if (m_italic == true)
		css += " font-style: italic;";
	if (m_bold == true)
		css += " font-weight: bold;";
	css += QString(" color: %1; }\n").arg(m_color.name());

	// Set the hover state class:
	QString hover;
	if (m_hoverUnder)
		hover = QString("text-decoration: %3;").arg(m_hoverUnder != m_underline ? "underline" : "none");
	if (m_hoverColor != m_color) {
		if (!hover.isEmpty())
			hover += " ";
		hover += QString("color: %4;").arg(m_hoverColor.name());
	}

	// But include it only if it contain a different style than non-hover state:
	if (!hover.isEmpty())
		css += QString("      .%1 a:hover { %2 }\n").arg(cssClass, hover);

	return css;
}

/** LinkLabel */

LinkLabel::LinkLabel(int hAlign, int vAlign, QWidget *parent, const char *name, WFlags f)
 : QFrame(parent, name, f), m_isSelected(false), m_isHovered(false), m_look(0)
{
	initLabel(hAlign, vAlign);
}

LinkLabel::LinkLabel(const QString &title, const QString &icon, LinkLook *look, int hAlign, int vAlign,
                     QWidget *parent, const char *name, WFlags f)
 : QFrame(parent, name, f), m_isSelected(false), m_isHovered(false), m_look(0)
{
	initLabel(hAlign, vAlign);
	setLink(title, icon, look);
}

void LinkLabel::initLabel(int hAlign, int vAlign)
{
	m_layout  = new QBoxLayout(this, QBoxLayout::LeftToRight);
	m_icon    = new QLabel(this);
	m_title   = new QLabel(this);
	m_spacer1 = new QSpacerItem(0, 0, QSizePolicy::Preferred/*Expanding*/, QSizePolicy::Preferred/*Expanding*/);
	m_spacer2 = new QSpacerItem(0, 0, QSizePolicy::Preferred/*Expanding*/, QSizePolicy::Preferred/*Expanding*/);

	m_hAlign = hAlign;
	m_vAlign = vAlign;

	m_title->setTextFormat(Qt::PlainText);

	// DEGUB:
	//m_icon->setPaletteBackgroundColor("lightblue");
	//m_title->setPaletteBackgroundColor("lightyellow");
}

LinkLabel::~LinkLabel()
{
}

void LinkLabel::setLink(const QString &title, const QString &icon, LinkLook *look)
{
	if (look)
		m_look = look; // Needed for icon size

	m_title->setText(title);
	m_title->setShown( ! title.isEmpty() );

	if (icon.isEmpty())
		m_icon->clear();
	else
		m_icon->setPixmap( DesktopIcon(icon, m_look->iconSize()) );
	m_icon->setShown( ! icon.isEmpty() );

	if (look)
		setLook(look);
}

void LinkLabel::setLook(LinkLook *look) // FIXME: called externaly (so, without setLink()) it's buggy (icon not
{
	m_look = look;

	QFont font;
	font.setBold( look->bold() );
	font.setUnderline( look->underline() );
	font.setItalic( look->italic() );
	m_title->setFont( font );
	m_title->setPaletteForegroundColor( m_isSelected ? KApplication::palette().active().highlightedText() : look->color() );

	m_icon->setShown( look->showIcon() && m_icon->pixmap() && ! m_icon->pixmap()->isNull() );

	setAlign(m_hAlign, m_vAlign);
}

void LinkLabel::setAlign(int hAlign, int vAlign)
{
	m_hAlign = hAlign;
	m_vAlign = vAlign;

	if (!m_look)
		return;

	// Define alignment flags :
	//FIXME TODO: Use directly flags !
	int hFlag, vFlag, wBreak;
	switch (hAlign) {
		default:
		case 0: hFlag = Qt::AlignLeft;    break;
		case 1: hFlag = Qt::AlignHCenter; break;
		case 2: hFlag = Qt::AlignRight;   break;
	}
	switch (vAlign) {
		case 0: vFlag = Qt::AlignTop;     break;
		default:
		case 1: vFlag = Qt::AlignVCenter; break;
		case 2: vFlag = Qt::AlignBottom;  break;
	}
	wBreak = Qt::WordBreak * !(!m_look->onTopIcon() && hAlign == 1);

	// Clear the widget :
	m_layout->removeItem(m_spacer1);
	m_layout->remove(m_icon);
	m_layout->remove(m_title);
	m_layout->removeItem(m_spacer2);

	// Otherwise, minimumSize will be incoherent (last size ? )
	m_layout->setResizeMode(QLayout::Minimum);

	// And re-populate the widget with the appropriates things and order
	bool addSpacers = m_look->onTopIcon() || hAlign == 1;
	if (m_look->onTopIcon()) {
		m_layout->setDirection(QBoxLayout::TopToBottom);
		//m_title->setSizePolicy( QSizePolicy(QSizePolicy::Preferred/*Expanding*/, QSizePolicy::Fixed, 0, 0, false) );
		m_icon->setSizePolicy( QSizePolicy(QSizePolicy::Preferred/*Expanding*/, QSizePolicy::Fixed, 0, 0, false) );
		m_spacer1->changeSize( 0, 0, QSizePolicy::Preferred, QSizePolicy::Preferred );
		m_spacer2->changeSize( 0, 0, QSizePolicy::Preferred, QSizePolicy::Preferred );
	} else {
		m_layout->setDirection(QBoxLayout::LeftToRight);
		//m_title->setSizePolicy( QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Maximum/*Expanding*/, 0, 0, false) );
		m_icon->setSizePolicy( QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Preferred/*Expanding*/, 0, 0, false) );
		m_spacer1->changeSize( 0, 0, QSizePolicy::Expanding, QSizePolicy::Preferred/*Expanding*/ );
		m_spacer2->changeSize( 0, 0, QSizePolicy::Expanding, QSizePolicy::Preferred/*Expanding*/ );
	}
	m_icon->setAlignment( hFlag | vFlag );
	m_title->setAlignment( hFlag | vFlag | wBreak );
	if ( addSpacers && !(m_look->onTopIcon() && vAlign == 0) ||
	   (!m_look->onTopIcon() && m_title->text().isEmpty() && hAlign == 2) )
		m_layout->addItem(m_spacer1);
	if ( !m_look->onTopIcon() && hAlign == 2 ) { // If align at right, icon is at right
		m_layout->addWidget(m_title);
		m_layout->addWidget(m_icon);
	} else {
		m_layout->addWidget(m_icon);
		m_layout->addWidget(m_title);
	}
	if ( addSpacers && !(m_look->onTopIcon() && vAlign == 2) ||
	   (!m_look->onTopIcon() && m_title->text().isEmpty() && hAlign == 0) )
		m_layout->addItem(m_spacer2);
}

void LinkLabel::enterEvent(QEvent*)
{
	Item *item = dynamic_cast<Item*>(parent());
	if (item && item->parentBasket()->isDuringEdit())
		return;

	m_isHovered = true;
	if ( ! m_isSelected )
		m_title->setPaletteForegroundColor(m_look->hoverColor());

	if (m_look->hoverUnder()) {
		QFont font = m_title->font();
		font.setUnderline( ! m_look->underline() );
		m_title->setFont( font );
	}
}

void LinkLabel::leaveEvent(QEvent*)
{
	m_isHovered = false;
	if ( ! m_isSelected )
		m_title->setPaletteForegroundColor(m_look->color());

	if (m_look->hoverUnder()) {
		QFont font = m_title->font();
		font.setUnderline( m_look->underline() );
		m_title->setFont( font );
	}
}

void LinkLabel::setSelected(bool selected)
{
	m_isSelected = selected;
	if (selected)
		m_title->setPaletteForegroundColor(KApplication::palette().active().highlightedText());
	else if (m_isHovered)
		m_title->setPaletteForegroundColor(m_look->hoverColor());
	else
		m_title->setPaletteForegroundColor(m_look->color());
}

void LinkLabel::setPaletteBackgroundColor(const QColor &color)
{
	QFrame::setPaletteBackgroundColor(color);
	m_title->setPaletteBackgroundColor(color);
}

int LinkLabel::heightForWidth(int w) const
{
	int iconS  = (m_icon->isShown())   ? m_look->iconSize()                 : 0; // Icon size
	int iconW  = (m_look->onTopIcon()) ? 0 : iconS;                              // Icon width to remove to w
	int titleH = (m_title->isShown())  ? m_title->heightForWidth(w - iconW) : 0; // Title height

	if (m_look->onTopIcon())
		return titleH + iconS;                     // No margin for the moment !
	else
		return (titleH >= iconS) ? titleH : iconS; // Idem !
}

QString LinkLabel::toHtml(const QString &imageName)
{
	QString begin = "<font color=" + m_look->color().name() + ">";
	QString end   = "</font>";
	if (m_look->underline()) {
		begin += "<u>";
		end.prepend("</u>");
	}
	if (m_look->italic()) {
		begin += "<i>";
		end.prepend("</i>");
	}
	if (m_look->bold()) {
		begin += "<b>";
		end.prepend("</b>");
	}
	if (m_look->showIcon() && m_icon->pixmap() != 0L) {
		QPixmap icon(*m_icon->pixmap());
		begin.prepend("<img src=" + imageName + " style=\"vertical-align: middle\">" +
			QString(m_look->onTopIcon() ? "<br>" : " "));
		QMimeSourceFactory::defaultFactory()->setPixmap(imageName, icon);
	} else
		QMimeSourceFactory::defaultFactory()->setData(imageName, 0L);
	return begin + Basket::textToHTMLWithoutP(m_title->text()) + end;
}

/** LinkLookEditWidget **/

LinkLookEditWidget::LinkLookEditWidget(LinkLook *look, const QString exTitle, const QString exIcon,
                                       QWidget *parent, const char *name, WFlags fl)
 : QWidget(parent, name, fl)
{
	QLabel      *label;
	QVBoxLayout *layout = new QVBoxLayout(this, 8, 6);

	m_look = look;

	m_italic = new QCheckBox(i18n("I&talic"), this);
	m_italic->setChecked(look->italic());
	layout->addWidget(m_italic);

	m_bold = new QCheckBox(i18n("&Bold"), this);
	m_bold->setChecked(look->bold());
	layout->addWidget(m_bold);

	QGridLayout *gl = new QGridLayout(layout, 4, 3);
	gl->addItem(new QSpacerItem(0, 0, QSizePolicy::Expanding), 1, 2);

	m_underline = new QComboBox(false, this);
	m_underline->insertItem(i18n("Always"));
	m_underline->insertItem(i18n("Never"));
	m_underline->insertItem(i18n("On mouse hovering"));
	m_underline->insertItem(i18n("When mouse is outside"));
	m_underline->setCurrentItem( look->underline() ? (look->hoverUnder()?3:0) : (look->hoverUnder()?2:1) );
	label = new QLabel(m_underline, i18n("&Underline:"), this);
	gl->addWidget(label,       0, 0);
	gl->addWidget(m_underline, 0, 1);

	m_color = new KColorButton(this);
	m_color->setColor(look->color());
	label = new QLabel(m_color, i18n("Colo&r:"), this);
	gl->addWidget(label,   1, 0);
	gl->addWidget(m_color, 1, 1);

	m_hoverColor = new KColorButton(this);
	m_hoverColor->setColor(look->hoverColor());
	label = new QLabel(m_hoverColor, i18n("&Mouse hover color:"), this);
	gl->addWidget(label,        2, 0);
	gl->addWidget(m_hoverColor, 2, 1);

	QHBoxLayout *icoLay = new QHBoxLayout(0L, 0, 6);
	m_onTopIcon = new QComboBox(false, this);
	m_onTopIcon->insertItem(i18n("No icon", "None"));
	m_onTopIcon->insertItem(i18n("On side"));
	m_onTopIcon->insertItem(i18n("On top"));
	if ( ! look->showIcon() )
		m_onTopIcon->setCurrentItem(0);
	else
		m_onTopIcon->setCurrentItem(look->onTopIcon() + 1);
	m_iconSize = new IconSizeCombo(false, this);
	m_iconSize->setSize(look->iconSize());
	icoLay->addWidget(m_onTopIcon);
	icoLay->addWidget(m_iconSize);
	label = new QLabel(m_onTopIcon, i18n("&Icon:"), this);
	gl->addWidget(label,  3, 0);
	gl->addItem(  icoLay, 3, 1);

	QGroupBox *gb = new QHGroupBox(i18n("Example"), this);
	m_exLook = new LinkLook(*look);
	m_example = new LinkLabel(exTitle, exIcon, m_exLook, 1, 1, gb);
	m_example->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
	m_example->setCursor(QCursor(Qt::PointingHandCursor));
	layout->addWidget(gb);
	m_exTitle = exTitle;
	m_exIcon  = exIcon;

	slotChangeLook();

	connect( m_underline,  SIGNAL(activated(int)),         this, SLOT(slotChangeLook())              );
	connect( m_italic,     SIGNAL(clicked()),              this, SLOT(slotChangeLook())              );
	connect( m_bold,       SIGNAL(clicked()),              this, SLOT(slotChangeLook())              );
	connect( m_color,      SIGNAL(changed(const QColor&)), this, SLOT(slotChangeLook(const QColor&)) );
	connect( m_hoverColor, SIGNAL(changed(const QColor&)), this, SLOT(slotChangeLook(const QColor&)) );
	connect( m_onTopIcon,  SIGNAL(activated(int)),         this, SLOT(slotChangeLook(int))           );
	connect( m_iconSize,   SIGNAL(activated(int)),         this, SLOT(slotChangeLook(int))           );
}


void LinkLookEditWidget::slotChangeLook(const QColor&)
{
	slotChangeLook();
}

void LinkLookEditWidget::slotChangeLook()
{
	m_iconSize->setEnabled(m_onTopIcon->currentItem() > 0);
	saveToLook(m_exLook);
//	m_example->setLook(m_exLook);                      // LookLabel doesn't remember icon
	m_example->setLink(m_exTitle, m_exIcon, m_exLook); // and can't reload it at another size
}

void LinkLookEditWidget::slotChangeLook(int)
{
	slotChangeLook();
}

LinkLookEditWidget::~LinkLookEditWidget()
{
}

void LinkLookEditWidget::saveChanges()
{
	saveToLook(m_look);
}

void LinkLookEditWidget::saveToLook(LinkLook *look)
{
	bool showIcon = (m_onTopIcon->currentItem() > 0);
	bool onTop    = (m_onTopIcon->currentItem() == 2);

	look->setLook( m_underline->currentItem() == 0 || m_underline->currentItem() == 3,
	               m_italic->isOn(), m_bold->isOn(),
	               m_underline->currentItem() == 2 || m_underline->currentItem() == 3,
	               m_color->color(), m_hoverColor->color(),
	               showIcon, m_iconSize->iconSize(), onTop );
}

void LinkLookEditWidget::slotShowIcon(bool show)
{
	m_iconProperties->setEnabled(show);
}

#include "linklabel.moc"
