# Arch Perl library, Copyright (C) 2005 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package AXP::Command::triggers::process;
use base 'AXP::Command::triggers';

use Arch::Name;
use Arch::Session;
use Arch::Util qw(save_file run_cmd arch_backend);
use Arch::TempFiles qw(temp_file_name);

sub optusage {
	"[options]"
}

sub infoline {
	"check all registered limits for new revisions"
}

sub options {
	(
		test    => { sh => 't', desc => "do not actually trigger actions and record new state" },
		verbose => { sh => 'v', desc => "be more verbose" },
		quiet   => { sh => 'q', desc => "be quiet" },
		log_to  => { sh => 'l', type => "=s", arg => 'FILE', desc => "log/append to FILE rather than stdout" },
		skip    => { sh => 's', desc => "just skip all existing revisions" },
	)
}

sub helptext {
	q{

		Check all registered limits for new revisions and run the
		associated command for every new revision.  If the command
		succeeds (i.e. email is sent), the new state is recorded and
		subsequent calls to 'process' will not report the revision again.

		This command is supposed to be run periodically from cron (use
		--quiet or --log-to options).
	}
}

sub _set_env ($$) {
	my $varname = shift;
	my $value = shift;

	$value = "" unless defined $value;
	$value =~ s/\\/\\\\/g;
	$value =~ s/'/\\'/g;
	$value =~ s/'/\\"/g;
	$ENV{$varname} = $value;
}

sub _set_env_all ($$$$$$) {
	my $vars = shift;
	my $revision = Arch::Name->new(shift());
	my $revision_desc = shift;
	my $log_body = shift;
	my $log_text = shift;
	my $log_file = shift;

	_set_env('ARCH_ARCHIVE', $revision->archive);
	_set_env('ARCH_REVISION', $revision->to_nonarch_string);
	_set_env('FULL_REVISION', $revision->to_string);
	_set_env('CATEGORY', $revision->category);
	_set_env('BRANCH', $revision->branch);
	_set_env('VERSION', $revision->version);
	_set_env('REVISION', $revision->revision);
	_set_env('LOG_SUMMARY', $revision_desc->{summary});
	_set_env('LOG_KIND', $revision_desc->{kind});
	_set_env('LOG_CREATOR', $revision_desc->{creator});
	_set_env('LOG_EMAIL', $revision_desc->{email});
	_set_env('LOG_DATE', $revision_desc->{date});
	_set_env('LOG_BODY', $log_body);
	_set_env('LOG_TEXT', $log_text);
	_set_env('LOG_FILE', $log_file);
	_set_env('AXP', $0);
	_set_env('ARCH_BACKEND', arch_backend());
	_set_env($_, $vars->{$_}) foreach keys %$vars;
}

sub execute {
	my $self = shift;
	my %opt = %{$self->{options}};

	die "The --test option is not implemented yet\n" if $opt{test};
	$ENV{DEBUG} = 1 if $opt{verbose};

	require IO::File;  # no 'use', speed up generation stuff
	my $file = $opt{log_to} || "-";
	my $out = new IO::File ">> $file" or die "Can't append to $file: $!";

	my $session = Arch::Session->new;

	my $tmp_file = temp_file_name();
	my $args = [ "--since", $tmp_file, "--omit-empty" ];

	foreach my $name (@{$self->all_trigger_names}) {
		print $out "* processing trigger $name\n" unless $opt{quiet};

		my $trigger = $self->load_trigger($name);
		my $action_file = $trigger->{action_file};
		my $vars = $trigger->{vars};

		$self->skip_revisions($trigger) if $opt{skip};

		my $old_state = $trigger->{state};
		save_file($tmp_file, $old_state);
		my $versions = $session->expanded_versions($name, $args);

		my %old_state = map { /(.*)--(.*)/ || die "Invalid snap line $_\n"; $1 => $_ } @$old_state;
		my $new_state = [];

		PROCESS_VERSION:
		foreach my $version (@$versions) {
			my $nonarch_version = $version;
			$nonarch_version =~ s!.*/!!;
			my $nonarch_revision = delete $old_state{$nonarch_version};
			push @$new_state, $nonarch_revision;

			save_file($tmp_file, [ $nonarch_revision || () ]);
			my $revision_descs = $session->get_revision_descs($version, $args);

			foreach my $revision_desc (@$revision_descs) {
				my $patchlevel = $revision_desc->{name};
				my $revision = "$version--$patchlevel";
				print $out "* triggering new revision $revision\n" unless $opt{quiet};

				my $log = $session->get_revision_log($revision);
				my $log_body = $log->body;
				my $log_text = $log->get_message;
				save_file($tmp_file, \$log_text);

				_set_env_all($vars, $revision, $revision_desc, $log_body, $log_text, $tmp_file);

				run_cmd($action_file);

				if ($? == 0) {
					$new_state->[-1] = "$nonarch_version--$patchlevel";
				} else {
					print $out "Action for $revision is failed with status $?\n";
					next PROCESS_VERSION;
				}
			}

			pop @$new_state unless defined $new_state->[-1];
		}

		push @$new_state, $old_state{$_} foreach keys %old_state;
		$new_state = [ sort @$new_state ];

		$trigger->{state} = $new_state;
		$self->save_trigger($trigger);
	}
}

1;
