# Arch Perl library, Copyright (C) 2005 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package AXP::Command::triggers::add;
use base 'AXP::Command::triggers';

use Arch::Util qw(arch_backend run_cmd);

sub optusage {
	"[options] limit [slave_limit ...]"
}

sub infoline {
	"add a new trigger"
}

sub options {
	(
		email   => { sh => 'e', desc => "use EMAIL_ADDR in standard send-mail action", type => "=s", arg => 'EMAIL_ADDR' },
		plain   => { sh => 'p', desc => "use plain-log-style rather than changelog-style" },
		action  => { sh => 'a', desc => "set SHELL_CMD to be the action (experts only)", type => "=s", arg => 'SHELL_CMD' },
		no_edit => { sh => 'n', desc => "do not invoke editor to tweak action" },
		no_skip => { sh => 'S', desc => "do not skip the already existing revisions" },
		master  => { sh => 'm', desc => "link parameter limits to MASTER (limit or number)", type => "=s", arg => 'MASTER' },
		var     => { sh => 'v', desc => 'set user vars: -v LABEL=prj1 -v EMAIL=me@me.org', type => "=s%", arg => 'VAR=VALUE' },
	)
}

sub helptext {
	my $prefix = $_[0]->{prefix};
	qq{
		Add a new trigger for LIMIT parameter.  LIMIT may be an arch
		archive, category, branch or version.

		Unless you use the --no-edit option, this will open an editor
		where you can enter commands that will be executed
		whenever a new revision is detected in LIMIT.  Revisions that
		already exist by the time the trigger is created will not
		cause the commands to be executed, unless you add the
		--no-skip option.

		The default command will send an email with the new revisions
		log to your account or the email provided by the --email option.

		You can change the list of commands later by using the
		'edit' command.

		With --master, do not create its own action for a new trigger,
		but share it with the action of already existing master trigger.
		So, 'add lim1 lim2' is just 'add lim1; add --master lim1 lim2'.

		Example:
			$prefix --email me\@me.org,list\@prj.org devel\@prg.org--2005
	}
}

sub execute {
	my $self = shift;
	my %opt = %{$self->{options}};

	my $master_trigger = undef;
	if ($opt{master}) {
		$master_trigger = $self->load_trigger($opt{master});
		goto PROCESS_SLAVES;
	}

	my $name = shift @ARGV;
	my $trigger = $self->create_trigger($name);

	$self->apply_vars($trigger, $opt{var});

	my $user = $ENV{USER} || $ENV{USERNAME} || getpwuid($>);
	my $email = $opt{email} || "$user\@localhost";

	my $plain_action = $opt{action} ||
		($opt{plain}? 'cat $LOG_FILE': '$AXP changelog --use-log $LOG_FILE')
		. qq( \\\n\t| mail -s "\$ARCH_REVISION commit: \$LOG_SUMMARY" $email);

	$trigger->{action} =
		"#!/bin/sh\n" .
		"#\n" .
		"# Run on every new $name revision.\n" .
		"# The special quote-escaped variables available in this script:\n" .
		'#   $ARCH_ARCHIVE, $ARCH_REVISION (full non-archive revision),' . "\n" .
		'#   $FULL_REVISION, $CATEGORY, $BRANCH, $VERSION, $REVISION,' . "\n" .
		'#   $LOG_SUMMARY, $LOG_KIND, $LOG_CREATOR, $LOG_EMAIL, $LOG_DATE,' . "\n" .
		'#   $LOG_BODY, $LOG_TEXT, $LOG_FILE,' . "\n" .
		'#   $AXP (' . $0 . '), $ARCH_BACKEND (' . arch_backend() . ').' . "\n" .
		"\n$plain_action\n";

	$self->edit_trigger($trigger)
		or die "Error or canceled by user; $name trigger is not added\n"
		unless $opt{no_edit};

	$self->skip_revisions($trigger) unless $opt{no_skip};
	$self->save_trigger($trigger);
	run_cmd("chmod +x", $trigger->{action_file});

	print "* trigger for $trigger->{name} is added\n";

	PROCESS_SLAVES:
	if ($master_trigger) {
		foreach my $name (@ARGV) {
			my $trigger = $self->create_trigger($name);
			$self->apply_vars($trigger, $opt{var});
			symlink($master_trigger->{action_file}, $trigger->{action_file});
			$self->skip_revisions($trigger) unless $opt{no_skip};
			$self->save_trigger($trigger);
			print "* slave trigger for $trigger->{name} is added\n";
		}
	}
}

1;
