/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	Class for formatting cd info into
 *	DB submitable form.
 *
 *	by Tony Sideris	(01:22PM Mar 20, 2002)
 *================================================*/
#include "arson.h"

#include <qtextstream.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qregexp.h>

#include <klocale.h>

#include "tempfile.h"
#include "submit.h"
#include "cdinfo.h"
#include "cddb.h"

/*========================================================*/
/*	Base class
 *========================================================*/

ArsonCdInfoSubmit::ArsonCdInfoSubmit (const ArsonCdInfo &info)
	: m_info(info),
	m_filename(QString::null)
{
	//	Nothing...
}

/*========================================================*/

ArsonCdInfoSubmit::~ArsonCdInfoSubmit (void)
{
	if (m_filename != QString::null)
	{
		Trace("ArsonCdinfoSubmit deleting file %s\n",
			m_filename.latin1());
		
		QFile::remove(m_filename);
	}
}

/*========================================================*/

bool ArsonCdInfoSubmit::writeFile (const QString &fn)
{
	bool res = false;
	QFile *pf = NULL;
	ArsonTempFile *pt = NULL;

	if (fn == QString::null)
	{
		pt = new ArsonTempFile("db", "txt");

		if (pt->status())
		{
			delete pt;
			pt = NULL;
		}
		else
		{
			pt->setAutoDelete(false);

			m_filename = pt->name();
			pf = pt->file();
		}
	}
	else
	{
		pf = new QFile(fn);

		if (!pf->open(IO_WriteOnly | IO_Truncate))
		{
			delete pf;
			pf = NULL;
		}
	}

	if (pf)
	{
		QTextStream ts (pf);
		res = writeContents(&ts);
	}
	else
		arsonErrorMsg(
			i18n("Failed to open DB file %1")
			.arg((fn == QString::null) ? m_filename : fn));

	if (!pt)
		delete pf;
	else
		delete pt;

	return res;
}

/*========================================================*/
/*	Create the FREEDB submit file.
 *========================================================*/

#define LINE_TOO_LONE		I18N_NOOP("Freedb: line too long (512 limit): %1")

class FdbHeader
{
public:
	FdbHeader (const QString &key, const QString &val)
		: m_key(key), m_val(val) { }

	const QString &m_key;
	const QString &m_val;
};

class FdbHeaderWriter
{
public:
	FdbHeaderWriter (QTextStream *ptr) : m_out(*ptr) { }

	FdbHeaderWriter &operator<< (const FdbHeader &hdr)
	{
		m_out << hdr.m_key << ":";

		if (hdr.m_val != QString::null)
			m_out << " " << hdr.m_val;

		m_out << "\r\n";
		return *this;
	}

private:
	QTextStream &m_out;
};

class FdbComment
{
public:
	FdbComment (QTextStream *ptr) : m_out(*ptr) { }

	FdbComment &operator<< (const QString &str)
	{
		if (str.length() + 3 > 512)
			throw ArsonError(
				i18n(LINE_TOO_LONE).arg(str));
		
		m_out << "#";

		if (str != QString::null)
			m_out << " " << str;

		m_out << "\n";
		return *this;
	}

private:
	QTextStream &m_out;
};

class FdbContent : public FdbHeader
{
public:
	FdbContent (const QString &key, const QString &val, bool validate = true)
		: FdbHeader(key, val)
	{
		if (validate && val.isEmpty())
			throw ArsonError(
				i18n("Freedb: Missing required field: %1").arg(key));
	}
};

class FdbContentWriter
{
public:
	FdbContentWriter (QTextStream *ptr) : m_out(*ptr) { }

	FdbContentWriter &operator<< (const FdbContent &obj)
	{
		if (obj.m_key.length() + obj.m_val.length() + 2 >  512)
			throw ArsonError(
				i18n(LINE_TOO_LONE).arg(obj.m_val + "=" + obj.m_val));

		m_out << obj.m_key << "=";

		if (obj.m_val != QString::null)
			m_out << obj.m_val;

		m_out << "\n";
		return *this;
	}

private:
	QTextStream &m_out;
};

/*========================================================*/

//#define MAXLINELEN			240
#define MAXLINELEN			16

void writeMultiLine (FdbContentWriter &writer,
	const QString &key, const QString &val)
{
	QString value (ArsonCddbParserBase::encode(val.latin1()));

	if (value.length() <= MAXLINELEN)
		writer << FdbContent(key, value, false);
	else
	{
		while (value.length() > MAXLINELEN)
		{
			writer << FdbContent(key, value.left(MAXLINELEN), false);
			value = value.right(value.length() - MAXLINELEN);
		}

		writer << FdbContent(key, value, false);
	}
}

/*========================================================*/

bool ArsonFreedbSubmit::writeContents (QTextStream *ptr) const
{
	//	If the contents have already been written, write the headers.
	if (m_contentLength > 0 && m_bWantHeaders)
		return writeHeaders(ptr, m_contentLength);

	//	Write the contents
	try
	{
		uint index;
		FdbComment comment(ptr);
		FdbContentWriter content(ptr);
		QString title;

		//	Writer the comment headers
		comment << "xmcd"
				<< QString::null
				<< "Track frame offsets:";

		for (index = 0; index < m_info.trackCount(); ++index)
			comment << QString("  ") + QString::number(m_info.track(index).offset());

		comment << QString::null
				<< QString("Disc length: ") + QString::number(m_info.totalSeconds()) + QString(" seconds")
				<< QString::null
				<< QString("Revision: 0")	//	FIXME: Add revision system
				<< QString("Submitted via: ") + PACKAGE " " VERSION
				<< QString("URL: " HOMEPAGE)
				<< QString::null;

		title = ((m_info.variousArtistDisk())
			? QString("Various Artists")
			: m_info.artist());

		title.append(" / ").append(m_info.title());

		//	Write the content
		content << FdbContent("DISCID", m_info.cddbID())
				<< FdbContent("DTITLE", title)
				<< FdbContent("DYEAR", m_info.year())
				<< FdbContent("DGENRE", m_info.genre());

		//	Write the track listing
		for (index = 0; index < m_info.trackCount(); ++index)
		{
			const ArsonCdInfo::Track track = m_info.track(index);
			QString trackTitle = track.title();
			bool ok;

			trackTitle.right(2).toInt(&ok);

			//	Make sure user didn't leave any track names blank or default
			if (ok && trackTitle.left(6) == "track-")
				throw ArsonError(
					i18n("Default track name found (%1), please edit this before submitting.").arg(trackTitle));

			if (m_info.variousArtistDisk())
				trackTitle = m_info.track(index).artist(&m_info)
					+ " / " + m_info.track(index).title();
			else
				trackTitle = m_info.track(index).title();

			content << FdbContent(
				QString("TTITLE") + QString::number(index),
				trackTitle);
		}

		//	Write the EXT data
		writeMultiLine(content, "EXTD", m_info.comment());

		for (index = 0; index < m_info.trackCount(); ++index)
			writeMultiLine(content,
				QString("EXTT") + QString::number(index),
				m_info.track(index).comment());

		//	Finally the PLAYORDER
		content << FdbContent("PLAYORDER",
			m_info.playOrder(), false);
	}
	catch (ArsonError &err) {
		err.report();
		return false;
	}

	return true;
}

/*========================================================*/

bool ArsonFreedbSubmit::writeHeaders (QTextStream *ptr, uint length) const
{
	try
	{
		FdbHeaderWriter hdr(ptr);

		hdr << FdbHeader("Category", m_info.categ())
			<< FdbHeader("Discid", m_info.cddbID())
			<< FdbHeader("User-Email", m_email)
			<< FdbHeader("Submit-Mode",
#ifdef ARSONDBG
				"test"
#else
				"submit"
#endif	//	ARSONDBG
				)
			<< FdbHeader("Content-Length", QString::number(length))
			<< FdbHeader("X-Cddbd-Note", QString("Submitted by ") + PACKAGE " " VERSION);

		(*ptr) << "\r\n";
	}
	catch (ArsonError &err) {
		err.report();
		return false;
	}

	return true;
}

/*========================================================*/

bool ArsonFreedbSubmit::writeFile (const QString &fn)
{
	//	Write the CDDB DB entry to file
	if (ArsonCdInfoSubmit::writeFile(fn))
	{
		const QString temp = filename();

		if (!m_bWantHeaders)
			return true;
		
		//	Get the length of the entry, then write the headers
		if ((m_contentLength = QFileInfo(temp).size()) > 0 &&
			ArsonCdInfoSubmit::writeFile())
		{
			QFile file (filename());
			QFile in (temp);

			//	Open the header file, and append the DB entry
			if (file.open(IO_WriteOnly | IO_Append) &&
				in.open(IO_ReadOnly))
			{
				QString line;

				while (in.readLine(line, 512) != -1)
					file.writeBlock(line.latin1(), line.length());

				file.close();
				in.close();

				//	Delete the temp DB entry file
				QFile::remove(temp);
				return true;
			}
		}
	}

	return false;
}

/*========================================================*/
/*	The container dialog
 *========================================================*/

ArsonCdInfoSubmitDlg::ArsonCdInfoSubmitDlg (ArsonCdInfoSubmit &submit,
	ArsonSendSocket *pSocket)
	: ArsonSocketWaitDlg(),
	m_pSocket(pSocket),
	m_info(submit)
{
	setMessage(
		i18n("Submitting CD information to online database..."));

	setCaption(
		i18n("CD Info Submit"));
}

/*========================================================*/

int ArsonCdInfoSubmitDlg::exec (void)
{
	if (m_info.writeFile())
	{
		m_pSocket->sendFile(m_info.filename());
		return ArsonSocketWaitDlg::exec();
	}

	return Rejected;
}

/*========================================================*/
