/* this file is a part of Ami software, (C) Hwang chi-deok 1999,2000,2001 */
/* This file includes a couple of arrays initialized with 
 * character strings in EUC-KR. If you don't have an editor that can
 * deal with EUC-KR, you have to make sure that your editor does not
 * haphazardly trashes the content.
 */

#include "config.h"

#include <stdio.h>
#include <ctype.h>
#include <sys/stat.h>
#include <unistd.h>
#include "ami.h"
#include "conf.h"
#include "keyparse.h"

#ifndef HANGUL_KEYBOARD
#define HANGUL_KEYBOARD "2bul"
#endif

static char * config_readline(FILE *fp);
static gboolean config_add_item(guchar *desc, guchar *key, guchar *val_string);
static int save_default_config_file(void);
static ConfigItem *ami_config = NULL;


static ConfigDesc ami_config_type[] =
{
    {"", STRING, &keyboard_name},
    {"Է ", INT, &ami_editing_mode},
    {"⺻ ۲", STRING, &default_fontname},
    {"ѿ ȯŰ", KEYSYM, &trigger_keys},
    {" ȯŰ", KEYSYM, &hanja_keys},
    {"Ư ԷŰ", KEYSYM, &special_char_keys},
    {"ѱۻ ", STRING, &hangul_mode_label},
    {" ", STRING, &english_mode_label},
    {" ѿ", BOOLEAN, &unique_han},
    {"flush key", KEYSYM, &flush_keys},
    {"last line scroll", BOOLEAN, &ami_line_wraping_mode},
    {"escape hangul toggle", BOOLEAN, &esc_han_toggle},
    {"â ", BOOLEAN, &support_status},
    {" ġȯ ", STRING, &ami_hanja_subst_format},
    {" ġȯ ", INT, &ami_hanja_def_subst_mode},
    {" ", BOOLEAN, &ami_use_underline},
    {"ڻ", STRING, &hanja_dic_filename},
    {"Ҹ ũ", INT, &bell_intensity},
    {"׸  ġ", STRING, &pix_dir_name},
    {"/߼ Է  ڹٲ ", BOOLEAN, &allow_shuffle},
    {"۲ (UTF-8 locale ۲ )",INT,&num_fonts_in_utf8_fs},
    {NULL, 0, NULL}
};

gboolean
ami_config_init(void)
{
    gboolean ret;
    int i;
    char *conf_file = g_strconcat(g_get_home_dir(), "/.ami/ami.conf", NULL);
    if (access(conf_file, R_OK) != 0) {
	if (!save_default_config_file())
	    g_error(_("Cannot write to %s\n"), conf_file);
    }
    ret = ami_config_load_file(conf_file, ami_config_type);
    on_keys = g_new(XIMTriggerKeys, 1);
    if (trigger_keys == NULL) {
	g_error(_("No key is specified for Korean-English KBD toggle."));
    }
    for(i=0;trigger_keys[i].keysym;i++);
    on_keys->count_keys = i;
    on_keys->keylist = trigger_keys;
    g_free(conf_file);
    ami_hanja_subst_mode = ami_hanja_def_subst_mode;
    if (!hangul_mode_label)
	hangul_mode_label = g_strdup(_(" [KO] "));
    if (!english_mode_label)
	english_mode_label = g_strdup(_(" [EN] "));
    if (!ami_hanja_subst_format)
	ami_hanja_subst_format = g_strdup(":(ѱ):ѱ()");
    if (!hanja_dic_filename) hanja_dic_filename = g_strdup(HANJA_DIC);
    if (!keyboard_name) keyboard_name = g_strdup(HANGUL_KEYBOARD);
    composer_set_shuffle_active((int) allow_shuffle);
    return ret;
}

static guchar *
line_skip_space(guchar *s)
{
    while(isspace(*s)) s++;
    return s;
}

static guchar *
line_rskip_space(guchar *s)
{
    while(isspace(*s)) s--;
    return s;
}

static gboolean
line_is_empty(guchar *s)
{
    while(*s) {
	if (!isspace(*s)) return FALSE;
	s++;
    }
    return TRUE;
}

static int
save_default_config_file(void)
{
    char *conf_dir = g_strdup_printf("%s/%s", g_get_home_dir(), ".ami");
    char *conf;
    extern const char *default_config;
    struct stat statbuf;
    FILE *fp;
    if (stat(conf_dir, &statbuf) != 0) {
	mkdir(conf_dir,0750);
    }
    conf = g_strdup_printf("%s/%s", conf_dir, "ami.conf");
    g_free(conf_dir);
    fp = fopen(conf, "w");
    if (fp == NULL) {
	g_error(_("Cannot save Ami configuration options to %s."), conf);
    }
    g_free(conf);
    fputs(default_config, fp);
    fclose(fp);
    return TRUE;
}

static char *
config_readline(FILE *fp)
{
    char buf[1024];
    if (fgets(buf, sizeof(buf), fp)) {
	return g_strdup(buf);
    }
    return NULL;
}

static ConfigItem *
config_find_config_item(guchar *key)
{
    ConfigItem *config = ami_config;
    while (config->key) {
	if (strcmp(config->key, key) == 0) return config;
	config++;
    }
    return NULL;
}

static gboolean
config_add_item(guchar *desc, guchar *key, guchar *val_string)
{
    ConfigItem *config_item;
    config_item = config_find_config_item(key);
    if (config_item == NULL) return FALSE;
    config_item->desc = g_strdup(desc);
    switch(config_item->type) {
	case INT:
	    *config_item->val.integer = atoi(val_string);
	    break;
	case BOOLEAN:
	    if (*val_string == 't'
	    	|| strcmp(val_string, "on") == 0
		|| *val_string == '0') *config_item->val.boolean = TRUE;
	    else if(*val_string == 'f'
	    	|| strcmp(val_string, "off") == 0
		|| *val_string == '1') *config_item->val.boolean = FALSE;
	    else {
		g_warning("%s: valid value is true or false", key);
	    }
	    break;
	case KEYSYM:
	    make_trigger_keys(config_item->val.keys, val_string);
	    break;
	case STRING:
	    if (*config_item->val.string) g_free(*config_item->val.string);
	    if (val_string[0] == '"' &&
	        val_string[strlen(val_string)-1] == '"') {
		val_string[strlen(val_string)-1] = 0;
		val_string++;
	    }
	    *config_item->val.string = g_strdup(val_string);
	    break;
    }
    return TRUE;
}

gboolean
ami_config_load_file(char *file, ConfigDesc *ami_config_type)
{
    int config_size;
    int i;
    GSList *list, *comment = NULL;
    FILE *fp;
    char *line;

    for(i=0;ami_config_type[i].key;i++);
    config_size = i;
    ami_config = g_new0(ConfigItem, config_size + 1);
    for(i=0;i<config_size;i++) {
	ami_config[i].key = ami_config_type[i].key;
	ami_config[i].type = ami_config_type[i].type;
	switch(ami_config[i].type) {
	    case STRING:
		ami_config[i].val.string = ami_config_type[i].val_pointer;
		break;
	    case INT:
		ami_config[i].val.integer = ami_config_type[i].val_pointer;
		break;
	    case KEYSYM:
		ami_config[i].val.keys = ami_config_type[i].val_pointer;
		break;
	    case BOOLEAN:
		ami_config[i].val.boolean = ami_config_type[i].val_pointer;
		break;
	}
    }
    fp = fopen(file, "r");
    if (fp == NULL) return FALSE;
    while ((line = config_readline(fp))) {
	if (line[0] == '#' || line_is_empty(line)) {
	    comment = g_slist_append(comment, line);
	} else {
	    char *sep = strchr(line, '=');
	    guchar * start, *end, *key, *val;
	    GString *desc_string;
	    if (sep == NULL) {
		g_warning(_("invalid config line: %s"), line);
		continue;
	    }
	    *sep = 0;
	    start = line_skip_space(line);
	    end = line_rskip_space(sep - 1);
	    if (isspace(*end)) *end = 0;
	    else *(end+1) = 0;
	    key = start;
	    sep += 1;
	    start = line_skip_space(sep);
	    end = line_rskip_space(sep + strlen(sep) - 1);
	    if (isspace(*end)) *end = 0;
	    else *(end+1) = 0;
	    val = start;
	    desc_string = g_string_new("");
	    list = comment;
	    while(list) {
		g_string_append(desc_string , list->data);
		g_free(list->data);
		list = list->next;
	    }
	    g_slist_free(comment);
	    comment = NULL;
	    if (!config_add_item(desc_string->str, key, val)) {
		g_warning(_("%s is not a valid key"), key);
	    };
	    g_string_free(desc_string, TRUE);
	    g_free(line);
	}
    }
    fclose(fp);
    return TRUE;
}

gboolean
ami_config_save_file(char *file)
{
    FILE *fp;
    ConfigItem *config;
    char *key_desc;
    if (file) {
	fp = fopen(file, "w");
    } else {
	file =g_strconcat(g_get_home_dir(), "/.ami/ami.conf", NULL);
	fp = fopen(file, "w");
	g_free(file);
    }
    if (fp == NULL) return FALSE;
    config = ami_config;
    while(config->key) {
	if (config->desc) fprintf (fp, "%s", config->desc);
	else fprintf (fp, "# %s\n", config->key);
	switch(config->type) {
	    case INT:
		fprintf (fp, "%s=%d\n", config->key, *config->val.integer);
		break;
	    case BOOLEAN:
		if(*config->val.boolean)
		    fprintf (fp, "%s=true\n", config->key);
		else
		    fprintf (fp, "%s=false\n", config->key);
		break;
	    case STRING:
		{
		    char *s = *config->val.string;
		    if (!s || !s[0]) s = "";
		    fprintf (fp, "%s=\"%s\"\n", config->key, s);
		}
		break;
	    case KEYSYM:
		key_desc = trigger_keys_to_str(*config->val.keys);
		fprintf (fp, "%s=%s\n", config->key, key_desc);
		g_free(key_desc);
		break;
	}
	config++;
    }
    fprintf(fp, "\n");
    fclose(fp);
    return TRUE;
}

/*
   '\n\' has to be used to make xgettext not complain about unterminated
  string literals. J. Shin
*/
const char *default_config = 
"\n\
# Caution: This file MUST be in EUC-KR encoding. If you use UTF-8 locale,\n\
# convert this file to UTF-8 before editing and convert back to EUC-KR\n\
# after you're done with editing. Otherwise, Ami wouldn't be able to\n\
# parse the file to get the values of configuration options.\n\
# ƹ  \n\
# :  Ű  ٲ ʽÿ.  \"Է \"̶ Ǿ\n\
#   ؼ Ȯ ̸̾ մϴ.\n\
\n\
# \n\
# 2bul, 3bul_390, 3bul_final, dvorak_2bul, dvorak_3bul_390,\n\
# dvorak_3bul_final Ami ǿ  ִ  Դϴ.\n\
# ߰ ٸ  ؼ   ֽϴ.\n\
=" HANGUL_KEYBOARD "\n\
\n\
# Է½  Ƽ   ݴϴ.\n\
# 1=ܾ  ϸ Է  ȭǥ Ű ̿ ̵ Ami \n\
#   óմϴ.\n\
# 2=ܾ  ϸ ȭǥ Ű ̿ ̵ Ŭ̾Ʈ α׷ \n\
#   Ͽ  ó ϵ մϴ. ϴ Դϴ.\n\
# 3=  մϴ.\n\
\n\
Է  = 2\n\
\n\
#  Ŭ̾Ʈ α׷  ѿ ¸   \n\
# false: ѿ ¸ â  մϴ. XIM ǥؿ  ϴ Դϴ.\n\
#  htt  Է±ʹ ޸ Ŭ̾Ʈ ִ   ʰ\n\
#   â ؼ gtk+  α׷   ϵ\n\
#  մϴ.\n\
# true:  Ŭ̾Ʈ α׷  ѿ ¸ ϴ. â \n\
#  ʽϴ.    ٸ     մϴ.\n\
\n\
 ѿ = false\n\
\n\
\n\
# ѿ ȯ  Ǵ Ű ݴϴ\n\
\n\
ѿ ȯŰ = Shift<Key>space, <Key>Hangul\n\
\n\
#  Է Ű\n\
\n\
 ȯŰ = <Key>F9, <Key>Hangul_Hanja\n\
\n\
# Ư  ԷŰ \n\
\n\
Ư ԷŰ = <Key>F3\n\
\n\
#  Է  Էâ  α׷   Ű\n\
\n\
flush key = Ctrl<Key>Return\n\
\n\
# â ȭ  Ʒ κп Ÿ (Ǵ gtk entry   ¥ \n\
# Էâ) Էâ  ʱ ѱ Է  Ǿ   Է\n\
# ȭ   κп ̷  ƴϸ  Էµǰ ִ  \n\
#  ũ ų   մϴ.\n\
# true̸ ׳ ũ ŵϴ.\n\
\n\
last line scroll = true\n\
\n\
# ⺻ ۲ \n\
# ƹ̿ ϴ α׷ ۲   ʰų ߸ ̸ 쿡\n\
#  ۲ ֽø ˴ϴ.   ۲,ѱ ۲\n\
# Ami UTF-8 locale ٸ Ʒ     Դϴ.\n\
# ׷ ʰ EUC-KR locale ٸ  Ʒ  ᵵ  , \n\
# \"*,*\"  ־ ϴ.\n\
\n\
⺻ ۲ = -*-*-medium-r-normal-ko-16-*-*-*-*-*-iso10646-1,-*-*-medium-r-normal--16-*-*-*-*-*-*-*\n\
\n\
# ѱ    â ǥõǴ ڿ ݴϴ.\n\
\n\
ѱۻ  = \" [KO] \"\n\
\n\
#  Է   â ǥõǴ ڿ  ݴϴ.\n\
\n\
  = \" [EN] \"\n\
\n\
# ѱ Է ¿ Escape    · ڵ  \n\
# ݴϴ. true/false\n\
\n\
escape hangul toggle = true\n\
\n\
# â    ݴϴ.\n\
# â Ʒʿ   â 彺 е  ɼ \n\
# false Ͻʽÿ. \n\
#  ѱ ¸ Ͻø  ɼǰ   â  ʽϴ.\n\
\n\
â  = true\n\
\n\
# ڸ ġȯϴ ¸ ݴϴ.\n\
# \"\" Ǿִ κ   ڷ ġȯǰ \"ѱ\"̶\n\
#  κ  ڿ شϴ ѱ ־ ˴ϴ.\n\
# µ   \":\" մϴ.\n\
\n\
 ġȯ  = :(ѱ):ѱ()\n\
\n\
#  ġȯ ¿  °  ⺻  \n\
# մϴ.\n\
# 1 \n\
\n\
 ġȯ  = 1\n\
\n\
#  Էâ  ִ ѱ ǥ  Ʒʿ   \n\
# ƴϸ   â  ٸ   \n\
\n\
  = true\n\
\n\
#   ġ \n\
\n\
ڻ = " HANJA_DIC "\n\
\n\
#  Ҹ ũ⸦ ݴϴ.\n\
# ȿ  -100 100̰  Ŭ Ҹ Ŀϴ.\n\
Ҹ ũ = 0\n\
\n\
# ѿ ǥø  ׸  ġϴ 丮 մϴ.\n\
# \"\" Ǿ  ƹ̿  ׸ մϴ.\n\
# 丮̸ '/' ϴ 丮 ̸   ּž մϴ.\n\
׸  ġ = \"\"\n\
\n\
# ʼ  ߼ Է  ʼ   Ÿ  ٲ  \n\
# ϰ ʼ+߼  Է  \n\
/߼ Է  ڹٲ  = false\n\
\n\
# UTF-8 locale   ۲ (fontset) ʿ ϴ\n\
# ۲  մϴ. $XROOT/locale/ko_KR.UTF-8/XLC_LOCALE\n\
#  ۲  ⿡  ˴ϴ. XFree86 4.x ִ\n\
# en_US.UTF-8/XLC_LOCALE 7 ۲ Ǿ ֽϴ.\n\
# ݸ鿡 X11R6.6 Solaris 8 Ȥ 9 ko_KR.UTF-8/XLC_LOCALE \n\
#  18 ۲ Ǿ ֽϴ. \n\
\n\
۲ (UTF-8 locale ۲ )= 7\n\
\n\
";
