// ---------------------------------------------------------------------------
// - Librarian.cpp                                                           -
// - aleph engine - librarian class implementation                           -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "System.hpp"
#include "Vector.hpp"
#include "Integer.hpp"
#include "Boolean.hpp"
#include "Runnable.hpp"
#include "Character.hpp"
#include "Librarian.hpp"
#include "Exception.hpp"
#include "InputFile.hpp"
#include "OutputFile.hpp"

namespace aleph {
  // the librarian supported quarks
  static const long QUARK_ADD     = String::intern ("add");
  static const long QUARK_WRITE   = String::intern ("write");
  static const long QUARK_LENGTH  = String::intern ("length");
  static const long QUARK_GETVEC  = String::intern ("get-names");
  static const long QUARK_EXISTS  = String::intern ("exists-p");
  static const long QUARK_EXTRACT = String::intern ("extract");

  // librarian constant]
  const long   AXL_MSIZE    = 4;
  const t_byte AXL_MAGIC[4] = {'\377', 'A', 'X', 'L'};
  const t_byte AXL_MAJOR    = 1;
  const t_byte AXL_MINOR    = 0;
  const t_byte AXL_FLAGS    = nilc;

  // the librarian header
  struct s_lhead {
    t_byte d_magic[AXL_MSIZE];
    t_byte d_major;
    t_byte d_minor;
    t_byte d_flags;
    t_octa d_hsize;
    // create a new header
    s_lhead (void) {
      for (long i = 0; i < AXL_MSIZE; i++) d_magic[i] = nilc;
      d_major = 0;
      d_minor = 0;
      d_flags = 0;
      d_hsize = 0;
    }
    // create a new header with a size
    s_lhead (const long size) {
      for (long i = 0; i < AXL_MSIZE; i++) d_magic[i] = AXL_MAGIC[i];
      d_major = AXL_MAJOR;
      d_minor = AXL_MINOR;
      d_flags = AXL_FLAGS;
      d_hsize = System::oswap (size);
    }
    // check this header
    bool check (void) {
      // check magic
      for (long i = 0; i < AXL_MSIZE; i++)
	if (d_magic[i] != AXL_MAGIC[i]) return false;
      // check major
      if (d_major != AXL_MAJOR) return false;
      // check minor
      if (d_minor > AXL_MINOR) return false;
      // look's ok
      return true;
    }
  };

  // this procedure format the file flags into a sequence of characters
  static String format_flags (const t_byte flags) {
    String result = "--------";
    return result;
  }

  // this procedure format the file size into a string  - 20 characters max
  static String format_fsize (const long fsize) {
    Integer ival (fsize);
    String result = ival.tostring ();
    return result.lfill (' ', 10);
  }

  // the file descriptor
  struct s_fdesc {
    // file path
    String d_fpath;
    // file name
    String d_fname;
    // the file size
    t_long d_fsize;
    // the librarian offset
    long   d_lfoff;
    // file flags
    t_byte d_flags;
    // next file in list
    s_fdesc* p_next;
    // create an empty descriptor
    s_fdesc (void) {
      d_fsize = 0;
      d_lfoff = 0;
      d_flags = 0;
      p_next  = nilp;
    }
    // create a new descriptor by name
    s_fdesc (const String& fpath, const long size) {
      d_fpath = fpath;
      d_fname = System::xname (d_fpath);
      d_fsize = size;
      d_lfoff = 0;
      d_flags = nilc;
      p_next  = nilp;
    }
    // delete a chain of descriptors
    ~s_fdesc (void) {
      delete p_next;
    }
    // append a descriptor at the end
    void append (s_fdesc* desc) {
      if (desc == nilp) return;
      s_fdesc* last = this;
      while (last->p_next != nilp) last = last->p_next;
      last->p_next = desc;
    }
    // return the serialized length
    long length (void) {
      long result = d_fname.length () + 1;
      result     += 8 + 1;
      return result;
    }
    // serialize this descriptor
    void wrstream (Output& os) {
      Integer   fsize = d_fsize;
      Character flags = d_flags;
      d_fname.wrstream (os);
      fsize.wrstream   (os);
      flags.wrstream   (os);
    }
    // deserialize this descriptor
    void rdstream (Input& is) {
      Integer   fsize;
      Character flags;
      d_fname.rdstream (is);
      fsize.rdstream   (is);
      flags.rdstream   (is);
      d_fpath = d_fname;
      d_fsize = fsize.tointeger ();
      d_flags = flags.tochar    ();
    }
    // format a descriptor to an output stream
    void format (Output& os) {
      String flags = format_flags (d_flags);
      String fsize = format_fsize (d_fsize);
      os << flags << ' ' << fsize << ' ' << d_fname << eolc;
    }
  };

  // this procedure compute the size of descritpor chain
  static long get_chain_length (s_fdesc* desc) {
    long result = 0;
    while (desc != nilp) {
      result += desc->length ();
      desc = desc->p_next;
    }
    return result;
  }

  // this procedure finds a descriptor by name
  static s_fdesc* get_named_desc (s_fdesc* desc, const String& name) {
    while (desc != nilp) {
      if (desc->d_fname == name) return desc;
      desc = desc->p_next;
    }
    return nilp;
  }

  // write the header on the output stream
  static void write_header (Output& os, s_fdesc* desc) {
    // get the librarian header
    s_lhead lhead (get_chain_length (desc));
    // write the librarian header
    os.write ((char*) &lhead, sizeof (lhead));
    // serialize the chain
    while (desc != nilp) {
      desc->wrstream (os);
      desc = desc->p_next;
    }
  }

  // read the header from an input stream
  static s_fdesc* read_header (const String& lname) {
    InputFile is (lname);
    // read the librarian header
    s_lhead lhead;
    Buffer* buf = is.Input::read (sizeof (lhead));
    if (buf->map (&lhead, sizeof (lhead)) != sizeof (lhead)) {
      delete buf;
      throw Exception ("librarian-error", "cannot read header");
    }
    delete buf;
    // check the header 
    if (lhead.check () == false)
      throw Exception ("librarian-error", "invalid librarian header");
    // check the input size
    long hsize = System::oswap (lhead.d_hsize);
    long lfoff = hsize + sizeof (s_lhead);
    if (hsize == 0) return nilp;
    // prepare for reading
    s_fdesc* result = nilp;
    s_fdesc* last   = nilp;
    // read until the size is null
    while (hsize != 0) {
      // read in one descriptor
      s_fdesc* desc = new s_fdesc;
      desc->rdstream (is);
      // update the file offset
      desc->d_lfoff = lfoff;
      lfoff += desc->d_fsize;
      // update result and size
      if (last == nilp) {
	result = desc;
	last   = desc;
      } else {
	last->p_next = desc;
	last = desc;
      }
      hsize -= desc->length ();
      if (hsize < 0) {
	delete result;
	throw Exception ("librarian-error", "cannot read file descriptors");
      }
    }
    return result;
  }

  // create an empty librarian

  Librarian::Librarian (void) {
    d_type = OUTPUT;
    p_desc = nilp;
  }

  // create a librarian by name
  
  Librarian::Librarian (const String& lname) {
    d_type  = INPUT;
    d_lname = lname;
    p_desc  = read_header (lname);
  }

  // destroy this librarian

  Librarian::~Librarian (void) {
    delete p_desc;
  }

  // return the class name

  String Librarian::repr (void) const {
    return "Librarian";
  }

  // return the librarian name

  String Librarian::getname (void) const {
    return d_lname;
  }

  // add a new file descriptor to the chain
  
  void Librarian::add (const String& path) {
    // check for type
    if (d_type == INPUT)
      throw Exception ("librarian-error", "cannot add file to librarian");
    // check for a file first
    InputFile is (path);
    if (is.length () == 0) return;
    // ok let's add it
    wrlock ();
    s_fdesc* fdesc = new s_fdesc (path, is.length ());
    if (p_desc == nilp)
      p_desc = fdesc;
    else
      p_desc->append (fdesc);
    unlock ();
  }

  // return the number of file in the librarian

  long Librarian::length (void) const {
    rdlock ();
    long result   = 0;
    s_fdesc* desc = p_desc;
    while (desc != nilp) {
      result++;
      desc = desc->p_next;
    }
    unlock ();
    return result;
  }

  // return true if the name exists in this librarian
  
  bool Librarian::exists (const String& name) const {
    rdlock ();
    s_fdesc* desc = p_desc;
    while (desc != nilp) {
      if (desc->d_fname == name) {
	unlock ();
	return true;
      }
      desc = desc->p_next;
    }
    unlock ();
    return false;
  }

  // return a list of file in this librarian

  Strvec Librarian::getlist (void) const {
    rdlock ();
    Strvec result;
    s_fdesc* desc = p_desc;
    while (desc != nilp) {
      result.add (desc->d_fname);
      desc = desc->p_next;
    }
    unlock ();
    return result;
  }

  // return a vector of file in this librarian

  Vector* Librarian::getstr (void) const {
    rdlock ();
    Vector* result = new Vector;
    s_fdesc* desc = p_desc;
    while (desc != nilp) {
      result->append (new String (desc->d_fname));
      desc = desc->p_next;
    }
    unlock ();
    return result;
  }

  // extract a file by name

  InputMapped* Librarian::extract (const String& name) const {
    if (d_type == OUTPUT)
      throw Exception ("librarian-error", "cannot extract from librarian");
    rdlock ();
    // get the descriptor by name
    s_fdesc* desc = get_named_desc (p_desc, name);
    if (desc == nilp) {
      unlock ();
      throw Exception ("extract-error", "cannot extract file", name);
    }
    // get the mapped file
    long size = desc->d_fsize;
    long foff = desc->d_lfoff;
    InputMapped* result = new InputMapped (d_lname, size, foff);
    unlock ();
    return result;
  }

  // write the librarian to an output file

  void Librarian::write (const String& lname) const {
    OutputFile os (lname);
    rdlock ();
    try {
      // write the header
      write_header (os, p_desc);
      // write all file sequentialy
      s_fdesc* desc = p_desc;
      while (desc != nilp) {
	InputFile is (desc->d_fpath);
	while (is.iseof () == false) os.write (is.read ());
	desc = desc->p_next;
      }
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // format the librarian content to an output stream

  void Librarian::format (Output& os) const {
    rdlock ();
    s_fdesc* desc = p_desc;
    while (desc != nilp) {
      desc->format (os);
      desc = desc->p_next;
    }
    unlock ();
  }

  // return true if the path is a valid librarian file

  bool Librarian::valid (const String& path) {
    try {
      read_header (path);
      return true;
    } catch (...) {
      return false;
    }
  }

  // create a new librarian in a generic way

  Object* Librarian::mknew (Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // check for 0 argument
    if (argc == 0) return new Librarian;

    // check for 1 argument
    if (argc == 1) {
      String fname = argv->getstring (0);
      return new Librarian (fname);
    }
    throw Exception ("argument-error", 
		     "invalid number of argument with librarian");
  }

  // apply a librarian method with a set of arguments and a quark

  Object* Librarian::apply (Runnable* robj, Nameset* nset, const long quark,
			 Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_LENGTH) return new Integer (length ());
      if (quark == QUARK_GETVEC) return getstr ();
    }

    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_EXISTS) {
	String fname = argv->getstring (0);
	return new Boolean (exists (fname));
      }
      if (quark == QUARK_ADD) {
	String fname = argv->getstring (0);
	add (fname);
	return nilp;
      }
      if (quark == QUARK_WRITE) {
	String fname = argv->getstring (0);
	write (fname);
	return nilp;
      }
      if (quark == QUARK_EXTRACT) {
	String fname = argv->getstring (0);
	Object* result = extract (fname);
	robj->post (result);
	return result;
      }
    }

    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }
}
