// ---------------------------------------------------------------------------
// - Character.hpp                                                           -
// - standard object library - character class definition                    -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef ALEPH_CHARACTER_HPP
#define ALEPH_CHARACTER_HPP

#ifndef  ALEPH_LITERAL_HPP
#include "Literal.hpp"
#endif

namespace aleph {

  /// The Character class is the object version of the basic 8-bit character.
  /// The Character class is derived from the literal class and implements
  /// standard comparision operators.
  /// @author amaury darsch

  class Character: public Literal {
  private:
    /// the character representation
    char d_value;

  public:
    /// create a new default character
    Character (void);

    /// create a new character from a native character
    /// @param value the value to create
    Character (const char value);

    /// create a new character from a string representation
    /// @param value the string to convert
    Character (const String& value);

    /// copy constructor for this character class
    /// @param that the character class to copy
    Character (const Character& that);

    /// @return the class name
    String repr (void) const;

    /// @return a literal representation of this character
    String toliteral (void) const;

    /// @return a string representation of this character
    String tostring (void) const;

    /// @return a clone of this character
    Object* clone (void) const;

    /// @return the character serial code
    t_byte serialid (void) const;

    /// serialize this character to an output stream
    /// @param os the output stream to write
    void wrstream (class Output& os) const;

    /// deserialize a character from an input stream
    /// @param is the input steam to read in
    void rdstream (class Input& is);

    /// @return a native character
    char tochar (void) const;

    /// assign a character with a native value
    /// @param value the value to assign
    Character& operator = (const char value);

    /// assign a character with a character value
    /// @param value the value to assign
    Character& operator = (const Character& value);

    /// compare this character with a native value
    /// @param value the value to compare
    /// @return true if they are equals
    bool operator == (const char value) const;

    /// compare this character with a native value
    /// @param value the value to compare
    /// @return true if they are not equals
    bool operator != (const char value) const;

    /// compare two characters
    /// @param value the value to compare
    /// @return true if they are equals
    bool operator == (const Character& value) const;

    /// compare two characters
    /// @param value the value to compare
    /// @return true if they are not equals
    bool operator != (const Character& value) const;

    /// compare two characters
    /// @param value the value to compare
    /// @return true if they are less
    bool operator < (const Character& value) const;

    /// compare two characters
    /// @param value the value to compare
    /// @return true if they are less or equal
    bool operator <= (const Character& value) const;

    /// compare two characters
    /// @param value the value to compare
    /// @return true if they are greater
    bool operator > (const Character& value) const;

    /// compare two characters
    /// @param value the value to compare
    /// @return true if they are greater or equal    
    bool operator >= (const Character& value) const;

    /// @return true if the character is an alphanumeric
    bool isalpha (void) const;

    /// @return true if the character is a digit
    bool isdigit (void) const;

    /// @return true if the character is a blank or tab
    bool isblank (void) const;

    /// @return true if the character is an eol
    bool iseol (void) const;

    /// @return true if the character is an eof
    bool iseof (void) const;

    /// @return true if the character is nil
    bool isnil (void) const;

    /// evaluate an object to a character value
    /// @param robj the current runnable
    /// @param nset the current nameset
    /// @param object the object to evaluate
    static char evalto (Runnable* robj, Nameset* nset, Object* object);

    /// generate a new character
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// operate this character with another object
    /// @param robj   the current runnable
    /// @param type   the operator type
    /// @param object the operand object
    Object* oper (Runnable* robj, t_oper type, Object* object);

    /// set an object to this character
    /// @param robj   the current runnable
    /// @param nset   the current nameset
    /// @param object the object to set
    Object* vdef (Runnable* robj, Nameset* nset, Object* object);

    /// apply this character with a set of arguments and a quark
    /// @param robj   the current runnable
    /// @param nset   the current nameset    
    /// @param quark  the name to apply these arguments
    /// @param argv   the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);

  public:
    // the memory allocation
    void* operator new    (const t_size size);
    void  operator delete (void* handle);
  };
}

#endif
