;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SDS-GLOBAL -*-

#|

DESC: base/tools.lisp - various general tools
Copyright (c) 1999-2000 - Stig Erik Sand

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

|#

(in-package :sds-global)


(defgeneric equal-to (a b)
  (:documentation "Equality like \\#'equal also for objects"))

(defmethod equal-to (a b)
  (equal a b))


(defun sys/getenv (var)
  "Return the value of the environment variable."
  #+allegro (sys::getenv (string var))
  #+clisp (sys::getenv (string var))
  #+cmu (cdr (assoc (string var) ext:*environment-list* :test #'equalp
                    :key #'string)) ; xlib::getenv
  #+gcl (si:getenv (string var))
  #+lispworks (lw:environment-variable (string var))
  #+lucid (lcl:environment-variable (string var))
  #-(or allegro clisp cmu gcl lispworks lucid)
  (error 'not-implemented :proc (list 'getenv var)))

;;;  Meta functions

(defun curry (fun &rest args)
  "a classic curry"
  (declare (type function fun))
  #'(lambda (&rest more)
      (apply fun (append args more))))

(defun rcurry (fun &rest args)
  "a classic right curry"
  (declare (type function fun))
  #'(lambda (&rest more)
      (apply fun (append more args))))

(defun compose (f g)
  "Returns a closure which is the composition of functions 
F and G."
  (declare (function f g))
  #'(lambda (&rest args)
      (funcall f (apply g args))))

;;;---

(defun force-to-list (ptr)
  "Makes ptr into a list if it's not already one, if the
arugment is NIL, NIL is returned"
  (if (listp ptr) ptr (list ptr)))

(defun vector-map (fn vec)
  "useful for me when I am lazy.."
  (map 'vector fn vec))



(defun flatten (l)
  "returns all non-conses in tree L in a single fresh list"
  (cond ((null l) nil)
        ((consp l) (nconc (flatten (first l))
                          (flatten (rest l))))
        (t (list l))))

(defmacro while (test &body body)
  "repeat BODY while TEST is true"
  `(do ()
       ((not ,test))
     ,@body))

(defmacro when-bind ((var expr) &body body)
  "generalisation of (let ((var expr)) (when var ...))."
  `(let ((,var ,expr))
    (when ,var
      ,@body)))

(defmacro unless-bind ((var expr) &body body)
  "generalisation of (let ((var expr)) (unless var ...))."
  `(let ((,var ,expr))
    (unless ,var
      ,@body)))

(defun positive-integer? (obj)
  "Returns T if obj is an integer and > 0."
  (and (integerp obj) (> obj 0)))

(defun non-negative-integer? (obj)
  "Returns T if obj is an integer and >= 0."
  (and (integerp obj) (>= obj 0)))


(defun split-seq-on (str &optional (ch #\Space))
  "returns a list of strings formed by breaking STR at every occurance
of CH (which is not included).  Works for any sequence, not just strings,
but optimized for vectors."
  (when str
    (do* ((prev-pos 0 (1+ next-pos))
          (next-pos (position ch str)
                    (position ch str :start prev-pos))
          (stuff (list (subseq str 0 next-pos))
                 (cons (subseq str prev-pos next-pos)
                       stuff)))
        ((null next-pos) (nreverse stuff)))))

(defun split-seq-using (str &optional (ch #\Space))
  "returns a list of strings.  Ignores multiple delimiters."
  (when str
    (do* ((prev-pos (position ch str :test-not #'eql)
                    (position ch str :test-not #'eql :start next-pos))
          (next-pos (when prev-pos (position ch str :start prev-pos))
                    (when prev-pos (position ch str :start prev-pos)))
          (stuff (when prev-pos
                   (list (subseq str prev-pos next-pos)))
                 (if prev-pos
                     (cons (subseq str prev-pos next-pos)
                           stuff)
                   stuff)))
        ((null next-pos) (nreverse stuff)))))


(defun filter (func lst)
  "returns a fresh list of all non-nil values generated by FUNC mapped
over LST"
  (declare (optimize (speed 3) (safety 1) (debug 0))
	   ;;(:explain :calls :types :boxing)
	   (type function func)
           )
  (let ((acc nil))
    (dolist (x lst)
      (let ((val (funcall func x)))
        (when val
          (push val acc))))
    (nreverse acc)))


;;; ugly..
(defun read-new-value ()
  (format t "Enter a new value: ")
  (let ((foo (list (car (multiple-value-list (read-line))))))
;;    (format t "returning ~a ~s~%" foo)
    foo))

(define-condition req-file-not-found (file-error)
  ()
  (:report (lambda (cond stream)
	     (format stream "Required file ~a not readable" 
		     (file-error-pathname cond)))))
  




(defun require-file (fname purpose)
  "Makes sure the file is there.. used for conditions"
  
  (declare (type string purpose))
;;  (format t "Checking out ~a~%" fname)
  (let ((is-file (probe-file fname))
	(report (strcat "Specify another file for " purpose)))
    (unless is-file
      (restart-case
	  (error 'req-file-not-found
		 :pathname fname)
	(use-new-file (new-file)
	    :report (lambda (stream) (write-string report stream))
	    :interactive read-new-value
	  ;;(format t "~a~%" (type-of new-file))
	  (require-file new-file purpose))))
    t))


(defun list-to-string (the-list)
  "Just returns the strings in the-list catenated together"
  (let ((result ""))
    (declare (type simple-base-string result))
    (dolist (x the-list)
      (declare (type simple-base-string x))
      (setq result (strcat result x)))
    
    result))

(defun list-to-sep-string (list &key (use-and-clause t) (and-word "and")
			   (separator ","))
  "Returns a string where list elements are separated by comma."
  (let ((len (length list)))
    
    (case len 
      (0 "")
      (1 (car list))
      (2 (strcat (car list) 
		 (if use-and-clause (strcat " " and-word) separator)  
		 " " 
		 (list-to-sep-string (cdr list) :use-and-clause use-and-clause :and-word and-word)))
      (otherwise
       (strcat (car list) separator " " (list-to-sep-string (cdr list) 
						   :use-and-clause use-and-clause 
						   :and-word and-word
						   :separator separator)))) 
    
    ))


(defun fill-info-obj (obj type val info)
  "This basically fills the obj with values"

  ;;    (warn "making info ~a ~a ~a" type val info)
  (when type
    (setf (slot-value obj 'type) (list type)))
  (when val
    (setf (slot-value obj 'value) (list val)))
  (when info
    (setf (slot-value obj 'info) (list info)))
  obj)

(defun get-info-except-types (info-list types)
  (flet ((filter-fun (x)
	   (dolist (i types)
	     (when (string-equal i (car (slot-value x 'type)))
	       (return-from filter-fun nil)))
	   x))
    
    (filter #'filter-fun info-list)))

(defun get-info-of-type (info-list type)
  "info-list is a list of sdoc-info and type is a string"
  (filter #'(lambda (x) (if (string-equal type (car (slot-value x 'type)))
			    x
			  nil))
	  info-list))

(defun strip-info-fields (info-list type-to-go)
  "Takes a list of sdoc-info and a string naming the field in each
info-field to remove. Result is returned as list of conses. The order
in the conses is as follows: (value . info), (type . info) and (type . value)"
  
  (flet ((get-right-fun ()
	   (cond 
	    ((string-equal type-to-go "type")
	     #'(lambda (x) (cons (slot-value x 'value)
				 (slot-value x 'info))))
	    ((string-equal type-to-go "value")
	     #'(lambda (x) (cons (slot-value x 'type)
				 (slot-value x 'info))))
	    ((string-equal type-to-go "info")
	     #'(lambda (x) (cons (slot-value x 'type)
				 (slot-value x 'value)))))))
	    
    (mapcar (get-right-fun) info-list)))

(defun collect-garbage ()
  "If possible, tells the system to do a GC"
  #+allegro (excl:gc t)
  #+clisp (ext:gc)
  #+cmucl (ext:gc)
  #+gcl (si::gbc)
  #+lispworks (normal-gc)
  
  t)


(defun figure-out-language (information)
  "returns a keyword or NIL.  Operates pretty fussy on information given."
  
  (block figure
    (cond ((consp information)
	   (dolist (i information)
	     (let ((lang (figure-out-language i)))
	       (unless (eq lang nil)
		 (return-from figure lang)))))
      
	  ((or (typep information 'sdoc-toplevel)
	       (typep information 'csf-toplevel))
	   (figure-out-language (slot-value information 'language)))
	  
	  ((stringp information)
	   (cond ((string-equal information "java")
		  :java)
		 ((string-equal information "c")
		  :c)
		 ((or (string-equal information "c++")
		      (string-equal information "cxx")
		      (string-equal information "cpp")
		      (string-equal information "cppp")
		      (string-equal information "opencxx"))
		  :c++)
		 ((string-equal information "python")
		  :python)
		 ((or (string-equal information "lisp")
		      (string-equal information "common-lisp"))
		  :lisp)
		 (t
		  (warn "Unknown language ~s" information)
		  nil)))
	  (t
	   (error "Unable to figure out language from ~s" information)))
    ))


(defun arrange-duplicates (obj-list) 
  "Rearranges the obj-list into a list and lets duplicates 
using GET-OBJECT-NAME be represented as a list."
  (let ((obj-table (make-hash-table :test #'equal)))   
    (dolist (i obj-list)  
      (push i (gethash (get-object-name i) obj-table)))
    (loop for v being the hash-values of obj-table
			  collecting v)))

(defun make-decent-filename (str)
  "removes odd characters from given string and returns result. optimise later."
  (substitute #\- #\* (substitute #\- #\Space str)))

(defun htbl-to-list (htbl)
  "Returns the values in a hash-table as a fresh list."
  (loop for x being each hash-value in htbl
	collecting x))
