/**********************************************************************
 ** Builder class: These functions control the builder activities. They
 **                create the builder and open the socket. They also log
 **                in the builder and load builder data, as well as handle
 **                all builder activities
 **
 ** Reviewed last: version 0.14
 **
 **
 ** Copyright (C) 2000 George Noel (Slate), Ed Boraas
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef BUILDER_C
#define BUILDER_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudtypes.h"
#include "mudobject.h"
#include "builder.h"
#include "player.h"
#include "location.h"
#include "btree.h"
#include "area_dbase.h"
#include "login.h"
#include "verb.h"
#include "parse.h"
#include "input.h"
#include "verb.h"
#include "verb_list.h"
#include "lexer.h"
#include "logs.h"
#include "errlog.h"
#include "object_list.h"
#ifdef WIN32
#include "../win32/winport.h"
#else
#include "port.h"
#endif
#include "mud.h"
#include "global.h"
#include "objtype.h"
#include "utils.h"
#include "editor.h"
#include "timespan.h"
#include "adminflags.h"
#include "action.h"
#include "gameflags.h"
#include "newfuncts.h"
#include "inp_funct.h"
#include "dirread.h"

/***********************************************************************
 ** _Builder (constructor) - creates a builder and opens up the socket
 **                         for business. Logs in the builder and loads
 **                         the builder information into the builder
 **                         object
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/
   
Builder::Builder(void)
{
	init_object(NULL);
}

/***********************************************************************
 ** _Builder (constructor) - creates a builder and opens up the socket
 **                         for business. Logs in the builder and loads
 **                         the builder information into the builder
 **                         object
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/
   
Builder::Builder(Connection *the_conn)
{
	init_object(the_conn);
}   

/***********************************************************************
 ** is_visible - is this builder totally visible
 **
 ** Parameters: Nothing
 **
 ** Returns: 1 for true, 0 for false
 **
 ***********************************************************************/

int Builder::is_visible()
{
    return (vislvl == VIS_VISIBLE);
}


/***********************************************************************
 ** is_vis_admin - is this builder admin invis
 **
 ** Parameters: Nothing
 **
 ** Returns: 1 for true, 0 for false
 **
 ***********************************************************************/

int Builder::is_vis_admin()
{
    return (vislvl == VIS_ADMIN);
}


/***********************************************************************
 ** get_vislvl - get the vislvl of the builder
 **
 ** Parameters: Nothing
 **
 ** Returns: 0, 1, 2, or 3
 **
 ***********************************************************************/

int Builder::get_vislvl()
{
    return vislvl;
}


/***********************************************************************
 ** get_vislvl_str - gets the string version of an builder's vislvl
 **
 ** Parameters: Nothing
 **
 ** Returns: Visible, GameInvis, Incognito, or AdminInvis
 **
 ***********************************************************************/

char *Builder::get_vislvl_str()
{
    return vis_table[vislvl];
}


/***********************************************************************
 ** set_vislvl - sets an builder's vislvl
 **
 ** Parameters: VIS_VISIBLE, VIS_GAME, VIS_INCOG, or VIS_ADMIN
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Builder::set_vislvl(int in_vislvl)
{
    vislvl = in_vislvl;
}

bool Builder::can_see_bldr(Builder *the_bldr)
{
    if (the_bldr->is_visible())
        return true;

    if (the_bldr->is_vis_admin() && get_adminflags()->get_flag(ADMINFLAG_ADMINISTRATOR))
        return true;

    return false;
}

/***********************************************************************
 ** init_object - initializes this object
 **
 ** Parameters: the_conn - the connection, if any, that has already been
 **                        opened for this object
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Builder::init_object(Connection *the_conn)
{
   Strings holder;
   Port    *buildport;
   int     has_valid_conn = 0;

   /* this should be the only place that this is set */
   obj_type = OBJ_TYPE_BUILDER;

   the_area = NULL;
   modifying = NULL;
   still_on = 1;
   quitting = 0;
   in_thread = 0;

   /* create a new connection, accepting the incoming player */
   buildport = mainstruct->get_build_socket();

#if 0
   if (the_conn == NULL)
      the_socket = new_Connection(buildport->get_socket(), this);
   else
   {
      the_socket = the_conn;
#ifdef WIN32
      the_socket->unlock();
#endif
      has_valid_conn = 1;
   }
#endif

   pager_lines = the_config.default_pager_lines; 

   /* set the prompt and push the main input handler onto the stack */
   set_prompt();

   input_handler.push_input_handler(main_bldr_handler, &prompt, 0, 0, 0);


#ifdef WIN32
   if (!has_valid_conn)
   {
       the_socket = new_Connection();

       if (buildport->accept_conn(the_socket, this) == -1)
	   { 
	      still_on = 0;
	      return -1;
	   }
   }
#else
   the_socket = new_Connection(buildport->get_socket(), this);
#endif

   the_socket->set_valid();
   holder.sprintf("login-%d", the_socket->get_socknum());
   set_pname(holder.str_show());

   if (the_socket->lost_link())
   {
      still_on = 0;
      return -1;
   }

   /* initalize variables */
   next_builder = NULL;
   set_area(_("builder"));

   login_builder((void *) this);
   return 1;
}


/***********************************************************************
 ** ~_Builder (destructor) - cleans up the object for deletion
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Builder::~Builder()
{
   /* clear away the next builder */
   next_builder = NULL;

   /* remove any locks we have on objects */
   if (the_area != NULL)
      unload_area();

   /* close out the socket */
   delete_Connection(the_socket);
}


/***********************************************************************
 ** get_long_input - like get_input, but this gets many strings of input
 **                  used for things like 'set desc'
 **
 ** Parameters: the_input - where we are putting the input into
 **
 ** Returns: the start_editor 
 **          -1 if failed
 **
 ***********************************************************************/
   
int Builder::get_long_input(Strings *the_input)
{ 
   Editor *input_editor;

   if (the_input == NULL)
      return -1;

   input_editor = new Editor(the_input->str_show());
   return input_editor->start_editor((MudObject *) this, the_input);
}


/***********************************************************************
 ** get_long_input - like get_input, but this gets many strings of input
 **                  used for things like 'set desc'
 **
 ** Parameters: the_input - where we are putting the input into
 **             remove_newline - should we remove the last newline when done
 **
 ** Returns: the start_editor 
 **          -1 if failed
 **
 ***********************************************************************/
   
int Builder::get_long_input(Strings *the_input, int remove_newline)
{ 
   Editor *input_editor;

   if (the_input == NULL)
      return -1;

   input_editor = new Editor(the_input->str_show());
   return input_editor->start_editor((MudObject *) this, the_input, 
                                                          remove_newline);
}


/***********************************************************************
 ** check_connection - goes to the connection and waits for input from
 **                    the player. Gets that input and handles it
 **
 ** Parameters: None
 **
 ** Returns: returns num chars gotten if successful
 **          -1 error with select
 **          -2 error with read
 **          -3 if empty packet found
 **          -4 if we lost connection (check errno)
 **
 ***********************************************************************/
   
int Builder::check_connection()
{
   Strings *the_input = NULL;
   Strings holder;
   int had_input = 0;
   int sock_result;

   if (((sock_result = the_socket->check_socket()) > 0) || 
        (the_socket->has_input()))
   {
       if (the_socket->has_input())
       {   
	 // Loop through and handle lines
	 for (int i=0; i<input_handler.get_lines_per_handle(); i++)
	 {
           the_input = the_socket->get_input();

           if (the_input != NULL)
           {
              holder.sprintf("                                   \r"
                             "&+g%s's input:&* %s", get_title(),
                                                       the_input->str_show());
              get_connection()->send_to_snoopers(holder.str_show());
              input_handler.exec_input_handler((MudObject *) this,
                                                     the_input->str_show());
	      delete the_input;
	   }
	   else
	   {
	     break;
	   }

	   // If this player has lost its connection, exit out
	   if (the_socket == NULL)
	   {
	     return -5;				
	   }

	   if (!is_quitting() && is_on())
	   {  
	     send_bldr("%s", input_handler.get_prompt());
	   }

           had_input = 1;
	 }

       }


       if (the_socket->has_output())
       {   
           if (!had_input && !is_quitting() && is_on())
	   {
              send_bldr("%s", input_handler.get_prompt());
           }
           the_socket->flush();
       }
#ifdef DEBUG_INPUT
       printf("Socket checking complete.\n");
#endif
       return 1;
   }
   else if (sock_result == -3)
   {
     //      set_off();
      return -1;
   }
   else if (sock_result == 0)
      return 0;
   else
      return -1;

   return 0;
}

/***********************************************************************
 ** send_bldr - sends a string to the builder's IO stream
 **
 ** Parameters: the_str - the string to be sent
 **
 ** Returns: the num of chars sent if successful, -1 if failed
 **
 ***********************************************************************/
   
int Builder::send_bldr(char *new_str, ...)
{
   int xlen;       /* holds the total characters resulting from vsprinf */
   va_list args;   /* holds the list of arguments */
   char *tmp_str;  /* used to move along the string */
   char *arg_str;  /* used to display the argument if we want to, testing */
   int  arg_int;   /* used to hold an integer argument, for testing */
   char arg_char;  /* used to hold a char argument, for testing */
   int  count = 0;
   int  tmp_int;


   /* we have to first estimate the size of a string we need to allocate
      to put this into. This is not an easy task, so we have to search
      along the va_list structs and count the size of each. So far it is
      only set up for strings, integers, and characters. Anything else and
      youre SOL */

   va_start( args, new_str );
   tmp_str = new_str;

   while (*tmp_str)
   {
      if (*tmp_str == '%')
      {
         if ((*(tmp_str+1)) && (*(tmp_str+1) == 's'))
	 {
            arg_str = va_arg(args, char *);
            if (arg_str != NULL) 
               count += (strlen(arg_str));
            else
	       count += 6;
         }
         else if ((*(tmp_str+1)) && (*(tmp_str+1) == 'd'))
	 {
            arg_int = va_arg(args, int);
            count += 10;
         }
         else if ((*(tmp_str+1)) && (*(tmp_str+1) == 'c'))
	 {
            arg_char = (char) va_arg(args, int);
            count += 3;
         }
         else if ((*(tmp_str+1)) && !isalpha(*(tmp_str+1)))
	 {
            arg_str = (char *) va_arg(args, void *);
            count += 100;
         }
         else
	 {
            arg_str = (char *) va_arg(args, void *);
	    count += 10;
         }
      }
      tmp_str++;
      count++;
   }
   /* increment two more just for safety */
   count += 2;

   /* now we reset the varg list and create the array, so we can load them
      all into the string */
   va_start( args, new_str );
   tmp_str = new char[count];
   xlen = vsprintf( tmp_str, new_str, args );
   
   tmp_int = the_socket->send_to_socket(tmp_str);

   /* now clean up after ourselves */
   va_end( args );

   delete tmp_str;

   return tmp_int;
}

/***********************************************************************
 ** get_next_builder - returns the pointer to the next builder in the list
 **
 ** Parameters: None
 **
 ** Returns: pointer to next builder
 **
 ***********************************************************************/
   
Builder *Builder::get_next_builder()
{
   return next_builder;
}

/***********************************************************************
 ** set_next_builder - sets the next builder in the list
 **
 ** Parameters: next_builder - the next builder to set to 
 **
 ** Returns: None
 **
 ***********************************************************************/
   
void Builder::set_next_builder(Builder *the_builder)
{
   next_builder = the_builder;
}


/***********************************************************************
 ** set_name - sets the name of the builder
 **
 ** Parameters: the_name - the name to set the player to
 **
 ** Returns:  1 for success
 **          -1 for failure
 **
 ***********************************************************************/
   
int Builder::set_pname(char *the_name)
{
   if (the_name == NULL)
      return -1;

   set_name(the_name);
   return 1;
}



/***********************************************************************
 ** is_on - is the player still connected and on
 **
 ** Parameters: None
 **
 ** Returns:   1 if player is on
 **            0 if player has quit
 **           -1 if player entry was bad
 **           -2 if player lost link
 **
 ***********************************************************************/
   
int Builder::is_on()
{
   return still_on;
}

/***********************************************************************
 ** set_off - set the player to not on anymore
 **
 ** Parameters: None
 **
 ** Returns:   Nothing
 **
 ***********************************************************************/
   
void Builder::set_off()
{
   still_on = 0;
   return;
}

/***********************************************************************
 ** set_quitting - returns if the player is quitting or not
 **
 ** Parameters: None
 **
 ** Returns:   1 if they are quitting
 **            0 if they are not quitting
 **
 ***********************************************************************/
   
int Builder::is_quitting()
{
   return quitting;
}

/***********************************************************************
 ** set_quitting - set the player to quitting
 **
 ** Parameters: None
 **
 ** Returns:   Nothing
 **
 ***********************************************************************/
   
void Builder::set_quitting()
{
   quitting = 1;
   return;
}


/***********************************************************************
 ** unload_area - unloads an area from this player
 **
 ** Parameters: None
 **
 ** Returns:  1 if successful, -1 if failed
 **
 ***********************************************************************/
   
int Builder::unload_area()
{
   Builder *builder_list;
   int elsewhere = 0;

   builder_list = mainstruct->get_first_builder();
   while (builder_list != NULL)
   {
      if ((builder_list->get_areaname() != NULL) && 
          (!STRCASECMP(get_areaname(), builder_list->get_areaname())))
      {
         if (STRCASECMP(get_name(), builder_list->get_name()))
            elsewhere = 1;
      }
      builder_list = builder_list->get_next_builder();
   }
   if (!elsewhere)
      delete_area();
   else
   {
      the_area = NULL;
      clr_modify();
   }
   name_area.truncate(0);
   return 1;
}


/***********************************************************************
 ** load_area - loads an area file into memory, assigning to the player
 **
 ** Parameters: areaname - the area name
 **             the_file - the file to load from
 **             error_log - the error log to write errors to
 **             build_format - if we should expect builder file format
 **
 ** Returns:  number read in if successful
 **           -1 if failed
 **
 ***********************************************************************/
   
int Builder::load_area(char *areaname, FILE *the_file, ErrLog *error_log, 
                                                         int build_format)
{
   Builder      *builder_list;
   Strings      holder;
   int count = 0;
   Flags        *admflags;

   if (name_area.str_show() != NULL)
      if (name_area.str_cmp(areaname))
      { 
         return -2;
      }

   /* see if somebody else is pointing to it, if not, delete it */
   if (the_area != NULL)
   {
      unload_area();
   }

   admflags = get_adminflags();

   /* see if somebody else has already loaded it, if they have, use theirs */
   builder_list = mainstruct->get_first_builder();
   while (builder_list != NULL)
   {
      if (builder_list->get_areaname() != NULL)
      {
         if (!STRCASECMP(areaname, builder_list->get_areaname()))
         {
            name_area = areaname;
            the_area = builder_list->get_area();

            if ((!the_area->can_read(this)) && 
                !(admflags->get_flag(ADMINFLAG_SETAREAS)))
	    {
               unload_area();
               return -3;
            }
            set_prompt();
            return 1;
         }
      }
      builder_list = builder_list->get_next_builder();
   }

   /* nobody has it already, load it */
   name_area = areaname;
   the_area = new Area_Dbase(name_area.str_show());

   the_area->load_area(the_file, error_log, build_format, 1);
   if ((!the_area->can_read(this)) &&
       !(admflags->get_flag(ADMINFLAG_SETAREAS)))
   {
      unload_area();
      return -3;
   }
   
   set_prompt();

   return count;
}

/***********************************************************************
 ** load_area - loads an area file into memory, assigning to the player
 **
 ** Parameters: the_dir - the directory that has the filenames
 **             error_log - the error log to write errors to
 **             build_format - if we should expect builder file format
 **
 ** Returns:  number read in if successful
 **           -1 if failed
 **
 ***********************************************************************/
   
int Builder::load_area(DirRead *the_dir, char *areaname, char *pattern, 
		       ErrLog *error_log, int build_format)
{
   Builder      *builder_list;
   Strings      holder;
   int count = 0;
   Flags        *admflags;

   if (name_area.str_show() != NULL)
      if (name_area.str_cmp(areaname))
      { 
         return -2;
      }

   /* see if somebody else is pointing to it, if not, delete it */
   if (the_area != NULL)
   {
      unload_area();
   }

   admflags = get_adminflags();

   /* see if somebody else has already loaded it, if they have, use theirs */
   builder_list = mainstruct->get_first_builder();
   while (builder_list != NULL)
   {
      if (builder_list->get_areaname() != NULL)
      {
         if (!STRCASECMP(areaname, builder_list->get_areaname()))
         {
            name_area = areaname;
            the_area = builder_list->get_area();

            if ((!the_area->can_read(this)) && 
                !(admflags->get_flag(ADMINFLAG_SETAREAS)))
	    {
               unload_area();
               return -3;
            }
            set_prompt();
            return 1;
         }
      }
      builder_list = builder_list->get_next_builder();
   }

   /* nobody has it already, load it */
   name_area = areaname;
   the_area = new Area_Dbase(name_area.str_show());

   the_area->load_area(the_dir, pattern, error_log, build_format, 1);
   if ((!the_area->can_read(this)) &&
       !(admflags->get_flag(ADMINFLAG_SETAREAS)))
   {
      unload_area();
      return -3;
   }
   
   set_prompt();

   return count;
}


/************************************************************************
 ** modify_object - loads an object into memory for modification
 **
 ** Parameters: the_name - the name of the object
 **
 ** Returns:  1 if successful
 **           -1 if there was no areafile loaded
 **           -2 if failed loading the object
 **           -3 if somebody is already modifying this object 
 **
 ***********************************************************************/
   
int Builder::modify_object(char *the_name)
{
   Entity *tmp_object;
   Builder *tmp_builder;

   if (the_area == NULL)
      return -1;

   tmp_object = the_area->get_areaobject(the_name);
   if (tmp_object == NULL)
      return -2;
   if (modifying != NULL)
      modifying->unlock();

   modifying = tmp_object;

   tmp_builder = mainstruct->get_first_builder();
   while (tmp_builder != NULL)
   {
      if ((tmp_object == tmp_builder->get_modify()) &&
          (tmp_builder != this))
      {
         modifying = NULL;
         return -3;
      }
      tmp_builder = tmp_builder->get_next_builder();
   }
 
   set_prompt();
   return 1;
}


/***********************************************************************
 ** get_areaname - returns the area name of the area that is currently loaded
 **
 ** Parameters: None
 **
 ** Returns:  a pointer to the areaname string
 **
 ***********************************************************************/
   
char *Builder::get_areaname(void)
{
   return name_area.str_show();
}


/***********************************************************************
 ** set_areaname - sets the area name (in the builder, not the actual area)
 **                of the area that is loaded
 **
 ** Parameters: new_name - the new name to set this to
 **
 ***********************************************************************/
   
void Builder::set_areaname(char *new_name)
{
   name_area = new_name;
}


/***********************************************************************
 ** get_area - returns a pointer to the area database
 **
 ** Parameters: None
 **
 ** Returns:  a pointer to the area database
 **
 ***********************************************************************/
   
Area_Dbase *Builder::get_area(void)
{
   return the_area;
}


/***********************************************************************
 ** delete_area - deletes an area we are pointing to
 **
 ** Parameters: None
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
int Builder::delete_area(void)
{
   if (the_area == NULL)
      return -1;

   clr_modify();
   name_area.truncate(0);

   delete the_area;
   the_area = NULL;

   set_prompt();
   return 1;
}

/***********************************************************************
 ** clr_modify - clears the pointer for modifying objects
 **
 ** Parameters: None
 **
 ** Returns:  None
 **
 ***********************************************************************/
   
void Builder::clr_modify(void)
{
   if (modifying != NULL)
   {
      modifying->unlock();
      modifying = NULL;
   }

   set_prompt();
}


/***********************************************************************
 ** has_modify - detects if there is an object in modify or not and
 **              returns it if it is there
 **
 ** Parameters: None
 **
 ** Returns:  the modifying object
 **           NULL if nothign there
 **
 ***********************************************************************/
   
Entity *Builder::get_modify()
{
   if (modifying == NULL)
      return NULL;
   return modifying;
}



/***********************************************************************
 ** is_in_thread - if the builder has a thread open still
 **
 ** Parameters: the_val - the value to set it to
 **
 ** Returns: 1 for in thread, 0 for not
 **
 ***********************************************************************/
   
int Builder::is_in_thread()
{
   return in_thread;
}


/***********************************************************************
 ** set_in_thread - set the builder as being in the thread
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/
   
void Builder::set_in_thread(int the_val)
{
   in_thread = the_val;
   return;
}


/***********************************************************************
 ** get_passwd - returns the character string with the encrypted passwd
 **
 ** Parameters: None
 **
 ** Returns: a pointer to the encrypted password string
 **
 ***********************************************************************/
   
char *Builder::get_passwd()
{
   return passwd.str_show();
}


/***********************************************************************
 ** set_passwd - allows the user to set the encrypted password string
 **
 ** Parameters: the_passwd - the string to set it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Builder::set_passwd(char *the_passwd)
{
   passwd = the_passwd;
   return 1;
}


/***********************************************************************
 ** echo_off - turns the echo off for the user's keystrokes
 **
 ** Parameters: Nothing
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/

int Builder::echo_off()
{
   return the_socket->echo_off();
}


/***********************************************************************
 ** echo_on - turns the echo on for the user's keystrokes
 **
 ** Parameters: Nothing
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/

int Builder::echo_on()
{
   return the_socket->echo_on();
}


/***********************************************************************
 ** find_host_name - finds the hostname using gethostbyname
 **                  (warning) can cause lag in the thread
 **
 ** Parameters: Nothing
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/

int Builder::find_host_name()
{
   return the_socket->find_host_addr();
}

/***********************************************************************
 ** get_ip_addr - gets the ip address string
 **
 ** Parameters: Nothing
 **
 ** Returns: a pointer to the ip addr string
 **
 ***********************************************************************/

char *Builder::get_ip_addr()
{
   return the_socket->get_ip_addr();
}

/***********************************************************************
 ** get_host_addr - gets the host address string
 **
 ** Parameters: Nothing
 **
 ** Returns: a pointer to the host address string
 **
 ***********************************************************************/

char *Builder::get_host_addr()
{
   return the_socket->get_host_addr();
}


/***********************************************************************
 ** get_flags - returns the command flags structure
 **
 ** Parameters: Nothing
 **
 ** Returns: a pointer to the command flags structure
 **
 ***********************************************************************/

Flags *Builder::get_comflags()
{
   return &command_flags;
}


/***********************************************************************
 ** set_comflags - sets the comflags for this player
 **
 ** Parameters: the_flags - the comflags we are setting to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Builder::set_comflags(save_flags *the_flags)
{
   command_flags.copy_from_save_struct(the_flags);

   return 1;
}

/***********************************************************************
 ** get_gameflags - gets the gameflags for this builder
 **
 ** Parameters: None
 **
 ** Returns: a pointer to the gameflags for success, NULL for failed
 **
 ***********************************************************************/

Flags *Builder::get_gameflags(void)
{
   return &game_flags;
}


/***********************************************************************
 ** set_gameflags - sets the gameflags for this player
 **
 ** Parameters: the_flags - the gameflags we are setting to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Builder::set_gameflags(save_flags *the_flags)
{
   game_flags.copy_from_save_struct(the_flags);

   return 1;
}



/***********************************************************************
 ** get_connection - gets the connection for this builder
 **
 ** Parameters: None
 **
 ** Returns: a pointer to the connection for success, NULL for failed
 **
 ***********************************************************************/

Connection *Builder::get_connection(void)
{
   return the_socket;
}


/***********************************************************************
 ** get_adminflags - gets the adminflags for this builder
 **
 ** Parameters: None
 **
 ** Returns: a pointer to the adminflags for success, NULL for failed
 **
 ***********************************************************************/

Flags *Builder::get_adminflags(void)
{
   return &admin_flags;
}


/***********************************************************************
 ** set_adminflags - sets the adminflags for this player
 **
 ** Parameters: the_flags - the adminflags we are setting to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Builder::set_adminflags(save_flags *the_flags)
{
   admin_flags.copy_from_save_struct(the_flags);

   return 1;
}



/***********************************************************************
 ** check_builder - checks the builder for input, output, and any events
 **
 ** Parameters: None
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/
   
int Builder::check_builder(void)
{
   int conn_result;

#ifdef WIN32
   // If we have windows, we need to take into account multiple threads that could
   // access the connection concurrently.  Handle this by locking the connection
   while (the_socket->set_locked() == FALSE)
	   Sleep(10);

#endif

   if ((conn_result = check_connection()) < 0)
   {
      if ((time(0) - the_socket->lost_link()) > the_config.secs_till_timeout)
      {
         Strings tmpholder;

         tmpholder.sprintf("Builder %s lost link and has timed out.",
                                                                 get_title());
         mainstruct->log_event(tmpholder.str_show());

         set_off();
#ifdef WIN32
		 the_socket->unlock();
#endif
         return -1;
      }
   }

   if (is_quitting())
   {
      set_off();
#ifdef WIN32
	 the_socket->unlock();
#endif
      return -1;
   }

   /* if the mud is shutting down, we remove all builders */
   if (mainstruct->is_shutdown())
   {
      send_bldr(_("Mud shutting down, closing connection.\n"));
      set_off();
#ifdef WIN32
 	  the_socket->unlock();
#endif

      return -1;
   }
#ifdef WIN32
   the_socket->unlock();
#endif

   return 1;
}


/***********************************************************************
 ** get_prompt - gets the prompt for the builder
 **
 ** Parameters: None
 **
 ** Returns: pointer to the prompt
 **
 ***********************************************************************/
   
char *Builder::get_prompt(void)
{
   return prompt.str_show();
}


/***********************************************************************
 ** set_prompt - sets the prompt to the appropriate objname and area
 **
 ** Parameters: None
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/
   
int Builder::set_prompt(void)
{
   char *obj_name;
   char *area_name;

   if (modifying == NULL)
      obj_name = _("[MUDOBJECT]");
   else
      obj_name = modifying->get_name();

   if (the_area == NULL)
       area_name = _("[AREA]");
   else
      area_name = name_area.str_show();

   prompt.sprintf("&+C(&+Y%s&+C:&+G%s&+C) %s", area_name, obj_name, 
                                           the_config.buildprompt.str_show());
   return 1;
}


/***********************************************************************
 ** get_input_handler - gets the input handler for this builder
 **
 ** Parameters: None
 **
 ** Returns: a pointer to the input handler
 **
 ***********************************************************************/
   
Inp_Handler *Builder::get_input_handler(void)
{
   return &input_handler;
}

/***********************************************************************
 ** swap_connection - swaps the connection of this builder with another
 **                   builder.  Essentially changes the ownership of the
 **                   builder with another socket
 **
 ** Parameters: swap_player - who we are swapping with
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/
   
int Builder::swap_connection(Builder *swap_builder)
{
   Connection *tmp_connection;
   Flags      *game_flags;

   game_flags = swap_builder->get_gameflags();
   tmp_connection = the_socket;
   the_socket = swap_builder->get_connection();
   if (game_flags->get_flag(GAMEFLAG_COLOR))
      the_socket->set_color(1);

   swap_builder->set_connection(tmp_connection);
   return 1;
}

/***********************************************************************
 ** set_connection - sets this connection to a new connection
 *
 **
 ***********************************************************************/
   
void Builder::set_connection(Connection *new_connection)
{
   the_socket = new_connection;
}

/***********************************************************************
 ** flush_builder - flushes the output of this builder
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/
   
void Builder::flush_builder(void)
{
   the_socket->flush();
}


/***********************************************************************
 ** display_prompt - displays the prompt to the builder and flushes output
 **
 **
 ***********************************************************************/
   
void Builder::display_prompt(void)
{
   send_bldr("%s", input_handler.get_prompt());
   flush_builder();
}


/***********************************************************************
 ** get_idle - gets the amount of time this player has been idle
 **
 ** Parameters: None
 **
 ** Returns:  time value (long int)
 **
 ***********************************************************************/

time_t Builder::get_idle()
{
   return the_socket->get_idle();
}


/***********************************************************************
 ** get_pager_lines - get the number of lines this pager is set to view
 **                   before getting a "press enter to continue" prompt
 **
 **
 ***********************************************************************/

int Builder::get_pager_lines()
{
   return pager_lines;
}


/***********************************************************************
 ** set_pager_lines - sets the pager lines value (see get_pager_lines)
 **
 ** Parameters: new_val - the new pager lines value
 **
 ***********************************************************************/

void Builder::set_pager_lines(int new_val)
{
   if (new_val > 0)
      pager_lines = new_val;
   else
      pager_lines = 0;
}




#endif

