/*
 * xmmplayer - mplayer plugin for xmms
 * Copyright (C) 2003 Hyriand - See COPYING
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "config.h"
#include "xmmplayer.h"
#include "videoformat.h"
#include "fileinfo.h"
#include "window.h"

#include <gdk/gdkx.h>
#include <gtk/gtk.h>
#include <glib.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/wait.h>
#include <sys/poll.h>
#include <sys/socket.h>
#include <xmms/configfile.h>
#include <xmms/util.h>

InputPlugin mplayer_plugin;
_xmmplayer_info xmmplayer_info;
_xmmplayer_config xmmplayer_config;

int decode_thread = -1;
int pipes[2];

static GtkWidget *about_window = 0;
static GtkWidget *error_dialog = 0;

static int xmmplayer_is_our_file (char *filename);

static void
xmmplayer_init ()
{
  ConfigFile *cfgfile = xmms_cfg_open_default_file ();
  if (!xmms_cfg_read_string
      (cfgfile, "XMMPlayer", "mplayer_path",
       &(xmmplayer_config.mplayer_path)))
    xmmplayer_config.mplayer_path = g_strdup (MPLAYER);
  if (!xmms_cfg_read_string
      (cfgfile, "XMMPlayer", "vo",
       &(xmmplayer_config.vo)))
    xmmplayer_config.vo = g_strdup ("");
  if (!xmms_cfg_read_string
      (cfgfile, "XMMPlayer", "ao",
       &(xmmplayer_config.ao)))
    xmmplayer_config.ao = g_strdup ("");
  if (!xmms_cfg_read_boolean
      (cfgfile, "XMMPlayer", "read_info", &(xmmplayer_config.read_info)))
    xmmplayer_config.read_info = FALSE;
  if (!xmms_cfg_read_boolean
      (cfgfile, "XMMPlayer", "fullscreen", &(xmmplayer_config.fullscreen)))
    xmmplayer_config.fullscreen = FALSE;
  if (!xmms_cfg_read_boolean
      (cfgfile, "XMMPlayer", "nocache", &(xmmplayer_config.nocache)))
    xmmplayer_config.nocache = FALSE;
  if (!xmms_cfg_read_boolean
      (cfgfile, "XMMPlayer", "noninterleaved",
       &(xmmplayer_config.noninterleaved)))
    xmmplayer_config.noninterleaved = FALSE;
  if (!xmms_cfg_read_int
      (cfgfile, "XMMPlayer", "index", &(xmmplayer_config.index)))
    xmmplayer_config.index = 0;
  if (!xmms_cfg_read_string
      (cfgfile, "XMMPlayer", "tv_command", &(xmmplayer_config.tv_c)))
    xmmplayer_config.tv_c = g_strdup("driver=v4l2:adevice=/dev/audio1:chanlist=europe-west:normid=0:input=0");
  if (!xmms_cfg_read_int
      (cfgfile, "XMMPlayer", "tv_width", &(xmmplayer_config.tv_w)))
    xmmplayer_config.tv_w = 384;
  if (!xmms_cfg_read_int
      (cfgfile, "XMMPlayer", "tv_height", &(xmmplayer_config.tv_h)))
    xmmplayer_config.tv_h = 288;
#ifdef LINUX_RTC
  if (!xmms_cfg_read_boolean
      (cfgfile, "XMMPlayer", "nortc", &(xmmplayer_config.nortc)))
    xmmplayer_config.nortc = FALSE;
#endif

  if (!xmms_cfg_read_boolean
      (cfgfile, "XMMPlayer", "videowin_scale",
       &(xmmplayer_config.videowin_scale)))
    xmmplayer_config.videowin_scale = TRUE;
  if (!xmms_cfg_read_boolean
      (cfgfile, "XMMPlayer", "videowin_visible",
       &(xmmplayer_config.videowin_visible)))
    xmmplayer_config.videowin_visible = TRUE;
  if (!xmms_cfg_read_boolean
      (cfgfile, "XMMPlayer", "videowin_autoclose",
       &(xmmplayer_config.videowin_autoclose)))
    xmmplayer_config.videowin_autoclose = TRUE;
  if (!xmms_cfg_read_boolean
      (cfgfile, "XMMPlayer", "videowin_resize",
       &(xmmplayer_config.videowin_resize)))
    xmmplayer_config.videowin_resize = TRUE;
  if (!xmms_cfg_read_int
      (cfgfile, "XMMPlayer", "videowin_width",
       &(xmmplayer_config.videowin_width)))
    xmmplayer_config.videowin_width = 400;
  if (!xmms_cfg_read_int
      (cfgfile, "XMMPlayer", "videowin_height",
       &(xmmplayer_config.videowin_height)))
    xmmplayer_config.videowin_height = 308;
  if (!xmms_cfg_read_int
      (cfgfile, "XMMPlayer", "videowin_x", &(xmmplayer_config.videowin_x)))
    xmmplayer_config.videowin_x = 0;
  if (!xmms_cfg_read_int
      (cfgfile, "XMMPlayer", "videowin_y", &(xmmplayer_config.videowin_y)))
    xmmplayer_config.videowin_y = 0;

  xmms_cfg_free (cfgfile);
}

gint xmmplayer_timeout_func (gpointer data)
{
  if (xmms_remote_is_playing (ctrlsocket_get_session_id()))
    {
      gchar *ext;
      gint pos;
      pos = xmms_remote_get_playlist_pos (ctrlsocket_get_session_id());
      ext = strrchr (xmms_remote_get_playlist_file (ctrlsocket_get_session_id(),pos), '.');

      if (xmmplayer_is_our_file(xmms_remote_get_playlist_file(ctrlsocket_get_session_id(),pos)))
        {
          return FALSE;
        }
      else
        {
          videowin_hide ();
          return FALSE;
        }
    }
  else
    {
      videowin_hide ();
      return FALSE;
    }
}

static void
xmmplayer_done (int pid, int pipes[2])
{
  int x;
  waitpid (pid, &x, 0);
  close (pipes[0]);
  close (pipes[1]);
  if (xmmplayer_config.videowin_visible && xmmplayer_config.videowin_autoclose)
    gtk_timeout_add (0, xmmplayer_timeout_func, NULL);
}

void
xmmplayer_stop ()
{
  xmmplayer_write_config (); 

  if (decode_thread != -1)
    {
      write (pipes[0], "quit\n", 5);
      write (pipes[0], "quit\n", 5);
      xmmplayer_done (decode_thread, pipes);
      decode_thread = -1;
    }
}

static void
xmmplayer_cleanup ()
{
  videowin_destroy ();
  g_free (xmmplayer_config.mplayer_path);
  g_free (xmmplayer_config.vo);
  g_free (xmmplayer_config.ao);

}

static int
pipe_mplayer (GList * args, int pipes[2])
{
  int pid;
  int p_in[2], p_out[2];

  pipe (p_in);
  pipe (p_out);

  pid = fork ();
  if (pid == 0)
    {
      char **argv =
	(char **) g_malloc ((g_list_length (args) + 2) * sizeof (char *));
      int i = 1;

      // Build the argument list
      argv[0] = xmmplayer_config.mplayer_path;
      args = g_list_first (args);
      while (args != NULL)
	{
	  argv[i] = args->data;
	  i++;
	  args = g_list_next (args);
	}
      argv[i] = NULL;

      // Hook up stdin and stdout with the pipes
      dup2 (p_in[0], fileno (stdin));
      dup2 (p_out[1], fileno (stdout));

      // Dispose of now unneeded pipes
      close (p_in[0]);
      close (p_in[1]);
      close (p_out[0]);
      close (p_out[1]);

      // Execute MPlayer
      execv (xmmplayer_config.mplayer_path, argv);
      return -1;
    }

  pipes[0] = p_in[1];
  pipes[1] = p_out[0];

  close (p_in[0]);
  close (p_out[1]);

  return pid;
}

void
xmmplayer_free_extended (_xmmplayer_info * info)
{
  if (info->clip_info)
    {
      g_free (info->clip_info);
      info->clip_info = NULL;
    }
  if (info->video_format)
    {
      g_free (info->video_format);
      info->video_format = NULL;
    }
  if (info->audio_codec)
    {
      g_free (info->audio_codec);
      info->audio_codec = NULL;
    }
}

void
_xmmplayer_ident_file (int pipes[2], _xmmplayer_info *info, int extended)
{
  char buffer[8192], *tmp;
  int r, ix = 0, clip_info = 0, tv = 0;
  
  info->identified = 0;
  info->length = -1;
  info->fps = 0.0;
  info->audio_rate = 0;
  info->audio_bitrate = 0;
  info->audio_channels = 0;
  info->pos = 0;
  info->tracks = 0;
  info->titles = 0;
  info->video_bitrate = 0;
  info->video_aspect = 0;
  info->video_width = -1;
  info->video_height = -1;

  info->clip_info = NULL;
  info->video_format = NULL;
  info->audio_codec = NULL;

  while (1)
    {
      r = read (pipes[1], &buffer[ix], 1);
      if (r < 1)
	break;
      buffer[ix + r] = '\0';
      ix++;
      if (buffer[ix - 1] != '\n')
	continue;
      buffer[ix - 1] = '\0';
      ix = 0;
      if ((clip_info) && (buffer[0] == ' '))
	{
	  if (info->clip_info)
	    {
	      tmp = info->clip_info;
	      info->clip_info = g_strdup_printf ("%s\n%s", tmp, buffer + 1);
	      g_free (tmp);
	    }
	  else
	    info->clip_info = g_strdup (&buffer[1]);
	  continue;
	}
      clip_info = 0;
      if (strncmp (buffer, "ID_FILENAME=", 12) == 0) {
        info->identified = 1;
        tv=1;
      }
      if (strncmp (buffer, "ID_FILENAME=tv://", 17) == 0) {
        tv=2;
        info->identified = 1;
      }

      sscanf (buffer, "There are %i titles on this DVD", &(info->titles));
      
      if (strncmp (buffer, "track ", 6) == 0)
        info->tracks++;

      if (tv==1) {
        sscanf (buffer, "ID_AUDIO_RATE=%i", &(info->audio_rate));
        sscanf (buffer, "ID_AUDIO_BITRATE=%i", &(info->audio_bitrate));
        sscanf (buffer, "ID_VIDEO_BITRATE=%i", &(info->video_bitrate));
        sscanf (buffer, "ID_VIDEO_ASPECT=%f", &(info->video_aspect));
        sscanf (buffer, "ID_VIDEO_WIDTH=%i", &(info->video_width));
        sscanf (buffer, "ID_VIDEO_HEIGHT=%i", &(info->video_height));
        sscanf (buffer, "ID_VIDEO_FPS=%f", &(info->fps));
        sscanf (buffer, "ID_AUDIO_NCH=%i", &(info->audio_channels));
        if (sscanf (buffer, "ID_LENGTH=%i", &(info->length)) == 1)
          info->length *= 1000;
        if (extended) {
          if (strncmp (buffer, "ID_VIDEO_FORMAT=", 16) == 0)
	    info->video_format = videoformat (&buffer[16]);
	  if (strcmp (buffer, "Clip info:") == 0)
	    clip_info = 1;
	  if (strncmp (buffer, "ID_AUDIO_CODEC=", 15) == 0)
	    info->audio_codec = g_strdup (&buffer[15]);
	}
      }
      if (tv==2) {
        info->audio_rate = 0;
        info->audio_bitrate = 0;
        info->video_bitrate = 0;
        info->video_aspect = 0.0;
        info->video_width = xmmplayer_config.tv_w;
        info->video_height = xmmplayer_config.tv_h;;
        info->fps = 25.0;
        info->audio_channels = 2;
        info->length = 0;
      }
    }
}

void
xmmplayer_ident_file (char *url, _xmmplayer_info * info, int extended)
{
  GList *args = NULL;
  int pipes[2], pid;

// Build argument list and pipe MPlayer
  args = g_list_append (args, "-nocache");
#ifdef LINUX_RTC
  if (xmmplayer_config.nortc)
    args = g_list_append (args, "-nortc");
#endif
  args = g_list_append (args, "-vo");
  args = g_list_append (args, "null");
  args = g_list_append (args, "-ao");
  args = g_list_append (args, "null");
  args = g_list_append (args, "-frames");
  args = g_list_append (args, "0");
  args = g_list_append (args, "-identify");
  args = g_list_append (args, url);
  pid = pipe_mplayer (args, pipes);
  g_list_free (args);

  _xmmplayer_ident_file (pipes, info, extended);

  xmmplayer_done (pid, pipes);
}

static int
xmmplayer_is_our_file (char *filename)
{
  int fin;
  char check[17];
  unsigned long *check2;

  check2 = (unsigned long *) check;

  if (strncmp ("vcd://", filename, 6) == 0)
    return 1;

  if (strncmp ("dvd://", filename, 6) == 0)
    return 1;

  if (strncmp ("tv://", filename, 5) == 0)
    return 1;

  if (strncmp ("mplayer://", filename, 10) == 0)
    return 1;

  if (strncmp ("mms://", filename, 6) == 0)
    return 1;

  if (strcasecmp (".ogm", filename + strlen (filename) - 4) == 0)
    return 1;

  fin = open (filename, O_RDONLY);
  if (fin >= 0 && read (fin, check, 17) == 17)
    {
      close (fin);
      if (strncmp (check, "RIFF", 4) == 0)
	{
	  if (strncmp (&check[8], "AVI", 3) == 0)
	    return 1;
	  return 0;
	}

      if ((*check2) == 0x75b22630)	// ASF
	return 1;
      if ((*check2) == 0xb3010000)	// MPEG
	return 1;
      if ((*check2) == 0xba010000)	// MPEG
	return 1;
      if (strncmp (&check[1], "RMF", 3) == 0)	// Real Media File
	return 1;
      if (strncmp (&check[6], "", 4) == 0) // bin
        return 1;

      if (strncmp (&check[4], "moov", 4) == 0)	// QT
	return 1;
      if (strncmp (&check[4], "mdat", 4) == 0)	// QT
	return 1;
      if (strncmp (&check[12], "cmov", 4) == 0)	// QT
	return 1;
      if (strncmp (&check[4], "moov", 4) == 0)	// QT
	return 1;
      if (strncmp (&check[4], "mdat", 4) == 0)	// QT
	return 1;
      if (strncmp (&check[4], "wide", 4) == 0)	// QT
	return 1;
      if (strncmp (&check[4], "free", 4) == 0)	// QT
	return 1;
      if (strncmp (&check[12], "cmov", 4) == 0)	// QT
	return 1;
      if (strncmp (&check[12], "mvhd", 4) == 0)	// QT
	return 1;
      if (strncmp (&check[12], "mdat", 4) == 0)	// QT
	return 1;

      return 0;
    }
  close (fin);
  return 0;
}

static void
xmmplayer_play_file (char *url)
{
  int vcdtrack = -1;
  char *title = NULL;
  GList *arglist = NULL;
  char tvcommand[8192];
  
  xmmplayer_stop ();

  if ((strncmp (url, "mplayer://", 10) != 0)
      && (strncmp (url, "mms://", 6) != 0))
    {
// Fork and start mplayer to identify the file
      xmmplayer_ident_file (url, &xmmplayer_info, 0);

// Check if the media could be identified
      if (xmmplayer_info.identified == 0)
	return;
    }

  if (strcmp (url, "vcd://") == 0)
    {
      gint i, pos = xmms_remote_get_playlist_pos (ctrlsocket_get_session_id());
      char *track;
      for (i = 0; i < xmmplayer_info.tracks; i++)
	{
	  track = g_strdup_printf ("vcd://%i", i + 1);
	  xmms_remote_playlist_ins_url_string (ctrlsocket_get_session_id(), track, pos + i + 1);
	  g_free (track);
	}
      return;
    }

  if (strcmp (url, "dvd://") == 0)
    {
      gint i, pos = xmms_remote_get_playlist_pos (ctrlsocket_get_session_id());
      char *track;
      for (i = 0; i < xmmplayer_info.titles; i++)
	{
	  track = g_strdup_printf ("dvd://%i", i + 1);
	  xmms_remote_playlist_ins_url_string (ctrlsocket_get_session_id(), track, pos + i + 1);
	  g_free (track);
	}
      return;
    }

  if (sscanf (url, "vcd://%i", &vcdtrack))
    title = g_strdup_printf ("VCD Track %i", vcdtrack);

  if (sscanf (url, "dvd://%i", &vcdtrack))
    title = g_strdup_printf ("DVD Title %i", vcdtrack);

  if (! strncmp (url, "tv://", 5)) {
    title = g_strdup_printf ("TV Channel %s", url + 5);
    arglist = g_list_append (arglist, "-tv");
    sprintf(tvcommand, "%s:width=%i:height=%i", xmmplayer_config.tv_c, xmmplayer_config.tv_w + 12, xmmplayer_config.tv_h + 8);
    arglist = g_list_append (arglist, tvcommand);
  }
  
  arglist = g_list_append (arglist, "-slave");

  if (strlen(xmmplayer_config.vo))
    {
      arglist = g_list_append (arglist, "-vo");
      arglist = g_list_append (arglist, xmmplayer_config.vo);
    }

  if (strlen(xmmplayer_config.ao))
    {
      arglist = g_list_append (arglist, "-ao");
      arglist = g_list_append (arglist, xmmplayer_config.ao);
    }

  if (xmmplayer_config.videowin_visible)
    {
      if(xmmplayer_info.identified != 0 && xmmplayer_info.video_width == -1) {
        videowin_hide();
      } else {
        videowin_create ();
        videowin_show ();
        if (xmmplayer_config.videowin_resize && xmmplayer_info.identified != 0) {
	  if (xmmplayer_info.video_aspect != 0.0)
	    videowin_force_resize (xmmplayer_info.video_width, -1,
				   xmmplayer_info.video_aspect);
	  else
	    videowin_force_resize (xmmplayer_info.video_width,
				   xmmplayer_info.video_height, 0.0);
	}
        arglist = g_list_append (arglist, "-wid");
        arglist = g_list_append (arglist, g_strdup_printf ("%i", (int) videocanvas));
        arglist = g_list_append (arglist, "-xy");
        arglist = g_list_append (arglist, g_strdup_printf ("%i", xmmplayer_config.videowin_width));
        if (xmmplayer_config.videowin_scale) {
	  arglist = g_list_append (arglist, "-vf");
	  arglist = g_list_append (arglist, "scale");
	  arglist = g_list_append (arglist, "-zoom");
	}
      }
    }
  else
    videowin_hide ();

  if (xmmplayer_config.fullscreen)
    arglist = g_list_append (arglist, "-fs");
  else
    arglist = g_list_append (arglist, "-nofs");

  if (xmmplayer_config.nocache)
    g_list_append (arglist, "-nocache");

  if (xmmplayer_config.noninterleaved)
    g_list_append (arglist, "-ni");

#ifdef LINUX_RTC
  if (xmmplayer_config.nortc)
    g_list_append (arglist, "-nortc");
#endif

  if (xmmplayer_config.index == 1)
    g_list_append (arglist, "-idx");
  else if (xmmplayer_config.index == 2)
    g_list_append (arglist, "-forceidx");

  if (vcdtrack != -1)
    {
      gint _frames =
	(gint) ((xmmplayer_info.length / 1000) * xmmplayer_info.fps);
      arglist = g_list_append (arglist, "-frames");
      arglist = g_list_append (arglist, g_strdup_printf ("%i", _frames));
    }

  if (strncmp (url, "mplayer://", 10) == 0)
    {
      char *newurl = g_strdup_printf ("http://%s", &url[10]);
      arglist = g_list_append (arglist, newurl);
      title = g_strdup (newurl);
    }
  else
    {
      if (strncmp (url, "mms://", 6) == 0)
	title = g_strdup (url);
      arglist = g_list_append (arglist, url);
    }

// Fork and start mplayer to play the file
  decode_thread = pipe_mplayer (arglist, pipes);

// Free the argument list
  g_list_free (arglist);

// Set XMMS file info
  mplayer_plugin.set_info (title, xmmplayer_info.length,
			   xmmplayer_info.audio_bitrate,
			   xmmplayer_info.audio_rate,
			   xmmplayer_info.audio_channels);
}

void
xmmplayer_fullscreen (void)
{
  if (decode_thread == -1)
    return;
  printf ("going fullscreen\n");
  write (pipes[0], "vo_fullscreen\n", 14);
}

static void
xmmplayer_seek (int time)
{
  gchar *buf;
  if (decode_thread == -1)
    return;
  buf = g_strdup_printf ("seek %i 2\n", time);
  write (pipes[0], buf, strlen (buf));
  g_free (buf);
}

static void
xmmplayer_pause (short paused)
{
  if (decode_thread != -1)
    write (pipes[0], "pause\n", 6);
}

static int
xmmplayer_get_time ()
{
  char buffer[8192], *key;
  int ret, mpos;
  float pos = 0.0;
  struct pollfd pfds[1] = { {0, POLLIN, 0} };
  if (decode_thread == -1)
    return -1;
  pfds[0].fd = pipes[1];
  pfds[0].revents = 0;
  poll (pfds, 1, 0);
  if ((pfds[0].revents & POLL_IN) == POLL_IN)
    {
      ret = read (pipes[1], buffer, 8191);
      buffer[ret] = 0;
      if ((ret < 1) || strstr (buffer, "\nExiting..."))
	{
	  xmmplayer_done (decode_thread, pipes);
	  decode_thread = -1;
	  return -1;
	}
      else
	{
	  key = strstr (buffer, "A:");
	  if (key && sscanf (key, "A:%i:%f", &mpos, &pos) == 2) {
	    pos += mpos * 60.0;
	    xmmplayer_info.pos = (int) (pos * 1000);
	  } else if (key && sscanf (key, "A:%f ", &pos)) {
	    xmmplayer_info.pos = (int) (pos * 1000);
	  }
	}
    }
  return xmmplayer_info.pos;
}

static void
xmmplayer_get_song_info (char *filename, char **title, int *length)
{
  _xmmplayer_info info;
  int vcdtrack;
  
  (*length) = -1;

  if (strncmp (filename, "mplayer://", 10) == 0)
    {
      (*title) = g_strdup_printf ("http://%s", &filename[10]);
      return;
    }

  if (strncmp (filename, "mms://", 6) == 0)
    {
      (*title) = g_strdup (filename);
      return;
    }

  if (strcmp (filename, "vcd://") == 0)
    {
      (*title) = g_strdup_printf ("VCD (all tracks)");
      return;
    }
  else if (sscanf (filename, "vcd://%i", &vcdtrack))
    (*title) = g_strdup_printf ("VCD Track %i", vcdtrack);

  if (strcmp (filename, "dvd://") == 0)
    {
      (*title) = g_strdup_printf ("DVD (all titles)");
      return;
    }
  else if (sscanf (filename, "dvd://%i", &vcdtrack))
    (*title) = g_strdup_printf ("DVD Title %i", vcdtrack);

  if (! strcmp (filename, "tv://"))
    {
      (*title) = g_strdup_printf ("TV (all channels)");
      return;
    }
   else if (! strncmp(filename, "tv://", 5))
    (*title) = g_strdup_printf ("TV Channel %s", filename + 5);
  
  if (!xmmplayer_config.read_info)
    return;
  
  xmmplayer_ident_file (filename, &info, 0);
  
  if (info.identified)
    (*length) = info.length;
}

/**
 * Pop-up GTK about box.
 * Taken from mpg123 plugin.
 */
static void
xmmplayer_about ()
{
  GtkWidget *dialog_vbox1;
  GtkWidget *hbox1;
  GtkWidget *label1;
  GtkWidget *dialog_action_area1;
  GtkWidget *about_exit;

  if (!about_window)
    {
      about_window = gtk_dialog_new ();
      gtk_object_set_data (GTK_OBJECT (about_window), "about_window",
			   about_window);
      gtk_window_set_title (GTK_WINDOW (about_window),
			    "About MPlyaer Plugin");
      gtk_window_set_policy (GTK_WINDOW (about_window), FALSE, FALSE, FALSE);
      gtk_window_set_position (GTK_WINDOW (about_window), GTK_WIN_POS_MOUSE);
      gtk_signal_connect (GTK_OBJECT (about_window), "destroy",
			  GTK_SIGNAL_FUNC (gtk_widget_destroyed),
			  &about_window);
      gtk_container_border_width (GTK_CONTAINER (about_window), 10);

      dialog_vbox1 = GTK_DIALOG (about_window)->vbox;
      gtk_object_set_data (GTK_OBJECT (about_window), "dialog_vbox1",
			   dialog_vbox1);
      gtk_widget_show (dialog_vbox1);
      gtk_container_border_width (GTK_CONTAINER (dialog_vbox1), 5);

      hbox1 = gtk_hbox_new (FALSE, 0);
      gtk_object_set_data (GTK_OBJECT (about_window), "hbox1", hbox1);
      gtk_widget_show (hbox1);
      gtk_box_pack_start (GTK_BOX (dialog_vbox1), hbox1, TRUE, TRUE, 0);
      gtk_container_border_width (GTK_CONTAINER (hbox1), 5);
      gtk_widget_realize (about_window);

      label1 =
	gtk_label_new (DESCRIPTION " v" VERSION
		       " by Hyriand\n\nhttp://thegraveyard.org/xmmplayer.php\n");
      gtk_object_set_data (GTK_OBJECT (about_window), "label1", label1);
      gtk_widget_show (label1);
      gtk_box_pack_start (GTK_BOX (hbox1), label1, TRUE, TRUE, 0);

      dialog_action_area1 = GTK_DIALOG (about_window)->action_area;
      gtk_object_set_data (GTK_OBJECT (about_window), "dialog_action_area1",
			   dialog_action_area1);
      gtk_widget_show (dialog_action_area1);
      gtk_container_border_width (GTK_CONTAINER (dialog_action_area1), 10);

      about_exit = gtk_button_new_with_label ("Ok");
      gtk_signal_connect_object (GTK_OBJECT (about_exit), "clicked",
				 GTK_SIGNAL_FUNC (gtk_widget_destroy),
				 GTK_OBJECT (about_window));

      gtk_object_set_data (GTK_OBJECT (about_window), "about_exit",
			   about_exit);
      gtk_widget_show (about_exit);
      gtk_box_pack_start (GTK_BOX (dialog_action_area1), about_exit, TRUE,
			  TRUE, 0);

      gtk_widget_show (about_window);
    }
  else
    {
      gdk_window_raise (about_window->window);
    }
}

/**
 * Display a GTK box containing the given error message.
 * Taken from mpg123 plugin.
 */
void
xmmplayer_error (char *error, ...)
{
  if (!error_dialog)
    {
      va_list args;
      char string[256];
      va_start (args, error);
      vsnprintf (string, 256, error, args);
      va_end (args);
      GDK_THREADS_ENTER ();
      error_dialog = xmms_show_message ("Error", string, "Ok", FALSE, 0, 0);
      gtk_signal_connect (GTK_OBJECT (error_dialog),
			  "destroy",
			  GTK_SIGNAL_FUNC (gtk_widget_destroyed),
			  &error_dialog);
      GDK_THREADS_LEAVE ();
    }
}

/**
 * XMMS input plugin hook.
 */
InputPlugin *
get_iplugin_info (void)
{
  memset (&mplayer_plugin, 0, sizeof (InputPlugin));
  mplayer_plugin.description = DESCRIPTION " " VERSION;
  mplayer_plugin.about = xmmplayer_about;
  mplayer_plugin.init = xmmplayer_init;
  mplayer_plugin.is_our_file = xmmplayer_is_our_file;
  mplayer_plugin.play_file = xmmplayer_play_file;
  mplayer_plugin.stop = xmmplayer_stop;
  mplayer_plugin.pause = xmmplayer_pause;
  mplayer_plugin.get_time = xmmplayer_get_time;
  mplayer_plugin.cleanup = xmmplayer_cleanup;
  mplayer_plugin.seek = xmmplayer_seek;
  mplayer_plugin.get_song_info = xmmplayer_get_song_info;
  mplayer_plugin.configure = xmmplayer_configure;
  mplayer_plugin.file_info_box = xmmplayer_file_info_box;

  return &mplayer_plugin;
}
