#include <string.h>
#include <time.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#ifdef HAVE_EDV2
# include <endeavour2.h>
#endif

#include "../include/string.h"
#include "guiutils.h"

#include "obj.h"
#include "win.h"
#include "wincb.h"
#include "core.h"
#include "config.h"


static gchar *COPY_SHORTEN_STRING(const gchar *s, gint max);
static const gchar *TIME_DURATION_STRING(
	core_struct *core, gulong dt, gulong t
);

gint WinScanListInsert(
	win_struct *win, gint row,
	obj_struct *obj			/* Transfered */
);
gint WinScanListAppend(
	win_struct *win,
	obj_struct *obj			/* Transfered */
);
void WinScanListRemove(win_struct *win, gint row);
void WinScanListClear(win_struct *win);
void WinScanListUpdateRow(
	win_struct *win, gint row, const obj_struct *obj
);

gint WinResultsListAppend(
	win_struct *win, const gchar *path, const gchar *virus_name
);
void WinResultsListClear(win_struct *win);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Returns a copy of the string that is no longer than max
 *	characters with appropriate shortened notation where
 *	appropriate.
 */
static gchar *COPY_SHORTEN_STRING(const gchar *s, gint max)
{
	gint len;

	if(s == NULL)
	    return(NULL);

	len = STRLEN(s);
	if((len > max) && (max > 3))
	{
	    /* Need to shorten string */
	    const gint i = len - max + 3;

	    return(g_strdup_printf(
		"...%s", (const gchar *)(s + i)
	    ));
	}
	else
	{   
	    return(STRDUP(s));
	}
}

/*
 *      Returns a statically allocated string describing the time
 *      lapsed specified by dt.
 */
static const gchar *TIME_DURATION_STRING(
	core_struct *core, gulong dt, gulong t
)
{
	gulong ct;
	static gchar buf[80];

	/* Less than one minute? */
	if(dt < 60l)
	{
	    g_snprintf(
		buf, sizeof(buf),
		"less than a minute ago"
	    );
	}
	/* Less than one hour? */
	else if(dt < (60 * 60))
	{
	    ct = MAX(dt / 60, 1);
	    g_snprintf(
		buf, sizeof(buf),
		"%ld minute%s ago",
		ct, (ct == 1) ? "" : "s"
	    );
	}
	/* Less than one day? */
	else if(dt < (24 * 60 * 60))
	{
	    ct = MAX(dt / 60 / 60, 1);
	    g_snprintf(
		buf, sizeof(buf),
		"%ld hour%s ago",
		ct, (ct == 1) ? "" : "s"
	    );
	}
	/* Less than one week? */   
	else if(dt < (7 * 24 * 60 * 60))
	{     
	    ct = MAX(dt / 60 / 60 / 24, 1);
	    g_snprintf(
		buf, sizeof(buf),
		"%ld day%s ago",
		ct, (ct == 1) ? "" : "s"
	    );
	}
	/* Less than one month (30 days)? */
	else if(dt < (30 * 24 * 60 * 60))
	{
	    ct = MAX(dt / 60 / 60 / 24 / 7, 1);
	    g_snprintf(
		buf, sizeof(buf),
		"%ld week%s ago",
		ct, (ct == 1) ? "" : "s"
	    );
	}
	/* Less than 6 months ago? */
	else if(dt < (6 * 30 * 24 * 60 * 60))
	{
	    ct = MAX(dt / 60 / 60 / 24 / 30, 1);
	    g_snprintf(
		buf, sizeof(buf),
		"%ld month%s",
		ct, (ct == 1) ? "" : "s"
	    );
	}
	else
	{
#ifdef HAVE_EDV2
	    strncpy(buf, EDVDateString(core->edv_ctx, t), sizeof(buf));
#else
	    gchar *s;
	    time_t tt = (time_t)t;
	    strncpy(buf, ctime(&tt), sizeof(buf));
	    s = strchr(buf, '\n');
	    if(s != NULL)
		*s = '\0';
#endif
	}

	buf[sizeof(buf) - 1] = '\0';

	return(buf);
}


/*
 *	Inserts the Object to the Win's List at the specified row.
 *
 *	The Object will be transfered and should not be referenced
 *	after this call.
 */
gint WinScanListInsert(
	win_struct *win, gint row,
	obj_struct *obj                 /* Transfered */
)
{
	gint i, columns, new_row;
	gchar **strv;
	GtkCList *clist = (win != NULL) ?
	    (GtkCList *)win->scan_clist : NULL;
	if(clist == NULL)
	{
	    ObjDelete(obj);
	    return(-1);
	}

	columns = clist->columns;
	if(columns <= 0)
	{
	    ObjDelete(obj);
	    return(-1);    
	}

	/* Allocate row cell values */
	strv = (gchar **)g_malloc(columns * sizeof(gchar *));
	for(i = 0; i < columns; i++)
	    strv[i] = "";

	gtk_clist_freeze(clist);

	/* Insert or append new row */
	if((row >= 0) && (row < clist->rows))
	    new_row = gtk_clist_insert(clist, row, strv);
	else
	    new_row = gtk_clist_append(clist, strv);

	/* Delete row cell values */
	g_free(strv);

	/* Failed to create new row? */
	if(new_row < 0)
	{
	    gtk_clist_thaw(clist);
	    ObjDelete(obj);
	    return(-1);    
	}

	if(obj != NULL)
	{
	    /* Set the Object as the row data */
	    gtk_clist_set_row_data_full(
		clist, new_row,
		obj, WinScanListItemDestroyCB
	    );

	    /* Update the row cell values with the new object */
	    WinScanListUpdateRow(win, new_row, obj);
	}

	gtk_clist_thaw(clist);

	return(new_row);
}

/*
 *	Appends the Object to the Win's List.
 *
 *	The Object will be transfered and should not be referenced
 *	after this call.
 */
gint WinScanListAppend(
	win_struct *win,
	obj_struct *obj                 /* Transfered */
)
{
	return(WinScanListInsert(win, -1, obj));
}

/*
 *	Removes the specified row on the Win's List.
 */
void WinScanListRemove(win_struct *win, gint row)
{
	GtkCList *clist = (win != NULL) ?
	    (GtkCList *)win->scan_clist : NULL;
	if(clist == NULL)
	    return;

	gtk_clist_freeze(clist);
	gtk_clist_remove(clist, row);
	gtk_clist_thaw(clist);
}

/*
 *	Clears all rows on the Win's List.
 */
void WinScanListClear(win_struct *win)
{
	GtkCList *clist = (win != NULL) ?
	    (GtkCList *)win->scan_clist : NULL;
	if(clist == NULL)
	    return;

	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	gtk_clist_thaw(clist);
}

/*
 *	Updates the specified row on the Win's List with the values
 *	specified in the Object.
 */
void WinScanListUpdateRow(
	win_struct *win, gint row, const obj_struct *obj
)
{
	gint column, columns;
	gulong cur_time = (gulong)time(NULL);
	GdkPixmap	*location_pixmap = NULL,
			*name_pixmap = NULL;
	GdkBitmap	*location_mask = NULL,
			*name_mask = NULL;
#ifdef HAVE_EDV2
	edv_context_struct *ctx;
#endif
	core_struct *core;
	GtkCList *clist = (win != NULL) ?
	    (GtkCList *)win->scan_clist : NULL;
	if(clist == NULL)
	    return;

	columns = clist->columns;
	if((row < 0) || (row >= clist->rows) || (columns <= 0))
	    return;

	core = CORE(win->core);
#ifdef HAVE_EDV2
	ctx = core->edv_ctx;
#endif

	/* Get the scan item pixmap */
	name_pixmap = core->scan_item_pixmap;
	name_mask = core->scan_item_mask;
	GDK_PIXMAP_REF(name_pixmap);
	GDK_BITMAP_REF(name_mask);

#ifdef HAVE_EDV2
	/* Match Device or MIME Type to get the location pixmap */
	if((obj != NULL) ? !STRISEMPTY(obj->path) : FALSE)
	{
	    const gchar *path = obj->path;
	    struct stat lstat_buf;

	    /* Match Device */
	    edv_device_struct *d = EDVDeviceMatch(ctx, path);
	    if(d != NULL)
	    {
		const edv_device_icon_state state =
		    EDV_DEVICE_ICON_STATE_STANDARD;
		location_pixmap = GDK_PIXMAP_NEW_FROM_XPM_FILE(
		    &location_mask, d->small_icon_file[state]
		);
	    }
	    /* Match MIME Type */
	    else if(!lstat(path, &lstat_buf))
	    {
		edv_mime_type_struct *m = EDVMimeTypeMatch(
		    ctx, path, &lstat_buf
		);
		if(m != NULL)
		{
		    const edv_mime_type_icon_state state =
			EDV_MIME_TYPE_ICON_STATE_STANDARD;
		    location_pixmap = GDK_PIXMAP_NEW_FROM_XPM_FILE(
			&location_mask, m->small_icon_file[state]
		    );
		}
	    }
	}
#endif	/* HAVE_EDV2 */


	/* Begin updating cells */
	gtk_clist_freeze(clist);

	/* Name */
	column = 0;
	if(column < columns)
	{
	    gchar *s = COPY_SHORTEN_STRING(
		(obj != NULL) ? obj->name : NULL,
		20
	    );
	    if(name_pixmap != NULL)
		gtk_clist_set_pixtext(
		    clist, row, column,
		    (s != NULL) ? s : "",
		    WIN_LIST_PIXMAP_TEXT_SPACING,
		    name_pixmap, name_mask
		);
	    else
		gtk_clist_set_text(
		    clist, row, column,
		    (s != NULL) ? s : ""
		);
	    g_free(s);
	}

	/* Location */
	column = 1;
	if(column < columns)
	{
	    gchar *s = COPY_SHORTEN_STRING(
		(obj != NULL) ? obj->path : NULL,
		25
	    );
	    if(location_pixmap != NULL)
		gtk_clist_set_pixtext(
		    clist, row, column,
		    (s != NULL) ? s : "",
		    WIN_LIST_PIXMAP_TEXT_SPACING,
		    location_pixmap, location_mask
		);
	    else  
	        gtk_clist_set_text(
		    clist, row, column,
		    (s != NULL) ? s : ""
	        );
	    g_free(s);
	}

	/* Last Runned */
	column = 2;
	if(column < columns)
	{
	    time_t t = (time_t)((obj != NULL) ? obj->last_runned : 0l);
	    const gchar *s = (t > 0l) ? TIME_DURATION_STRING(
		core, cur_time - t, t
	    ) : "Unknown";
	    gtk_clist_set_text(
		clist, row, column,
		(s != NULL) ? s : ""
	    );
	}     

	/* Resize columns */
	gtk_clist_columns_autosize(clist);

	gtk_clist_thaw(clist);

	GDK_PIXMAP_UNREF(name_pixmap);
	GDK_BITMAP_UNREF(name_mask);
	GDK_PIXMAP_UNREF(location_pixmap);
	GDK_BITMAP_UNREF(location_mask);
}


/*
 *	Appends the row to the Win's Results List.
 */
gint WinResultsListAppend(
	win_struct *win, const gchar *path, const gchar *virus_name
)
{
	gint i, column, row, new_row, columns;
	gchar **strv;
	GdkPixmap *pixmap = NULL;
	GdkBitmap *mask = NULL;
#ifdef HAVE_EDV2
	edv_context_struct *ctx;
#endif
	obj_struct *obj;
	core_struct *core;
	GtkCList *clist = (win != NULL) ?
	    (GtkCList *)win->results_clist : NULL;
	if(clist == NULL)
	    return(-1);

	columns = clist->columns;
	if(columns <= 0)
	    return(-1);

	core = CORE(win->core);
#ifdef HAVE_EDV2
	ctx = core->edv_ctx;
#endif

	/* Allocate row cell values */
	strv = (gchar **)g_malloc(columns * sizeof(gchar *));
	for(i = 0; i < columns; i++)                         
	    strv[i] = "";

	gtk_clist_freeze(clist);

	/* Append new row */
	new_row = gtk_clist_append(clist, strv);

	/* Delete row cell values */
	g_free(strv);

	/* Failed to create new row? */
	if(new_row < 0)                
	{
	    gtk_clist_thaw(clist);
	    return(-1);
	}

	/* Create an object to hold the result information and set it
	 * as the new row's data
	 */
	obj = ObjNew();
	if(obj != NULL)
	{
	    obj->path = STRDUP(path);
	    gtk_clist_set_row_data_full(
		clist, new_row,
		obj, WinResultsListItemDestroyCB
	    );
	}

	/* Infected with virus? */
	if((virus_name != NULL) ?
	    (strcasestr(virus_name, "virus") != NULL) : FALSE
	)
	{
	    pixmap = core->file_infected_pixmap;
	    mask = core->file_infected_mask;
	    GDK_PIXMAP_REF(pixmap);
	    GDK_BITMAP_REF(mask);
	}
	/* Match Device or MIME Type */
	else if(!STRISEMPTY(path))
	{
#ifdef HAVE_EDV2
	    struct stat lstat_buf;

	    /* Match Device */
	    edv_device_struct *d = EDVDeviceMatch(ctx, path);
	    if(d != NULL)     
	    {
		const edv_device_icon_state state =
		    EDV_DEVICE_ICON_STATE_STANDARD;
		pixmap = GDK_PIXMAP_NEW_FROM_XPM_FILE(
		    &mask, d->small_icon_file[state]
		);
	    }     
	    /* Match MIME Type */
	    else if(!lstat(path, &lstat_buf))
	    {
		edv_mime_type_struct *m = EDVMimeTypeMatch(
		    ctx, path, &lstat_buf
		);
		if(m != NULL)
		{
		    const edv_mime_type_icon_state state =
			EDV_MIME_TYPE_ICON_STATE_STANDARD;
		    pixmap = GDK_PIXMAP_NEW_FROM_XPM_FILE(
			&mask, m->small_icon_file[state]
		    );
		}
	    }
#endif	/* HAVE_EDV2 */
	}


	/* Begin setting cells */
	row = new_row;

	/* Object */
	column = 0;
	if(column < columns)
	{
	    gchar *s = COPY_SHORTEN_STRING(path, 32);
	    if(pixmap != NULL)
		gtk_clist_set_pixtext(
		    clist, row, column,
		    (s != NULL) ? s : "",
		    WIN_LIST_PIXMAP_TEXT_SPACING,
		    pixmap, mask 
		);
	    else
		gtk_clist_set_text(  
		    clist, row, column,
		    (s != NULL) ? s : ""
		);
	    g_free(s);
	}     

	/* Virus/Problem */
	column = 1;
	if(column < columns)
	{
	    const gchar *s = virus_name;
	    gtk_clist_set_text(
		clist, row, column,
		(s != NULL) ? s : ""   
	    );
	}

	/* Resize columns */
	gtk_clist_columns_autosize(clist);

	gtk_clist_thaw(clist);

	GDK_PIXMAP_UNREF(pixmap);
	GDK_BITMAP_UNREF(mask);

	return(new_row);
}


/*
 *      Clears all rows on the Win's Results List.
 */
void WinResultsListClear(win_struct *win)
{
	GtkCList *clist = (win != NULL) ?
	    (GtkCList *)win->results_clist : NULL;
	if(clist == NULL)
	    return;

	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	gtk_clist_thaw(clist);
} 
