/*
 *   Copyright (C) 2002-2004 by Jonathan Naylor G4KLX
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <sys/time.h>
#include <math.h>

#include "common/FFT.h"
#include "common/SFFT.h"
#include "common/SoundFile.h"
#include "common/Exception.h"
#include "common/Average.h"

#include "jt6m/JT6MLookups.h"
#include "jt6m/JT6MDefs.h"
#include "jt6m/JT6MFoldMessages.h"

#include "JT6MDecoder.h"

int main(int argc, char **argv)
{
	if (argc < 2) {
		::fprintf(stderr, "Usage: JT6MDecoder <filename>\n");
		return 1;
	}

	wxString fileName = wxString(argv[1]);

	try {
		CJT6MDecoder decoder(fileName);
		decoder.run();
	}
	catch (CException& ex) {
		::fprintf(stderr, "Error: %s\n", ex.getMessage().c_str());
		return 1;
	}
	catch (...) {
		::fprintf(stderr, "An exception has occurred\n");
		return 1;
	}

	return 0;
}

CJT6MDecoder::CJT6MDecoder(const wxString& fileName) :
m_fileName(fileName),
m_audioSamples(NULL),
m_correlations(NULL),
m_samplesCount(0)
{
	m_audioSamples = new double[JT6M_MAX_AUDIO_DATA];

        int corrCount = (JT6M_SYNCBIN_LAST - JT6M_SYNCBIN_FIRST) * (3 * JT6M_SYMBOL_LENGTH) / JT6M_SKIP_FACTOR;
        m_correlations = new CCorrelation[corrCount];
}

CJT6MDecoder::~CJT6MDecoder()
{
	delete[] m_audioSamples;
	delete[] m_correlations;
}

void CJT6MDecoder::run()
{
	// Reset the state variables
	int corrCount = (JT6M_SYNCBIN_LAST - JT6M_SYNCBIN_FIRST) * (3 * JT6M_SYMBOL_LENGTH) / JT6M_SKIP_FACTOR;
	for (int i = 0; i < corrCount; i++)
		m_correlations[i].clear();
	m_samplesCount = 0;

	CSoundFile file(m_fileName, JT6M_SAMPLE_RATE);
	file.openRead();

	while (m_samplesCount < (JT6M_MAX_AUDIO_DATA - JT6M_SOUNDBUF_LENGTH)) {
		int len = JT6M_SOUNDBUF_LENGTH;

		file.read(m_audioSamples + m_samplesCount, len);

		if (len <= 0) break;

		m_samplesCount += len;
	}

	file.close();

	::fprintf(stdout, "Received %d samples of data\n", m_samplesCount);

	CFFT fft(JT6M_FFT_LENGTH);

	double bins[JT6M_FFT_LENGTH];

	for (int samplesCount = 0; samplesCount < m_samplesCount; samplesCount++) {
		if (samplesCount >= JT6M_FFT_LENGTH && (samplesCount % JT6M_SKIP_FACTOR) == 0) {
			fft.process(m_audioSamples + samplesCount - JT6M_FFT_LENGTH, bins);

			storeCorrelations(bins, samplesCount);
		}
	}

	int syncBin, timeOffset;
	findCorrelation(syncBin, timeOffset);

	double quality;
	correlate(syncBin, timeOffset, quality);

	decode(syncBin, timeOffset);
}

void CJT6MDecoder::storeCorrelations(double* data, int samplesCount)
{
	wxASSERT(m_correlations != NULL);
	wxASSERT(data != NULL);
	wxASSERT(samplesCount >= 0);

	int sampleNo = samplesCount / JT6M_SKIP_FACTOR;

	for (int t = 0; t < (3 * JT6M_SYMBOL_LENGTH) / JT6M_SKIP_FACTOR; t++) {
		int pos = (sampleNo + t) % (3 * JT6M_SYMBOL_LENGTH / JT6M_SKIP_FACTOR);

		if ((pos % (JT6M_SYMBOL_LENGTH / JT6M_SKIP_FACTOR)) == 0) {
			double mult;
			switch (pos / (JT6M_SYMBOL_LENGTH / JT6M_SKIP_FACTOR)) {
				case 0:
				case 1:
					mult = -0.5;
					break;
				case 2:
					mult = +1.0;
					break;
				default:
					mult = 0.0;
					fprintf(stderr, "Nonsense\n");
					break;
			}

			for (int bin = 0; bin < (JT6M_SYNCBIN_LAST - JT6M_SYNCBIN_FIRST); bin++) {
				int index = getCorrelationsIndex(bin, t);
				m_correlations[index].addProduct(mult, data[bin + JT6M_SYNCBIN_FIRST]);
			}
		}
	}
}

void CJT6MDecoder::findCorrelation(int& syncBin, int& offset) const
{
	wxASSERT(m_correlations != NULL);

	double quality = 0.0;
	int maxI = 0, maxJ = 0;

	for (int i = 0; i < (JT6M_SYNCBIN_LAST - JT6M_SYNCBIN_FIRST); i++) {
		for (int j = 0; j < (1 * JT6M_SYMBOL_LENGTH) / JT6M_SKIP_FACTOR; j++) {

			int n = getCorrelationsIndex(i, j);

			double value;
			bool ret = m_correlations[n].getRawValue(value);

			if (ret && value > quality) {
				quality = value;
				maxJ    = j;
				maxI    = i;
			}
		}
	}

	offset  = maxJ * JT6M_SKIP_FACTOR;;
	syncBin = maxI + JT6M_SYNCBIN_FIRST;

	::fprintf(stdout, "Best sync bin is %d with time offset of %d and quality of %f\n", syncBin, offset, quality);
}

void CJT6MDecoder::correlate(int syncBin, int& offset, double& quality) const
{
	wxASSERT(m_audioSamples != NULL);
	wxASSERT(m_samplesCount > 0);
	wxASSERT(syncBin >= JT6M_SYNCBIN_FIRST);

	double* syncSamples = new double[m_samplesCount];

	CSFFT sfft(JT6M_FFT_LENGTH, syncBin, syncBin + 1);

	double bins[JT6M_FFT_LENGTH];

	for (int i = 0; i < m_samplesCount; i++) {
		sfft.process(m_audioSamples[i], bins);
		syncSamples[i] = bins[syncBin];
	}

//	int startOffset = offset - JT6M_SKIP_FACTOR;
	int startOffset = offset - (3 * JT6M_SYMBOL_LENGTH);
	if (startOffset < 0) startOffset = 0;
//	int   endOffset = startOffset + 2 * JT6M_SKIP_FACTOR;
	int   endOffset = startOffset + 6 * JT6M_SYMBOL_LENGTH;

	double bestCorrelation = 0.0;
	for (int timeOffset = startOffset; timeOffset < endOffset; timeOffset++) {
		CCorrelation correlation;

		int index = 0;
		for (int pos = timeOffset; pos < m_samplesCount; pos += JT6M_SYMBOL_LENGTH) {

			double mult;
			switch (index) {
				case 0:
				case 1:
					mult = -0.5;
					break;
				case 2:
					mult = +1.0;
					break;
				default:
					mult = 0.0;
					fprintf(stderr, "Nonsense\n");
					break;
			}

			correlation.addProduct(mult, syncSamples[pos]);

			index = (index + 1) % 3;
		}

		double value;
		bool ret = correlation.getRawValue(value);

		if (ret && value > bestCorrelation) {
			bestCorrelation = value;
			offset          = timeOffset;
		}
	}

	delete[] syncSamples;

	::fprintf(stdout, "Raw offset %d with value %f\n", offset, bestCorrelation);

	offset %= (3 * JT6M_SYMBOL_LENGTH);
	offset -= 2 * JT6M_SYMBOL_LENGTH;

	if (offset < 0)
		offset += 3 * JT6M_SYMBOL_LENGTH;

	::fprintf(stdout, "Maximum correlation in bin %d at time offset of %d\n", syncBin, offset);
}

void CJT6MDecoder::decode(int syncBin, int timeOffset) const
{
	CJT6MFoldMessages* messages[JT6M_MAX_MESSAGE_LENGTH / 2];
	for (int i = 0; i < (JT6M_MAX_MESSAGE_LENGTH / 2); i++)
		messages[i] = new CJT6MFoldMessages(2 + i * 2);

	CFFT fft(JT6M_FFT_LENGTH);

	double bins[JT6M_FFT_LENGTH];

	CJT6MLookups lookups;

	int pos = 0;
	bool inBurst = false;
	int burstStart = 0;

	for (int index = timeOffset; index < (m_samplesCount - JT6M_FFT_LENGTH); index += JT6M_SYMBOL_LENGTH) {
		fft.process(m_audioSamples + index, bins);

// fprintf(stderr, "%d %f\n", pos % 3, bins[syncBin]);

		if ((pos % 3) == 0) {
			CAverage noise;
			noise.addValue(bins[syncBin - 2]);
			noise.addValue(bins[syncBin - 3]);
			noise.addValue(bins[syncBin - 4]);
			noise.addValue(bins[syncBin - 5]);

			if (!inBurst && bins[syncBin] > noise.getAverage()) {
				burstStart = index;
				inBurst    = true;
			} else if (inBurst && bins[syncBin] <= noise.getAverage()) {
				double timStart = double(burstStart - JT6M_FFT_LENGTH) / double(JT6M_SAMPLE_RATE);
				double  timDiff = double(index - burstStart) / double(JT6M_SAMPLE_RATE);
				if (timDiff >= 0.15)
					fprintf(stderr, "%d/%.1f %d/%.1f\n", burstStart, timStart, index - burstStart, timDiff);
				inBurst = false;
			}
		} else {
//			if (inBurst) {
				for (int i = 0; i < JT6M_MAX_MESSAGE_LENGTH / 2; i++)
					messages[i]->addBins(pos, bins + syncBin + 1);
//			}
		}

		pos++;
	}

	if (inBurst) {
		double timStart = double(burstStart - JT6M_FFT_LENGTH) / double(JT6M_SAMPLE_RATE);
		double  timDiff = double(m_samplesCount - burstStart) / double(JT6M_SAMPLE_RATE);
		if (timDiff >= 0.15)
			fprintf(stderr, "%d/%.1f %d/%.1f\n", burstStart, timStart, m_samplesCount - burstStart, timDiff);
	}

	int       bestI = 0;
	double bestQual = messages[0]->getQuality();

	fprintf(stderr, "2: %f\n", bestQual);

	for (int i = 1; i < JT6M_MAX_MESSAGE_LENGTH / 2; i++) {
		double qual = messages[i]->getQuality();

		fprintf(stderr, "%d: %f\n", 2 + i * 2, qual);

		if (qual > bestQual) {
			bestQual = qual;
			bestI    = i;
		}
	}

	fprintf(stderr, "%d: %f %s\n", messages[bestI]->getLength(),
				       messages[bestI]->getQuality(),
				       messages[bestI]->getMessage().c_str());

	for (int i = 0; i < JT6M_MAX_MESSAGE_LENGTH / 2; i++)
		delete messages[i];
}

int CJT6MDecoder::getCorrelationsIndex(int bin, int offset) const
{
	int nBins   = JT6M_SYNCBIN_LAST - JT6M_SYNCBIN_FIRST;
	int nOffset = (3 * JT6M_SYMBOL_LENGTH) / JT6M_SKIP_FACTOR;

	wxASSERT(bin >= 0 && bin < nBins);
	wxASSERT(offset >= 0 && offset < nOffset);

	return bin * nOffset + offset;
}
