/*
 *   Copyright (C) 2002-2004 by Jonathan Naylor G4KLX
 *   Copyright (C) 1999-2001 by Thomas Sailor HB9JNX
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "SerialPTT.h"
#include "Exception.h"

#include <wx/debug.h>
#include <wx/log.h>

#include <sys/types.h>

#ifdef __WINDOWS__

#include <winioctl.h>

CSerialPTT::CSerialPTT(const wxString& device) :
CPTTPort(),
m_device(device),
m_handle(INVALID_HANDLE_VALUE)
{
}

CSerialPTT::CSerialPTT() :
CPTTPort(),
m_device(wxEmptyString),
m_handle(INVALID_HANDLE_VALUE)
{
}

CSerialPTT::~CSerialPTT()
{
}

void CSerialPTT::open(const wxString& device)
{
	m_device = device;

	open();
}

void CSerialPTT::open()
{
	wxASSERT(m_handle == INVALID_HANDLE_VALUE);

	m_handle = ::CreateFile(m_device.mb_str(), GENERIC_READ | GENERIC_WRITE, 0, NULL, OPEN_EXISTING, 0, NULL);
	if (m_handle == INVALID_HANDLE_VALUE) {
		wxString text;
		text.Printf(wxT("Cannot open the serial port - 0x%lX"), ::GetLastError());
		throw CException(text);
	}

	DCB dcb;
	if (::GetCommState(m_handle, &dcb) == 0) {
		close();
		throw CException(wxT("Device is not a serial port"));
	}

	unkeyTX();
}

void CSerialPTT::keyTX()
{
	wxASSERT(m_handle != INVALID_HANDLE_VALUE);

	if (::EscapeCommFunction(m_handle, SETRTS) == 0) {
		wxString text;
		text.Printf(wxT("Cannot key the transmitter - 0x%lX"), ::GetLastError());
		throw CException(text);
	}

	if (::EscapeCommFunction(m_handle, SETDTR) == 0) {
		wxString text;
		text.Printf(wxT("Cannot key the transmitter - 0x%lX"), ::GetLastError());
		throw CException(text);
	}
}

void CSerialPTT::unkeyTX()
{
	wxASSERT(m_handle != INVALID_HANDLE_VALUE);

	if (::EscapeCommFunction(m_handle, CLRRTS) == 0) {
		wxString text;
		text.Printf(wxT("Cannot key the transmitter - 0x%lX"), ::GetLastError());
		throw CException(text);
	}

	if (::EscapeCommFunction(m_handle, CLRDTR) == 0) {
		wxString text;
		text.Printf(wxT("Cannot key the transmitter - 0x%lX"), ::GetLastError());
		throw CException(text);
	}
}

void CSerialPTT::close()
{
	wxASSERT(m_handle != INVALID_HANDLE_VALUE);

	unkeyTX();

	::CloseHandle(m_handle);
	m_handle = INVALID_HANDLE_VALUE;
}

wxArrayString CSerialPTT::getDevices()
{
	wxArrayString devices;

	devices.Alloc(4);

	devices.Add(wxT("\\\\.\\COM1"));
	devices.Add(wxT("\\\\.\\COM2"));
	devices.Add(wxT("\\\\.\\COM3"));
	devices.Add(wxT("\\\\.\\COM4"));

	return devices;
}

#else

#include <sys/ioctl.h>
#include <sys/stat.h>

#include <cerrno>
#include <fcntl.h>
#include <unistd.h>

CSerialPTT::CSerialPTT(const wxString& device) :
CPTTPort(),
m_device(device),
m_fd(-1)
{
}

CSerialPTT::CSerialPTT() :
CPTTPort(),
m_device(wxEmptyString),
m_fd(-1)
{
}

CSerialPTT::~CSerialPTT()
{
}

void CSerialPTT::open(const wxString& device)
{
	m_device = device;

	open();
}

void CSerialPTT::open()
{
	wxASSERT(m_fd == -1);

	m_fd = ::open(m_device.mb_str(), O_RDWR, 0);
	if (m_fd < 0) {
		wxString text;
		text.Printf(wxT("Cannot open the serial port - %d, %s"), errno, strerror(errno));
		throw CException(text);
	}

	if (::isatty(m_fd) == 0) {
		close();
		throw CException(wxT("Device is not a serial port"));
	}

	unkeyTX();
}

void CSerialPTT::keyTX()
{
	wxASSERT(m_fd != -1);

	unsigned int y;
	if (::ioctl(m_fd, TIOCMGET, &y) < 0) {
		wxString text;
		text.Printf(wxT("Cannot key the transmitter - %d, %s"), errno, strerror(errno));
		throw CException(text);
	}

	y |= TIOCM_RTS;
	y |= TIOCM_DTR;

	if (::ioctl(m_fd, TIOCMSET, &y) < 0) {
		wxString text;
		text.Printf(wxT("Cannot key the transmitter - %d, %s"), errno, strerror(errno));
		throw CException(text);
	}
}

void CSerialPTT::unkeyTX()
{
	wxASSERT(m_fd != -1);

	unsigned int y;
	if (::ioctl(m_fd, TIOCMGET, &y) < 0) {
		wxString text;
		text.Printf(wxT("Cannot unkey the transmitter - %d, %s"), errno, strerror(errno));
		throw CException(text);
	}

	y &= ~TIOCM_RTS;
	y &= ~TIOCM_DTR;

	if (::ioctl(m_fd, TIOCMSET, &y) < 0) {
		wxString text;
		text.Printf(wxT("Cannot unkey the transmitter - %d, %s"), errno, strerror(errno));
		throw CException(text);
	}
}

void CSerialPTT::close()
{
	wxASSERT(m_fd != -1);

	unkeyTX();

	::close(m_fd);
	m_fd = -1;
}

wxArrayString CSerialPTT::getDevices()
{
	wxArrayString devices;

	devices.Alloc(4);

	devices.Add(wxT("/dev/ttyS0"));
	devices.Add(wxT("/dev/ttyS1"));
	devices.Add(wxT("/dev/ttyS2"));
	devices.Add(wxT("/dev/ttyS3"));

	return devices;
}

#endif
