/* Copyright 2001 Matt Flax <flatmax@ieee.org>
   This file is part of MFFM Time Scale Modification for Audio.

   MFFM Time Scale Modification for Audio is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   MFFM Time Scale Modification for Audio is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You have received a copy of the GNU General Public License
   along with MFFM Time Scale Modification for Audio
 */

/*
  This example file is for use when you are streaming files to and from disk
  and changing their speed at the same time. It is a simpler API if that is
  the case.
  If you want to stream to or from something other then a file, you are more
  interested in the embedded example file.
 */
#include <stdlib.h>

#include <mffm/libSndFileWrapper.H>

#include "WSOLA.H"

void exitRoutine(void){std::cout<<std::endl;}

int main(int argc, char *argv[]){

  atexit(exitRoutine);

  std::cout<<"WSOLA version "<<VERSION_NUM<<"\nAuthor : mffm <flatmax@ieee.org>"<<std::endl;

  if (argc<4){
    std::cerr<<"Useage :\n"<<argv[0]<<" inputFile outputFile factor"<<std::endl;
    exit(-1);
  }

  double tau=atof(argv[3]);
  std::cout<<"Factor="<<tau<<std::endl;

  int cnt, fs, channels;

  //open the input file
  //open the input file
  //short int *input=readWav(argv[1], &cnt, &fs, &channels);
  SF_INFO sfInfoIn;
  sfInfoIn.format=0;
  SNDFILE* inputFile=sf_open(argv[1], SFM_READ,&sfInfoIn);
  if (inputFile==NULL){
    perror("couldn't open input file\n");
    sf_perror(inputFile);
    exit(-1);
  }

  channels=sfInfoIn.channels;
  fs=sfInfoIn.samplerate;
  if (channels<1){
    std::cerr<<argv[0]<<":: input channels must be more then 1"<<std::endl;
    closeSndFile(inputFile); //Make sure you close the input file
    exit(-3);
  }

  //open the output file
  SF_INFO sfInfoOut;
  sfInfoOut.format=0;
  SNDFILE* outputFile=openWavWrite(&sfInfoOut, argv[2], fs, channels, 16);
  if (outputFile==NULL){
    closeSndFile(inputFile); //Make sure you close the input file
    cout<<"Couldn't open the output file "<<argv[2]<<" aborting"<<endl;
    return -2;
  }

  std::cout<<"Processing for "<<channels<<" channels at "<<fs<<" Hz"<<std::endl;

  int windowSize=channels*HANNING_LENGTH(fs);

  cout<<"windowSize="<<windowSize<<endl;

  // Make sure that the half window size is an integer number of channels
  while (remainder((double)windowSize/2.0/(double)channels, floor((double)windowSize/2.0/(double)channels)) != 0.0){
  cout<<"windowSize/2/channels "<<(double)windowSize/2.0/(double)channels<<endl;
  cout<<"remainder "<<remainder((double)windowSize/2.0/(double)channels, floor((double)windowSize/2.0/(double)channels))<<endl;
    windowSize++;
  }
  cout<<"windowSize="<<windowSize<<endl;

  WSOLA wsola(windowSize, fs, channels);//Version 4 instantiation
  cout<<"initting"<<endl;
  //Init with the tau value for the first window
  wsola.initProcess(inputFile, tau);
  cout<<"about to process"<<endl;

  int ret;
  while ((ret=wsola.processFrame(inputFile, outputFile, tau))>=0){
  }

  if (ret==WSOLA::INPUT_READ2DF_ERR)
    cout<<"Destination frame read error, most probably using a factor >=1.0 and at the end of the input file"<<endl;
  else if (ret==WSOLA::INPUT_READ2SF_ERR)
    cout<<"Destination frame read error, most probably using a factor <1.0, and at the end of the input file"<<endl;
  else if (ret==WSOLA::WRITE_ERR)
    cout<<"Is your disk full, we encountered a write error"<<endl;
  else if (ret==WSOLA::DEFAULT_ERR)
    cout<<"Un-known default error encountered"<<endl;
  else //Handle other errors here
    cout<<"Unknown error - unreliable failure"<<endl;

  closeSndFile(inputFile); //Make sure you close the files
  closeSndFile(outputFile);

  return 1;
}
