/* @(#)  display VIPS IMAGE descriptor for debugging
 * @(#)  a valid IMAGE descriptor is needed
 * @(#) HANDLESIMAGE
 * @(#)  Static strings must be modified if header files are changed
 * @(#) void im_printdesc(image)
 * @(#) IMAGE *image;
 *
 * Copyright: Nicos Dessipris
 * Written on: 15/01/1990
 * Modified on : 3/6/92 Kirk Martinez
 * 15/4/93 JC
 *	- partial regions dumped too
 *	- resolution bug fixed
 * 11/5/93 JC
 *	- formatting changed for ctags
 * 1/7/93 JC
 *	- adapted for partial v2
 * 15/11/94 JC
 *	- new Coding types added
 *	- new Type values added
 *	- new Compression type added
 * 2/3/98 JC
 *	- IM_ANY added
 * 5/11/98 JC
 *	- prints byte-order too
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/
#include <vips/intl.h>

#include <stdio.h>
#include <string.h>

#include <vips/vips.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

static const char *im_Type[] = {
	"IM_TYPE_MULTIBAND", "IM_TYPE_B_W", 
	"LUMINACE", "XRAY", "IR", "YUV", 
	"RED_ONLY", "GREEN_ONLY", "BLUE_ONLY", 
	"POWER_SPECTRUM", "IM_TYPE_HISTOGRAM", "LUT", "IM_TYPE_XYZ",
	"IM_TYPE_LAB", "CMC", "IM_TYPE_CMYK", "IM_TYPE_LABQ", "IM_TYPE_RGB", 
	"IM_TYPE_UCS", "IM_TYPE_LCH", "IM_TYPE_LABS",
	"<unknown>", "IM_TYPE_sRGB", "IM_TYPE_YXY", "IM_TYPE_FOURIER"
};

static const char *im_BandFmt[] = {
	"IM_BANDFMT_UCHAR", "IM_BANDFMT_CHAR", "IM_BANDFMT_USHORT", 
	"IM_BANDFMT_SHORT", "IM_BANDFMT_UINT", "IM_BANDFMT_INT", 
	"IM_BANDFMT_FLOAT", "IM_BANDFMT_COMPLEX", "IM_BANDFMT_DOUBLE", 
	"IM_BANDFMT_DPCOMPLEX"
};

static const char *im_Coding[] = {
	"IM_CODING_NONE", "COLQUANT8", "IM_CODING_LABQ", 
	"IM_CODING_LABQ_COMPRESSED"
};

static const char *im_Compression[] = {
	"NO_COMPRESSION", "TCSF_COMPRESSION", "JPEG_COMPRESSION"
};

static const char *im_dtype[] = {
	"IM_NONE", "IM_SETBUF", "IM_SETBUF_FOREIGN", "IM_OPENIN", "IM_MMAPIN", 
	"IM_MMAPINRW", "IM_OPENOUT", "IM_PARTIAL"
};

static const char *im_dhint[] = {
	"IM_SMALLTILE", "IM_FATSTRIP", "IM_THINSTRIP", "IM_ANY"
};

/* Stuff to decode an enum.
 */
typedef struct _EnumTable {
	const char *error;	/* eg. "<bad Coding>" */
	const char **names;	/* eg. {"IM_CODING_NONE",..} */
	int nnames;
} EnumTable;

static EnumTable enumType = {
	N_( "<bad Type>" ),
	im_Type,
	IM_NUMBER( im_Type )
};

static EnumTable enumBandFmt = {
	N_( "<bad BandFmt>" ),
	im_BandFmt,
	IM_NUMBER( im_BandFmt )
};

static EnumTable enumCoding = {
	N_( "<bad Coding>" ),
	im_Coding,
	IM_NUMBER( im_Coding )
};

static EnumTable enumCompression = {
	N_( "<bad Compression>" ),
	im_Compression,
	IM_NUMBER( im_Compression )
};

static EnumTable enumdtype = {
	N_( "<bad dtype>" ),
	im_dtype,
	IM_NUMBER( im_dtype )
};

static EnumTable enumdhint = {
	N_( "<bad dhint>" ),
	im_dhint,
	IM_NUMBER( im_dhint )
};

static const char *
enum2char( EnumTable *etable, int n )
{
	if( n < 0 || n > etable->nnames ) 
		return( _( etable->error ) );
	else
		return( etable->names[n] );
}

static int
char2enum( EnumTable *etable, const char *name )
{
	int i;

	for( i = 0; i < etable->nnames; i++ )
		if( strcasecmp( etable->names[i], name ) == 0 )
			return( i );

	im_errormsg( "char2enum", _( etable->error ) );

	return( -1 );
}


/* Prettyprint various header fields.
 */
const char *im_Type2char( int n ) 
	{ return( enum2char( &enumType, n ) ); }
const char *im_BandFmt2char( int n ) 
	{ return( enum2char( &enumBandFmt, n ) ); }
const char *im_Coding2char( int n ) 
	{ return( enum2char( &enumCoding, n ) ); }
const char *im_Compression2char( int n ) 
	{ return( enum2char( &enumCompression, n ) ); }
const char *im_dtype2char( im_desc_type n ) 
	{ return( enum2char( &enumdtype, n ) ); }
const char *im_dhint2char( im_demand_type n ) 
	{ return( enum2char( &enumdhint, n ) ); }

int im_char2Type( const char *str ) 
	{ return( char2enum( &enumType, str ) ); }
int im_char2BandFmt( const char *str ) 
	{ return( char2enum( &enumBandFmt, str ) ); }
int im_char2Coding( const char *str ) 
	{ return( char2enum( &enumCoding, str ) ); }
int im_char2Compression( const char *str ) 
	{ return( char2enum( &enumCompression, str ) ); }
im_desc_type im_char2dtype( const char *str ) 
	{ return( char2enum( &enumdtype, str ) ); }
im_demand_type im_char2dhint( const char *str ) 
	{ return( char2enum( &enumdhint, str ) ); }

static void *
print_region( REGION *reg, void *a, void *b )
{	
	printf( "Region defined for area at %dx%d size %dx%d\n",
		reg->valid.left, reg->valid.top,
		reg->valid.width, reg->valid.height );
	if( reg->seq )
		printf( "sequence in progress on region\n" );
	if( reg->buf )
		printf( "local memory allocated\n" );

	return( NULL );
}

void 
im_printdesc( IMAGE *image )
{	
	if( !image ) {
		printf( "NULL descriptor\n" );
		return;
	}

	if( im_isMSBfirst( image ) )
		printf( "SPARC (MSB first) " );
	else
		printf( "Intel (LSB first) " );
	printf( "byte order image, on a " );
	if( im_amiMSBfirst() )
		printf( "SPARC (MSB first) " );
	else
		printf( "Intel (LSB first) " );
	printf( "byte order machine\n" );
 
	printf( "Xsize: %d\nYsize: %d\nBands: %d\nBbits: %d\n",
		image->Xsize, image->Ysize, 
		image->Bands, image->Bbits );

	printf( "BandFmt: %d --> %s\nCoding: %d --> %s\nType: %d --> %s\n",
		image->BandFmt, im_BandFmt2char( image->BandFmt ),
		image->Coding, im_Coding2char( image->Coding ),
		image->Type, im_Type2char( image->Type ) );

	printf( "Xres: %g\nYres: %g\n", 
		image->Xres, image->Yres );

	printf( "Xoffset: %d\nYoffset: %d\n", 
		image->Xoffset, image->Yoffset ); 

	if( image->Hist ) 
		printf( "Hist:\n%s\n", image->Hist );

	if( image->filename ) 
		printf( "filename: %s\n", image->filename );
	if( image->generate )
		printf( "user generate function attached\n" );
	if( image->closefns )
		printf( "user close callbacks attached\n" );
	if( image->evalfns )
		printf( "user eval callbacks attached\n" );
	if( image->evalendfns )
		printf( "user evalend callbacks attached\n" );
	if( image->regions ) {
		printf( "%d regions present\n", 
			g_slist_length( image->regions ) );
		im_slist_map2( image->regions, 
			(VSListMap2Fn) print_region, NULL, NULL );
	}
	if( image->kill )
		printf( "kill flag set\n" );
	if( image->closing )
		printf( "closing flag set\n" );
	if( image->close_pending )
		printf( "close_pending flag set\n" );
	printf( "dhint: %s\n", im_dhint2char( image->dhint ) );
	printf( "dtype: %s\n", im_dtype2char( image->dtype ) );
}
